/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is the latest version of <b>AWS WAF</b>, named AWS WAFV2, released in November, 2019. For information, including
 * how to migrate your AWS WAF resources from the prior release, see the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">AWS WAF Developer Guide</a>.
 * </p>
 * </note>
 * <p>
 * Defines an association between Amazon Kinesis Data Firehose destinations and a web ACL resource, for logging from AWS
 * WAF. As part of the association, you can specify parts of the standard logging fields to keep out of the logs.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LoggingConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<LoggingConfiguration.Builder, LoggingConfiguration> {
    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LoggingConfiguration::resourceArn)).setter(setter(Builder::resourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceArn").build()).build();

    private static final SdkField<List<String>> LOG_DESTINATION_CONFIGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(LoggingConfiguration::logDestinationConfigs))
            .setter(setter(Builder::logDestinationConfigs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogDestinationConfigs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<FieldToMatch>> REDACTED_FIELDS_FIELD = SdkField
            .<List<FieldToMatch>> builder(MarshallingType.LIST)
            .getter(getter(LoggingConfiguration::redactedFields))
            .setter(setter(Builder::redactedFields))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RedactedFields").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<FieldToMatch> builder(MarshallingType.SDK_POJO)
                                            .constructor(FieldToMatch::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_ARN_FIELD,
            LOG_DESTINATION_CONFIGS_FIELD, REDACTED_FIELDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String resourceArn;

    private final List<String> logDestinationConfigs;

    private final List<FieldToMatch> redactedFields;

    private LoggingConfiguration(BuilderImpl builder) {
        this.resourceArn = builder.resourceArn;
        this.logDestinationConfigs = builder.logDestinationConfigs;
        this.redactedFields = builder.redactedFields;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the web ACL that you want to associate with <code>LogDestinationConfigs</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the web ACL that you want to associate with
     *         <code>LogDestinationConfigs</code>.
     */
    public String resourceArn() {
        return resourceArn;
    }

    /**
     * Returns true if the LogDestinationConfigs property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasLogDestinationConfigs() {
        return logDestinationConfigs != null && !(logDestinationConfigs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Kinesis Data Firehose Amazon Resource Name (ARNs) that you want to associate with the web ACL.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLogDestinationConfigs()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The Amazon Kinesis Data Firehose Amazon Resource Name (ARNs) that you want to associate with the web ACL.
     */
    public List<String> logDestinationConfigs() {
        return logDestinationConfigs;
    }

    /**
     * Returns true if the RedactedFields property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRedactedFields() {
        return redactedFields != null && !(redactedFields instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The parts of the request that you want to keep out of the logs. For example, if you redact the cookie field, the
     * cookie field in the firehose will be <code>xxx</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRedactedFields()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The parts of the request that you want to keep out of the logs. For example, if you redact the cookie
     *         field, the cookie field in the firehose will be <code>xxx</code>.
     */
    public List<FieldToMatch> redactedFields() {
        return redactedFields;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(logDestinationConfigs());
        hashCode = 31 * hashCode + Objects.hashCode(redactedFields());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LoggingConfiguration)) {
            return false;
        }
        LoggingConfiguration other = (LoggingConfiguration) obj;
        return Objects.equals(resourceArn(), other.resourceArn())
                && Objects.equals(logDestinationConfigs(), other.logDestinationConfigs())
                && Objects.equals(redactedFields(), other.redactedFields());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("LoggingConfiguration").add("ResourceArn", resourceArn())
                .add("LogDestinationConfigs", logDestinationConfigs()).add("RedactedFields", redactedFields()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResourceArn":
            return Optional.ofNullable(clazz.cast(resourceArn()));
        case "LogDestinationConfigs":
            return Optional.ofNullable(clazz.cast(logDestinationConfigs()));
        case "RedactedFields":
            return Optional.ofNullable(clazz.cast(redactedFields()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LoggingConfiguration, T> g) {
        return obj -> g.apply((LoggingConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LoggingConfiguration> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the web ACL that you want to associate with
         * <code>LogDestinationConfigs</code>.
         * </p>
         * 
         * @param resourceArn
         *        The Amazon Resource Name (ARN) of the web ACL that you want to associate with
         *        <code>LogDestinationConfigs</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArn(String resourceArn);

        /**
         * <p>
         * The Amazon Kinesis Data Firehose Amazon Resource Name (ARNs) that you want to associate with the web ACL.
         * </p>
         * 
         * @param logDestinationConfigs
         *        The Amazon Kinesis Data Firehose Amazon Resource Name (ARNs) that you want to associate with the web
         *        ACL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logDestinationConfigs(Collection<String> logDestinationConfigs);

        /**
         * <p>
         * The Amazon Kinesis Data Firehose Amazon Resource Name (ARNs) that you want to associate with the web ACL.
         * </p>
         * 
         * @param logDestinationConfigs
         *        The Amazon Kinesis Data Firehose Amazon Resource Name (ARNs) that you want to associate with the web
         *        ACL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logDestinationConfigs(String... logDestinationConfigs);

        /**
         * <p>
         * The parts of the request that you want to keep out of the logs. For example, if you redact the cookie field,
         * the cookie field in the firehose will be <code>xxx</code>.
         * </p>
         * 
         * @param redactedFields
         *        The parts of the request that you want to keep out of the logs. For example, if you redact the cookie
         *        field, the cookie field in the firehose will be <code>xxx</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder redactedFields(Collection<FieldToMatch> redactedFields);

        /**
         * <p>
         * The parts of the request that you want to keep out of the logs. For example, if you redact the cookie field,
         * the cookie field in the firehose will be <code>xxx</code>.
         * </p>
         * 
         * @param redactedFields
         *        The parts of the request that you want to keep out of the logs. For example, if you redact the cookie
         *        field, the cookie field in the firehose will be <code>xxx</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder redactedFields(FieldToMatch... redactedFields);

        /**
         * <p>
         * The parts of the request that you want to keep out of the logs. For example, if you redact the cookie field,
         * the cookie field in the firehose will be <code>xxx</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<FieldToMatch>.Builder} avoiding the need to
         * create one manually via {@link List<FieldToMatch>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<FieldToMatch>.Builder#build()} is called immediately and its
         * result is passed to {@link #redactedFields(List<FieldToMatch>)}.
         * 
         * @param redactedFields
         *        a consumer that will call methods on {@link List<FieldToMatch>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #redactedFields(List<FieldToMatch>)
         */
        Builder redactedFields(Consumer<FieldToMatch.Builder>... redactedFields);
    }

    static final class BuilderImpl implements Builder {
        private String resourceArn;

        private List<String> logDestinationConfigs = DefaultSdkAutoConstructList.getInstance();

        private List<FieldToMatch> redactedFields = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LoggingConfiguration model) {
            resourceArn(model.resourceArn);
            logDestinationConfigs(model.logDestinationConfigs);
            redactedFields(model.redactedFields);
        }

        public final String getResourceArn() {
            return resourceArn;
        }

        @Override
        public final Builder resourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
            return this;
        }

        public final void setResourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
        }

        public final Collection<String> getLogDestinationConfigs() {
            return logDestinationConfigs;
        }

        @Override
        public final Builder logDestinationConfigs(Collection<String> logDestinationConfigs) {
            this.logDestinationConfigs = LogDestinationConfigsCopier.copy(logDestinationConfigs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder logDestinationConfigs(String... logDestinationConfigs) {
            logDestinationConfigs(Arrays.asList(logDestinationConfigs));
            return this;
        }

        public final void setLogDestinationConfigs(Collection<String> logDestinationConfigs) {
            this.logDestinationConfigs = LogDestinationConfigsCopier.copy(logDestinationConfigs);
        }

        public final Collection<FieldToMatch.Builder> getRedactedFields() {
            return redactedFields != null ? redactedFields.stream().map(FieldToMatch::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder redactedFields(Collection<FieldToMatch> redactedFields) {
            this.redactedFields = RedactedFieldsCopier.copy(redactedFields);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder redactedFields(FieldToMatch... redactedFields) {
            redactedFields(Arrays.asList(redactedFields));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder redactedFields(Consumer<FieldToMatch.Builder>... redactedFields) {
            redactedFields(Stream.of(redactedFields).map(c -> FieldToMatch.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRedactedFields(Collection<FieldToMatch.BuilderImpl> redactedFields) {
            this.redactedFields = RedactedFieldsCopier.copyFromBuilder(redactedFields);
        }

        @Override
        public LoggingConfiguration build() {
            return new LoggingConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
