/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.voiceid.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The registration configuration to be used during the batch fraudster registration job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RegistrationConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<RegistrationConfig.Builder, RegistrationConfig> {
    private static final SdkField<String> DUPLICATE_REGISTRATION_ACTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DuplicateRegistrationAction")
            .getter(getter(RegistrationConfig::duplicateRegistrationActionAsString))
            .setter(setter(Builder::duplicateRegistrationAction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DuplicateRegistrationAction")
                    .build()).build();

    private static final SdkField<Integer> FRAUDSTER_SIMILARITY_THRESHOLD_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("FraudsterSimilarityThreshold")
            .getter(getter(RegistrationConfig::fraudsterSimilarityThreshold))
            .setter(setter(Builder::fraudsterSimilarityThreshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FraudsterSimilarityThreshold")
                    .build()).build();

    private static final SdkField<List<String>> WATCHLIST_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("WatchlistIds")
            .getter(getter(RegistrationConfig::watchlistIds))
            .setter(setter(Builder::watchlistIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WatchlistIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            DUPLICATE_REGISTRATION_ACTION_FIELD, FRAUDSTER_SIMILARITY_THRESHOLD_FIELD, WATCHLIST_IDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String duplicateRegistrationAction;

    private final Integer fraudsterSimilarityThreshold;

    private final List<String> watchlistIds;

    private RegistrationConfig(BuilderImpl builder) {
        this.duplicateRegistrationAction = builder.duplicateRegistrationAction;
        this.fraudsterSimilarityThreshold = builder.fraudsterSimilarityThreshold;
        this.watchlistIds = builder.watchlistIds;
    }

    /**
     * <p>
     * The action to take when a fraudster is identified as a duplicate. The default action is <code>SKIP</code>, which
     * skips registering the duplicate fraudster. Setting the value to <code>REGISTER_AS_NEW</code> always registers a
     * new fraudster into the specified domain.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #duplicateRegistrationAction} will return {@link DuplicateRegistrationAction#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #duplicateRegistrationActionAsString}.
     * </p>
     * 
     * @return The action to take when a fraudster is identified as a duplicate. The default action is <code>SKIP</code>
     *         , which skips registering the duplicate fraudster. Setting the value to <code>REGISTER_AS_NEW</code>
     *         always registers a new fraudster into the specified domain.
     * @see DuplicateRegistrationAction
     */
    public final DuplicateRegistrationAction duplicateRegistrationAction() {
        return DuplicateRegistrationAction.fromValue(duplicateRegistrationAction);
    }

    /**
     * <p>
     * The action to take when a fraudster is identified as a duplicate. The default action is <code>SKIP</code>, which
     * skips registering the duplicate fraudster. Setting the value to <code>REGISTER_AS_NEW</code> always registers a
     * new fraudster into the specified domain.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #duplicateRegistrationAction} will return {@link DuplicateRegistrationAction#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #duplicateRegistrationActionAsString}.
     * </p>
     * 
     * @return The action to take when a fraudster is identified as a duplicate. The default action is <code>SKIP</code>
     *         , which skips registering the duplicate fraudster. Setting the value to <code>REGISTER_AS_NEW</code>
     *         always registers a new fraudster into the specified domain.
     * @see DuplicateRegistrationAction
     */
    public final String duplicateRegistrationActionAsString() {
        return duplicateRegistrationAction;
    }

    /**
     * <p>
     * The minimum similarity score between the new and old fraudsters in order to consider the new fraudster a
     * duplicate.
     * </p>
     * 
     * @return The minimum similarity score between the new and old fraudsters in order to consider the new fraudster a
     *         duplicate.
     */
    public final Integer fraudsterSimilarityThreshold() {
        return fraudsterSimilarityThreshold;
    }

    /**
     * For responses, this returns true if the service returned a value for the WatchlistIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasWatchlistIds() {
        return watchlistIds != null && !(watchlistIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The identifiers of watchlists that a fraudster is registered to. If a watchlist isn't provided, the fraudsters
     * are registered to the default watchlist.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasWatchlistIds} method.
     * </p>
     * 
     * @return The identifiers of watchlists that a fraudster is registered to. If a watchlist isn't provided, the
     *         fraudsters are registered to the default watchlist.
     */
    public final List<String> watchlistIds() {
        return watchlistIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(duplicateRegistrationActionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(fraudsterSimilarityThreshold());
        hashCode = 31 * hashCode + Objects.hashCode(hasWatchlistIds() ? watchlistIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RegistrationConfig)) {
            return false;
        }
        RegistrationConfig other = (RegistrationConfig) obj;
        return Objects.equals(duplicateRegistrationActionAsString(), other.duplicateRegistrationActionAsString())
                && Objects.equals(fraudsterSimilarityThreshold(), other.fraudsterSimilarityThreshold())
                && hasWatchlistIds() == other.hasWatchlistIds() && Objects.equals(watchlistIds(), other.watchlistIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RegistrationConfig").add("DuplicateRegistrationAction", duplicateRegistrationActionAsString())
                .add("FraudsterSimilarityThreshold", fraudsterSimilarityThreshold())
                .add("WatchlistIds", hasWatchlistIds() ? watchlistIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DuplicateRegistrationAction":
            return Optional.ofNullable(clazz.cast(duplicateRegistrationActionAsString()));
        case "FraudsterSimilarityThreshold":
            return Optional.ofNullable(clazz.cast(fraudsterSimilarityThreshold()));
        case "WatchlistIds":
            return Optional.ofNullable(clazz.cast(watchlistIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("DuplicateRegistrationAction", DUPLICATE_REGISTRATION_ACTION_FIELD);
        map.put("FraudsterSimilarityThreshold", FRAUDSTER_SIMILARITY_THRESHOLD_FIELD);
        map.put("WatchlistIds", WATCHLIST_IDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RegistrationConfig, T> g) {
        return obj -> g.apply((RegistrationConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RegistrationConfig> {
        /**
         * <p>
         * The action to take when a fraudster is identified as a duplicate. The default action is <code>SKIP</code>,
         * which skips registering the duplicate fraudster. Setting the value to <code>REGISTER_AS_NEW</code> always
         * registers a new fraudster into the specified domain.
         * </p>
         * 
         * @param duplicateRegistrationAction
         *        The action to take when a fraudster is identified as a duplicate. The default action is
         *        <code>SKIP</code>, which skips registering the duplicate fraudster. Setting the value to
         *        <code>REGISTER_AS_NEW</code> always registers a new fraudster into the specified domain.
         * @see DuplicateRegistrationAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DuplicateRegistrationAction
         */
        Builder duplicateRegistrationAction(String duplicateRegistrationAction);

        /**
         * <p>
         * The action to take when a fraudster is identified as a duplicate. The default action is <code>SKIP</code>,
         * which skips registering the duplicate fraudster. Setting the value to <code>REGISTER_AS_NEW</code> always
         * registers a new fraudster into the specified domain.
         * </p>
         * 
         * @param duplicateRegistrationAction
         *        The action to take when a fraudster is identified as a duplicate. The default action is
         *        <code>SKIP</code>, which skips registering the duplicate fraudster. Setting the value to
         *        <code>REGISTER_AS_NEW</code> always registers a new fraudster into the specified domain.
         * @see DuplicateRegistrationAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DuplicateRegistrationAction
         */
        Builder duplicateRegistrationAction(DuplicateRegistrationAction duplicateRegistrationAction);

        /**
         * <p>
         * The minimum similarity score between the new and old fraudsters in order to consider the new fraudster a
         * duplicate.
         * </p>
         * 
         * @param fraudsterSimilarityThreshold
         *        The minimum similarity score between the new and old fraudsters in order to consider the new fraudster
         *        a duplicate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fraudsterSimilarityThreshold(Integer fraudsterSimilarityThreshold);

        /**
         * <p>
         * The identifiers of watchlists that a fraudster is registered to. If a watchlist isn't provided, the
         * fraudsters are registered to the default watchlist.
         * </p>
         * 
         * @param watchlistIds
         *        The identifiers of watchlists that a fraudster is registered to. If a watchlist isn't provided, the
         *        fraudsters are registered to the default watchlist.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder watchlistIds(Collection<String> watchlistIds);

        /**
         * <p>
         * The identifiers of watchlists that a fraudster is registered to. If a watchlist isn't provided, the
         * fraudsters are registered to the default watchlist.
         * </p>
         * 
         * @param watchlistIds
         *        The identifiers of watchlists that a fraudster is registered to. If a watchlist isn't provided, the
         *        fraudsters are registered to the default watchlist.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder watchlistIds(String... watchlistIds);
    }

    static final class BuilderImpl implements Builder {
        private String duplicateRegistrationAction;

        private Integer fraudsterSimilarityThreshold;

        private List<String> watchlistIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RegistrationConfig model) {
            duplicateRegistrationAction(model.duplicateRegistrationAction);
            fraudsterSimilarityThreshold(model.fraudsterSimilarityThreshold);
            watchlistIds(model.watchlistIds);
        }

        public final String getDuplicateRegistrationAction() {
            return duplicateRegistrationAction;
        }

        public final void setDuplicateRegistrationAction(String duplicateRegistrationAction) {
            this.duplicateRegistrationAction = duplicateRegistrationAction;
        }

        @Override
        public final Builder duplicateRegistrationAction(String duplicateRegistrationAction) {
            this.duplicateRegistrationAction = duplicateRegistrationAction;
            return this;
        }

        @Override
        public final Builder duplicateRegistrationAction(DuplicateRegistrationAction duplicateRegistrationAction) {
            this.duplicateRegistrationAction(duplicateRegistrationAction == null ? null : duplicateRegistrationAction.toString());
            return this;
        }

        public final Integer getFraudsterSimilarityThreshold() {
            return fraudsterSimilarityThreshold;
        }

        public final void setFraudsterSimilarityThreshold(Integer fraudsterSimilarityThreshold) {
            this.fraudsterSimilarityThreshold = fraudsterSimilarityThreshold;
        }

        @Override
        public final Builder fraudsterSimilarityThreshold(Integer fraudsterSimilarityThreshold) {
            this.fraudsterSimilarityThreshold = fraudsterSimilarityThreshold;
            return this;
        }

        public final Collection<String> getWatchlistIds() {
            if (watchlistIds instanceof SdkAutoConstructList) {
                return null;
            }
            return watchlistIds;
        }

        public final void setWatchlistIds(Collection<String> watchlistIds) {
            this.watchlistIds = RegistrationConfigWatchlistIdsCopier.copy(watchlistIds);
        }

        @Override
        public final Builder watchlistIds(Collection<String> watchlistIds) {
            this.watchlistIds = RegistrationConfigWatchlistIdsCopier.copy(watchlistIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder watchlistIds(String... watchlistIds) {
            watchlistIds(Arrays.asList(watchlistIds));
            return this;
        }

        @Override
        public RegistrationConfig build() {
            return new RegistrationConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
