/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.transfer.model.ConflictException;
import software.amazon.awssdk.services.transfer.model.CreateServerRequest;
import software.amazon.awssdk.services.transfer.model.CreateServerResponse;
import software.amazon.awssdk.services.transfer.model.CreateUserRequest;
import software.amazon.awssdk.services.transfer.model.CreateUserResponse;
import software.amazon.awssdk.services.transfer.model.DeleteServerRequest;
import software.amazon.awssdk.services.transfer.model.DeleteServerResponse;
import software.amazon.awssdk.services.transfer.model.DeleteSshPublicKeyRequest;
import software.amazon.awssdk.services.transfer.model.DeleteSshPublicKeyResponse;
import software.amazon.awssdk.services.transfer.model.DeleteUserRequest;
import software.amazon.awssdk.services.transfer.model.DeleteUserResponse;
import software.amazon.awssdk.services.transfer.model.DescribeServerRequest;
import software.amazon.awssdk.services.transfer.model.DescribeServerResponse;
import software.amazon.awssdk.services.transfer.model.DescribeUserRequest;
import software.amazon.awssdk.services.transfer.model.DescribeUserResponse;
import software.amazon.awssdk.services.transfer.model.ImportSshPublicKeyRequest;
import software.amazon.awssdk.services.transfer.model.ImportSshPublicKeyResponse;
import software.amazon.awssdk.services.transfer.model.InternalServiceErrorException;
import software.amazon.awssdk.services.transfer.model.InvalidNextTokenException;
import software.amazon.awssdk.services.transfer.model.InvalidRequestException;
import software.amazon.awssdk.services.transfer.model.ListServersRequest;
import software.amazon.awssdk.services.transfer.model.ListServersResponse;
import software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.transfer.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.transfer.model.ListUsersRequest;
import software.amazon.awssdk.services.transfer.model.ListUsersResponse;
import software.amazon.awssdk.services.transfer.model.ResourceExistsException;
import software.amazon.awssdk.services.transfer.model.ResourceNotFoundException;
import software.amazon.awssdk.services.transfer.model.ServiceUnavailableException;
import software.amazon.awssdk.services.transfer.model.StartServerRequest;
import software.amazon.awssdk.services.transfer.model.StartServerResponse;
import software.amazon.awssdk.services.transfer.model.StopServerRequest;
import software.amazon.awssdk.services.transfer.model.StopServerResponse;
import software.amazon.awssdk.services.transfer.model.TagResourceRequest;
import software.amazon.awssdk.services.transfer.model.TagResourceResponse;
import software.amazon.awssdk.services.transfer.model.TestIdentityProviderRequest;
import software.amazon.awssdk.services.transfer.model.TestIdentityProviderResponse;
import software.amazon.awssdk.services.transfer.model.ThrottlingException;
import software.amazon.awssdk.services.transfer.model.TransferException;
import software.amazon.awssdk.services.transfer.model.UntagResourceRequest;
import software.amazon.awssdk.services.transfer.model.UntagResourceResponse;
import software.amazon.awssdk.services.transfer.model.UpdateServerRequest;
import software.amazon.awssdk.services.transfer.model.UpdateServerResponse;
import software.amazon.awssdk.services.transfer.model.UpdateUserRequest;
import software.amazon.awssdk.services.transfer.model.UpdateUserResponse;
import software.amazon.awssdk.services.transfer.paginators.ListServersIterable;
import software.amazon.awssdk.services.transfer.paginators.ListTagsForResourceIterable;
import software.amazon.awssdk.services.transfer.paginators.ListUsersIterable;

/**
 * Service client for accessing AWS Transfer. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * AWS Transfer for SFTP is a fully managed service that enables the transfer of files directly into and out of Amazon
 * S3 using the Secure File Transfer Protocol (SFTP)—also known as Secure Shell (SSH) File Transfer Protocol. AWS helps
 * you seamlessly migrate your file transfer workflows to AWS Transfer for SFTP—by integrating with existing
 * authentication systems, and providing DNS routing with Amazon Route 53—so nothing changes for your customers and
 * partners, or their applications. With your data in S3, you can use it with AWS services for processing, analytics,
 * machine learning, and archiving. Getting started with AWS Transfer for SFTP (AWS SFTP) is easy; there is no
 * infrastructure to buy and set up.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public interface TransferClient extends SdkClient {
    String SERVICE_NAME = "transfer";

    /**
     * Create a {@link TransferClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static TransferClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link TransferClient}.
     */
    static TransferClientBuilder builder() {
        return new DefaultTransferClientBuilder();
    }

    /**
     * <p>
     * Instantiates an autoscaling virtual server based on Secure File Transfer Protocol (SFTP) in AWS. When you make
     * updates to your server or when you work with users, use the service-generated <code>ServerId</code> property that
     * is assigned to the newly created server.
     * </p>
     *
     * @param createServerRequest
     * @return Result of the CreateServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceExistsException
     *         The requested resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.CreateServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/CreateServer" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateServerResponse createServer(CreateServerRequest createServerRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceExistsException, AwsServiceException,
            SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Instantiates an autoscaling virtual server based on Secure File Transfer Protocol (SFTP) in AWS. When you make
     * updates to your server or when you work with users, use the service-generated <code>ServerId</code> property that
     * is assigned to the newly created server.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateServerRequest.Builder} avoiding the need to
     * create one manually via {@link CreateServerRequest#builder()}
     * </p>
     *
     * @param createServerRequest
     *        A {@link Consumer} that will call methods on {@link CreateServerRequest.Builder} to create a request.
     * @return Result of the CreateServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceExistsException
     *         The requested resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.CreateServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/CreateServer" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateServerResponse createServer(Consumer<CreateServerRequest.Builder> createServerRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException, ResourceExistsException,
            AwsServiceException, SdkClientException, TransferException {
        return createServer(CreateServerRequest.builder().applyMutation(createServerRequest).build());
    }

    /**
     * <p>
     * Creates a user and associates them with an existing Secure File Transfer Protocol (SFTP) server. You can only
     * create and associate users with SFTP servers that have the <code>IdentityProviderType</code> set to
     * <code>SERVICE_MANAGED</code>. Using parameters for <code>CreateUser</code>, you can specify the user name, set
     * the home directory, store the user's public key, and assign the user's AWS Identity and Access Management (IAM)
     * role. You can also optionally add a scope-down policy, and assign metadata with tags that can be used to group
     * and search for users.
     * </p>
     *
     * @param createUserRequest
     * @return Result of the CreateUser operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceExistsException
     *         The requested resource does not exist.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.CreateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateUserResponse createUser(CreateUserRequest createUserRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceExistsException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a user and associates them with an existing Secure File Transfer Protocol (SFTP) server. You can only
     * create and associate users with SFTP servers that have the <code>IdentityProviderType</code> set to
     * <code>SERVICE_MANAGED</code>. Using parameters for <code>CreateUser</code>, you can specify the user name, set
     * the home directory, store the user's public key, and assign the user's AWS Identity and Access Management (IAM)
     * role. You can also optionally add a scope-down policy, and assign metadata with tags that can be used to group
     * and search for users.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateUserRequest.Builder} avoiding the need to
     * create one manually via {@link CreateUserRequest#builder()}
     * </p>
     *
     * @param createUserRequest
     *        A {@link Consumer} that will call methods on {@link CreateUserRequest.Builder} to create a request.
     * @return Result of the CreateUser operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceExistsException
     *         The requested resource does not exist.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.CreateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateUserResponse createUser(Consumer<CreateUserRequest.Builder> createUserRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException, ResourceExistsException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, TransferException {
        return createUser(CreateUserRequest.builder().applyMutation(createUserRequest).build());
    }

    /**
     * <p>
     * Deletes the Secure File Transfer Protocol (SFTP) server that you specify.
     * </p>
     * <p>
     * No response returns from this operation.
     * </p>
     *
     * @param deleteServerRequest
     * @return Result of the DeleteServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.DeleteServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteServer" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteServerResponse deleteServer(DeleteServerRequest deleteServerRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the Secure File Transfer Protocol (SFTP) server that you specify.
     * </p>
     * <p>
     * No response returns from this operation.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteServerRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteServerRequest#builder()}
     * </p>
     *
     * @param deleteServerRequest
     *        A {@link Consumer} that will call methods on {@link DeleteServerRequest.Builder} to create a request.
     * @return Result of the DeleteServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.DeleteServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteServer" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteServerResponse deleteServer(Consumer<DeleteServerRequest.Builder> deleteServerRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, TransferException {
        return deleteServer(DeleteServerRequest.builder().applyMutation(deleteServerRequest).build());
    }

    /**
     * <p>
     * Deletes a user's Secure Shell (SSH) public key.
     * </p>
     * <p>
     * No response is returned from this operation.
     * </p>
     *
     * @param deleteSshPublicKeyRequest
     * @return Result of the DeleteSshPublicKey operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.DeleteSshPublicKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteSshPublicKey" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteSshPublicKeyResponse deleteSshPublicKey(DeleteSshPublicKeyRequest deleteSshPublicKeyRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a user's Secure Shell (SSH) public key.
     * </p>
     * <p>
     * No response is returned from this operation.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteSshPublicKeyRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteSshPublicKeyRequest#builder()}
     * </p>
     *
     * @param deleteSshPublicKeyRequest
     *        A {@link Consumer} that will call methods on {@link DeleteSshPublicKeyRequest.Builder} to create a
     *        request.
     * @return Result of the DeleteSshPublicKey operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.DeleteSshPublicKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteSshPublicKey" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteSshPublicKeyResponse deleteSshPublicKey(Consumer<DeleteSshPublicKeyRequest.Builder> deleteSshPublicKeyRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, TransferException {
        return deleteSshPublicKey(DeleteSshPublicKeyRequest.builder().applyMutation(deleteSshPublicKeyRequest).build());
    }

    /**
     * <p>
     * Deletes the user belonging to the server you specify.
     * </p>
     * <p>
     * No response returns from this operation.
     * </p>
     * <note>
     * <p>
     * When you delete a user from a server, the user's information is lost.
     * </p>
     * </note>
     *
     * @param deleteUserRequest
     * @return Result of the DeleteUser operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.DeleteUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteUserResponse deleteUser(DeleteUserRequest deleteUserRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the user belonging to the server you specify.
     * </p>
     * <p>
     * No response returns from this operation.
     * </p>
     * <note>
     * <p>
     * When you delete a user from a server, the user's information is lost.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteUserRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteUserRequest#builder()}
     * </p>
     *
     * @param deleteUserRequest
     *        A {@link Consumer} that will call methods on {@link DeleteUserRequest.Builder} to create a request.
     * @return Result of the DeleteUser operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.DeleteUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteUserResponse deleteUser(Consumer<DeleteUserRequest.Builder> deleteUserRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, TransferException {
        return deleteUser(DeleteUserRequest.builder().applyMutation(deleteUserRequest).build());
    }

    /**
     * <p>
     * Describes the server that you specify by passing the <code>ServerId</code> parameter.
     * </p>
     * <p>
     * The response contains a description of the server's properties. When you set <code>EndpointType</code> to VPC,
     * the response will contain the <code>EndpointDetails</code>.
     * </p>
     *
     * @param describeServerRequest
     * @return Result of the DescribeServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.DescribeServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeServer" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeServerResponse describeServer(DescribeServerRequest describeServerRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes the server that you specify by passing the <code>ServerId</code> parameter.
     * </p>
     * <p>
     * The response contains a description of the server's properties. When you set <code>EndpointType</code> to VPC,
     * the response will contain the <code>EndpointDetails</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeServerRequest.Builder} avoiding the need to
     * create one manually via {@link DescribeServerRequest#builder()}
     * </p>
     *
     * @param describeServerRequest
     *        A {@link Consumer} that will call methods on {@link DescribeServerRequest.Builder} to create a request.
     * @return Result of the DescribeServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.DescribeServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeServer" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeServerResponse describeServer(Consumer<DescribeServerRequest.Builder> describeServerRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, TransferException {
        return describeServer(DescribeServerRequest.builder().applyMutation(describeServerRequest).build());
    }

    /**
     * <p>
     * Describes the user assigned to a specific server, as identified by its <code>ServerId</code> property.
     * </p>
     * <p>
     * The response from this call returns the properties of the user associated with the <code>ServerId</code> value
     * that was specified.
     * </p>
     *
     * @param describeUserRequest
     * @return Result of the DescribeUser operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.DescribeUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeUser" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeUserResponse describeUser(DescribeUserRequest describeUserRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes the user assigned to a specific server, as identified by its <code>ServerId</code> property.
     * </p>
     * <p>
     * The response from this call returns the properties of the user associated with the <code>ServerId</code> value
     * that was specified.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeUserRequest.Builder} avoiding the need to
     * create one manually via {@link DescribeUserRequest#builder()}
     * </p>
     *
     * @param describeUserRequest
     *        A {@link Consumer} that will call methods on {@link DescribeUserRequest.Builder} to create a request.
     * @return Result of the DescribeUser operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.DescribeUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeUser" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeUserResponse describeUser(Consumer<DescribeUserRequest.Builder> describeUserRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, TransferException {
        return describeUser(DescribeUserRequest.builder().applyMutation(describeUserRequest).build());
    }

    /**
     * <p>
     * Adds a Secure Shell (SSH) public key to a user account identified by a <code>UserName</code> value assigned to a
     * specific server, identified by <code>ServerId</code>.
     * </p>
     * <p>
     * The response returns the <code>UserName</code> value, the <code>ServerId</code> value, and the name of the
     * <code>SshPublicKeyId</code>.
     * </p>
     *
     * @param importSshPublicKeyRequest
     * @return Result of the ImportSshPublicKey operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceExistsException
     *         The requested resource does not exist.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ImportSshPublicKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ImportSshPublicKey" target="_top">AWS
     *      API Documentation</a>
     */
    default ImportSshPublicKeyResponse importSshPublicKey(ImportSshPublicKeyRequest importSshPublicKeyRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException, ResourceExistsException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds a Secure Shell (SSH) public key to a user account identified by a <code>UserName</code> value assigned to a
     * specific server, identified by <code>ServerId</code>.
     * </p>
     * <p>
     * The response returns the <code>UserName</code> value, the <code>ServerId</code> value, and the name of the
     * <code>SshPublicKeyId</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ImportSshPublicKeyRequest.Builder} avoiding the
     * need to create one manually via {@link ImportSshPublicKeyRequest#builder()}
     * </p>
     *
     * @param importSshPublicKeyRequest
     *        A {@link Consumer} that will call methods on {@link ImportSshPublicKeyRequest.Builder} to create a
     *        request.
     * @return Result of the ImportSshPublicKey operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceExistsException
     *         The requested resource does not exist.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ImportSshPublicKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ImportSshPublicKey" target="_top">AWS
     *      API Documentation</a>
     */
    default ImportSshPublicKeyResponse importSshPublicKey(Consumer<ImportSshPublicKeyRequest.Builder> importSshPublicKeyRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException, ResourceExistsException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, TransferException {
        return importSshPublicKey(ImportSshPublicKeyRequest.builder().applyMutation(importSshPublicKeyRequest).build());
    }

    /**
     * <p>
     * Lists the Secure File Transfer Protocol (SFTP) servers that are associated with your AWS account.
     * </p>
     *
     * @return Result of the ListServers operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListServers
     * @see #listServers(ListServersRequest)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListServersResponse listServers() throws ServiceUnavailableException, InternalServiceErrorException,
            InvalidNextTokenException, InvalidRequestException, AwsServiceException, SdkClientException, TransferException {
        return listServers(ListServersRequest.builder().build());
    }

    /**
     * <p>
     * Lists the Secure File Transfer Protocol (SFTP) servers that are associated with your AWS account.
     * </p>
     *
     * @param listServersRequest
     * @return Result of the ListServers operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListServers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListServersResponse listServers(ListServersRequest listServersRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidNextTokenException, InvalidRequestException, AwsServiceException,
            SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the Secure File Transfer Protocol (SFTP) servers that are associated with your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListServersRequest.Builder} avoiding the need to
     * create one manually via {@link ListServersRequest#builder()}
     * </p>
     *
     * @param listServersRequest
     *        A {@link Consumer} that will call methods on {@link ListServersRequest.Builder} to create a request.
     * @return Result of the ListServers operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListServers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListServersResponse listServers(Consumer<ListServersRequest.Builder> listServersRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidNextTokenException,
            InvalidRequestException, AwsServiceException, SdkClientException, TransferException {
        return listServers(ListServersRequest.builder().applyMutation(listServersRequest).build());
    }

    /**
     * <p>
     * Lists the Secure File Transfer Protocol (SFTP) servers that are associated with your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersIterable responses = client.listServersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transfer.paginators.ListServersIterable responses = client.listServersPaginator(request);
     *     for (software.amazon.awssdk.services.transfer.model.ListServersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersIterable responses = client.listServersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)} operation.</b>
     * </p>
     *
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListServers
     * @see #listServersPaginator(ListServersRequest)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListServersIterable listServersPaginator() throws ServiceUnavailableException, InternalServiceErrorException,
            InvalidNextTokenException, InvalidRequestException, AwsServiceException, SdkClientException, TransferException {
        return listServersPaginator(ListServersRequest.builder().build());
    }

    /**
     * <p>
     * Lists the Secure File Transfer Protocol (SFTP) servers that are associated with your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersIterable responses = client.listServersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transfer.paginators.ListServersIterable responses = client.listServersPaginator(request);
     *     for (software.amazon.awssdk.services.transfer.model.ListServersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersIterable responses = client.listServersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)} operation.</b>
     * </p>
     *
     * @param listServersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListServers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListServersIterable listServersPaginator(ListServersRequest listServersRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidNextTokenException, InvalidRequestException, AwsServiceException,
            SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the Secure File Transfer Protocol (SFTP) servers that are associated with your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersIterable responses = client.listServersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transfer.paginators.ListServersIterable responses = client.listServersPaginator(request);
     *     for (software.amazon.awssdk.services.transfer.model.ListServersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersIterable responses = client.listServersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListServersRequest.Builder} avoiding the need to
     * create one manually via {@link ListServersRequest#builder()}
     * </p>
     *
     * @param listServersRequest
     *        A {@link Consumer} that will call methods on {@link ListServersRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListServers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListServersIterable listServersPaginator(Consumer<ListServersRequest.Builder> listServersRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidNextTokenException,
            InvalidRequestException, AwsServiceException, SdkClientException, TransferException {
        return listServersPaginator(ListServersRequest.builder().applyMutation(listServersRequest).build());
    }

    /**
     * <p>
     * Lists all of the tags associated with the Amazon Resource Number (ARN) you specify. The resource can be a user,
     * server, or role.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidNextTokenException,
            InvalidRequestException, AwsServiceException, SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all of the tags associated with the Amazon Resource Number (ARN) you specify. The resource can be a user,
     * server, or role.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidNextTokenException, InvalidRequestException, AwsServiceException,
            SdkClientException, TransferException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Lists all of the tags associated with the Amazon Resource Number (ARN) you specify. The resource can be a user,
     * server, or role.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transfer.paginators.ListTagsForResourceIterable responses = client
     *             .listTagsForResourcePaginator(request);
     *     for (software.amazon.awssdk.services.transfer.model.ListTagsForResourceResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default ListTagsForResourceIterable listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidNextTokenException,
            InvalidRequestException, AwsServiceException, SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all of the tags associated with the Amazon Resource Number (ARN) you specify. The resource can be a user,
     * server, or role.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transfer.paginators.ListTagsForResourceIterable responses = client
     *             .listTagsForResourcePaginator(request);
     *     for (software.amazon.awssdk.services.transfer.model.ListTagsForResourceResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default ListTagsForResourceIterable listTagsForResourcePaginator(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidNextTokenException, InvalidRequestException, AwsServiceException,
            SdkClientException, TransferException {
        return listTagsForResourcePaginator(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest)
                .build());
    }

    /**
     * <p>
     * Lists the users for the server that you specify by passing the <code>ServerId</code> parameter.
     * </p>
     *
     * @param listUsersRequest
     * @return Result of the ListUsers operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListUsers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListUsersResponse listUsers(ListUsersRequest listUsersRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidNextTokenException, InvalidRequestException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the users for the server that you specify by passing the <code>ServerId</code> parameter.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListUsersRequest.Builder} avoiding the need to
     * create one manually via {@link ListUsersRequest#builder()}
     * </p>
     *
     * @param listUsersRequest
     *        A {@link Consumer} that will call methods on {@link ListUsersRequest.Builder} to create a request.
     * @return Result of the ListUsers operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListUsers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListUsersResponse listUsers(Consumer<ListUsersRequest.Builder> listUsersRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidNextTokenException, InvalidRequestException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, TransferException {
        return listUsers(ListUsersRequest.builder().applyMutation(listUsersRequest).build());
    }

    /**
     * <p>
     * Lists the users for the server that you specify by passing the <code>ServerId</code> parameter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listUsers(software.amazon.awssdk.services.transfer.model.ListUsersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListUsersIterable responses = client.listUsersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transfer.paginators.ListUsersIterable responses = client.listUsersPaginator(request);
     *     for (software.amazon.awssdk.services.transfer.model.ListUsersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListUsersIterable responses = client.listUsersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listUsers(software.amazon.awssdk.services.transfer.model.ListUsersRequest)} operation.</b>
     * </p>
     *
     * @param listUsersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListUsers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListUsersIterable listUsersPaginator(ListUsersRequest listUsersRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidNextTokenException, InvalidRequestException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the users for the server that you specify by passing the <code>ServerId</code> parameter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listUsers(software.amazon.awssdk.services.transfer.model.ListUsersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListUsersIterable responses = client.listUsersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transfer.paginators.ListUsersIterable responses = client.listUsersPaginator(request);
     *     for (software.amazon.awssdk.services.transfer.model.ListUsersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListUsersIterable responses = client.listUsersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listUsers(software.amazon.awssdk.services.transfer.model.ListUsersRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListUsersRequest.Builder} avoiding the need to
     * create one manually via {@link ListUsersRequest#builder()}
     * </p>
     *
     * @param listUsersRequest
     *        A {@link Consumer} that will call methods on {@link ListUsersRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> parameter that was passed is invalid.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.ListUsers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListUsersIterable listUsersPaginator(Consumer<ListUsersRequest.Builder> listUsersRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidNextTokenException,
            InvalidRequestException, ResourceNotFoundException, AwsServiceException, SdkClientException, TransferException {
        return listUsersPaginator(ListUsersRequest.builder().applyMutation(listUsersRequest).build());
    }

    /**
     * <p>
     * Changes the state of a Secure File Transfer Protocol (SFTP) server from <code>OFFLINE</code> to
     * <code>ONLINE</code>. It has no impact on an SFTP server that is already <code>ONLINE</code>. An
     * <code>ONLINE</code> server can accept and process file transfer jobs.
     * </p>
     * <p>
     * The state of <code>STARTING</code> indicates that the server is in an intermediate state, either not fully able
     * to respond, or not fully online. The values of <code>START_FAILED</code> can indicate an error condition.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     *
     * @param startServerRequest
     * @return Result of the StartServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.StartServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/StartServer" target="_top">AWS API
     *      Documentation</a>
     */
    default StartServerResponse startServer(StartServerRequest startServerRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Changes the state of a Secure File Transfer Protocol (SFTP) server from <code>OFFLINE</code> to
     * <code>ONLINE</code>. It has no impact on an SFTP server that is already <code>ONLINE</code>. An
     * <code>ONLINE</code> server can accept and process file transfer jobs.
     * </p>
     * <p>
     * The state of <code>STARTING</code> indicates that the server is in an intermediate state, either not fully able
     * to respond, or not fully online. The values of <code>START_FAILED</code> can indicate an error condition.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartServerRequest.Builder} avoiding the need to
     * create one manually via {@link StartServerRequest#builder()}
     * </p>
     *
     * @param startServerRequest
     *        A {@link Consumer} that will call methods on {@link StartServerRequest.Builder} to create a request.
     * @return Result of the StartServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.StartServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/StartServer" target="_top">AWS API
     *      Documentation</a>
     */
    default StartServerResponse startServer(Consumer<StartServerRequest.Builder> startServerRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, TransferException {
        return startServer(StartServerRequest.builder().applyMutation(startServerRequest).build());
    }

    /**
     * <p>
     * Changes the state of an SFTP server from <code>ONLINE</code> to <code>OFFLINE</code>. An <code>OFFLINE</code>
     * server cannot accept and process file transfer jobs. Information tied to your server such as server and user
     * properties are not affected by stopping your server. Stopping a server will not reduce or impact your Secure File
     * Transfer Protocol (SFTP) endpoint billing.
     * </p>
     * <p>
     * The state of <code>STOPPING</code> indicates that the server is in an intermediate state, either not fully able
     * to respond, or not fully offline. The values of <code>STOP_FAILED</code> can indicate an error condition.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     *
     * @param stopServerRequest
     * @return Result of the StopServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.StopServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/StopServer" target="_top">AWS API
     *      Documentation</a>
     */
    default StopServerResponse stopServer(StopServerRequest stopServerRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Changes the state of an SFTP server from <code>ONLINE</code> to <code>OFFLINE</code>. An <code>OFFLINE</code>
     * server cannot accept and process file transfer jobs. Information tied to your server such as server and user
     * properties are not affected by stopping your server. Stopping a server will not reduce or impact your Secure File
     * Transfer Protocol (SFTP) endpoint billing.
     * </p>
     * <p>
     * The state of <code>STOPPING</code> indicates that the server is in an intermediate state, either not fully able
     * to respond, or not fully offline. The values of <code>STOP_FAILED</code> can indicate an error condition.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StopServerRequest.Builder} avoiding the need to
     * create one manually via {@link StopServerRequest#builder()}
     * </p>
     *
     * @param stopServerRequest
     *        A {@link Consumer} that will call methods on {@link StopServerRequest.Builder} to create a request.
     * @return Result of the StopServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.StopServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/StopServer" target="_top">AWS API
     *      Documentation</a>
     */
    default StopServerResponse stopServer(Consumer<StopServerRequest.Builder> stopServerRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, TransferException {
        return stopServer(StopServerRequest.builder().applyMutation(stopServerRequest).build());
    }

    /**
     * <p>
     * Attaches a key-value pair to a resource, as identified by its Amazon Resource Name (ARN). Resources are users,
     * servers, roles, and other entities.
     * </p>
     * <p>
     * There is no response returned from this call.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Attaches a key-value pair to a resource, as identified by its Amazon Resource Name (ARN). Resources are users,
     * servers, roles, and other entities.
     * </p>
     * <p>
     * There is no response returned from this call.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on {@link TagResourceRequest.Builder} to create a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, TransferException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * If the <code>IdentityProviderType</code> of the server is <code>API_Gateway</code>, tests whether your API
     * Gateway is set up successfully. We highly recommend that you call this operation to test your authentication
     * method as soon as you create your server. By doing so, you can troubleshoot issues with the API Gateway
     * integration to ensure that your users can successfully use the service.
     * </p>
     *
     * @param testIdentityProviderRequest
     * @return Result of the TestIdentityProvider operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.TestIdentityProvider
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/TestIdentityProvider" target="_top">AWS
     *      API Documentation</a>
     */
    default TestIdentityProviderResponse testIdentityProvider(TestIdentityProviderRequest testIdentityProviderRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * If the <code>IdentityProviderType</code> of the server is <code>API_Gateway</code>, tests whether your API
     * Gateway is set up successfully. We highly recommend that you call this operation to test your authentication
     * method as soon as you create your server. By doing so, you can troubleshoot issues with the API Gateway
     * integration to ensure that your users can successfully use the service.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TestIdentityProviderRequest.Builder} avoiding the
     * need to create one manually via {@link TestIdentityProviderRequest#builder()}
     * </p>
     *
     * @param testIdentityProviderRequest
     *        A {@link Consumer} that will call methods on {@link TestIdentityProviderRequest.Builder} to create a
     *        request.
     * @return Result of the TestIdentityProvider operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.TestIdentityProvider
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/TestIdentityProvider" target="_top">AWS
     *      API Documentation</a>
     */
    default TestIdentityProviderResponse testIdentityProvider(
            Consumer<TestIdentityProviderRequest.Builder> testIdentityProviderRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, TransferException {
        return testIdentityProvider(TestIdentityProviderRequest.builder().applyMutation(testIdentityProviderRequest).build());
    }

    /**
     * <p>
     * Detaches a key-value pair from a resource, as identified by its Amazon Resource Name (ARN). Resources are users,
     * servers, roles, and other entities.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Detaches a key-value pair from a resource, as identified by its Amazon Resource Name (ARN). Resources are users,
     * servers, roles, and other entities.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on {@link UntagResourceRequest.Builder} to create a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, TransferException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates the server properties after that server has been created.
     * </p>
     * <p>
     * The <code>UpdateServer</code> call returns the <code>ServerId</code> of the Secure File Transfer Protocol (SFTP)
     * server you updated.
     * </p>
     *
     * @param updateServerRequest
     * @return Result of the UpdateServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws ConflictException
     *         This exception is thrown when the <code>UpdatServer</code> is called for a server that has VPC as the
     *         endpoint type and the server's <code>VpcEndpointID</code> is not in the available state.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceExistsException
     *         The requested resource does not exist.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.UpdateServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UpdateServer" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateServerResponse updateServer(UpdateServerRequest updateServerRequest) throws ServiceUnavailableException,
            ConflictException, InternalServiceErrorException, InvalidRequestException, ResourceExistsException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the server properties after that server has been created.
     * </p>
     * <p>
     * The <code>UpdateServer</code> call returns the <code>ServerId</code> of the Secure File Transfer Protocol (SFTP)
     * server you updated.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateServerRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateServerRequest#builder()}
     * </p>
     *
     * @param updateServerRequest
     *        A {@link Consumer} that will call methods on {@link UpdateServerRequest.Builder} to create a request.
     * @return Result of the UpdateServer operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws ConflictException
     *         This exception is thrown when the <code>UpdatServer</code> is called for a server that has VPC as the
     *         endpoint type and the server's <code>VpcEndpointID</code> is not in the available state.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceExistsException
     *         The requested resource does not exist.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.UpdateServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UpdateServer" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateServerResponse updateServer(Consumer<UpdateServerRequest.Builder> updateServerRequest)
            throws ServiceUnavailableException, ConflictException, InternalServiceErrorException, InvalidRequestException,
            ResourceExistsException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            TransferException {
        return updateServer(UpdateServerRequest.builder().applyMutation(updateServerRequest).build());
    }

    /**
     * <p>
     * Assigns new properties to a user. Parameters you pass modify any or all of the following: the home directory,
     * role, and policy for the <code>UserName</code> and <code>ServerId</code> you specify.
     * </p>
     * <p>
     * The response returns the <code>ServerId</code> and the <code>UserName</code> for the updated user.
     * </p>
     *
     * @param updateUserRequest
     * @return Result of the UpdateUser operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.UpdateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateUserResponse updateUser(UpdateUserRequest updateUserRequest) throws ServiceUnavailableException,
            InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, TransferException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Assigns new properties to a user. Parameters you pass modify any or all of the following: the home directory,
     * role, and policy for the <code>UserName</code> and <code>ServerId</code> you specify.
     * </p>
     * <p>
     * The response returns the <code>ServerId</code> and the <code>UserName</code> for the updated user.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateUserRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateUserRequest#builder()}
     * </p>
     *
     * @param updateUserRequest
     *        A {@link Consumer} that will call methods on {@link UpdateUserRequest.Builder} to create a request.
     * @return Result of the UpdateUser operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request has failed because the AWS Transfer for SFTP service is not available.
     * @throws InternalServiceErrorException
     *         This exception is thrown when an error occurs in the AWS Transfer for SFTP service.
     * @throws InvalidRequestException
     *         This exception is thrown when the client submits a malformed request.
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource is not found by the AWS Transfer for SFTP service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TransferException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TransferClient.UpdateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateUserResponse updateUser(Consumer<UpdateUserRequest.Builder> updateUserRequest)
            throws ServiceUnavailableException, InternalServiceErrorException, InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, TransferException {
        return updateUser(UpdateUserRequest.builder().applyMutation(updateUserRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of("transfer");
    }
}
