/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribestreaming.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains encryption related settings to be used for data encryption with Key Management Service, including
 * KmsEncryptionContext and KmsKeyId. The KmsKeyId is required, while KmsEncryptionContext is optional for additional
 * layer of security.
 * </p>
 * <p>
 * By default, Amazon Web Services HealthScribe provides encryption at rest to protect sensitive customer data using
 * Amazon S3-managed keys. HealthScribe uses the KMS key you specify as a second layer of encryption.
 * </p>
 * <p>
 * Your <code>ResourceAccessRoleArn</code> must permission to use your KMS key. For more information, see <a
 * href="https://docs.aws.amazon.com/transcribe/latest/dg/health-scribe-encryption.html">Data Encryption at rest for
 * Amazon Web Services HealthScribe</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MedicalScribeEncryptionSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<MedicalScribeEncryptionSettings.Builder, MedicalScribeEncryptionSettings> {
    private static final SdkField<Map<String, String>> KMS_ENCRYPTION_CONTEXT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("KmsEncryptionContext")
            .getter(getter(MedicalScribeEncryptionSettings::kmsEncryptionContext))
            .setter(setter(Builder::kmsEncryptionContext))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsEncryptionContext").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(MedicalScribeEncryptionSettings::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KMS_ENCRYPTION_CONTEXT_FIELD,
            KMS_KEY_ID_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Map<String, String> kmsEncryptionContext;

    private final String kmsKeyId;

    private MedicalScribeEncryptionSettings(BuilderImpl builder) {
        this.kmsEncryptionContext = builder.kmsEncryptionContext;
        this.kmsKeyId = builder.kmsKeyId;
    }

    /**
     * For responses, this returns true if the service returned a value for the KmsEncryptionContext property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasKmsEncryptionContext() {
        return kmsEncryptionContext != null && !(kmsEncryptionContext instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of plain text, non-secret key:value pairs, known as encryption context pairs, that provide an added layer
     * of security for your data. For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/key-management.html#kms-context">KMSencryption context
     * </a> and <a href="https://docs.aws.amazon.com/transcribe/latest/dg/symmetric-asymmetric.html">Asymmetric keys in
     * KMS </a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasKmsEncryptionContext} method.
     * </p>
     * 
     * @return A map of plain text, non-secret key:value pairs, known as encryption context pairs, that provide an added
     *         layer of security for your data. For more information, see <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/key-management.html#kms-context">KMSencryption
     *         context </a> and <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/symmetric-asymmetric.html">Asymmetric keys in KMS
     *         </a>.
     */
    public final Map<String, String> kmsEncryptionContext() {
        return kmsEncryptionContext;
    }

    /**
     * <p>
     * The ID of the KMS key you want to use for your streaming session. You can specify its KMS key ID, key Amazon
     * Resource Name (ARN), alias name, or alias ARN. When using an alias name, prefix it with <code>"alias/"</code>. To
     * specify a KMS key in a different Amazon Web Services account, you must use the key ARN or alias ARN.
     * </p>
     * <p>
     * For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID: 1234abcd-12ab-34cd-56ef-1234567890ab
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN: arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name: alias/ExampleAlias
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias ARN: arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias
     * </p>
     * </li>
     * </ul>
     * <p>
     * To get the key ID and key ARN for a KMS key, use the <a
     * href="https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeys.html">ListKeys</a> or <a
     * href="https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html">DescribeKey</a> KMS API
     * operations. To get the alias name and alias ARN, use <a
     * href="https://docs.aws.amazon.com/kms/latest/APIReference/API_ListAliases.html">ListKeys</a> API operation.
     * </p>
     * 
     * @return The ID of the KMS key you want to use for your streaming session. You can specify its KMS key ID, key
     *         Amazon Resource Name (ARN), alias name, or alias ARN. When using an alias name, prefix it with
     *         <code>"alias/"</code>. To specify a KMS key in a different Amazon Web Services account, you must use the
     *         key ARN or alias ARN.</p>
     *         <p>
     *         For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Key ID: 1234abcd-12ab-34cd-56ef-1234567890ab
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key ARN: arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias name: alias/ExampleAlias
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias ARN: arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To get the key ID and key ARN for a KMS key, use the <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeys.html">ListKeys</a> or <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html">DescribeKey</a> KMS API
     *         operations. To get the alias name and alias ARN, use <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_ListAliases.html">ListKeys</a> API
     *         operation.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasKmsEncryptionContext() ? kmsEncryptionContext() : null);
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MedicalScribeEncryptionSettings)) {
            return false;
        }
        MedicalScribeEncryptionSettings other = (MedicalScribeEncryptionSettings) obj;
        return hasKmsEncryptionContext() == other.hasKmsEncryptionContext()
                && Objects.equals(kmsEncryptionContext(), other.kmsEncryptionContext())
                && Objects.equals(kmsKeyId(), other.kmsKeyId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MedicalScribeEncryptionSettings")
                .add("KmsEncryptionContext", hasKmsEncryptionContext() ? kmsEncryptionContext() : null)
                .add("KmsKeyId", kmsKeyId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "KmsEncryptionContext":
            return Optional.ofNullable(clazz.cast(kmsEncryptionContext()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("KmsEncryptionContext", KMS_ENCRYPTION_CONTEXT_FIELD);
        map.put("KmsKeyId", KMS_KEY_ID_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MedicalScribeEncryptionSettings, T> g) {
        return obj -> g.apply((MedicalScribeEncryptionSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MedicalScribeEncryptionSettings> {
        /**
         * <p>
         * A map of plain text, non-secret key:value pairs, known as encryption context pairs, that provide an added
         * layer of security for your data. For more information, see <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/key-management.html#kms-context">KMSencryption context
         * </a> and <a href="https://docs.aws.amazon.com/transcribe/latest/dg/symmetric-asymmetric.html">Asymmetric keys
         * in KMS </a>.
         * </p>
         * 
         * @param kmsEncryptionContext
         *        A map of plain text, non-secret key:value pairs, known as encryption context pairs, that provide an
         *        added layer of security for your data. For more information, see <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/key-management.html#kms-context">KMSencryption
         *        context </a> and <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/symmetric-asymmetric.html">Asymmetric keys in
         *        KMS </a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsEncryptionContext(Map<String, String> kmsEncryptionContext);

        /**
         * <p>
         * The ID of the KMS key you want to use for your streaming session. You can specify its KMS key ID, key Amazon
         * Resource Name (ARN), alias name, or alias ARN. When using an alias name, prefix it with <code>"alias/"</code>
         * . To specify a KMS key in a different Amazon Web Services account, you must use the key ARN or alias ARN.
         * </p>
         * <p>
         * For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Key ID: 1234abcd-12ab-34cd-56ef-1234567890ab
         * </p>
         * </li>
         * <li>
         * <p>
         * Key ARN: arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias name: alias/ExampleAlias
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias ARN: arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias
         * </p>
         * </li>
         * </ul>
         * <p>
         * To get the key ID and key ARN for a KMS key, use the <a
         * href="https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeys.html">ListKeys</a> or <a
         * href="https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html">DescribeKey</a> KMS API
         * operations. To get the alias name and alias ARN, use <a
         * href="https://docs.aws.amazon.com/kms/latest/APIReference/API_ListAliases.html">ListKeys</a> API operation.
         * </p>
         * 
         * @param kmsKeyId
         *        The ID of the KMS key you want to use for your streaming session. You can specify its KMS key ID, key
         *        Amazon Resource Name (ARN), alias name, or alias ARN. When using an alias name, prefix it with
         *        <code>"alias/"</code>. To specify a KMS key in a different Amazon Web Services account, you must use
         *        the key ARN or alias ARN.</p>
         *        <p>
         *        For example:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Key ID: 1234abcd-12ab-34cd-56ef-1234567890ab
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Key ARN: arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias name: alias/ExampleAlias
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias ARN: arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To get the key ID and key ARN for a KMS key, use the <a
         *        href="https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeys.html">ListKeys</a> or <a
         *        href="https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html">DescribeKey</a> KMS
         *        API operations. To get the alias name and alias ARN, use <a
         *        href="https://docs.aws.amazon.com/kms/latest/APIReference/API_ListAliases.html">ListKeys</a> API
         *        operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);
    }

    static final class BuilderImpl implements Builder {
        private Map<String, String> kmsEncryptionContext = DefaultSdkAutoConstructMap.getInstance();

        private String kmsKeyId;

        private BuilderImpl() {
        }

        private BuilderImpl(MedicalScribeEncryptionSettings model) {
            kmsEncryptionContext(model.kmsEncryptionContext);
            kmsKeyId(model.kmsKeyId);
        }

        public final Map<String, String> getKmsEncryptionContext() {
            if (kmsEncryptionContext instanceof SdkAutoConstructMap) {
                return null;
            }
            return kmsEncryptionContext;
        }

        public final void setKmsEncryptionContext(Map<String, String> kmsEncryptionContext) {
            this.kmsEncryptionContext = KMSEncryptionContextMapCopier.copy(kmsEncryptionContext);
        }

        @Override
        public final Builder kmsEncryptionContext(Map<String, String> kmsEncryptionContext) {
            this.kmsEncryptionContext = KMSEncryptionContextMapCopier.copy(kmsEncryptionContext);
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        @Override
        public MedicalScribeEncryptionSettings build() {
            return new MedicalScribeEncryptionSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
