/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.transcribe.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.transcribe.internal.TranscribeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.transcribe.model.BadRequestException;
import software.amazon.awssdk.services.transcribe.model.ConflictException;
import software.amazon.awssdk.services.transcribe.model.CreateCallAnalyticsCategoryRequest;
import software.amazon.awssdk.services.transcribe.model.CreateCallAnalyticsCategoryResponse;
import software.amazon.awssdk.services.transcribe.model.CreateLanguageModelRequest;
import software.amazon.awssdk.services.transcribe.model.CreateLanguageModelResponse;
import software.amazon.awssdk.services.transcribe.model.CreateMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.CreateMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteCallAnalyticsCategoryRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteCallAnalyticsCategoryResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteCallAnalyticsJobRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteCallAnalyticsJobResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteLanguageModelRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteLanguageModelResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalScribeJobRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalScribeJobResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.DescribeLanguageModelRequest;
import software.amazon.awssdk.services.transcribe.model.DescribeLanguageModelResponse;
import software.amazon.awssdk.services.transcribe.model.GetCallAnalyticsCategoryRequest;
import software.amazon.awssdk.services.transcribe.model.GetCallAnalyticsCategoryResponse;
import software.amazon.awssdk.services.transcribe.model.GetCallAnalyticsJobRequest;
import software.amazon.awssdk.services.transcribe.model.GetCallAnalyticsJobResponse;
import software.amazon.awssdk.services.transcribe.model.GetMedicalScribeJobRequest;
import software.amazon.awssdk.services.transcribe.model.GetMedicalScribeJobResponse;
import software.amazon.awssdk.services.transcribe.model.GetMedicalTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.GetMedicalTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.GetMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.GetMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.GetTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.GetTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.InternalFailureException;
import software.amazon.awssdk.services.transcribe.model.LimitExceededException;
import software.amazon.awssdk.services.transcribe.model.ListCallAnalyticsCategoriesRequest;
import software.amazon.awssdk.services.transcribe.model.ListCallAnalyticsCategoriesResponse;
import software.amazon.awssdk.services.transcribe.model.ListCallAnalyticsJobsRequest;
import software.amazon.awssdk.services.transcribe.model.ListCallAnalyticsJobsResponse;
import software.amazon.awssdk.services.transcribe.model.ListLanguageModelsRequest;
import software.amazon.awssdk.services.transcribe.model.ListLanguageModelsResponse;
import software.amazon.awssdk.services.transcribe.model.ListMedicalScribeJobsRequest;
import software.amazon.awssdk.services.transcribe.model.ListMedicalScribeJobsResponse;
import software.amazon.awssdk.services.transcribe.model.ListMedicalTranscriptionJobsRequest;
import software.amazon.awssdk.services.transcribe.model.ListMedicalTranscriptionJobsResponse;
import software.amazon.awssdk.services.transcribe.model.ListMedicalVocabulariesRequest;
import software.amazon.awssdk.services.transcribe.model.ListMedicalVocabulariesResponse;
import software.amazon.awssdk.services.transcribe.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.transcribe.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsRequest;
import software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsResponse;
import software.amazon.awssdk.services.transcribe.model.ListVocabulariesRequest;
import software.amazon.awssdk.services.transcribe.model.ListVocabulariesResponse;
import software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersRequest;
import software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersResponse;
import software.amazon.awssdk.services.transcribe.model.NotFoundException;
import software.amazon.awssdk.services.transcribe.model.StartCallAnalyticsJobRequest;
import software.amazon.awssdk.services.transcribe.model.StartCallAnalyticsJobResponse;
import software.amazon.awssdk.services.transcribe.model.StartMedicalScribeJobRequest;
import software.amazon.awssdk.services.transcribe.model.StartMedicalScribeJobResponse;
import software.amazon.awssdk.services.transcribe.model.StartMedicalTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.StartMedicalTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.StartTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.StartTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.TagResourceRequest;
import software.amazon.awssdk.services.transcribe.model.TagResourceResponse;
import software.amazon.awssdk.services.transcribe.model.TranscribeException;
import software.amazon.awssdk.services.transcribe.model.UntagResourceRequest;
import software.amazon.awssdk.services.transcribe.model.UntagResourceResponse;
import software.amazon.awssdk.services.transcribe.model.UpdateCallAnalyticsCategoryRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateCallAnalyticsCategoryResponse;
import software.amazon.awssdk.services.transcribe.model.UpdateMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyResponse;
import software.amazon.awssdk.services.transcribe.transform.CreateCallAnalyticsCategoryRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.CreateLanguageModelRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.CreateMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.CreateVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.CreateVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteCallAnalyticsCategoryRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteCallAnalyticsJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteLanguageModelRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteMedicalScribeJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteMedicalTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DescribeLanguageModelRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetCallAnalyticsCategoryRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetCallAnalyticsJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetMedicalScribeJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetMedicalTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListCallAnalyticsCategoriesRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListCallAnalyticsJobsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListLanguageModelsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListMedicalScribeJobsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListMedicalTranscriptionJobsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListMedicalVocabulariesRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListTranscriptionJobsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListVocabulariesRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListVocabularyFiltersRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.StartCallAnalyticsJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.StartMedicalScribeJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.StartMedicalTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.StartTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateCallAnalyticsCategoryRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateVocabularyRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link TranscribeClient}.
 *
 * @see TranscribeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTranscribeClient implements TranscribeClient {
    private static final Logger log = Logger.loggerFor(DefaultTranscribeClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTranscribeClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Transcribe" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new Call Analytics category.
     * </p>
     * <p>
     * All categories are automatically applied to your Call Analytics transcriptions. Note that in order to apply
     * categories to your transcriptions, you must create them before submitting your transcription request, as
     * categories cannot be applied retroactively.
     * </p>
     * <p>
     * When creating a new category, you can use the <code>InputType</code> parameter to label the category as a
     * <code>POST_CALL</code> or a <code>REAL_TIME</code> category. <code>POST_CALL</code> categories can only be
     * applied to post-call transcriptions and <code>REAL_TIME</code> categories can only be applied to real-time
     * transcriptions. If you do not include <code>InputType</code>, your category is created as a
     * <code>POST_CALL</code> category by default.
     * </p>
     * <p>
     * Call Analytics categories are composed of rules. For each category, you must create between 1 and 20 rules. Rules
     * can include these parameters: , , , and .
     * </p>
     * <p>
     * To update an existing category, see .
     * </p>
     * <p>
     * To learn more about Call Analytics categories, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/tca-categories-batch.html">Creating categories for
     * post-call transcriptions</a> and <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/tca-categories-stream.html">Creating categories for
     * real-time transcriptions</a>.
     * </p>
     *
     * @param createCallAnalyticsCategoryRequest
     * @return Result of the CreateCallAnalyticsCategory operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws ConflictException
     *         A resource already exists with this name. Resource names must be unique within an Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.CreateCallAnalyticsCategory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateCallAnalyticsCategory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateCallAnalyticsCategoryResponse createCallAnalyticsCategory(
            CreateCallAnalyticsCategoryRequest createCallAnalyticsCategoryRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, ConflictException, AwsServiceException, SdkClientException,
            TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCallAnalyticsCategoryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateCallAnalyticsCategoryResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCallAnalyticsCategoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCallAnalyticsCategoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCallAnalyticsCategory");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateCallAnalyticsCategoryRequest, CreateCallAnalyticsCategoryResponse>()
                            .withOperationName("CreateCallAnalyticsCategory").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createCallAnalyticsCategoryRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateCallAnalyticsCategoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new custom language model.
     * </p>
     * <p>
     * When creating a new custom language model, you must specify:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you want a Wideband (audio sample rates over 16,000 Hz) or Narrowband (audio sample rates under 16,000 Hz)
     * base model
     * </p>
     * </li>
     * <li>
     * <p>
     * The location of your training and tuning files (this must be an Amazon S3 URI)
     * </p>
     * </li>
     * <li>
     * <p>
     * The language of your model
     * </p>
     * </li>
     * <li>
     * <p>
     * A unique name for your model
     * </p>
     * </li>
     * </ul>
     *
     * @param createLanguageModelRequest
     * @return Result of the CreateLanguageModel operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws ConflictException
     *         A resource already exists with this name. Resource names must be unique within an Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.CreateLanguageModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateLanguageModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateLanguageModelResponse createLanguageModel(CreateLanguageModelRequest createLanguageModelRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, ConflictException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateLanguageModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateLanguageModelResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createLanguageModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLanguageModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLanguageModel");

            return clientHandler.execute(new ClientExecutionParams<CreateLanguageModelRequest, CreateLanguageModelResponse>()
                    .withOperationName("CreateLanguageModel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createLanguageModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateLanguageModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new custom medical vocabulary.
     * </p>
     * <p>
     * Before creating a new custom medical vocabulary, you must first upload a text file that contains your vocabulary
     * table into an Amazon S3 bucket. Note that this differs from , where you can include a list of terms within your
     * request using the <code>Phrases</code> flag; <code>CreateMedicalVocabulary</code> does not support the
     * <code>Phrases</code> flag and only accepts vocabularies in table format.
     * </p>
     * <p>
     * Each language has a character set that contains all allowed characters for that specific language. If you use
     * unsupported characters, your custom vocabulary request fails. Refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/charsets.html">Character Sets for Custom Vocabularies</a>
     * to get the character set for your language.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/custom-vocabulary.html">Custom vocabularies</a>.
     * </p>
     *
     * @param createMedicalVocabularyRequest
     * @return Result of the CreateMedicalVocabulary operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws ConflictException
     *         A resource already exists with this name. Resource names must be unique within an Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.CreateMedicalVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateMedicalVocabularyResponse createMedicalVocabulary(CreateMedicalVocabularyRequest createMedicalVocabularyRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, ConflictException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateMedicalVocabularyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMedicalVocabularyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMedicalVocabulary");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateMedicalVocabularyRequest, CreateMedicalVocabularyResponse>()
                            .withOperationName("CreateMedicalVocabulary").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createMedicalVocabularyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateMedicalVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new custom vocabulary.
     * </p>
     * <p>
     * When creating a new custom vocabulary, you can either upload a text file that contains your new entries, phrases,
     * and terms into an Amazon S3 bucket and include the URI in your request. Or you can include a list of terms
     * directly in your request using the <code>Phrases</code> flag.
     * </p>
     * <p>
     * Each language has a character set that contains all allowed characters for that specific language. If you use
     * unsupported characters, your custom vocabulary request fails. Refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/charsets.html">Character Sets for Custom Vocabularies</a>
     * to get the character set for your language.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/custom-vocabulary.html">Custom vocabularies</a>.
     * </p>
     *
     * @param createVocabularyRequest
     * @return Result of the CreateVocabulary operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws ConflictException
     *         A resource already exists with this name. Resource names must be unique within an Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.CreateVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateVocabulary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateVocabularyResponse createVocabulary(CreateVocabularyRequest createVocabularyRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, ConflictException, AwsServiceException, SdkClientException,
            TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateVocabularyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVocabularyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVocabulary");

            return clientHandler.execute(new ClientExecutionParams<CreateVocabularyRequest, CreateVocabularyResponse>()
                    .withOperationName("CreateVocabulary").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createVocabularyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new custom vocabulary filter.
     * </p>
     * <p>
     * You can use custom vocabulary filters to mask, delete, or flag specific words from your transcript. Custom
     * vocabulary filters are commonly used to mask profanity in transcripts.
     * </p>
     * <p>
     * Each language has a character set that contains all allowed characters for that specific language. If you use
     * unsupported characters, your custom vocabulary filter request fails. Refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/charsets.html">Character Sets for Custom Vocabularies</a>
     * to get the character set for your language.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/vocabulary-filtering.html">Vocabulary filtering</a>.
     * </p>
     *
     * @param createVocabularyFilterRequest
     * @return Result of the CreateVocabularyFilter operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws ConflictException
     *         A resource already exists with this name. Resource names must be unique within an Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.CreateVocabularyFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateVocabularyFilterResponse createVocabularyFilter(CreateVocabularyFilterRequest createVocabularyFilterRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, ConflictException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateVocabularyFilterResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVocabularyFilterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVocabularyFilter");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateVocabularyFilterRequest, CreateVocabularyFilterResponse>()
                            .withOperationName("CreateVocabularyFilter").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createVocabularyFilterRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateVocabularyFilterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a Call Analytics category. To use this operation, specify the name of the category you want to delete
     * using <code>CategoryName</code>. Category names are case sensitive.
     * </p>
     *
     * @param deleteCallAnalyticsCategoryRequest
     * @return Result of the DeleteCallAnalyticsCategory operation returned by the service.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DeleteCallAnalyticsCategory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteCallAnalyticsCategory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteCallAnalyticsCategoryResponse deleteCallAnalyticsCategory(
            DeleteCallAnalyticsCategoryRequest deleteCallAnalyticsCategoryRequest) throws NotFoundException,
            LimitExceededException, BadRequestException, InternalFailureException, AwsServiceException, SdkClientException,
            TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCallAnalyticsCategoryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteCallAnalyticsCategoryResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCallAnalyticsCategoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCallAnalyticsCategoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCallAnalyticsCategory");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteCallAnalyticsCategoryRequest, DeleteCallAnalyticsCategoryResponse>()
                            .withOperationName("DeleteCallAnalyticsCategory").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteCallAnalyticsCategoryRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteCallAnalyticsCategoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a Call Analytics job. To use this operation, specify the name of the job you want to delete using
     * <code>CallAnalyticsJobName</code>. Job names are case sensitive.
     * </p>
     *
     * @param deleteCallAnalyticsJobRequest
     * @return Result of the DeleteCallAnalyticsJob operation returned by the service.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DeleteCallAnalyticsJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteCallAnalyticsJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteCallAnalyticsJobResponse deleteCallAnalyticsJob(DeleteCallAnalyticsJobRequest deleteCallAnalyticsJobRequest)
            throws LimitExceededException, BadRequestException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCallAnalyticsJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteCallAnalyticsJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCallAnalyticsJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCallAnalyticsJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCallAnalyticsJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteCallAnalyticsJobRequest, DeleteCallAnalyticsJobResponse>()
                            .withOperationName("DeleteCallAnalyticsJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteCallAnalyticsJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteCallAnalyticsJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a custom language model. To use this operation, specify the name of the language model you want to delete
     * using <code>ModelName</code>. custom language model names are case sensitive.
     * </p>
     *
     * @param deleteLanguageModelRequest
     * @return Result of the DeleteLanguageModel operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DeleteLanguageModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteLanguageModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteLanguageModelResponse deleteLanguageModel(DeleteLanguageModelRequest deleteLanguageModelRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLanguageModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteLanguageModelResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLanguageModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLanguageModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLanguageModel");

            return clientHandler.execute(new ClientExecutionParams<DeleteLanguageModelRequest, DeleteLanguageModelResponse>()
                    .withOperationName("DeleteLanguageModel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteLanguageModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteLanguageModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a Medical Scribe job. To use this operation, specify the name of the job you want to delete using
     * <code>MedicalScribeJobName</code>. Job names are case sensitive.
     * </p>
     *
     * @param deleteMedicalScribeJobRequest
     * @return Result of the DeleteMedicalScribeJob operation returned by the service.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DeleteMedicalScribeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteMedicalScribeJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteMedicalScribeJobResponse deleteMedicalScribeJob(DeleteMedicalScribeJobRequest deleteMedicalScribeJobRequest)
            throws LimitExceededException, BadRequestException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMedicalScribeJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteMedicalScribeJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMedicalScribeJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMedicalScribeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMedicalScribeJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteMedicalScribeJobRequest, DeleteMedicalScribeJobResponse>()
                            .withOperationName("DeleteMedicalScribeJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteMedicalScribeJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteMedicalScribeJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a medical transcription job. To use this operation, specify the name of the job you want to delete using
     * <code>MedicalTranscriptionJobName</code>. Job names are case sensitive.
     * </p>
     *
     * @param deleteMedicalTranscriptionJobRequest
     * @return Result of the DeleteMedicalTranscriptionJob operation returned by the service.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DeleteMedicalTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteMedicalTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteMedicalTranscriptionJobResponse deleteMedicalTranscriptionJob(
            DeleteMedicalTranscriptionJobRequest deleteMedicalTranscriptionJobRequest) throws LimitExceededException,
            BadRequestException, InternalFailureException, AwsServiceException, SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMedicalTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteMedicalTranscriptionJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMedicalTranscriptionJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteMedicalTranscriptionJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMedicalTranscriptionJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteMedicalTranscriptionJobRequest, DeleteMedicalTranscriptionJobResponse>()
                            .withOperationName("DeleteMedicalTranscriptionJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteMedicalTranscriptionJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteMedicalTranscriptionJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a custom medical vocabulary. To use this operation, specify the name of the custom vocabulary you want to
     * delete using <code>VocabularyName</code>. Custom vocabulary names are case sensitive.
     * </p>
     *
     * @param deleteMedicalVocabularyRequest
     * @return Result of the DeleteMedicalVocabulary operation returned by the service.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DeleteMedicalVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteMedicalVocabularyResponse deleteMedicalVocabulary(DeleteMedicalVocabularyRequest deleteMedicalVocabularyRequest)
            throws NotFoundException, LimitExceededException, BadRequestException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteMedicalVocabularyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMedicalVocabularyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMedicalVocabulary");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteMedicalVocabularyRequest, DeleteMedicalVocabularyResponse>()
                            .withOperationName("DeleteMedicalVocabulary").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteMedicalVocabularyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteMedicalVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a transcription job. To use this operation, specify the name of the job you want to delete using
     * <code>TranscriptionJobName</code>. Job names are case sensitive.
     * </p>
     *
     * @param deleteTranscriptionJobRequest
     * @return Result of the DeleteTranscriptionJob operation returned by the service.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DeleteTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteTranscriptionJobResponse deleteTranscriptionJob(DeleteTranscriptionJobRequest deleteTranscriptionJobRequest)
            throws LimitExceededException, BadRequestException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteTranscriptionJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTranscriptionJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTranscriptionJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteTranscriptionJobRequest, DeleteTranscriptionJobResponse>()
                            .withOperationName("DeleteTranscriptionJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteTranscriptionJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteTranscriptionJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a custom vocabulary. To use this operation, specify the name of the custom vocabulary you want to delete
     * using <code>VocabularyName</code>. Custom vocabulary names are case sensitive.
     * </p>
     *
     * @param deleteVocabularyRequest
     * @return Result of the DeleteVocabulary operation returned by the service.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DeleteVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteVocabulary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteVocabularyResponse deleteVocabulary(DeleteVocabularyRequest deleteVocabularyRequest) throws NotFoundException,
            LimitExceededException, BadRequestException, InternalFailureException, AwsServiceException, SdkClientException,
            TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteVocabularyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVocabularyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVocabulary");

            return clientHandler.execute(new ClientExecutionParams<DeleteVocabularyRequest, DeleteVocabularyResponse>()
                    .withOperationName("DeleteVocabulary").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteVocabularyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a custom vocabulary filter. To use this operation, specify the name of the custom vocabulary filter you
     * want to delete using <code>VocabularyFilterName</code>. Custom vocabulary filter names are case sensitive.
     * </p>
     *
     * @param deleteVocabularyFilterRequest
     * @return Result of the DeleteVocabularyFilter operation returned by the service.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DeleteVocabularyFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteVocabularyFilterResponse deleteVocabularyFilter(DeleteVocabularyFilterRequest deleteVocabularyFilterRequest)
            throws NotFoundException, LimitExceededException, BadRequestException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteVocabularyFilterResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVocabularyFilterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVocabularyFilter");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteVocabularyFilterRequest, DeleteVocabularyFilterResponse>()
                            .withOperationName("DeleteVocabularyFilter").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteVocabularyFilterRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteVocabularyFilterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about the specified custom language model.
     * </p>
     * <p>
     * This operation also shows if the base language model that you used to create your custom language model has been
     * updated. If Amazon Transcribe has updated the base model, you can create a new custom language model using the
     * updated base model.
     * </p>
     * <p>
     * If you tried to create a new custom language model and the request wasn't successful, you can use
     * <code>DescribeLanguageModel</code> to help identify the reason for this failure.
     * </p>
     *
     * @param describeLanguageModelRequest
     * @return Result of the DescribeLanguageModel operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DescribeLanguageModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DescribeLanguageModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeLanguageModelResponse describeLanguageModel(DescribeLanguageModelRequest describeLanguageModelRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, NotFoundException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeLanguageModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeLanguageModelResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeLanguageModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLanguageModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLanguageModel");

            return clientHandler.execute(new ClientExecutionParams<DescribeLanguageModelRequest, DescribeLanguageModelResponse>()
                    .withOperationName("DescribeLanguageModel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeLanguageModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeLanguageModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about the specified Call Analytics category.
     * </p>
     * <p>
     * To get a list of your Call Analytics categories, use the operation.
     * </p>
     *
     * @param getCallAnalyticsCategoryRequest
     * @return Result of the GetCallAnalyticsCategory operation returned by the service.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.GetCallAnalyticsCategory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetCallAnalyticsCategory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetCallAnalyticsCategoryResponse getCallAnalyticsCategory(
            GetCallAnalyticsCategoryRequest getCallAnalyticsCategoryRequest) throws NotFoundException, LimitExceededException,
            InternalFailureException, BadRequestException, AwsServiceException, SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCallAnalyticsCategoryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetCallAnalyticsCategoryResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCallAnalyticsCategoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCallAnalyticsCategoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCallAnalyticsCategory");

            return clientHandler
                    .execute(new ClientExecutionParams<GetCallAnalyticsCategoryRequest, GetCallAnalyticsCategoryResponse>()
                            .withOperationName("GetCallAnalyticsCategory").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getCallAnalyticsCategoryRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetCallAnalyticsCategoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about the specified Call Analytics job.
     * </p>
     * <p>
     * To view the job's status, refer to <code>CallAnalyticsJobStatus</code>. If the status is <code>COMPLETED</code>,
     * the job is finished. You can find your completed transcript at the URI specified in
     * <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>, <code>FailureReason</code> provides details
     * on why your transcription job failed.
     * </p>
     * <p>
     * If you enabled personally identifiable information (PII) redaction, the redacted transcript appears at the
     * location specified in <code>RedactedTranscriptFileUri</code>.
     * </p>
     * <p>
     * If you chose to redact the audio in your media file, you can find your redacted media file at the location
     * specified in <code>RedactedMediaFileUri</code>.
     * </p>
     * <p>
     * To get a list of your Call Analytics jobs, use the operation.
     * </p>
     *
     * @param getCallAnalyticsJobRequest
     * @return Result of the GetCallAnalyticsJob operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.GetCallAnalyticsJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetCallAnalyticsJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetCallAnalyticsJobResponse getCallAnalyticsJob(GetCallAnalyticsJobRequest getCallAnalyticsJobRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, NotFoundException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCallAnalyticsJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetCallAnalyticsJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCallAnalyticsJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCallAnalyticsJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCallAnalyticsJob");

            return clientHandler.execute(new ClientExecutionParams<GetCallAnalyticsJobRequest, GetCallAnalyticsJobResponse>()
                    .withOperationName("GetCallAnalyticsJob").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getCallAnalyticsJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetCallAnalyticsJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about the specified Medical Scribe job.
     * </p>
     * <p>
     * To view the status of the specified medical transcription job, check the <code>MedicalScribeJobStatus</code>
     * field. If the status is <code>COMPLETED</code>, the job is finished. You can find the results at the location
     * specified in <code>MedicalScribeOutput</code>. If the status is <code>FAILED</code>, <code>FailureReason</code>
     * provides details on why your Medical Scribe job failed.
     * </p>
     * <p>
     * To get a list of your Medical Scribe jobs, use the operation.
     * </p>
     *
     * @param getMedicalScribeJobRequest
     * @return Result of the GetMedicalScribeJob operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.GetMedicalScribeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetMedicalScribeJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetMedicalScribeJobResponse getMedicalScribeJob(GetMedicalScribeJobRequest getMedicalScribeJobRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, NotFoundException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMedicalScribeJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetMedicalScribeJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMedicalScribeJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMedicalScribeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMedicalScribeJob");

            return clientHandler.execute(new ClientExecutionParams<GetMedicalScribeJobRequest, GetMedicalScribeJobResponse>()
                    .withOperationName("GetMedicalScribeJob").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getMedicalScribeJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMedicalScribeJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about the specified medical transcription job.
     * </p>
     * <p>
     * To view the status of the specified medical transcription job, check the <code>TranscriptionJobStatus</code>
     * field. If the status is <code>COMPLETED</code>, the job is finished. You can find the results at the location
     * specified in <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>, <code>FailureReason</code>
     * provides details on why your transcription job failed.
     * </p>
     * <p>
     * To get a list of your medical transcription jobs, use the operation.
     * </p>
     *
     * @param getMedicalTranscriptionJobRequest
     * @return Result of the GetMedicalTranscriptionJob operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.GetMedicalTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetMedicalTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetMedicalTranscriptionJobResponse getMedicalTranscriptionJob(
            GetMedicalTranscriptionJobRequest getMedicalTranscriptionJobRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, NotFoundException, AwsServiceException, SdkClientException,
            TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMedicalTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetMedicalTranscriptionJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMedicalTranscriptionJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMedicalTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMedicalTranscriptionJob");

            return clientHandler
                    .execute(new ClientExecutionParams<GetMedicalTranscriptionJobRequest, GetMedicalTranscriptionJobResponse>()
                            .withOperationName("GetMedicalTranscriptionJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getMedicalTranscriptionJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetMedicalTranscriptionJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about the specified custom medical vocabulary.
     * </p>
     * <p>
     * To view the status of the specified custom medical vocabulary, check the <code>VocabularyState</code> field. If
     * the status is <code>READY</code>, your custom vocabulary is available to use. If the status is
     * <code>FAILED</code>, <code>FailureReason</code> provides details on why your vocabulary failed.
     * </p>
     * <p>
     * To get a list of your custom medical vocabularies, use the operation.
     * </p>
     *
     * @param getMedicalVocabularyRequest
     * @return Result of the GetMedicalVocabulary operation returned by the service.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.GetMedicalVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetMedicalVocabularyResponse getMedicalVocabulary(GetMedicalVocabularyRequest getMedicalVocabularyRequest)
            throws NotFoundException, LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetMedicalVocabularyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMedicalVocabularyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMedicalVocabulary");

            return clientHandler.execute(new ClientExecutionParams<GetMedicalVocabularyRequest, GetMedicalVocabularyResponse>()
                    .withOperationName("GetMedicalVocabulary").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getMedicalVocabularyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMedicalVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about the specified transcription job.
     * </p>
     * <p>
     * To view the status of the specified transcription job, check the <code>TranscriptionJobStatus</code> field. If
     * the status is <code>COMPLETED</code>, the job is finished. You can find the results at the location specified in
     * <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>, <code>FailureReason</code> provides details
     * on why your transcription job failed.
     * </p>
     * <p>
     * If you enabled content redaction, the redacted transcript can be found at the location specified in
     * <code>RedactedTranscriptFileUri</code>.
     * </p>
     * <p>
     * To get a list of your transcription jobs, use the operation.
     * </p>
     *
     * @param getTranscriptionJobRequest
     * @return Result of the GetTranscriptionJob operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.GetTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetTranscriptionJobResponse getTranscriptionJob(GetTranscriptionJobRequest getTranscriptionJobRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, NotFoundException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetTranscriptionJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTranscriptionJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTranscriptionJob");

            return clientHandler.execute(new ClientExecutionParams<GetTranscriptionJobRequest, GetTranscriptionJobResponse>()
                    .withOperationName("GetTranscriptionJob").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getTranscriptionJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTranscriptionJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about the specified custom vocabulary.
     * </p>
     * <p>
     * To view the status of the specified custom vocabulary, check the <code>VocabularyState</code> field. If the
     * status is <code>READY</code>, your custom vocabulary is available to use. If the status is <code>FAILED</code>,
     * <code>FailureReason</code> provides details on why your custom vocabulary failed.
     * </p>
     * <p>
     * To get a list of your custom vocabularies, use the operation.
     * </p>
     *
     * @param getVocabularyRequest
     * @return Result of the GetVocabulary operation returned by the service.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.GetVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetVocabulary" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetVocabularyResponse getVocabulary(GetVocabularyRequest getVocabularyRequest) throws NotFoundException,
            LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException, SdkClientException,
            TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetVocabularyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getVocabularyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVocabulary");

            return clientHandler.execute(new ClientExecutionParams<GetVocabularyRequest, GetVocabularyResponse>()
                    .withOperationName("GetVocabulary").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getVocabularyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about the specified custom vocabulary filter.
     * </p>
     * <p>
     * To get a list of your custom vocabulary filters, use the operation.
     * </p>
     *
     * @param getVocabularyFilterRequest
     * @return Result of the GetVocabularyFilter operation returned by the service.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.GetVocabularyFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetVocabularyFilterResponse getVocabularyFilter(GetVocabularyFilterRequest getVocabularyFilterRequest)
            throws NotFoundException, LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetVocabularyFilterResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getVocabularyFilterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVocabularyFilter");

            return clientHandler.execute(new ClientExecutionParams<GetVocabularyFilterRequest, GetVocabularyFilterResponse>()
                    .withOperationName("GetVocabularyFilter").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getVocabularyFilterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetVocabularyFilterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of Call Analytics categories, including all rules that make up each category.
     * </p>
     * <p>
     * To get detailed information about a specific Call Analytics category, use the operation.
     * </p>
     *
     * @param listCallAnalyticsCategoriesRequest
     * @return Result of the ListCallAnalyticsCategories operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListCallAnalyticsCategories
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListCallAnalyticsCategories"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCallAnalyticsCategoriesResponse listCallAnalyticsCategories(
            ListCallAnalyticsCategoriesRequest listCallAnalyticsCategoriesRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, AwsServiceException, SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCallAnalyticsCategoriesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListCallAnalyticsCategoriesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCallAnalyticsCategoriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCallAnalyticsCategoriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCallAnalyticsCategories");

            return clientHandler
                    .execute(new ClientExecutionParams<ListCallAnalyticsCategoriesRequest, ListCallAnalyticsCategoriesResponse>()
                            .withOperationName("ListCallAnalyticsCategories").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listCallAnalyticsCategoriesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListCallAnalyticsCategoriesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of Call Analytics jobs that match the specified criteria. If no criteria are specified, all Call
     * Analytics jobs are returned.
     * </p>
     * <p>
     * To get detailed information about a specific Call Analytics job, use the operation.
     * </p>
     *
     * @param listCallAnalyticsJobsRequest
     * @return Result of the ListCallAnalyticsJobs operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListCallAnalyticsJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListCallAnalyticsJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCallAnalyticsJobsResponse listCallAnalyticsJobs(ListCallAnalyticsJobsRequest listCallAnalyticsJobsRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCallAnalyticsJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListCallAnalyticsJobsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCallAnalyticsJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCallAnalyticsJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCallAnalyticsJobs");

            return clientHandler.execute(new ClientExecutionParams<ListCallAnalyticsJobsRequest, ListCallAnalyticsJobsResponse>()
                    .withOperationName("ListCallAnalyticsJobs").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listCallAnalyticsJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListCallAnalyticsJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of custom language models that match the specified criteria. If no criteria are specified, all
     * custom language models are returned.
     * </p>
     * <p>
     * To get detailed information about a specific custom language model, use the operation.
     * </p>
     *
     * @param listLanguageModelsRequest
     * @return Result of the ListLanguageModels operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListLanguageModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListLanguageModels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListLanguageModelsResponse listLanguageModels(ListLanguageModelsRequest listLanguageModelsRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLanguageModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListLanguageModelsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLanguageModelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLanguageModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLanguageModels");

            return clientHandler.execute(new ClientExecutionParams<ListLanguageModelsRequest, ListLanguageModelsResponse>()
                    .withOperationName("ListLanguageModels").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listLanguageModelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListLanguageModelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of Medical Scribe jobs that match the specified criteria. If no criteria are specified, all
     * Medical Scribe jobs are returned.
     * </p>
     * <p>
     * To get detailed information about a specific Medical Scribe job, use the operation.
     * </p>
     *
     * @param listMedicalScribeJobsRequest
     * @return Result of the ListMedicalScribeJobs operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListMedicalScribeJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListMedicalScribeJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMedicalScribeJobsResponse listMedicalScribeJobs(ListMedicalScribeJobsRequest listMedicalScribeJobsRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMedicalScribeJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListMedicalScribeJobsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMedicalScribeJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMedicalScribeJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMedicalScribeJobs");

            return clientHandler.execute(new ClientExecutionParams<ListMedicalScribeJobsRequest, ListMedicalScribeJobsResponse>()
                    .withOperationName("ListMedicalScribeJobs").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listMedicalScribeJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMedicalScribeJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of medical transcription jobs that match the specified criteria. If no criteria are specified,
     * all medical transcription jobs are returned.
     * </p>
     * <p>
     * To get detailed information about a specific medical transcription job, use the operation.
     * </p>
     *
     * @param listMedicalTranscriptionJobsRequest
     * @return Result of the ListMedicalTranscriptionJobs operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListMedicalTranscriptionJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListMedicalTranscriptionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMedicalTranscriptionJobsResponse listMedicalTranscriptionJobs(
            ListMedicalTranscriptionJobsRequest listMedicalTranscriptionJobsRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, AwsServiceException, SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMedicalTranscriptionJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListMedicalTranscriptionJobsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMedicalTranscriptionJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMedicalTranscriptionJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMedicalTranscriptionJobs");

            return clientHandler
                    .execute(new ClientExecutionParams<ListMedicalTranscriptionJobsRequest, ListMedicalTranscriptionJobsResponse>()
                            .withOperationName("ListMedicalTranscriptionJobs").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listMedicalTranscriptionJobsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListMedicalTranscriptionJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of custom medical vocabularies that match the specified criteria. If no criteria are specified,
     * all custom medical vocabularies are returned.
     * </p>
     * <p>
     * To get detailed information about a specific custom medical vocabulary, use the operation.
     * </p>
     *
     * @param listMedicalVocabulariesRequest
     * @return Result of the ListMedicalVocabularies operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListMedicalVocabularies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListMedicalVocabularies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMedicalVocabulariesResponse listMedicalVocabularies(ListMedicalVocabulariesRequest listMedicalVocabulariesRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMedicalVocabulariesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListMedicalVocabulariesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMedicalVocabulariesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMedicalVocabulariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMedicalVocabularies");

            return clientHandler
                    .execute(new ClientExecutionParams<ListMedicalVocabulariesRequest, ListMedicalVocabulariesResponse>()
                            .withOperationName("ListMedicalVocabularies").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listMedicalVocabulariesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListMedicalVocabulariesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all tags associated with the specified transcription job, vocabulary, model, or resource.
     * </p>
     * <p>
     * To learn more about using tags with Amazon Transcribe, refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/tagging.html">Tagging resources</a>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws BadRequestException, NotFoundException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of transcription jobs that match the specified criteria. If no criteria are specified, all
     * transcription jobs are returned.
     * </p>
     * <p>
     * To get detailed information about a specific transcription job, use the operation.
     * </p>
     *
     * @param listTranscriptionJobsRequest
     * @return Result of the ListTranscriptionJobs operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListTranscriptionJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListTranscriptionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTranscriptionJobsResponse listTranscriptionJobs(ListTranscriptionJobsRequest listTranscriptionJobsRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTranscriptionJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTranscriptionJobsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTranscriptionJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTranscriptionJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTranscriptionJobs");

            return clientHandler.execute(new ClientExecutionParams<ListTranscriptionJobsRequest, ListTranscriptionJobsResponse>()
                    .withOperationName("ListTranscriptionJobs").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTranscriptionJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTranscriptionJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of custom vocabularies that match the specified criteria. If no criteria are specified, all
     * custom vocabularies are returned.
     * </p>
     * <p>
     * To get detailed information about a specific custom vocabulary, use the operation.
     * </p>
     *
     * @param listVocabulariesRequest
     * @return Result of the ListVocabularies operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListVocabularies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListVocabulariesResponse listVocabularies(ListVocabulariesRequest listVocabulariesRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, AwsServiceException, SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVocabulariesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListVocabulariesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVocabulariesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVocabulariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVocabularies");

            return clientHandler.execute(new ClientExecutionParams<ListVocabulariesRequest, ListVocabulariesResponse>()
                    .withOperationName("ListVocabularies").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listVocabulariesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListVocabulariesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of custom vocabulary filters that match the specified criteria. If no criteria are specified, all
     * custom vocabularies are returned.
     * </p>
     * <p>
     * To get detailed information about a specific custom vocabulary filter, use the operation.
     * </p>
     *
     * @param listVocabularyFiltersRequest
     * @return Result of the ListVocabularyFilters operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListVocabularyFilters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularyFilters"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListVocabularyFiltersResponse listVocabularyFilters(ListVocabularyFiltersRequest listVocabularyFiltersRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVocabularyFiltersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListVocabularyFiltersResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVocabularyFiltersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVocabularyFiltersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVocabularyFilters");

            return clientHandler.execute(new ClientExecutionParams<ListVocabularyFiltersRequest, ListVocabularyFiltersResponse>()
                    .withOperationName("ListVocabularyFilters").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listVocabularyFiltersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListVocabularyFiltersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Transcribes the audio from a customer service call and applies any additional Request Parameters you choose to
     * include in your request.
     * </p>
     * <p>
     * In addition to many standard transcription features, Call Analytics provides you with call characteristics, call
     * summarization, speaker sentiment, and optional redaction of your text transcript and your audio file. You can
     * also apply custom categories to flag specified conditions. To learn more about these features and insights, refer
     * to <a href="https://docs.aws.amazon.com/transcribe/latest/dg/call-analytics.html">Analyzing call center audio
     * with Call Analytics</a>.
     * </p>
     * <p>
     * If you want to apply categories to your Call Analytics job, you must create them before submitting your job
     * request. Categories cannot be retroactively applied to a job. To create a new category, use the operation. To
     * learn more about Call Analytics categories, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/tca-categories-batch.html">Creating categories for
     * post-call transcriptions</a> and <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/tca-categories-stream.html">Creating categories for
     * real-time transcriptions</a>.
     * </p>
     * <p>
     * To make a <code>StartCallAnalyticsJob</code> request, you must first upload your media file into an Amazon S3
     * bucket; you can then specify the Amazon S3 location of the file using the <code>Media</code> parameter.
     * </p>
     * <p>
     * Job queuing is available for Call Analytics jobs. If you pass a <code>DataAccessRoleArn</code> in your request
     * and you exceed your Concurrent Job Limit, your job will automatically be added to a queue to be processed once
     * your concurrent job count is below the limit.
     * </p>
     * <p>
     * You must include the following parameters in your <code>StartCallAnalyticsJob</code> request:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>region</code>: The Amazon Web Services Region where you are making your request. For a list of Amazon Web
     * Services Regions supported with Amazon Transcribe, refer to <a
     * href="https://docs.aws.amazon.com/general/latest/gr/transcribe.html">Amazon Transcribe endpoints and quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CallAnalyticsJobName</code>: A custom name that you create for your transcription job that's unique within
     * your Amazon Web Services account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Media</code> (<code>MediaFileUri</code> or <code>RedactedMediaFileUri</code>): The Amazon S3 location of
     * your media file.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * With Call Analytics, you can redact the audio contained in your media file by including
     * <code>RedactedMediaFileUri</code>, instead of <code>MediaFileUri</code>, to specify the location of your input
     * audio. If you choose to redact your audio, you can find your redacted media at the location specified in the
     * <code>RedactedMediaFileUri</code> field of your response.
     * </p>
     * </note>
     *
     * @param startCallAnalyticsJobRequest
     * @return Result of the StartCallAnalyticsJob operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws ConflictException
     *         A resource already exists with this name. Resource names must be unique within an Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.StartCallAnalyticsJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/StartCallAnalyticsJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartCallAnalyticsJobResponse startCallAnalyticsJob(StartCallAnalyticsJobRequest startCallAnalyticsJobRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, ConflictException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartCallAnalyticsJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartCallAnalyticsJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startCallAnalyticsJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startCallAnalyticsJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartCallAnalyticsJob");

            return clientHandler.execute(new ClientExecutionParams<StartCallAnalyticsJobRequest, StartCallAnalyticsJobResponse>()
                    .withOperationName("StartCallAnalyticsJob").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startCallAnalyticsJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartCallAnalyticsJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Transcribes patient-clinician conversations and generates clinical notes.
     * </p>
     * <p>
     * Amazon Web Services HealthScribe automatically provides rich conversation transcripts, identifies speaker roles,
     * classifies dialogues, extracts medical terms, and generates preliminary clinical notes. To learn more about these
     * features, refer to <a href="https://docs.aws.amazon.com/transcribe/latest/dg/health-scribe.html">Amazon Web
     * Services HealthScribe</a>.
     * </p>
     * <p>
     * To make a <code>StartMedicalScribeJob</code> request, you must first upload your media file into an Amazon S3
     * bucket; you can then specify the Amazon S3 location of the file using the <code>Media</code> parameter.
     * </p>
     * <p>
     * You must include the following parameters in your <code>StartMedicalTranscriptionJob</code> request:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DataAccessRoleArn</code>: The ARN of an IAM role with the these minimum permissions: read permission on
     * input file Amazon S3 bucket specified in <code>Media</code>, write permission on the Amazon S3 bucket specified
     * in <code>OutputBucketName</code>, and full permissions on the KMS key specified in
     * <code>OutputEncryptionKMSKeyId</code> (if set). The role should also allow <code>transcribe.amazonaws.com</code>
     * to assume it.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Media</code> (<code>MediaFileUri</code>): The Amazon S3 location of your media file.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MedicalScribeJobName</code>: A custom name you create for your MedicalScribe job that is unique within your
     * Amazon Web Services account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OutputBucketName</code>: The Amazon S3 bucket where you want your output files stored.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Settings</code>: A <code>MedicalScribeSettings</code> object that must set exactly one of
     * <code>ShowSpeakerLabels</code> or <code>ChannelIdentification</code> to true. If <code>ShowSpeakerLabels</code>
     * is true, <code>MaxSpeakerLabels</code> must also be set.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ChannelDefinitions</code>: A <code>MedicalScribeChannelDefinitions</code> array should be set if and only
     * if the <code>ChannelIdentification</code> value of <code>Settings</code> is set to true.
     * </p>
     * </li>
     * </ul>
     *
     * @param startMedicalScribeJobRequest
     * @return Result of the StartMedicalScribeJob operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws ConflictException
     *         A resource already exists with this name. Resource names must be unique within an Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.StartMedicalScribeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/StartMedicalScribeJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartMedicalScribeJobResponse startMedicalScribeJob(StartMedicalScribeJobRequest startMedicalScribeJobRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, ConflictException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartMedicalScribeJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartMedicalScribeJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startMedicalScribeJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMedicalScribeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMedicalScribeJob");

            return clientHandler.execute(new ClientExecutionParams<StartMedicalScribeJobRequest, StartMedicalScribeJobResponse>()
                    .withOperationName("StartMedicalScribeJob").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startMedicalScribeJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartMedicalScribeJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Transcribes the audio from a medical dictation or conversation and applies any additional Request Parameters you
     * choose to include in your request.
     * </p>
     * <p>
     * In addition to many standard transcription features, Amazon Transcribe Medical provides you with a robust medical
     * vocabulary and, optionally, content identification, which adds flags to personal health information (PHI). To
     * learn more about these features, refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works-med.html">How Amazon Transcribe Medical
     * works</a>.
     * </p>
     * <p>
     * To make a <code>StartMedicalTranscriptionJob</code> request, you must first upload your media file into an Amazon
     * S3 bucket; you can then specify the Amazon S3 location of the file using the <code>Media</code> parameter.
     * </p>
     * <p>
     * You must include the following parameters in your <code>StartMedicalTranscriptionJob</code> request:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>region</code>: The Amazon Web Services Region where you are making your request. For a list of Amazon Web
     * Services Regions supported with Amazon Transcribe, refer to <a
     * href="https://docs.aws.amazon.com/general/latest/gr/transcribe.html">Amazon Transcribe endpoints and quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MedicalTranscriptionJobName</code>: A custom name you create for your transcription job that is unique
     * within your Amazon Web Services account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Media</code> (<code>MediaFileUri</code>): The Amazon S3 location of your media file.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LanguageCode</code>: This must be <code>en-US</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OutputBucketName</code>: The Amazon S3 bucket where you want your transcript stored. If you want your
     * output stored in a sub-folder of this bucket, you must also include <code>OutputKey</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Specialty</code>: This must be <code>PRIMARYCARE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Type</code>: Choose whether your audio is a conversation or a dictation.
     * </p>
     * </li>
     * </ul>
     *
     * @param startMedicalTranscriptionJobRequest
     * @return Result of the StartMedicalTranscriptionJob operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws ConflictException
     *         A resource already exists with this name. Resource names must be unique within an Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.StartMedicalTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/StartMedicalTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartMedicalTranscriptionJobResponse startMedicalTranscriptionJob(
            StartMedicalTranscriptionJobRequest startMedicalTranscriptionJobRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, ConflictException, AwsServiceException, SdkClientException,
            TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartMedicalTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartMedicalTranscriptionJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startMedicalTranscriptionJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMedicalTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMedicalTranscriptionJob");

            return clientHandler
                    .execute(new ClientExecutionParams<StartMedicalTranscriptionJobRequest, StartMedicalTranscriptionJobResponse>()
                            .withOperationName("StartMedicalTranscriptionJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startMedicalTranscriptionJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartMedicalTranscriptionJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Transcribes the audio from a media file and applies any additional Request Parameters you choose to include in
     * your request.
     * </p>
     * <p>
     * To make a <code>StartTranscriptionJob</code> request, you must first upload your media file into an Amazon S3
     * bucket; you can then specify the Amazon S3 location of the file using the <code>Media</code> parameter.
     * </p>
     * <p>
     * You must include the following parameters in your <code>StartTranscriptionJob</code> request:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>region</code>: The Amazon Web Services Region where you are making your request. For a list of Amazon Web
     * Services Regions supported with Amazon Transcribe, refer to <a
     * href="https://docs.aws.amazon.com/general/latest/gr/transcribe.html">Amazon Transcribe endpoints and quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TranscriptionJobName</code>: A custom name you create for your transcription job that is unique within your
     * Amazon Web Services account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Media</code> (<code>MediaFileUri</code>): The Amazon S3 location of your media file.
     * </p>
     * </li>
     * <li>
     * <p>
     * One of <code>LanguageCode</code>, <code>IdentifyLanguage</code>, or <code>IdentifyMultipleLanguages</code>: If
     * you know the language of your media file, specify it using the <code>LanguageCode</code> parameter; you can find
     * all valid language codes in the <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a> table.
     * If you do not know the languages spoken in your media, use either <code>IdentifyLanguage</code> or
     * <code>IdentifyMultipleLanguages</code> and let Amazon Transcribe identify the languages for you.
     * </p>
     * </li>
     * </ul>
     *
     * @param startTranscriptionJobRequest
     * @return Result of the StartTranscriptionJob operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws ConflictException
     *         A resource already exists with this name. Resource names must be unique within an Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.StartTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/StartTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartTranscriptionJobResponse startTranscriptionJob(StartTranscriptionJobRequest startTranscriptionJobRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, ConflictException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartTranscriptionJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startTranscriptionJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTranscriptionJob");

            return clientHandler.execute(new ClientExecutionParams<StartTranscriptionJobRequest, StartTranscriptionJobResponse>()
                    .withOperationName("StartTranscriptionJob").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startTranscriptionJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartTranscriptionJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more custom tags, each in the form of a key:value pair, to the specified resource.
     * </p>
     * <p>
     * To learn more about using tags with Amazon Transcribe, refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/tagging.html">Tagging resources</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws ConflictException
     *         A resource already exists with this name. Resource names must be unique within an Amazon Web Services
     *         account.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws BadRequestException, ConflictException,
            NotFoundException, LimitExceededException, InternalFailureException, AwsServiceException, SdkClientException,
            TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified Amazon Transcribe resource.
     * </p>
     * <p>
     * If you include <code>UntagResource</code> in your request, you must also include <code>ResourceArn</code> and
     * <code>TagKeys</code>.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws ConflictException
     *         A resource already exists with this name. Resource names must be unique within an Amazon Web Services
     *         account.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws LimitExceededException,
            BadRequestException, ConflictException, NotFoundException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified Call Analytics category with new rules. Note that the
     * <code>UpdateCallAnalyticsCategory</code> operation overwrites all existing rules contained in the specified
     * category. You cannot append additional rules onto an existing category.
     * </p>
     * <p>
     * To create a new category, see .
     * </p>
     *
     * @param updateCallAnalyticsCategoryRequest
     * @return Result of the UpdateCallAnalyticsCategory operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws ConflictException
     *         A resource already exists with this name. Resource names must be unique within an Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.UpdateCallAnalyticsCategory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateCallAnalyticsCategory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateCallAnalyticsCategoryResponse updateCallAnalyticsCategory(
            UpdateCallAnalyticsCategoryRequest updateCallAnalyticsCategoryRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, NotFoundException, ConflictException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateCallAnalyticsCategoryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateCallAnalyticsCategoryResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateCallAnalyticsCategoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCallAnalyticsCategoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCallAnalyticsCategory");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateCallAnalyticsCategoryRequest, UpdateCallAnalyticsCategoryResponse>()
                            .withOperationName("UpdateCallAnalyticsCategory").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateCallAnalyticsCategoryRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateCallAnalyticsCategoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing custom medical vocabulary with new values. This operation overwrites all existing information
     * with your new values; you cannot append new terms onto an existing custom vocabulary.
     * </p>
     *
     * @param updateMedicalVocabularyRequest
     * @return Result of the UpdateMedicalVocabulary operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws ConflictException
     *         A resource already exists with this name. Resource names must be unique within an Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.UpdateMedicalVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateMedicalVocabularyResponse updateMedicalVocabulary(UpdateMedicalVocabularyRequest updateMedicalVocabularyRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, NotFoundException, ConflictException,
            AwsServiceException, SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateMedicalVocabularyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateMedicalVocabularyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMedicalVocabulary");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateMedicalVocabularyRequest, UpdateMedicalVocabularyResponse>()
                            .withOperationName("UpdateMedicalVocabulary").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateMedicalVocabularyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateMedicalVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing custom vocabulary with new values. This operation overwrites all existing information with
     * your new values; you cannot append new terms onto an existing custom vocabulary.
     * </p>
     *
     * @param updateVocabularyRequest
     * @return Result of the UpdateVocabulary operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws ConflictException
     *         A resource already exists with this name. Resource names must be unique within an Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.UpdateVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateVocabulary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateVocabularyResponse updateVocabulary(UpdateVocabularyRequest updateVocabularyRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, NotFoundException, ConflictException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateVocabularyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateVocabularyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVocabulary");

            return clientHandler.execute(new ClientExecutionParams<UpdateVocabularyRequest, UpdateVocabularyResponse>()
                    .withOperationName("UpdateVocabulary").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateVocabularyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing custom vocabulary filter with a new list of words. The new list you provide overwrites all
     * previous entries; you cannot append new terms onto an existing custom vocabulary filter.
     * </p>
     *
     * @param updateVocabularyFilterRequest
     * @return Result of the UpdateVocabularyFilter operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. This can occur when the entity you're trying to
     *         delete doesn't exist or if it's in a non-terminal state (such as <code>IN PROGRESS</code>). See the
     *         exception message field for more information.
     * @throws LimitExceededException
     *         You've either sent too many requests or your input file is too long. Wait before retrying your request,
     *         or use a smaller file and try your request again.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message, correct the issue, and try your request again.
     * @throws NotFoundException
     *         We can't find the requested resource. Check that the specified name is correct and try your request
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.UpdateVocabularyFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateVocabularyFilterResponse updateVocabularyFilter(UpdateVocabularyFilterRequest updateVocabularyFilterRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, NotFoundException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateVocabularyFilterResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InternalFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalFailureException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateVocabularyFilterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVocabularyFilter");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateVocabularyFilterRequest, UpdateVocabularyFilterResponse>()
                            .withOperationName("UpdateVocabularyFilter").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateVocabularyFilterRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateVocabularyFilterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        TranscribeServiceClientConfigurationBuilder serviceConfigBuilder = new TranscribeServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(TranscribeException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.1");
    }

    @Override
    public final TranscribeServiceClientConfiguration serviceClientConfiguration() {
        return new TranscribeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
