/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.transcribe.model.BadRequestException;
import software.amazon.awssdk.services.transcribe.model.ConflictException;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.GetTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.GetTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.InternalFailureException;
import software.amazon.awssdk.services.transcribe.model.LimitExceededException;
import software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsRequest;
import software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsResponse;
import software.amazon.awssdk.services.transcribe.model.ListVocabulariesRequest;
import software.amazon.awssdk.services.transcribe.model.ListVocabulariesResponse;
import software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersRequest;
import software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersResponse;
import software.amazon.awssdk.services.transcribe.model.NotFoundException;
import software.amazon.awssdk.services.transcribe.model.StartTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.StartTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.TranscribeException;
import software.amazon.awssdk.services.transcribe.model.TranscribeRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyResponse;
import software.amazon.awssdk.services.transcribe.paginators.ListTranscriptionJobsPublisher;
import software.amazon.awssdk.services.transcribe.paginators.ListVocabulariesPublisher;
import software.amazon.awssdk.services.transcribe.paginators.ListVocabularyFiltersPublisher;
import software.amazon.awssdk.services.transcribe.transform.CreateVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.CreateVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListTranscriptionJobsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListVocabulariesRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListVocabularyFiltersRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.StartTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateVocabularyRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link TranscribeAsyncClient}.
 *
 * @see TranscribeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTranscribeAsyncClient implements TranscribeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultTranscribeAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTranscribeAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new custom vocabulary that you can use to change the way Amazon Transcribe handles transcription of an
     * audio file.
     * </p>
     *
     * @param createVocabularyRequest
     * @return A Java Future containing the result of the CreateVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>ConflictException When you are using the <code>CreateVocabulary</code> operation, the
     *         <code>JobName</code> field is a duplicate of a previously entered job name. Resend your request with a
     *         different name.</p>
     *         <p>
     *         When you are using the <code>UpdateVocabulary</code> operation, there are two jobs running at the same
     *         time. Resend the second request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.CreateVocabulary
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateVocabulary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVocabularyResponse> createVocabulary(CreateVocabularyRequest createVocabularyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVocabularyRequest, CreateVocabularyResponse>()
                            .withOperationName("CreateVocabulary")
                            .withMarshaller(new CreateVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createVocabularyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new vocabulary filter that you can use to filter words, such as profane words, from the output of a
     * transcription job.
     * </p>
     *
     * @param createVocabularyFilterRequest
     * @return A Java Future containing the result of the CreateVocabularyFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>ConflictException When you are using the <code>CreateVocabulary</code> operation, the
     *         <code>JobName</code> field is a duplicate of a previously entered job name. Resend your request with a
     *         different name.</p>
     *         <p>
     *         When you are using the <code>UpdateVocabulary</code> operation, there are two jobs running at the same
     *         time. Resend the second request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.CreateVocabularyFilter
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVocabularyFilterResponse> createVocabularyFilter(
            CreateVocabularyFilterRequest createVocabularyFilterRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVocabularyFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVocabularyFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVocabularyFilterRequest, CreateVocabularyFilterResponse>()
                            .withOperationName("CreateVocabularyFilter")
                            .withMarshaller(new CreateVocabularyFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createVocabularyFilterRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a previously submitted transcription job along with any other generated results such as the
     * transcription, models, and so on.
     * </p>
     *
     * @param deleteTranscriptionJobRequest
     * @return A Java Future containing the result of the DeleteTranscriptionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteTranscriptionJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTranscriptionJobResponse> deleteTranscriptionJob(
            DeleteTranscriptionJobRequest deleteTranscriptionJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTranscriptionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTranscriptionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTranscriptionJobRequest, DeleteTranscriptionJobResponse>()
                            .withOperationName("DeleteTranscriptionJob")
                            .withMarshaller(new DeleteTranscriptionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteTranscriptionJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a vocabulary from Amazon Transcribe.
     * </p>
     *
     * @param deleteVocabularyRequest
     * @return A Java Future containing the result of the DeleteVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteVocabulary
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteVocabulary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVocabularyResponse> deleteVocabulary(DeleteVocabularyRequest deleteVocabularyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVocabularyRequest, DeleteVocabularyResponse>()
                            .withOperationName("DeleteVocabulary")
                            .withMarshaller(new DeleteVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteVocabularyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a vocabulary filter.
     * </p>
     *
     * @param deleteVocabularyFilterRequest
     * @return A Java Future containing the result of the DeleteVocabularyFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteVocabularyFilter
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVocabularyFilterResponse> deleteVocabularyFilter(
            DeleteVocabularyFilterRequest deleteVocabularyFilterRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVocabularyFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVocabularyFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVocabularyFilterRequest, DeleteVocabularyFilterResponse>()
                            .withOperationName("DeleteVocabularyFilter")
                            .withMarshaller(new DeleteVocabularyFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteVocabularyFilterRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a transcription job. To see the status of the job, check the
     * <code>TranscriptionJobStatus</code> field. If the status is <code>COMPLETED</code>, the job is finished and you
     * can find the results at the location specified in the <code>TranscriptionFileUri</code> field.
     * </p>
     *
     * @param getTranscriptionJobRequest
     * @return A Java Future containing the result of the GetTranscriptionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetTranscriptionJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetTranscriptionJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTranscriptionJobResponse> getTranscriptionJob(
            GetTranscriptionJobRequest getTranscriptionJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTranscriptionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTranscriptionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTranscriptionJobRequest, GetTranscriptionJobResponse>()
                            .withOperationName("GetTranscriptionJob")
                            .withMarshaller(new GetTranscriptionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getTranscriptionJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a vocabulary.
     * </p>
     *
     * @param getVocabularyRequest
     * @return A Java Future containing the result of the GetVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetVocabulary
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetVocabulary" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetVocabularyResponse> getVocabulary(GetVocabularyRequest getVocabularyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetVocabularyRequest, GetVocabularyResponse>()
                            .withOperationName("GetVocabulary")
                            .withMarshaller(new GetVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getVocabularyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a vocabulary filter.
     * </p>
     *
     * @param getVocabularyFilterRequest
     * @return A Java Future containing the result of the GetVocabularyFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetVocabularyFilter
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetVocabularyFilter" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetVocabularyFilterResponse> getVocabularyFilter(
            GetVocabularyFilterRequest getVocabularyFilterRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetVocabularyFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetVocabularyFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetVocabularyFilterRequest, GetVocabularyFilterResponse>()
                            .withOperationName("GetVocabularyFilter")
                            .withMarshaller(new GetVocabularyFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getVocabularyFilterRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists transcription jobs with the specified status.
     * </p>
     *
     * @param listTranscriptionJobsRequest
     * @return A Java Future containing the result of the ListTranscriptionJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListTranscriptionJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListTranscriptionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTranscriptionJobsResponse> listTranscriptionJobs(
            ListTranscriptionJobsRequest listTranscriptionJobsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTranscriptionJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTranscriptionJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTranscriptionJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTranscriptionJobsRequest, ListTranscriptionJobsResponse>()
                            .withOperationName("ListTranscriptionJobs")
                            .withMarshaller(new ListTranscriptionJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTranscriptionJobsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists transcription jobs with the specified status.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTranscriptionJobs(software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListTranscriptionJobsPublisher publisher = client.listTranscriptionJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListTranscriptionJobsPublisher publisher = client.listTranscriptionJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTranscriptionJobs(software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTranscriptionJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListTranscriptionJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListTranscriptionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    public ListTranscriptionJobsPublisher listTranscriptionJobsPaginator(ListTranscriptionJobsRequest listTranscriptionJobsRequest) {
        return new ListTranscriptionJobsPublisher(this, applyPaginatorUserAgent(listTranscriptionJobsRequest));
    }

    /**
     * <p>
     * Returns a list of vocabularies that match the specified criteria. If no criteria are specified, returns the
     * entire list of vocabularies.
     * </p>
     *
     * @param listVocabulariesRequest
     * @return A Java Future containing the result of the ListVocabularies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListVocabularies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVocabulariesResponse> listVocabularies(ListVocabulariesRequest listVocabulariesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVocabulariesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVocabulariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVocabulariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVocabulariesRequest, ListVocabulariesResponse>()
                            .withOperationName("ListVocabularies")
                            .withMarshaller(new ListVocabulariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listVocabulariesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of vocabularies that match the specified criteria. If no criteria are specified, returns the
     * entire list of vocabularies.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listVocabularies(software.amazon.awssdk.services.transcribe.model.ListVocabulariesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListVocabulariesPublisher publisher = client.listVocabulariesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListVocabulariesPublisher publisher = client.listVocabulariesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transcribe.model.ListVocabulariesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transcribe.model.ListVocabulariesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVocabularies(software.amazon.awssdk.services.transcribe.model.ListVocabulariesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listVocabulariesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListVocabularies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularies" target="_top">AWS
     *      API Documentation</a>
     */
    public ListVocabulariesPublisher listVocabulariesPaginator(ListVocabulariesRequest listVocabulariesRequest) {
        return new ListVocabulariesPublisher(this, applyPaginatorUserAgent(listVocabulariesRequest));
    }

    /**
     * <p>
     * Gets information about vocabulary filters.
     * </p>
     *
     * @param listVocabularyFiltersRequest
     * @return A Java Future containing the result of the ListVocabularyFilters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListVocabularyFilters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularyFilters"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVocabularyFiltersResponse> listVocabularyFilters(
            ListVocabularyFiltersRequest listVocabularyFiltersRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVocabularyFiltersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVocabularyFiltersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVocabularyFiltersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVocabularyFiltersRequest, ListVocabularyFiltersResponse>()
                            .withOperationName("ListVocabularyFilters")
                            .withMarshaller(new ListVocabularyFiltersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listVocabularyFiltersRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about vocabulary filters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listVocabularyFilters(software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListVocabularyFiltersPublisher publisher = client.listVocabularyFiltersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListVocabularyFiltersPublisher publisher = client.listVocabularyFiltersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVocabularyFilters(software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listVocabularyFiltersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListVocabularyFilters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularyFilters"
     *      target="_top">AWS API Documentation</a>
     */
    public ListVocabularyFiltersPublisher listVocabularyFiltersPaginator(ListVocabularyFiltersRequest listVocabularyFiltersRequest) {
        return new ListVocabularyFiltersPublisher(this, applyPaginatorUserAgent(listVocabularyFiltersRequest));
    }

    /**
     * <p>
     * Starts an asynchronous job to transcribe speech to text.
     * </p>
     *
     * @param startTranscriptionJobRequest
     * @return A Java Future containing the result of the StartTranscriptionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>ConflictException When you are using the <code>CreateVocabulary</code> operation, the
     *         <code>JobName</code> field is a duplicate of a previously entered job name. Resend your request with a
     *         different name.</p>
     *         <p>
     *         When you are using the <code>UpdateVocabulary</code> operation, there are two jobs running at the same
     *         time. Resend the second request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.StartTranscriptionJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/StartTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartTranscriptionJobResponse> startTranscriptionJob(
            StartTranscriptionJobRequest startTranscriptionJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartTranscriptionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartTranscriptionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartTranscriptionJobRequest, StartTranscriptionJobResponse>()
                            .withOperationName("StartTranscriptionJob")
                            .withMarshaller(new StartTranscriptionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(startTranscriptionJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing vocabulary with new values. The <code>UpdateVocabulary</code> operation overwrites all of the
     * existing information with the values that you provide in the request.
     * </p>
     *
     * @param updateVocabularyRequest
     * @return A Java Future containing the result of the UpdateVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>ConflictException When you are using the <code>CreateVocabulary</code> operation, the
     *         <code>JobName</code> field is a duplicate of a previously entered job name. Resend your request with a
     *         different name.</p>
     *         <p>
     *         When you are using the <code>UpdateVocabulary</code> operation, there are two jobs running at the same
     *         time. Resend the second request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.UpdateVocabulary
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateVocabulary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateVocabularyResponse> updateVocabulary(UpdateVocabularyRequest updateVocabularyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVocabularyRequest, UpdateVocabularyResponse>()
                            .withOperationName("UpdateVocabulary")
                            .withMarshaller(new UpdateVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateVocabularyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a vocabulary filter with a new list of filtered words.
     * </p>
     *
     * @param updateVocabularyFilterRequest
     * @return A Java Future containing the result of the UpdateVocabularyFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the
     *         transcription you're trying to delete doesn't exist or if it is in a non-terminal state (for example,
     *         it's "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.UpdateVocabularyFilter
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateVocabularyFilterResponse> updateVocabularyFilter(
            UpdateVocabularyFilterRequest updateVocabularyFilterRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateVocabularyFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVocabularyFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVocabularyFilterRequest, UpdateVocabularyFilterResponse>()
                            .withOperationName("UpdateVocabularyFilter")
                            .withMarshaller(new UpdateVocabularyFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateVocabularyFilterRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TranscribeException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    private <T extends TranscribeRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
