/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sns.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the details of a single Amazon SNS message along with an <code>Id</code> that identifies a message within
 * the batch.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PublishBatchRequestEntry implements SdkPojo, Serializable,
        ToCopyableBuilder<PublishBatchRequestEntry.Builder, PublishBatchRequestEntry> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(PublishBatchRequestEntry::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Message")
            .getter(getter(PublishBatchRequestEntry::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Message").build()).build();

    private static final SdkField<String> SUBJECT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Subject")
            .getter(getter(PublishBatchRequestEntry::subject)).setter(setter(Builder::subject))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subject").build()).build();

    private static final SdkField<String> MESSAGE_STRUCTURE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MessageStructure").getter(getter(PublishBatchRequestEntry::messageStructure))
            .setter(setter(Builder::messageStructure))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageStructure").build()).build();

    private static final SdkField<Map<String, MessageAttributeValue>> MESSAGE_ATTRIBUTES_FIELD = SdkField
            .<Map<String, MessageAttributeValue>> builder(MarshallingType.MAP)
            .memberName("MessageAttributes")
            .getter(getter(PublishBatchRequestEntry::messageAttributes))
            .setter(setter(Builder::messageAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageAttributes").build(),
                    MapTrait.builder()
                            .keyLocationName("Name")
                            .valueLocationName("Value")
                            .valueFieldInfo(
                                    SdkField.<MessageAttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageAttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Value").build()).build()).build()).build();

    private static final SdkField<String> MESSAGE_DEDUPLICATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MessageDeduplicationId").getter(getter(PublishBatchRequestEntry::messageDeduplicationId))
            .setter(setter(Builder::messageDeduplicationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageDeduplicationId").build())
            .build();

    private static final SdkField<String> MESSAGE_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MessageGroupId").getter(getter(PublishBatchRequestEntry::messageGroupId))
            .setter(setter(Builder::messageGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageGroupId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, MESSAGE_FIELD,
            SUBJECT_FIELD, MESSAGE_STRUCTURE_FIELD, MESSAGE_ATTRIBUTES_FIELD, MESSAGE_DEDUPLICATION_ID_FIELD,
            MESSAGE_GROUP_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String message;

    private final String subject;

    private final String messageStructure;

    private final Map<String, MessageAttributeValue> messageAttributes;

    private final String messageDeduplicationId;

    private final String messageGroupId;

    private PublishBatchRequestEntry(BuilderImpl builder) {
        this.id = builder.id;
        this.message = builder.message;
        this.subject = builder.subject;
        this.messageStructure = builder.messageStructure;
        this.messageAttributes = builder.messageAttributes;
        this.messageDeduplicationId = builder.messageDeduplicationId;
        this.messageGroupId = builder.messageGroupId;
    }

    /**
     * <p>
     * An identifier for the message in this batch.
     * </p>
     * <note>
     * <p>
     * The <code>Ids</code> of a batch request must be unique within a request.
     * </p>
     * <p>
     * This identifier can have up to 80 characters. The following characters are accepted: alphanumeric characters,
     * hyphens(-), and underscores (_).
     * </p>
     * </note>
     * 
     * @return An identifier for the message in this batch.</p> <note>
     *         <p>
     *         The <code>Ids</code> of a batch request must be unique within a request.
     *         </p>
     *         <p>
     *         This identifier can have up to 80 characters. The following characters are accepted: alphanumeric
     *         characters, hyphens(-), and underscores (_).
     *         </p>
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The body of the message.
     * </p>
     * 
     * @return The body of the message.
     */
    public final String message() {
        return message;
    }

    /**
     * <p>
     * The subject of the batch message.
     * </p>
     * 
     * @return The subject of the batch message.
     */
    public final String subject() {
        return subject;
    }

    /**
     * <p>
     * Set <code>MessageStructure</code> to <code>json</code> if you want to send a different message for each protocol.
     * For example, using one publish action, you can send a short message to your SMS subscribers and a longer message
     * to your email subscribers. If you set <code>MessageStructure</code> to <code>json</code>, the value of the
     * <code>Message</code> parameter must:
     * </p>
     * <ul>
     * <li>
     * <p>
     * be a syntactically valid JSON object; and
     * </p>
     * </li>
     * <li>
     * <p>
     * contain at least a top-level JSON key of "default" with a value that is a string.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can define other top-level keys that define the message you want to send to a specific transport protocol
     * (e.g. http).
     * </p>
     * 
     * @return Set <code>MessageStructure</code> to <code>json</code> if you want to send a different message for each
     *         protocol. For example, using one publish action, you can send a short message to your SMS subscribers and
     *         a longer message to your email subscribers. If you set <code>MessageStructure</code> to <code>json</code>
     *         , the value of the <code>Message</code> parameter must: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         be a syntactically valid JSON object; and
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         contain at least a top-level JSON key of "default" with a value that is a string.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         You can define other top-level keys that define the message you want to send to a specific transport
     *         protocol (e.g. http).
     */
    public final String messageStructure() {
        return messageStructure;
    }

    /**
     * For responses, this returns true if the service returned a value for the MessageAttributes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMessageAttributes() {
        return messageAttributes != null && !(messageAttributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For more
     * information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html">Amazon SNS
     * message attributes</a> in the Amazon SNS Developer Guide.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMessageAttributes} method.
     * </p>
     * 
     * @return Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html">Amazon SNS message
     *         attributes</a> in the Amazon SNS Developer Guide.
     */
    public final Map<String, MessageAttributeValue> messageAttributes() {
        return messageAttributes;
    }

    /**
     * <p>
     * This parameter applies only to FIFO (first-in-first-out) topics.
     * </p>
     * <p>
     * The token used for deduplication of messages within a 5-minute minimum deduplication interval. If a message with
     * a particular <code>MessageDeduplicationId</code> is sent successfully, subsequent messages with the same
     * <code>MessageDeduplicationId</code> are accepted successfully but aren't delivered.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Every message must have a unique <code>MessageDeduplicationId</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * You may provide a <code>MessageDeduplicationId</code> explicitly.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
     * <code>ContentBasedDeduplication</code> for your topic, Amazon SNS uses a SHA-256 hash to generate the
     * <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the message).
     * </p>
     * </li>
     * <li>
     * <p>
     * If you don't provide a <code>MessageDeduplicationId</code> and the topic doesn't have
     * <code>ContentBasedDeduplication</code> set, the action fails with an error.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the topic has a <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code> overrides
     * the generated one.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within the
     * deduplication interval are treated as duplicates and only one copy of the message is delivered.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you send one message with <code>ContentBasedDeduplication</code> enabled, and then another message with a
     * <code>MessageDeduplicationId</code> that is the same as the one generated for the first
     * <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of the message
     * is delivered.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>MessageDeduplicationId</code> is available to the consumer of the message (this can be useful for
     * troubleshooting delivery issues).
     * </p>
     * <p>
     * If a message is sent successfully but the acknowledgement is lost and the message is resent with the same
     * <code>MessageDeduplicationId</code> after the deduplication interval, Amazon SNS can't detect duplicate messages.
     * </p>
     * <p>
     * Amazon SNS continues to keep track of the message deduplication ID even after the message is received and
     * deleted.
     * </p>
     * </note>
     * <p>
     * The length of <code>MessageDeduplicationId</code> is 128 characters.
     * </p>
     * <p>
     * <code>MessageDeduplicationId</code> can contain alphanumeric characters <code>(a-z, A-Z, 0-9)</code> and
     * punctuation <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
     * </p>
     * 
     * @return This parameter applies only to FIFO (first-in-first-out) topics.</p>
     *         <p>
     *         The token used for deduplication of messages within a 5-minute minimum deduplication interval. If a
     *         message with a particular <code>MessageDeduplicationId</code> is sent successfully, subsequent messages
     *         with the same <code>MessageDeduplicationId</code> are accepted successfully but aren't delivered.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Every message must have a unique <code>MessageDeduplicationId</code>.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You may provide a <code>MessageDeduplicationId</code> explicitly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
     *         <code>ContentBasedDeduplication</code> for your topic, Amazon SNS uses a SHA-256 hash to generate the
     *         <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the
     *         message).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you don't provide a <code>MessageDeduplicationId</code> and the topic doesn't have
     *         <code>ContentBasedDeduplication</code> set, the action fails with an error.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the topic has a <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code>
     *         overrides the generated one.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within the
     *         deduplication interval are treated as duplicates and only one copy of the message is delivered.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you send one message with <code>ContentBasedDeduplication</code> enabled, and then another message
     *         with a <code>MessageDeduplicationId</code> that is the same as the one generated for the first
     *         <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of the
     *         message is delivered.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The <code>MessageDeduplicationId</code> is available to the consumer of the message (this can be useful
     *         for troubleshooting delivery issues).
     *         </p>
     *         <p>
     *         If a message is sent successfully but the acknowledgement is lost and the message is resent with the same
     *         <code>MessageDeduplicationId</code> after the deduplication interval, Amazon SNS can't detect duplicate
     *         messages.
     *         </p>
     *         <p>
     *         Amazon SNS continues to keep track of the message deduplication ID even after the message is received and
     *         deleted.
     *         </p>
     *         </note>
     *         <p>
     *         The length of <code>MessageDeduplicationId</code> is 128 characters.
     *         </p>
     *         <p>
     *         <code>MessageDeduplicationId</code> can contain alphanumeric characters <code>(a-z, A-Z, 0-9)</code> and
     *         punctuation <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
     */
    public final String messageDeduplicationId() {
        return messageDeduplicationId;
    }

    /**
     * <p>
     * This parameter applies only to FIFO (first-in-first-out) topics.
     * </p>
     * <p>
     * The tag that specifies that a message belongs to a specific message group. Messages that belong to the same
     * message group are processed in a FIFO manner (however, messages in different message groups might be processed
     * out of order). To interleave multiple ordered streams within a single topic, use <code>MessageGroupId</code>
     * values (for example, session data for multiple users). In this scenario, multiple consumers can process the
     * topic, but the session data of each user is processed in a FIFO fashion.
     * </p>
     * <p>
     * You must associate a non-empty <code>MessageGroupId</code> with a message. If you don't provide a
     * <code>MessageGroupId</code>, the action fails.
     * </p>
     * <p>
     * The length of <code>MessageGroupId</code> is 128 characters.
     * </p>
     * <p>
     * <code>MessageGroupId</code> can contain alphanumeric characters <code>(a-z, A-Z, 0-9)</code> and punctuation
     * <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
     * </p>
     * <important>
     * <p>
     * <code>MessageGroupId</code> is required for FIFO topics. You can't use it for standard topics.
     * </p>
     * </important>
     * 
     * @return This parameter applies only to FIFO (first-in-first-out) topics.</p>
     *         <p>
     *         The tag that specifies that a message belongs to a specific message group. Messages that belong to the
     *         same message group are processed in a FIFO manner (however, messages in different message groups might be
     *         processed out of order). To interleave multiple ordered streams within a single topic, use
     *         <code>MessageGroupId</code> values (for example, session data for multiple users). In this scenario,
     *         multiple consumers can process the topic, but the session data of each user is processed in a FIFO
     *         fashion.
     *         </p>
     *         <p>
     *         You must associate a non-empty <code>MessageGroupId</code> with a message. If you don't provide a
     *         <code>MessageGroupId</code>, the action fails.
     *         </p>
     *         <p>
     *         The length of <code>MessageGroupId</code> is 128 characters.
     *         </p>
     *         <p>
     *         <code>MessageGroupId</code> can contain alphanumeric characters <code>(a-z, A-Z, 0-9)</code> and
     *         punctuation <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
     *         </p>
     *         <important>
     *         <p>
     *         <code>MessageGroupId</code> is required for FIFO topics. You can't use it for standard topics.
     *         </p>
     */
    public final String messageGroupId() {
        return messageGroupId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(subject());
        hashCode = 31 * hashCode + Objects.hashCode(messageStructure());
        hashCode = 31 * hashCode + Objects.hashCode(hasMessageAttributes() ? messageAttributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(messageDeduplicationId());
        hashCode = 31 * hashCode + Objects.hashCode(messageGroupId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PublishBatchRequestEntry)) {
            return false;
        }
        PublishBatchRequestEntry other = (PublishBatchRequestEntry) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(message(), other.message())
                && Objects.equals(subject(), other.subject()) && Objects.equals(messageStructure(), other.messageStructure())
                && hasMessageAttributes() == other.hasMessageAttributes()
                && Objects.equals(messageAttributes(), other.messageAttributes())
                && Objects.equals(messageDeduplicationId(), other.messageDeduplicationId())
                && Objects.equals(messageGroupId(), other.messageGroupId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PublishBatchRequestEntry").add("Id", id()).add("Message", message()).add("Subject", subject())
                .add("MessageStructure", messageStructure())
                .add("MessageAttributes", hasMessageAttributes() ? messageAttributes() : null)
                .add("MessageDeduplicationId", messageDeduplicationId()).add("MessageGroupId", messageGroupId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Message":
            return Optional.ofNullable(clazz.cast(message()));
        case "Subject":
            return Optional.ofNullable(clazz.cast(subject()));
        case "MessageStructure":
            return Optional.ofNullable(clazz.cast(messageStructure()));
        case "MessageAttributes":
            return Optional.ofNullable(clazz.cast(messageAttributes()));
        case "MessageDeduplicationId":
            return Optional.ofNullable(clazz.cast(messageDeduplicationId()));
        case "MessageGroupId":
            return Optional.ofNullable(clazz.cast(messageGroupId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PublishBatchRequestEntry, T> g) {
        return obj -> g.apply((PublishBatchRequestEntry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PublishBatchRequestEntry> {
        /**
         * <p>
         * An identifier for the message in this batch.
         * </p>
         * <note>
         * <p>
         * The <code>Ids</code> of a batch request must be unique within a request.
         * </p>
         * <p>
         * This identifier can have up to 80 characters. The following characters are accepted: alphanumeric characters,
         * hyphens(-), and underscores (_).
         * </p>
         * </note>
         * 
         * @param id
         *        An identifier for the message in this batch.</p> <note>
         *        <p>
         *        The <code>Ids</code> of a batch request must be unique within a request.
         *        </p>
         *        <p>
         *        This identifier can have up to 80 characters. The following characters are accepted: alphanumeric
         *        characters, hyphens(-), and underscores (_).
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The body of the message.
         * </p>
         * 
         * @param message
         *        The body of the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * The subject of the batch message.
         * </p>
         * 
         * @param subject
         *        The subject of the batch message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subject(String subject);

        /**
         * <p>
         * Set <code>MessageStructure</code> to <code>json</code> if you want to send a different message for each
         * protocol. For example, using one publish action, you can send a short message to your SMS subscribers and a
         * longer message to your email subscribers. If you set <code>MessageStructure</code> to <code>json</code>, the
         * value of the <code>Message</code> parameter must:
         * </p>
         * <ul>
         * <li>
         * <p>
         * be a syntactically valid JSON object; and
         * </p>
         * </li>
         * <li>
         * <p>
         * contain at least a top-level JSON key of "default" with a value that is a string.
         * </p>
         * </li>
         * </ul>
         * <p>
         * You can define other top-level keys that define the message you want to send to a specific transport protocol
         * (e.g. http).
         * </p>
         * 
         * @param messageStructure
         *        Set <code>MessageStructure</code> to <code>json</code> if you want to send a different message for
         *        each protocol. For example, using one publish action, you can send a short message to your SMS
         *        subscribers and a longer message to your email subscribers. If you set <code>MessageStructure</code>
         *        to <code>json</code>, the value of the <code>Message</code> parameter must: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        be a syntactically valid JSON object; and
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        contain at least a top-level JSON key of "default" with a value that is a string.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        You can define other top-level keys that define the message you want to send to a specific transport
         *        protocol (e.g. http).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageStructure(String messageStructure);

        /**
         * <p>
         * Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For more
         * information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html">Amazon SNS
         * message attributes</a> in the Amazon SNS Developer Guide.
         * </p>
         * 
         * @param messageAttributes
         *        Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html">Amazon SNS message
         *        attributes</a> in the Amazon SNS Developer Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageAttributes(Map<String, MessageAttributeValue> messageAttributes);

        /**
         * <p>
         * This parameter applies only to FIFO (first-in-first-out) topics.
         * </p>
         * <p>
         * The token used for deduplication of messages within a 5-minute minimum deduplication interval. If a message
         * with a particular <code>MessageDeduplicationId</code> is sent successfully, subsequent messages with the same
         * <code>MessageDeduplicationId</code> are accepted successfully but aren't delivered.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Every message must have a unique <code>MessageDeduplicationId</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * You may provide a <code>MessageDeduplicationId</code> explicitly.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
         * <code>ContentBasedDeduplication</code> for your topic, Amazon SNS uses a SHA-256 hash to generate the
         * <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the message).
         * </p>
         * </li>
         * <li>
         * <p>
         * If you don't provide a <code>MessageDeduplicationId</code> and the topic doesn't have
         * <code>ContentBasedDeduplication</code> set, the action fails with an error.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the topic has a <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code>
         * overrides the generated one.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within the
         * deduplication interval are treated as duplicates and only one copy of the message is delivered.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you send one message with <code>ContentBasedDeduplication</code> enabled, and then another message with a
         * <code>MessageDeduplicationId</code> that is the same as the one generated for the first
         * <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of the
         * message is delivered.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The <code>MessageDeduplicationId</code> is available to the consumer of the message (this can be useful for
         * troubleshooting delivery issues).
         * </p>
         * <p>
         * If a message is sent successfully but the acknowledgement is lost and the message is resent with the same
         * <code>MessageDeduplicationId</code> after the deduplication interval, Amazon SNS can't detect duplicate
         * messages.
         * </p>
         * <p>
         * Amazon SNS continues to keep track of the message deduplication ID even after the message is received and
         * deleted.
         * </p>
         * </note>
         * <p>
         * The length of <code>MessageDeduplicationId</code> is 128 characters.
         * </p>
         * <p>
         * <code>MessageDeduplicationId</code> can contain alphanumeric characters <code>(a-z, A-Z, 0-9)</code> and
         * punctuation <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
         * </p>
         * 
         * @param messageDeduplicationId
         *        This parameter applies only to FIFO (first-in-first-out) topics.</p>
         *        <p>
         *        The token used for deduplication of messages within a 5-minute minimum deduplication interval. If a
         *        message with a particular <code>MessageDeduplicationId</code> is sent successfully, subsequent
         *        messages with the same <code>MessageDeduplicationId</code> are accepted successfully but aren't
         *        delivered.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Every message must have a unique <code>MessageDeduplicationId</code>.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        You may provide a <code>MessageDeduplicationId</code> explicitly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
         *        <code>ContentBasedDeduplication</code> for your topic, Amazon SNS uses a SHA-256 hash to generate the
         *        <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the
         *        message).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you don't provide a <code>MessageDeduplicationId</code> and the topic doesn't have
         *        <code>ContentBasedDeduplication</code> set, the action fails with an error.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the topic has a <code>ContentBasedDeduplication</code> set, your
         *        <code>MessageDeduplicationId</code> overrides the generated one.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within
         *        the deduplication interval are treated as duplicates and only one copy of the message is delivered.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you send one message with <code>ContentBasedDeduplication</code> enabled, and then another message
         *        with a <code>MessageDeduplicationId</code> that is the same as the one generated for the first
         *        <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of
         *        the message is delivered.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The <code>MessageDeduplicationId</code> is available to the consumer of the message (this can be
         *        useful for troubleshooting delivery issues).
         *        </p>
         *        <p>
         *        If a message is sent successfully but the acknowledgement is lost and the message is resent with the
         *        same <code>MessageDeduplicationId</code> after the deduplication interval, Amazon SNS can't detect
         *        duplicate messages.
         *        </p>
         *        <p>
         *        Amazon SNS continues to keep track of the message deduplication ID even after the message is received
         *        and deleted.
         *        </p>
         *        </note>
         *        <p>
         *        The length of <code>MessageDeduplicationId</code> is 128 characters.
         *        </p>
         *        <p>
         *        <code>MessageDeduplicationId</code> can contain alphanumeric characters <code>(a-z, A-Z, 0-9)</code>
         *        and punctuation <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageDeduplicationId(String messageDeduplicationId);

        /**
         * <p>
         * This parameter applies only to FIFO (first-in-first-out) topics.
         * </p>
         * <p>
         * The tag that specifies that a message belongs to a specific message group. Messages that belong to the same
         * message group are processed in a FIFO manner (however, messages in different message groups might be
         * processed out of order). To interleave multiple ordered streams within a single topic, use
         * <code>MessageGroupId</code> values (for example, session data for multiple users). In this scenario, multiple
         * consumers can process the topic, but the session data of each user is processed in a FIFO fashion.
         * </p>
         * <p>
         * You must associate a non-empty <code>MessageGroupId</code> with a message. If you don't provide a
         * <code>MessageGroupId</code>, the action fails.
         * </p>
         * <p>
         * The length of <code>MessageGroupId</code> is 128 characters.
         * </p>
         * <p>
         * <code>MessageGroupId</code> can contain alphanumeric characters <code>(a-z, A-Z, 0-9)</code> and punctuation
         * <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
         * </p>
         * <important>
         * <p>
         * <code>MessageGroupId</code> is required for FIFO topics. You can't use it for standard topics.
         * </p>
         * </important>
         * 
         * @param messageGroupId
         *        This parameter applies only to FIFO (first-in-first-out) topics.</p>
         *        <p>
         *        The tag that specifies that a message belongs to a specific message group. Messages that belong to the
         *        same message group are processed in a FIFO manner (however, messages in different message groups might
         *        be processed out of order). To interleave multiple ordered streams within a single topic, use
         *        <code>MessageGroupId</code> values (for example, session data for multiple users). In this scenario,
         *        multiple consumers can process the topic, but the session data of each user is processed in a FIFO
         *        fashion.
         *        </p>
         *        <p>
         *        You must associate a non-empty <code>MessageGroupId</code> with a message. If you don't provide a
         *        <code>MessageGroupId</code>, the action fails.
         *        </p>
         *        <p>
         *        The length of <code>MessageGroupId</code> is 128 characters.
         *        </p>
         *        <p>
         *        <code>MessageGroupId</code> can contain alphanumeric characters <code>(a-z, A-Z, 0-9)</code> and
         *        punctuation <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
         *        </p>
         *        <important>
         *        <p>
         *        <code>MessageGroupId</code> is required for FIFO topics. You can't use it for standard topics.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageGroupId(String messageGroupId);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String message;

        private String subject;

        private String messageStructure;

        private Map<String, MessageAttributeValue> messageAttributes = DefaultSdkAutoConstructMap.getInstance();

        private String messageDeduplicationId;

        private String messageGroupId;

        private BuilderImpl() {
        }

        private BuilderImpl(PublishBatchRequestEntry model) {
            id(model.id);
            message(model.message);
            subject(model.subject);
            messageStructure(model.messageStructure);
            messageAttributes(model.messageAttributes);
            messageDeduplicationId(model.messageDeduplicationId);
            messageGroupId(model.messageGroupId);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getMessage() {
            return message;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final String getSubject() {
            return subject;
        }

        public final void setSubject(String subject) {
            this.subject = subject;
        }

        @Override
        public final Builder subject(String subject) {
            this.subject = subject;
            return this;
        }

        public final String getMessageStructure() {
            return messageStructure;
        }

        public final void setMessageStructure(String messageStructure) {
            this.messageStructure = messageStructure;
        }

        @Override
        public final Builder messageStructure(String messageStructure) {
            this.messageStructure = messageStructure;
            return this;
        }

        public final Map<String, MessageAttributeValue.Builder> getMessageAttributes() {
            Map<String, MessageAttributeValue.Builder> result = MessageAttributeMapCopier.copyToBuilder(this.messageAttributes);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setMessageAttributes(Map<String, MessageAttributeValue.BuilderImpl> messageAttributes) {
            this.messageAttributes = MessageAttributeMapCopier.copyFromBuilder(messageAttributes);
        }

        @Override
        public final Builder messageAttributes(Map<String, MessageAttributeValue> messageAttributes) {
            this.messageAttributes = MessageAttributeMapCopier.copy(messageAttributes);
            return this;
        }

        public final String getMessageDeduplicationId() {
            return messageDeduplicationId;
        }

        public final void setMessageDeduplicationId(String messageDeduplicationId) {
            this.messageDeduplicationId = messageDeduplicationId;
        }

        @Override
        public final Builder messageDeduplicationId(String messageDeduplicationId) {
            this.messageDeduplicationId = messageDeduplicationId;
            return this;
        }

        public final String getMessageGroupId() {
            return messageGroupId;
        }

        public final void setMessageGroupId(String messageGroupId) {
            this.messageGroupId = messageGroupId;
        }

        @Override
        public final Builder messageGroupId(String messageGroupId) {
            this.messageGroupId = messageGroupId;
            return this;
        }

        @Override
        public PublishBatchRequestEntry build() {
            return new PublishBatchRequestEntry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
