/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.snowball.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Each <code>LongTermPricingListEntry</code> object contains information about a long-term pricing type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LongTermPricingListEntry implements SdkPojo, Serializable,
        ToCopyableBuilder<LongTermPricingListEntry.Builder, LongTermPricingListEntry> {
    private static final SdkField<String> LONG_TERM_PRICING_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LongTermPricingId").getter(getter(LongTermPricingListEntry::longTermPricingId))
            .setter(setter(Builder::longTermPricingId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LongTermPricingId").build()).build();

    private static final SdkField<Instant> LONG_TERM_PRICING_END_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LongTermPricingEndDate").getter(getter(LongTermPricingListEntry::longTermPricingEndDate))
            .setter(setter(Builder::longTermPricingEndDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LongTermPricingEndDate").build())
            .build();

    private static final SdkField<Instant> LONG_TERM_PRICING_START_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT).memberName("LongTermPricingStartDate")
            .getter(getter(LongTermPricingListEntry::longTermPricingStartDate)).setter(setter(Builder::longTermPricingStartDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LongTermPricingStartDate").build())
            .build();

    private static final SdkField<String> LONG_TERM_PRICING_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LongTermPricingType").getter(getter(LongTermPricingListEntry::longTermPricingTypeAsString))
            .setter(setter(Builder::longTermPricingType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LongTermPricingType").build())
            .build();

    private static final SdkField<String> CURRENT_ACTIVE_JOB_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CurrentActiveJob").getter(getter(LongTermPricingListEntry::currentActiveJob))
            .setter(setter(Builder::currentActiveJob))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CurrentActiveJob").build()).build();

    private static final SdkField<String> REPLACEMENT_JOB_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReplacementJob").getter(getter(LongTermPricingListEntry::replacementJob))
            .setter(setter(Builder::replacementJob))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplacementJob").build()).build();

    private static final SdkField<Boolean> IS_LONG_TERM_PRICING_AUTO_RENEW_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsLongTermPricingAutoRenew")
            .getter(getter(LongTermPricingListEntry::isLongTermPricingAutoRenew))
            .setter(setter(Builder::isLongTermPricingAutoRenew))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsLongTermPricingAutoRenew").build())
            .build();

    private static final SdkField<String> LONG_TERM_PRICING_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LongTermPricingStatus").getter(getter(LongTermPricingListEntry::longTermPricingStatus))
            .setter(setter(Builder::longTermPricingStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LongTermPricingStatus").build())
            .build();

    private static final SdkField<String> SNOWBALL_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SnowballType").getter(getter(LongTermPricingListEntry::snowballTypeAsString))
            .setter(setter(Builder::snowballType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnowballType").build()).build();

    private static final SdkField<List<String>> JOB_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("JobIds")
            .getter(getter(LongTermPricingListEntry::jobIds))
            .setter(setter(Builder::jobIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LONG_TERM_PRICING_ID_FIELD,
            LONG_TERM_PRICING_END_DATE_FIELD, LONG_TERM_PRICING_START_DATE_FIELD, LONG_TERM_PRICING_TYPE_FIELD,
            CURRENT_ACTIVE_JOB_FIELD, REPLACEMENT_JOB_FIELD, IS_LONG_TERM_PRICING_AUTO_RENEW_FIELD,
            LONG_TERM_PRICING_STATUS_FIELD, SNOWBALL_TYPE_FIELD, JOB_IDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String longTermPricingId;

    private final Instant longTermPricingEndDate;

    private final Instant longTermPricingStartDate;

    private final String longTermPricingType;

    private final String currentActiveJob;

    private final String replacementJob;

    private final Boolean isLongTermPricingAutoRenew;

    private final String longTermPricingStatus;

    private final String snowballType;

    private final List<String> jobIds;

    private LongTermPricingListEntry(BuilderImpl builder) {
        this.longTermPricingId = builder.longTermPricingId;
        this.longTermPricingEndDate = builder.longTermPricingEndDate;
        this.longTermPricingStartDate = builder.longTermPricingStartDate;
        this.longTermPricingType = builder.longTermPricingType;
        this.currentActiveJob = builder.currentActiveJob;
        this.replacementJob = builder.replacementJob;
        this.isLongTermPricingAutoRenew = builder.isLongTermPricingAutoRenew;
        this.longTermPricingStatus = builder.longTermPricingStatus;
        this.snowballType = builder.snowballType;
        this.jobIds = builder.jobIds;
    }

    /**
     * <p>
     * The ID of the long-term pricing type for the device.
     * </p>
     * 
     * @return The ID of the long-term pricing type for the device.
     */
    public final String longTermPricingId() {
        return longTermPricingId;
    }

    /**
     * <p>
     * The end date the long-term pricing contract.
     * </p>
     * 
     * @return The end date the long-term pricing contract.
     */
    public final Instant longTermPricingEndDate() {
        return longTermPricingEndDate;
    }

    /**
     * <p>
     * The start date of the long-term pricing contract.
     * </p>
     * 
     * @return The start date of the long-term pricing contract.
     */
    public final Instant longTermPricingStartDate() {
        return longTermPricingStartDate;
    }

    /**
     * <p>
     * The type of long-term pricing that was selected for the device.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #longTermPricingType} will return {@link LongTermPricingType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #longTermPricingTypeAsString}.
     * </p>
     * 
     * @return The type of long-term pricing that was selected for the device.
     * @see LongTermPricingType
     */
    public final LongTermPricingType longTermPricingType() {
        return LongTermPricingType.fromValue(longTermPricingType);
    }

    /**
     * <p>
     * The type of long-term pricing that was selected for the device.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #longTermPricingType} will return {@link LongTermPricingType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #longTermPricingTypeAsString}.
     * </p>
     * 
     * @return The type of long-term pricing that was selected for the device.
     * @see LongTermPricingType
     */
    public final String longTermPricingTypeAsString() {
        return longTermPricingType;
    }

    /**
     * <p>
     * The current active jobs on the device the long-term pricing type.
     * </p>
     * 
     * @return The current active jobs on the device the long-term pricing type.
     */
    public final String currentActiveJob() {
        return currentActiveJob;
    }

    /**
     * <p>
     * A new device that replaces a device that is ordered with long-term pricing.
     * </p>
     * 
     * @return A new device that replaces a device that is ordered with long-term pricing.
     */
    public final String replacementJob() {
        return replacementJob;
    }

    /**
     * <p>
     * If set to <code>true</code>, specifies that the current long-term pricing type for the device should be
     * automatically renewed before the long-term pricing contract expires.
     * </p>
     * 
     * @return If set to <code>true</code>, specifies that the current long-term pricing type for the device should be
     *         automatically renewed before the long-term pricing contract expires.
     */
    public final Boolean isLongTermPricingAutoRenew() {
        return isLongTermPricingAutoRenew;
    }

    /**
     * <p>
     * The status of the long-term pricing type.
     * </p>
     * 
     * @return The status of the long-term pricing type.
     */
    public final String longTermPricingStatus() {
        return longTermPricingStatus;
    }

    /**
     * <p>
     * The type of Snow Family devices associated with this long-term pricing job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #snowballType} will
     * return {@link SnowballType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #snowballTypeAsString}.
     * </p>
     * 
     * @return The type of Snow Family devices associated with this long-term pricing job.
     * @see SnowballType
     */
    public final SnowballType snowballType() {
        return SnowballType.fromValue(snowballType);
    }

    /**
     * <p>
     * The type of Snow Family devices associated with this long-term pricing job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #snowballType} will
     * return {@link SnowballType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #snowballTypeAsString}.
     * </p>
     * 
     * @return The type of Snow Family devices associated with this long-term pricing job.
     * @see SnowballType
     */
    public final String snowballTypeAsString() {
        return snowballType;
    }

    /**
     * For responses, this returns true if the service returned a value for the JobIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasJobIds() {
        return jobIds != null && !(jobIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the jobs that are associated with a long-term pricing type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasJobIds} method.
     * </p>
     * 
     * @return The IDs of the jobs that are associated with a long-term pricing type.
     */
    public final List<String> jobIds() {
        return jobIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(longTermPricingId());
        hashCode = 31 * hashCode + Objects.hashCode(longTermPricingEndDate());
        hashCode = 31 * hashCode + Objects.hashCode(longTermPricingStartDate());
        hashCode = 31 * hashCode + Objects.hashCode(longTermPricingTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(currentActiveJob());
        hashCode = 31 * hashCode + Objects.hashCode(replacementJob());
        hashCode = 31 * hashCode + Objects.hashCode(isLongTermPricingAutoRenew());
        hashCode = 31 * hashCode + Objects.hashCode(longTermPricingStatus());
        hashCode = 31 * hashCode + Objects.hashCode(snowballTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasJobIds() ? jobIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LongTermPricingListEntry)) {
            return false;
        }
        LongTermPricingListEntry other = (LongTermPricingListEntry) obj;
        return Objects.equals(longTermPricingId(), other.longTermPricingId())
                && Objects.equals(longTermPricingEndDate(), other.longTermPricingEndDate())
                && Objects.equals(longTermPricingStartDate(), other.longTermPricingStartDate())
                && Objects.equals(longTermPricingTypeAsString(), other.longTermPricingTypeAsString())
                && Objects.equals(currentActiveJob(), other.currentActiveJob())
                && Objects.equals(replacementJob(), other.replacementJob())
                && Objects.equals(isLongTermPricingAutoRenew(), other.isLongTermPricingAutoRenew())
                && Objects.equals(longTermPricingStatus(), other.longTermPricingStatus())
                && Objects.equals(snowballTypeAsString(), other.snowballTypeAsString()) && hasJobIds() == other.hasJobIds()
                && Objects.equals(jobIds(), other.jobIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LongTermPricingListEntry").add("LongTermPricingId", longTermPricingId())
                .add("LongTermPricingEndDate", longTermPricingEndDate())
                .add("LongTermPricingStartDate", longTermPricingStartDate())
                .add("LongTermPricingType", longTermPricingTypeAsString()).add("CurrentActiveJob", currentActiveJob())
                .add("ReplacementJob", replacementJob()).add("IsLongTermPricingAutoRenew", isLongTermPricingAutoRenew())
                .add("LongTermPricingStatus", longTermPricingStatus()).add("SnowballType", snowballTypeAsString())
                .add("JobIds", hasJobIds() ? jobIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LongTermPricingId":
            return Optional.ofNullable(clazz.cast(longTermPricingId()));
        case "LongTermPricingEndDate":
            return Optional.ofNullable(clazz.cast(longTermPricingEndDate()));
        case "LongTermPricingStartDate":
            return Optional.ofNullable(clazz.cast(longTermPricingStartDate()));
        case "LongTermPricingType":
            return Optional.ofNullable(clazz.cast(longTermPricingTypeAsString()));
        case "CurrentActiveJob":
            return Optional.ofNullable(clazz.cast(currentActiveJob()));
        case "ReplacementJob":
            return Optional.ofNullable(clazz.cast(replacementJob()));
        case "IsLongTermPricingAutoRenew":
            return Optional.ofNullable(clazz.cast(isLongTermPricingAutoRenew()));
        case "LongTermPricingStatus":
            return Optional.ofNullable(clazz.cast(longTermPricingStatus()));
        case "SnowballType":
            return Optional.ofNullable(clazz.cast(snowballTypeAsString()));
        case "JobIds":
            return Optional.ofNullable(clazz.cast(jobIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LongTermPricingListEntry, T> g) {
        return obj -> g.apply((LongTermPricingListEntry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LongTermPricingListEntry> {
        /**
         * <p>
         * The ID of the long-term pricing type for the device.
         * </p>
         * 
         * @param longTermPricingId
         *        The ID of the long-term pricing type for the device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder longTermPricingId(String longTermPricingId);

        /**
         * <p>
         * The end date the long-term pricing contract.
         * </p>
         * 
         * @param longTermPricingEndDate
         *        The end date the long-term pricing contract.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder longTermPricingEndDate(Instant longTermPricingEndDate);

        /**
         * <p>
         * The start date of the long-term pricing contract.
         * </p>
         * 
         * @param longTermPricingStartDate
         *        The start date of the long-term pricing contract.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder longTermPricingStartDate(Instant longTermPricingStartDate);

        /**
         * <p>
         * The type of long-term pricing that was selected for the device.
         * </p>
         * 
         * @param longTermPricingType
         *        The type of long-term pricing that was selected for the device.
         * @see LongTermPricingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LongTermPricingType
         */
        Builder longTermPricingType(String longTermPricingType);

        /**
         * <p>
         * The type of long-term pricing that was selected for the device.
         * </p>
         * 
         * @param longTermPricingType
         *        The type of long-term pricing that was selected for the device.
         * @see LongTermPricingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LongTermPricingType
         */
        Builder longTermPricingType(LongTermPricingType longTermPricingType);

        /**
         * <p>
         * The current active jobs on the device the long-term pricing type.
         * </p>
         * 
         * @param currentActiveJob
         *        The current active jobs on the device the long-term pricing type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentActiveJob(String currentActiveJob);

        /**
         * <p>
         * A new device that replaces a device that is ordered with long-term pricing.
         * </p>
         * 
         * @param replacementJob
         *        A new device that replaces a device that is ordered with long-term pricing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replacementJob(String replacementJob);

        /**
         * <p>
         * If set to <code>true</code>, specifies that the current long-term pricing type for the device should be
         * automatically renewed before the long-term pricing contract expires.
         * </p>
         * 
         * @param isLongTermPricingAutoRenew
         *        If set to <code>true</code>, specifies that the current long-term pricing type for the device should
         *        be automatically renewed before the long-term pricing contract expires.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isLongTermPricingAutoRenew(Boolean isLongTermPricingAutoRenew);

        /**
         * <p>
         * The status of the long-term pricing type.
         * </p>
         * 
         * @param longTermPricingStatus
         *        The status of the long-term pricing type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder longTermPricingStatus(String longTermPricingStatus);

        /**
         * <p>
         * The type of Snow Family devices associated with this long-term pricing job.
         * </p>
         * 
         * @param snowballType
         *        The type of Snow Family devices associated with this long-term pricing job.
         * @see SnowballType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SnowballType
         */
        Builder snowballType(String snowballType);

        /**
         * <p>
         * The type of Snow Family devices associated with this long-term pricing job.
         * </p>
         * 
         * @param snowballType
         *        The type of Snow Family devices associated with this long-term pricing job.
         * @see SnowballType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SnowballType
         */
        Builder snowballType(SnowballType snowballType);

        /**
         * <p>
         * The IDs of the jobs that are associated with a long-term pricing type.
         * </p>
         * 
         * @param jobIds
         *        The IDs of the jobs that are associated with a long-term pricing type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobIds(Collection<String> jobIds);

        /**
         * <p>
         * The IDs of the jobs that are associated with a long-term pricing type.
         * </p>
         * 
         * @param jobIds
         *        The IDs of the jobs that are associated with a long-term pricing type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobIds(String... jobIds);
    }

    static final class BuilderImpl implements Builder {
        private String longTermPricingId;

        private Instant longTermPricingEndDate;

        private Instant longTermPricingStartDate;

        private String longTermPricingType;

        private String currentActiveJob;

        private String replacementJob;

        private Boolean isLongTermPricingAutoRenew;

        private String longTermPricingStatus;

        private String snowballType;

        private List<String> jobIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LongTermPricingListEntry model) {
            longTermPricingId(model.longTermPricingId);
            longTermPricingEndDate(model.longTermPricingEndDate);
            longTermPricingStartDate(model.longTermPricingStartDate);
            longTermPricingType(model.longTermPricingType);
            currentActiveJob(model.currentActiveJob);
            replacementJob(model.replacementJob);
            isLongTermPricingAutoRenew(model.isLongTermPricingAutoRenew);
            longTermPricingStatus(model.longTermPricingStatus);
            snowballType(model.snowballType);
            jobIds(model.jobIds);
        }

        public final String getLongTermPricingId() {
            return longTermPricingId;
        }

        public final void setLongTermPricingId(String longTermPricingId) {
            this.longTermPricingId = longTermPricingId;
        }

        @Override
        public final Builder longTermPricingId(String longTermPricingId) {
            this.longTermPricingId = longTermPricingId;
            return this;
        }

        public final Instant getLongTermPricingEndDate() {
            return longTermPricingEndDate;
        }

        public final void setLongTermPricingEndDate(Instant longTermPricingEndDate) {
            this.longTermPricingEndDate = longTermPricingEndDate;
        }

        @Override
        public final Builder longTermPricingEndDate(Instant longTermPricingEndDate) {
            this.longTermPricingEndDate = longTermPricingEndDate;
            return this;
        }

        public final Instant getLongTermPricingStartDate() {
            return longTermPricingStartDate;
        }

        public final void setLongTermPricingStartDate(Instant longTermPricingStartDate) {
            this.longTermPricingStartDate = longTermPricingStartDate;
        }

        @Override
        public final Builder longTermPricingStartDate(Instant longTermPricingStartDate) {
            this.longTermPricingStartDate = longTermPricingStartDate;
            return this;
        }

        public final String getLongTermPricingType() {
            return longTermPricingType;
        }

        public final void setLongTermPricingType(String longTermPricingType) {
            this.longTermPricingType = longTermPricingType;
        }

        @Override
        public final Builder longTermPricingType(String longTermPricingType) {
            this.longTermPricingType = longTermPricingType;
            return this;
        }

        @Override
        public final Builder longTermPricingType(LongTermPricingType longTermPricingType) {
            this.longTermPricingType(longTermPricingType == null ? null : longTermPricingType.toString());
            return this;
        }

        public final String getCurrentActiveJob() {
            return currentActiveJob;
        }

        public final void setCurrentActiveJob(String currentActiveJob) {
            this.currentActiveJob = currentActiveJob;
        }

        @Override
        public final Builder currentActiveJob(String currentActiveJob) {
            this.currentActiveJob = currentActiveJob;
            return this;
        }

        public final String getReplacementJob() {
            return replacementJob;
        }

        public final void setReplacementJob(String replacementJob) {
            this.replacementJob = replacementJob;
        }

        @Override
        public final Builder replacementJob(String replacementJob) {
            this.replacementJob = replacementJob;
            return this;
        }

        public final Boolean getIsLongTermPricingAutoRenew() {
            return isLongTermPricingAutoRenew;
        }

        public final void setIsLongTermPricingAutoRenew(Boolean isLongTermPricingAutoRenew) {
            this.isLongTermPricingAutoRenew = isLongTermPricingAutoRenew;
        }

        @Override
        public final Builder isLongTermPricingAutoRenew(Boolean isLongTermPricingAutoRenew) {
            this.isLongTermPricingAutoRenew = isLongTermPricingAutoRenew;
            return this;
        }

        public final String getLongTermPricingStatus() {
            return longTermPricingStatus;
        }

        public final void setLongTermPricingStatus(String longTermPricingStatus) {
            this.longTermPricingStatus = longTermPricingStatus;
        }

        @Override
        public final Builder longTermPricingStatus(String longTermPricingStatus) {
            this.longTermPricingStatus = longTermPricingStatus;
            return this;
        }

        public final String getSnowballType() {
            return snowballType;
        }

        public final void setSnowballType(String snowballType) {
            this.snowballType = snowballType;
        }

        @Override
        public final Builder snowballType(String snowballType) {
            this.snowballType = snowballType;
            return this;
        }

        @Override
        public final Builder snowballType(SnowballType snowballType) {
            this.snowballType(snowballType == null ? null : snowballType.toString());
            return this;
        }

        public final Collection<String> getJobIds() {
            if (jobIds instanceof SdkAutoConstructList) {
                return null;
            }
            return jobIds;
        }

        public final void setJobIds(Collection<String> jobIds) {
            this.jobIds = LongTermPricingAssociatedJobIdListCopier.copy(jobIds);
        }

        @Override
        public final Builder jobIds(Collection<String> jobIds) {
            this.jobIds = LongTermPricingAssociatedJobIdListCopier.copy(jobIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder jobIds(String... jobIds) {
            jobIds(Arrays.asList(jobIds));
            return this;
        }

        @Override
        public LongTermPricingListEntry build() {
            return new LongTermPricingListEntry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
