/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sms;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.sms.model.CreateAppRequest;
import software.amazon.awssdk.services.sms.model.CreateAppResponse;
import software.amazon.awssdk.services.sms.model.CreateReplicationJobRequest;
import software.amazon.awssdk.services.sms.model.CreateReplicationJobResponse;
import software.amazon.awssdk.services.sms.model.DeleteAppLaunchConfigurationRequest;
import software.amazon.awssdk.services.sms.model.DeleteAppLaunchConfigurationResponse;
import software.amazon.awssdk.services.sms.model.DeleteAppReplicationConfigurationRequest;
import software.amazon.awssdk.services.sms.model.DeleteAppReplicationConfigurationResponse;
import software.amazon.awssdk.services.sms.model.DeleteAppRequest;
import software.amazon.awssdk.services.sms.model.DeleteAppResponse;
import software.amazon.awssdk.services.sms.model.DeleteReplicationJobRequest;
import software.amazon.awssdk.services.sms.model.DeleteReplicationJobResponse;
import software.amazon.awssdk.services.sms.model.DeleteServerCatalogRequest;
import software.amazon.awssdk.services.sms.model.DeleteServerCatalogResponse;
import software.amazon.awssdk.services.sms.model.DisassociateConnectorRequest;
import software.amazon.awssdk.services.sms.model.DisassociateConnectorResponse;
import software.amazon.awssdk.services.sms.model.GenerateChangeSetRequest;
import software.amazon.awssdk.services.sms.model.GenerateChangeSetResponse;
import software.amazon.awssdk.services.sms.model.GenerateTemplateRequest;
import software.amazon.awssdk.services.sms.model.GenerateTemplateResponse;
import software.amazon.awssdk.services.sms.model.GetAppLaunchConfigurationRequest;
import software.amazon.awssdk.services.sms.model.GetAppLaunchConfigurationResponse;
import software.amazon.awssdk.services.sms.model.GetAppReplicationConfigurationRequest;
import software.amazon.awssdk.services.sms.model.GetAppReplicationConfigurationResponse;
import software.amazon.awssdk.services.sms.model.GetAppRequest;
import software.amazon.awssdk.services.sms.model.GetAppResponse;
import software.amazon.awssdk.services.sms.model.GetConnectorsRequest;
import software.amazon.awssdk.services.sms.model.GetConnectorsResponse;
import software.amazon.awssdk.services.sms.model.GetReplicationJobsRequest;
import software.amazon.awssdk.services.sms.model.GetReplicationJobsResponse;
import software.amazon.awssdk.services.sms.model.GetReplicationRunsRequest;
import software.amazon.awssdk.services.sms.model.GetReplicationRunsResponse;
import software.amazon.awssdk.services.sms.model.GetServersRequest;
import software.amazon.awssdk.services.sms.model.GetServersResponse;
import software.amazon.awssdk.services.sms.model.ImportServerCatalogRequest;
import software.amazon.awssdk.services.sms.model.ImportServerCatalogResponse;
import software.amazon.awssdk.services.sms.model.InternalErrorException;
import software.amazon.awssdk.services.sms.model.InvalidParameterException;
import software.amazon.awssdk.services.sms.model.LaunchAppRequest;
import software.amazon.awssdk.services.sms.model.LaunchAppResponse;
import software.amazon.awssdk.services.sms.model.ListAppsRequest;
import software.amazon.awssdk.services.sms.model.ListAppsResponse;
import software.amazon.awssdk.services.sms.model.MissingRequiredParameterException;
import software.amazon.awssdk.services.sms.model.NoConnectorsAvailableException;
import software.amazon.awssdk.services.sms.model.OperationNotPermittedException;
import software.amazon.awssdk.services.sms.model.PutAppLaunchConfigurationRequest;
import software.amazon.awssdk.services.sms.model.PutAppLaunchConfigurationResponse;
import software.amazon.awssdk.services.sms.model.PutAppReplicationConfigurationRequest;
import software.amazon.awssdk.services.sms.model.PutAppReplicationConfigurationResponse;
import software.amazon.awssdk.services.sms.model.ReplicationJobAlreadyExistsException;
import software.amazon.awssdk.services.sms.model.ReplicationJobNotFoundException;
import software.amazon.awssdk.services.sms.model.ReplicationRunLimitExceededException;
import software.amazon.awssdk.services.sms.model.ServerCannotBeReplicatedException;
import software.amazon.awssdk.services.sms.model.SmsException;
import software.amazon.awssdk.services.sms.model.SmsRequest;
import software.amazon.awssdk.services.sms.model.StartAppReplicationRequest;
import software.amazon.awssdk.services.sms.model.StartAppReplicationResponse;
import software.amazon.awssdk.services.sms.model.StartOnDemandReplicationRunRequest;
import software.amazon.awssdk.services.sms.model.StartOnDemandReplicationRunResponse;
import software.amazon.awssdk.services.sms.model.StopAppReplicationRequest;
import software.amazon.awssdk.services.sms.model.StopAppReplicationResponse;
import software.amazon.awssdk.services.sms.model.TemporarilyUnavailableException;
import software.amazon.awssdk.services.sms.model.TerminateAppRequest;
import software.amazon.awssdk.services.sms.model.TerminateAppResponse;
import software.amazon.awssdk.services.sms.model.UnauthorizedOperationException;
import software.amazon.awssdk.services.sms.model.UpdateAppRequest;
import software.amazon.awssdk.services.sms.model.UpdateAppResponse;
import software.amazon.awssdk.services.sms.model.UpdateReplicationJobRequest;
import software.amazon.awssdk.services.sms.model.UpdateReplicationJobResponse;
import software.amazon.awssdk.services.sms.paginators.GetConnectorsPublisher;
import software.amazon.awssdk.services.sms.paginators.GetReplicationJobsPublisher;
import software.amazon.awssdk.services.sms.paginators.GetReplicationRunsPublisher;
import software.amazon.awssdk.services.sms.paginators.GetServersPublisher;
import software.amazon.awssdk.services.sms.transform.CreateAppRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.CreateReplicationJobRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.DeleteAppLaunchConfigurationRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.DeleteAppReplicationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.DeleteAppRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.DeleteReplicationJobRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.DeleteServerCatalogRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.DisassociateConnectorRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.GenerateChangeSetRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.GenerateTemplateRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.GetAppLaunchConfigurationRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.GetAppReplicationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.GetAppRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.GetConnectorsRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.GetReplicationJobsRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.GetReplicationRunsRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.GetServersRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.ImportServerCatalogRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.LaunchAppRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.ListAppsRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.PutAppLaunchConfigurationRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.PutAppReplicationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.StartAppReplicationRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.StartOnDemandReplicationRunRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.StopAppReplicationRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.TerminateAppRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.UpdateAppRequestMarshaller;
import software.amazon.awssdk.services.sms.transform.UpdateReplicationJobRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SmsAsyncClient}.
 *
 * @see SmsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSmsAsyncClient implements SmsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSmsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSmsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an application. An application consists of one or more server groups. Each server group contain one or
     * more servers.
     * </p>
     *
     * @param createAppRequest
     * @return A Java Future containing the result of the CreateApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.CreateApp
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/CreateApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAppResponse> createApp(CreateAppRequest createAppRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAppResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAppRequest, CreateAppResponse>().withOperationName("CreateApp")
                            .withMarshaller(new CreateAppRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createAppRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a replication job. The replication job schedules periodic replication runs to replicate your server to
     * AWS. Each replication run creates an Amazon Machine Image (AMI).
     * </p>
     *
     * @param createReplicationJobRequest
     * @return A Java Future containing the result of the CreateReplicationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>ServerCannotBeReplicatedException The specified server cannot be replicated.</li>
     *         <li>ReplicationJobAlreadyExistsException The specified replication job already exists.</li>
     *         <li>NoConnectorsAvailableException There are no connectors available.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>TemporarilyUnavailableException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.CreateReplicationJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/CreateReplicationJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateReplicationJobResponse> createReplicationJob(
            CreateReplicationJobRequest createReplicationJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateReplicationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateReplicationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateReplicationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateReplicationJobRequest, CreateReplicationJobResponse>()
                            .withOperationName("CreateReplicationJob")
                            .withMarshaller(new CreateReplicationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createReplicationJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing application. Optionally deletes the launched stack associated with the application and all
     * AWS SMS replication jobs for servers in the application.
     * </p>
     *
     * @param deleteAppRequest
     * @return A Java Future containing the result of the DeleteApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.DeleteApp
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/DeleteApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppResponse> deleteApp(DeleteAppRequest deleteAppRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAppResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppRequest, DeleteAppResponse>().withOperationName("DeleteApp")
                            .withMarshaller(new DeleteAppRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteAppRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes existing launch configuration for an application.
     * </p>
     *
     * @param deleteAppLaunchConfigurationRequest
     * @return A Java Future containing the result of the DeleteAppLaunchConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.DeleteAppLaunchConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/DeleteAppLaunchConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppLaunchConfigurationResponse> deleteAppLaunchConfiguration(
            DeleteAppLaunchConfigurationRequest deleteAppLaunchConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppLaunchConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAppLaunchConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppLaunchConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppLaunchConfigurationRequest, DeleteAppLaunchConfigurationResponse>()
                            .withOperationName("DeleteAppLaunchConfiguration")
                            .withMarshaller(new DeleteAppLaunchConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteAppLaunchConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes existing replication configuration for an application.
     * </p>
     *
     * @param deleteAppReplicationConfigurationRequest
     * @return A Java Future containing the result of the DeleteAppReplicationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.DeleteAppReplicationConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/DeleteAppReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppReplicationConfigurationResponse> deleteAppReplicationConfiguration(
            DeleteAppReplicationConfigurationRequest deleteAppReplicationConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppReplicationConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteAppReplicationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppReplicationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppReplicationConfigurationRequest, DeleteAppReplicationConfigurationResponse>()
                            .withOperationName("DeleteAppReplicationConfiguration")
                            .withMarshaller(new DeleteAppReplicationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteAppReplicationConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified replication job.
     * </p>
     * <p>
     * After you delete a replication job, there are no further replication runs. AWS deletes the contents of the Amazon
     * S3 bucket used to store AWS SMS artifacts. The AMIs created by the replication runs are not deleted.
     * </p>
     *
     * @param deleteReplicationJobRequest
     * @return A Java Future containing the result of the DeleteReplicationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>ReplicationJobNotFoundException The specified replication job does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.DeleteReplicationJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/DeleteReplicationJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteReplicationJobResponse> deleteReplicationJob(
            DeleteReplicationJobRequest deleteReplicationJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteReplicationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteReplicationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteReplicationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteReplicationJobRequest, DeleteReplicationJobResponse>()
                            .withOperationName("DeleteReplicationJob")
                            .withMarshaller(new DeleteReplicationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteReplicationJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes all servers from your server catalog.
     * </p>
     *
     * @param deleteServerCatalogRequest
     * @return A Java Future containing the result of the DeleteServerCatalog operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.DeleteServerCatalog
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/DeleteServerCatalog" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteServerCatalogResponse> deleteServerCatalog(
            DeleteServerCatalogRequest deleteServerCatalogRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteServerCatalogResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteServerCatalogResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteServerCatalogResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteServerCatalogRequest, DeleteServerCatalogResponse>()
                            .withOperationName("DeleteServerCatalog")
                            .withMarshaller(new DeleteServerCatalogRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteServerCatalogRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates the specified connector from AWS SMS.
     * </p>
     * <p>
     * After you disassociate a connector, it is no longer available to support replication jobs.
     * </p>
     *
     * @param disassociateConnectorRequest
     * @return A Java Future containing the result of the DisassociateConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.DisassociateConnector
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/DisassociateConnector" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateConnectorResponse> disassociateConnector(
            DisassociateConnectorRequest disassociateConnectorRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateConnectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateConnectorRequest, DisassociateConnectorResponse>()
                            .withOperationName("DisassociateConnector")
                            .withMarshaller(new DisassociateConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disassociateConnectorRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates a target change set for a currently launched stack and writes it to an Amazon S3 object in the
     * customer’s Amazon S3 bucket.
     * </p>
     *
     * @param generateChangeSetRequest
     * @return A Java Future containing the result of the GenerateChangeSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.GenerateChangeSet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/GenerateChangeSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GenerateChangeSetResponse> generateChangeSet(GenerateChangeSetRequest generateChangeSetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GenerateChangeSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GenerateChangeSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GenerateChangeSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GenerateChangeSetRequest, GenerateChangeSetResponse>()
                            .withOperationName("GenerateChangeSet")
                            .withMarshaller(new GenerateChangeSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(generateChangeSetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates an Amazon CloudFormation template based on the current launch configuration and writes it to an Amazon
     * S3 object in the customer’s Amazon S3 bucket.
     * </p>
     *
     * @param generateTemplateRequest
     * @return A Java Future containing the result of the GenerateTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.GenerateTemplate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/GenerateTemplate" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GenerateTemplateResponse> generateTemplate(GenerateTemplateRequest generateTemplateRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GenerateTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GenerateTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GenerateTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GenerateTemplateRequest, GenerateTemplateResponse>()
                            .withOperationName("GenerateTemplate")
                            .withMarshaller(new GenerateTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(generateTemplateRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve information about an application.
     * </p>
     *
     * @param getAppRequest
     * @return A Java Future containing the result of the GetApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.GetApp
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/GetApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAppResponse> getApp(GetAppRequest getAppRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAppResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAppRequest, GetAppResponse>().withOperationName("GetApp")
                            .withMarshaller(new GetAppRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getAppRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the application launch configuration associated with an application.
     * </p>
     *
     * @param getAppLaunchConfigurationRequest
     * @return A Java Future containing the result of the GetAppLaunchConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.GetAppLaunchConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/GetAppLaunchConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAppLaunchConfigurationResponse> getAppLaunchConfiguration(
            GetAppLaunchConfigurationRequest getAppLaunchConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAppLaunchConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAppLaunchConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAppLaunchConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAppLaunchConfigurationRequest, GetAppLaunchConfigurationResponse>()
                            .withOperationName("GetAppLaunchConfiguration")
                            .withMarshaller(new GetAppLaunchConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getAppLaunchConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves an application replication configuration associatd with an application.
     * </p>
     *
     * @param getAppReplicationConfigurationRequest
     * @return A Java Future containing the result of the GetAppReplicationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.GetAppReplicationConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/GetAppReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAppReplicationConfigurationResponse> getAppReplicationConfiguration(
            GetAppReplicationConfigurationRequest getAppReplicationConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAppReplicationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAppReplicationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAppReplicationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAppReplicationConfigurationRequest, GetAppReplicationConfigurationResponse>()
                            .withOperationName("GetAppReplicationConfiguration")
                            .withMarshaller(new GetAppReplicationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getAppReplicationConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the connectors registered with the AWS SMS.
     * </p>
     *
     * @param getConnectorsRequest
     * @return A Java Future containing the result of the GetConnectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.GetConnectors
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/GetConnectors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetConnectorsResponse> getConnectors(GetConnectorsRequest getConnectorsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConnectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetConnectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConnectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConnectorsRequest, GetConnectorsResponse>()
                            .withOperationName("GetConnectors")
                            .withMarshaller(new GetConnectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getConnectorsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the connectors registered with the AWS SMS.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getConnectors(software.amazon.awssdk.services.sms.model.GetConnectorsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sms.paginators.GetConnectorsPublisher publisher = client.getConnectorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sms.paginators.GetConnectorsPublisher publisher = client.getConnectorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sms.model.GetConnectorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sms.model.GetConnectorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getConnectors(software.amazon.awssdk.services.sms.model.GetConnectorsRequest)} operation.</b>
     * </p>
     *
     * @param getConnectorsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.GetConnectors
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/GetConnectors" target="_top">AWS API
     *      Documentation</a>
     */
    public GetConnectorsPublisher getConnectorsPaginator(GetConnectorsRequest getConnectorsRequest) {
        return new GetConnectorsPublisher(this, applyPaginatorUserAgent(getConnectorsRequest));
    }

    /**
     * <p>
     * Describes the specified replication job or all of your replication jobs.
     * </p>
     *
     * @param getReplicationJobsRequest
     * @return A Java Future containing the result of the GetReplicationJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.GetReplicationJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/GetReplicationJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetReplicationJobsResponse> getReplicationJobs(GetReplicationJobsRequest getReplicationJobsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetReplicationJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetReplicationJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetReplicationJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetReplicationJobsRequest, GetReplicationJobsResponse>()
                            .withOperationName("GetReplicationJobs")
                            .withMarshaller(new GetReplicationJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getReplicationJobsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified replication job or all of your replication jobs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getReplicationJobs(software.amazon.awssdk.services.sms.model.GetReplicationJobsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sms.paginators.GetReplicationJobsPublisher publisher = client.getReplicationJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sms.paginators.GetReplicationJobsPublisher publisher = client.getReplicationJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sms.model.GetReplicationJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sms.model.GetReplicationJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getReplicationJobs(software.amazon.awssdk.services.sms.model.GetReplicationJobsRequest)} operation.</b>
     * </p>
     *
     * @param getReplicationJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.GetReplicationJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/GetReplicationJobs" target="_top">AWS API
     *      Documentation</a>
     */
    public GetReplicationJobsPublisher getReplicationJobsPaginator(GetReplicationJobsRequest getReplicationJobsRequest) {
        return new GetReplicationJobsPublisher(this, applyPaginatorUserAgent(getReplicationJobsRequest));
    }

    /**
     * <p>
     * Describes the replication runs for the specified replication job.
     * </p>
     *
     * @param getReplicationRunsRequest
     * @return A Java Future containing the result of the GetReplicationRuns operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.GetReplicationRuns
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/GetReplicationRuns" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetReplicationRunsResponse> getReplicationRuns(GetReplicationRunsRequest getReplicationRunsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetReplicationRunsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetReplicationRunsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetReplicationRunsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetReplicationRunsRequest, GetReplicationRunsResponse>()
                            .withOperationName("GetReplicationRuns")
                            .withMarshaller(new GetReplicationRunsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getReplicationRunsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the replication runs for the specified replication job.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getReplicationRuns(software.amazon.awssdk.services.sms.model.GetReplicationRunsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sms.paginators.GetReplicationRunsPublisher publisher = client.getReplicationRunsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sms.paginators.GetReplicationRunsPublisher publisher = client.getReplicationRunsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sms.model.GetReplicationRunsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sms.model.GetReplicationRunsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getReplicationRuns(software.amazon.awssdk.services.sms.model.GetReplicationRunsRequest)} operation.</b>
     * </p>
     *
     * @param getReplicationRunsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.GetReplicationRuns
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/GetReplicationRuns" target="_top">AWS API
     *      Documentation</a>
     */
    public GetReplicationRunsPublisher getReplicationRunsPaginator(GetReplicationRunsRequest getReplicationRunsRequest) {
        return new GetReplicationRunsPublisher(this, applyPaginatorUserAgent(getReplicationRunsRequest));
    }

    /**
     * <p>
     * Describes the servers in your server catalog.
     * </p>
     * <p>
     * Before you can describe your servers, you must import them using <a>ImportServerCatalog</a>.
     * </p>
     *
     * @param getServersRequest
     * @return A Java Future containing the result of the GetServers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.GetServers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/GetServers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetServersResponse> getServers(GetServersRequest getServersRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetServersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServersRequest, GetServersResponse>().withOperationName("GetServers")
                            .withMarshaller(new GetServersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getServersRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the servers in your server catalog.
     * </p>
     * <p>
     * Before you can describe your servers, you must import them using <a>ImportServerCatalog</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getServers(software.amazon.awssdk.services.sms.model.GetServersRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sms.paginators.GetServersPublisher publisher = client.getServersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sms.paginators.GetServersPublisher publisher = client.getServersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sms.model.GetServersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sms.model.GetServersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getServers(software.amazon.awssdk.services.sms.model.GetServersRequest)} operation.</b>
     * </p>
     *
     * @param getServersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.GetServers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/GetServers" target="_top">AWS API
     *      Documentation</a>
     */
    public GetServersPublisher getServersPaginator(GetServersRequest getServersRequest) {
        return new GetServersPublisher(this, applyPaginatorUserAgent(getServersRequest));
    }

    /**
     * <p>
     * Gathers a complete list of on-premises servers. Connectors must be installed and monitoring all servers that you
     * want to import.
     * </p>
     * <p>
     * This call returns immediately, but might take additional time to retrieve all the servers.
     * </p>
     *
     * @param importServerCatalogRequest
     * @return A Java Future containing the result of the ImportServerCatalog operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>NoConnectorsAvailableException There are no connectors available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.ImportServerCatalog
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/ImportServerCatalog" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ImportServerCatalogResponse> importServerCatalog(
            ImportServerCatalogRequest importServerCatalogRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportServerCatalogResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ImportServerCatalogResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ImportServerCatalogResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportServerCatalogRequest, ImportServerCatalogResponse>()
                            .withOperationName("ImportServerCatalog")
                            .withMarshaller(new ImportServerCatalogRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(importServerCatalogRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Launches an application stack.
     * </p>
     *
     * @param launchAppRequest
     * @return A Java Future containing the result of the LaunchApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.LaunchApp
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/LaunchApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<LaunchAppResponse> launchApp(LaunchAppRequest launchAppRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<LaunchAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    LaunchAppResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<LaunchAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<LaunchAppRequest, LaunchAppResponse>().withOperationName("LaunchApp")
                            .withMarshaller(new LaunchAppRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(launchAppRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of summaries for all applications.
     * </p>
     *
     * @param listAppsRequest
     * @return A Java Future containing the result of the ListApps operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.ListApps
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/ListApps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppsResponse> listApps(ListAppsRequest listAppsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAppsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppsRequest, ListAppsResponse>().withOperationName("ListApps")
                            .withMarshaller(new ListAppsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listAppsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a launch configuration for an application.
     * </p>
     *
     * @param putAppLaunchConfigurationRequest
     * @return A Java Future containing the result of the PutAppLaunchConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.PutAppLaunchConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/PutAppLaunchConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAppLaunchConfigurationResponse> putAppLaunchConfiguration(
            PutAppLaunchConfigurationRequest putAppLaunchConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAppLaunchConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutAppLaunchConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAppLaunchConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAppLaunchConfigurationRequest, PutAppLaunchConfigurationResponse>()
                            .withOperationName("PutAppLaunchConfiguration")
                            .withMarshaller(new PutAppLaunchConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putAppLaunchConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates a replication configuration for an application.
     * </p>
     *
     * @param putAppReplicationConfigurationRequest
     * @return A Java Future containing the result of the PutAppReplicationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.PutAppReplicationConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/PutAppReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAppReplicationConfigurationResponse> putAppReplicationConfiguration(
            PutAppReplicationConfigurationRequest putAppReplicationConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAppReplicationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutAppReplicationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAppReplicationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAppReplicationConfigurationRequest, PutAppReplicationConfigurationResponse>()
                            .withOperationName("PutAppReplicationConfiguration")
                            .withMarshaller(new PutAppReplicationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putAppReplicationConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts replicating an application.
     * </p>
     *
     * @param startAppReplicationRequest
     * @return A Java Future containing the result of the StartAppReplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.StartAppReplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/StartAppReplication" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartAppReplicationResponse> startAppReplication(
            StartAppReplicationRequest startAppReplicationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartAppReplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartAppReplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartAppReplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartAppReplicationRequest, StartAppReplicationResponse>()
                            .withOperationName("StartAppReplication")
                            .withMarshaller(new StartAppReplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(startAppReplicationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an on-demand replication run for the specified replication job. This replication run starts immediately.
     * This replication run is in addition to the ones already scheduled.
     * </p>
     * <p>
     * There is a limit on the number of on-demand replications runs you can request in a 24-hour period.
     * </p>
     *
     * @param startOnDemandReplicationRunRequest
     * @return A Java Future containing the result of the StartOnDemandReplicationRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>ReplicationRunLimitExceededException You have exceeded the number of on-demand replication runs you
     *         can request in a 24-hour period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.StartOnDemandReplicationRun
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/StartOnDemandReplicationRun"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartOnDemandReplicationRunResponse> startOnDemandReplicationRun(
            StartOnDemandReplicationRunRequest startOnDemandReplicationRunRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartOnDemandReplicationRunResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartOnDemandReplicationRunResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartOnDemandReplicationRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartOnDemandReplicationRunRequest, StartOnDemandReplicationRunResponse>()
                            .withOperationName("StartOnDemandReplicationRun")
                            .withMarshaller(new StartOnDemandReplicationRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(startOnDemandReplicationRunRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops replicating an application.
     * </p>
     *
     * @param stopAppReplicationRequest
     * @return A Java Future containing the result of the StopAppReplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.StopAppReplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/StopAppReplication" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopAppReplicationResponse> stopAppReplication(StopAppReplicationRequest stopAppReplicationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopAppReplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopAppReplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopAppReplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopAppReplicationRequest, StopAppReplicationResponse>()
                            .withOperationName("StopAppReplication")
                            .withMarshaller(new StopAppReplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(stopAppReplicationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Terminates the stack for an application.
     * </p>
     *
     * @param terminateAppRequest
     * @return A Java Future containing the result of the TerminateApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.TerminateApp
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/TerminateApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TerminateAppResponse> terminateApp(TerminateAppRequest terminateAppRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TerminateAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TerminateAppResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TerminateAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TerminateAppRequest, TerminateAppResponse>()
                            .withOperationName("TerminateApp").withMarshaller(new TerminateAppRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(terminateAppRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an application.
     * </p>
     *
     * @param updateAppRequest
     * @return A Java Future containing the result of the UpdateApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.UpdateApp
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/UpdateApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAppResponse> updateApp(UpdateAppRequest updateAppRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateAppResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAppRequest, UpdateAppResponse>().withOperationName("UpdateApp")
                            .withMarshaller(new UpdateAppRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateAppRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified settings for the specified replication job.
     * </p>
     *
     * @param updateReplicationJobRequest
     * @return A Java Future containing the result of the UpdateReplicationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A specified parameter is not valid.</li>
     *         <li>MissingRequiredParameterException A required parameter is missing.</li>
     *         <li>OperationNotPermittedException This operation is not allowed.</li>
     *         <li>UnauthorizedOperationException You lack permissions needed to perform this operation. Check your IAM
     *         policies, and ensure that you are using the correct access keys.</li>
     *         <li>ServerCannotBeReplicatedException The specified server cannot be replicated.</li>
     *         <li>ReplicationJobNotFoundException The specified replication job does not exist.</li>
     *         <li>InternalErrorException An internal error occurred.</li>
     *         <li>TemporarilyUnavailableException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SmsAsyncClient.UpdateReplicationJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sms-2016-10-24/UpdateReplicationJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateReplicationJobResponse> updateReplicationJob(
            UpdateReplicationJobRequest updateReplicationJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateReplicationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateReplicationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateReplicationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateReplicationJobRequest, UpdateReplicationJobResponse>()
                            .withOperationName("UpdateReplicationJob")
                            .withMarshaller(new UpdateReplicationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateReplicationJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SmsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TemporarilyUnavailableException")
                                .exceptionBuilderSupplier(TemporarilyUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingRequiredParameterException")
                                .exceptionBuilderSupplier(MissingRequiredParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReplicationJobNotFoundException")
                                .exceptionBuilderSupplier(ReplicationJobNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationNotPermittedException")
                                .exceptionBuilderSupplier(OperationNotPermittedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalError")
                                .exceptionBuilderSupplier(InternalErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerCannotBeReplicatedException")
                                .exceptionBuilderSupplier(ServerCannotBeReplicatedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReplicationRunLimitExceededException")
                                .exceptionBuilderSupplier(ReplicationRunLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedOperationException")
                                .exceptionBuilderSupplier(UnauthorizedOperationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReplicationJobAlreadyExistsException")
                                .exceptionBuilderSupplier(ReplicationJobAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoConnectorsAvailableException")
                                .exceptionBuilderSupplier(NoConnectorsAvailableException::builder).build());
    }

    private <T extends SmsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
