/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.signer;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.signer.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.signer.internal.SignerServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.signer.model.AccessDeniedException;
import software.amazon.awssdk.services.signer.model.AddProfilePermissionRequest;
import software.amazon.awssdk.services.signer.model.AddProfilePermissionResponse;
import software.amazon.awssdk.services.signer.model.BadRequestException;
import software.amazon.awssdk.services.signer.model.CancelSigningProfileRequest;
import software.amazon.awssdk.services.signer.model.CancelSigningProfileResponse;
import software.amazon.awssdk.services.signer.model.ConflictException;
import software.amazon.awssdk.services.signer.model.DescribeSigningJobRequest;
import software.amazon.awssdk.services.signer.model.DescribeSigningJobResponse;
import software.amazon.awssdk.services.signer.model.GetRevocationStatusRequest;
import software.amazon.awssdk.services.signer.model.GetRevocationStatusResponse;
import software.amazon.awssdk.services.signer.model.GetSigningPlatformRequest;
import software.amazon.awssdk.services.signer.model.GetSigningPlatformResponse;
import software.amazon.awssdk.services.signer.model.GetSigningProfileRequest;
import software.amazon.awssdk.services.signer.model.GetSigningProfileResponse;
import software.amazon.awssdk.services.signer.model.InternalServiceErrorException;
import software.amazon.awssdk.services.signer.model.ListProfilePermissionsRequest;
import software.amazon.awssdk.services.signer.model.ListProfilePermissionsResponse;
import software.amazon.awssdk.services.signer.model.ListSigningJobsRequest;
import software.amazon.awssdk.services.signer.model.ListSigningJobsResponse;
import software.amazon.awssdk.services.signer.model.ListSigningPlatformsRequest;
import software.amazon.awssdk.services.signer.model.ListSigningPlatformsResponse;
import software.amazon.awssdk.services.signer.model.ListSigningProfilesRequest;
import software.amazon.awssdk.services.signer.model.ListSigningProfilesResponse;
import software.amazon.awssdk.services.signer.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.signer.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.signer.model.NotFoundException;
import software.amazon.awssdk.services.signer.model.PutSigningProfileRequest;
import software.amazon.awssdk.services.signer.model.PutSigningProfileResponse;
import software.amazon.awssdk.services.signer.model.RemoveProfilePermissionRequest;
import software.amazon.awssdk.services.signer.model.RemoveProfilePermissionResponse;
import software.amazon.awssdk.services.signer.model.ResourceNotFoundException;
import software.amazon.awssdk.services.signer.model.RevokeSignatureRequest;
import software.amazon.awssdk.services.signer.model.RevokeSignatureResponse;
import software.amazon.awssdk.services.signer.model.RevokeSigningProfileRequest;
import software.amazon.awssdk.services.signer.model.RevokeSigningProfileResponse;
import software.amazon.awssdk.services.signer.model.ServiceLimitExceededException;
import software.amazon.awssdk.services.signer.model.SignPayloadRequest;
import software.amazon.awssdk.services.signer.model.SignPayloadResponse;
import software.amazon.awssdk.services.signer.model.SignerException;
import software.amazon.awssdk.services.signer.model.StartSigningJobRequest;
import software.amazon.awssdk.services.signer.model.StartSigningJobResponse;
import software.amazon.awssdk.services.signer.model.TagResourceRequest;
import software.amazon.awssdk.services.signer.model.TagResourceResponse;
import software.amazon.awssdk.services.signer.model.ThrottlingException;
import software.amazon.awssdk.services.signer.model.TooManyRequestsException;
import software.amazon.awssdk.services.signer.model.UntagResourceRequest;
import software.amazon.awssdk.services.signer.model.UntagResourceResponse;
import software.amazon.awssdk.services.signer.model.ValidationException;
import software.amazon.awssdk.services.signer.transform.AddProfilePermissionRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.CancelSigningProfileRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.DescribeSigningJobRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.GetRevocationStatusRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.GetSigningPlatformRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.GetSigningProfileRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.ListProfilePermissionsRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.ListSigningJobsRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.ListSigningPlatformsRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.ListSigningProfilesRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.PutSigningProfileRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.RemoveProfilePermissionRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.RevokeSignatureRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.RevokeSigningProfileRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.SignPayloadRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.StartSigningJobRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.signer.waiters.SignerWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link SignerClient}.
 *
 * @see SignerClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSignerClient implements SignerClient {
    private static final Logger log = Logger.loggerFor(DefaultSignerClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSignerClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "signer" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Adds cross-account permissions to a signing profile.
     * </p>
     *
     * @param addProfilePermissionRequest
     * @return Result of the AddProfilePermission operation returned by the service.
     * @throws ValidationException
     *         You signing certificate could not be validated.
     * @throws ResourceNotFoundException
     *         A specified resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ServiceLimitExceededException
     *         The client is making a request that exceeds service limits.
     * @throws ConflictException
     *         The resource encountered a conflicting state.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.AddProfilePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/AddProfilePermission" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AddProfilePermissionResponse addProfilePermission(AddProfilePermissionRequest addProfilePermissionRequest)
            throws ValidationException, ResourceNotFoundException, AccessDeniedException, ServiceLimitExceededException,
            ConflictException, TooManyRequestsException, InternalServiceErrorException, AwsServiceException, SdkClientException,
            SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddProfilePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AddProfilePermissionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(addProfilePermissionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addProfilePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddProfilePermission");

            return clientHandler.execute(new ClientExecutionParams<AddProfilePermissionRequest, AddProfilePermissionResponse>()
                    .withOperationName("AddProfilePermission").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(addProfilePermissionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AddProfilePermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Changes the state of an <code>ACTIVE</code> signing profile to <code>CANCELED</code>. A canceled profile is still
     * viewable with the <code>ListSigningProfiles</code> operation, but it cannot perform new signing jobs, and is
     * deleted two years after cancelation.
     * </p>
     *
     * @param cancelSigningProfileRequest
     * @return Result of the CancelSigningProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         A specified resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.CancelSigningProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/CancelSigningProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CancelSigningProfileResponse cancelSigningProfile(CancelSigningProfileRequest cancelSigningProfileRequest)
            throws ResourceNotFoundException, AccessDeniedException, TooManyRequestsException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelSigningProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CancelSigningProfileResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelSigningProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelSigningProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelSigningProfile");

            return clientHandler.execute(new ClientExecutionParams<CancelSigningProfileRequest, CancelSigningProfileResponse>()
                    .withOperationName("CancelSigningProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(cancelSigningProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelSigningProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a specific code signing job. You specify the job by using the <code>jobId</code> value
     * that is returned by the <a>StartSigningJob</a> operation.
     * </p>
     *
     * @param describeSigningJobRequest
     * @return Result of the DescribeSigningJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         A specified resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.DescribeSigningJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/DescribeSigningJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeSigningJobResponse describeSigningJob(DescribeSigningJobRequest describeSigningJobRequest)
            throws ResourceNotFoundException, AccessDeniedException, TooManyRequestsException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSigningJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSigningJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSigningJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSigningJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSigningJob");

            return clientHandler.execute(new ClientExecutionParams<DescribeSigningJobRequest, DescribeSigningJobResponse>()
                    .withOperationName("DescribeSigningJob").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeSigningJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeSigningJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the revocation status of one or more of the signing profile, signing job, and signing certificate.
     * </p>
     *
     * @param getRevocationStatusRequest
     * @return Result of the GetRevocationStatus operation returned by the service.
     * @throws ValidationException
     *         You signing certificate could not be validated.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.GetRevocationStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/GetRevocationStatus" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetRevocationStatusResponse getRevocationStatus(GetRevocationStatusRequest getRevocationStatusRequest)
            throws ValidationException, AccessDeniedException, TooManyRequestsException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRevocationStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRevocationStatusResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRevocationStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRevocationStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRevocationStatus");
            String hostPrefix = "verification.";
            String resolvedHostExpression = "verification.";

            return clientHandler.execute(new ClientExecutionParams<GetRevocationStatusRequest, GetRevocationStatusResponse>()
                    .withOperationName("GetRevocationStatus").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .hostPrefixExpression(resolvedHostExpression).withRequestConfiguration(clientConfiguration)
                    .withInput(getRevocationStatusRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRevocationStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information on a specific signing platform.
     * </p>
     *
     * @param getSigningPlatformRequest
     * @return Result of the GetSigningPlatform operation returned by the service.
     * @throws ResourceNotFoundException
     *         A specified resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.GetSigningPlatform
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/GetSigningPlatform" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSigningPlatformResponse getSigningPlatform(GetSigningPlatformRequest getSigningPlatformRequest)
            throws ResourceNotFoundException, AccessDeniedException, TooManyRequestsException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSigningPlatformResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSigningPlatformResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSigningPlatformRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSigningPlatformRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSigningPlatform");

            return clientHandler.execute(new ClientExecutionParams<GetSigningPlatformRequest, GetSigningPlatformResponse>()
                    .withOperationName("GetSigningPlatform").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getSigningPlatformRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSigningPlatformRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information on a specific signing profile.
     * </p>
     *
     * @param getSigningProfileRequest
     * @return Result of the GetSigningProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         A specified resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.GetSigningProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/GetSigningProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSigningProfileResponse getSigningProfile(GetSigningProfileRequest getSigningProfileRequest)
            throws ResourceNotFoundException, AccessDeniedException, TooManyRequestsException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSigningProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSigningProfileResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSigningProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSigningProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSigningProfile");

            return clientHandler.execute(new ClientExecutionParams<GetSigningProfileRequest, GetSigningProfileResponse>()
                    .withOperationName("GetSigningProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getSigningProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSigningProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the cross-account permissions associated with a signing profile.
     * </p>
     *
     * @param listProfilePermissionsRequest
     * @return Result of the ListProfilePermissions operation returned by the service.
     * @throws ValidationException
     *         You signing certificate could not be validated.
     * @throws ResourceNotFoundException
     *         A specified resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.ListProfilePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/ListProfilePermissions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListProfilePermissionsResponse listProfilePermissions(ListProfilePermissionsRequest listProfilePermissionsRequest)
            throws ValidationException, ResourceNotFoundException, AccessDeniedException, TooManyRequestsException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProfilePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListProfilePermissionsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProfilePermissionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProfilePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProfilePermissions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListProfilePermissionsRequest, ListProfilePermissionsResponse>()
                            .withOperationName("ListProfilePermissions").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listProfilePermissionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListProfilePermissionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all your signing jobs. You can use the <code>maxResults</code> parameter to limit the number of signing
     * jobs that are returned in the response. If additional jobs remain to be listed, AWS Signer returns a
     * <code>nextToken</code> value. Use this value in subsequent calls to <code>ListSigningJobs</code> to fetch the
     * remaining values. You can continue calling <code>ListSigningJobs</code> with your <code>maxResults</code>
     * parameter and with new values that Signer returns in the <code>nextToken</code> parameter until all of your
     * signing jobs have been returned.
     * </p>
     *
     * @param listSigningJobsRequest
     * @return Result of the ListSigningJobs operation returned by the service.
     * @throws ValidationException
     *         You signing certificate could not be validated.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.ListSigningJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/ListSigningJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSigningJobsResponse listSigningJobs(ListSigningJobsRequest listSigningJobsRequest) throws ValidationException,
            AccessDeniedException, TooManyRequestsException, InternalServiceErrorException, AwsServiceException,
            SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSigningJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSigningJobsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSigningJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSigningJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSigningJobs");

            return clientHandler.execute(new ClientExecutionParams<ListSigningJobsRequest, ListSigningJobsResponse>()
                    .withOperationName("ListSigningJobs").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listSigningJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSigningJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all signing platforms available in AWS Signer that match the request parameters. If additional jobs remain
     * to be listed, Signer returns a <code>nextToken</code> value. Use this value in subsequent calls to
     * <code>ListSigningJobs</code> to fetch the remaining values. You can continue calling <code>ListSigningJobs</code>
     * with your <code>maxResults</code> parameter and with new values that Signer returns in the <code>nextToken</code>
     * parameter until all of your signing jobs have been returned.
     * </p>
     *
     * @param listSigningPlatformsRequest
     * @return Result of the ListSigningPlatforms operation returned by the service.
     * @throws ValidationException
     *         You signing certificate could not be validated.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.ListSigningPlatforms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/ListSigningPlatforms" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSigningPlatformsResponse listSigningPlatforms(ListSigningPlatformsRequest listSigningPlatformsRequest)
            throws ValidationException, AccessDeniedException, TooManyRequestsException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSigningPlatformsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSigningPlatformsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSigningPlatformsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSigningPlatformsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSigningPlatforms");

            return clientHandler.execute(new ClientExecutionParams<ListSigningPlatformsRequest, ListSigningPlatformsResponse>()
                    .withOperationName("ListSigningPlatforms").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listSigningPlatformsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSigningPlatformsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all available signing profiles in your AWS account. Returns only profiles with an <code>ACTIVE</code>
     * status unless the <code>includeCanceled</code> request field is set to <code>true</code>. If additional jobs
     * remain to be listed, AWS Signer returns a <code>nextToken</code> value. Use this value in subsequent calls to
     * <code>ListSigningJobs</code> to fetch the remaining values. You can continue calling <code>ListSigningJobs</code>
     * with your <code>maxResults</code> parameter and with new values that Signer returns in the <code>nextToken</code>
     * parameter until all of your signing jobs have been returned.
     * </p>
     *
     * @param listSigningProfilesRequest
     * @return Result of the ListSigningProfiles operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.ListSigningProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/ListSigningProfiles" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSigningProfilesResponse listSigningProfiles(ListSigningProfilesRequest listSigningProfilesRequest)
            throws AccessDeniedException, TooManyRequestsException, InternalServiceErrorException, AwsServiceException,
            SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSigningProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSigningProfilesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSigningProfilesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSigningProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSigningProfiles");

            return clientHandler.execute(new ClientExecutionParams<ListSigningProfilesRequest, ListSigningProfilesResponse>()
                    .withOperationName("ListSigningProfiles").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listSigningProfilesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSigningProfilesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the tags associated with a signing profile resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws BadRequestException
     *         The request contains invalid parameters for the ARN or tags. This exception also occurs when you call a
     *         tagging API on a cancelled signing profile.
     * @throws NotFoundException
     *         The signing profile was not found.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServiceErrorException, BadRequestException, NotFoundException, TooManyRequestsException,
            AwsServiceException, SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a signing profile. A signing profile is a code-signing template that can be used to carry out a
     * pre-defined signing job.
     * </p>
     *
     * @param putSigningProfileRequest
     * @return Result of the PutSigningProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         A specified resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         You signing certificate could not be validated.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.PutSigningProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/PutSigningProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutSigningProfileResponse putSigningProfile(PutSigningProfileRequest putSigningProfileRequest)
            throws ResourceNotFoundException, AccessDeniedException, ValidationException, TooManyRequestsException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutSigningProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutSigningProfileResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putSigningProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putSigningProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutSigningProfile");

            return clientHandler.execute(new ClientExecutionParams<PutSigningProfileRequest, PutSigningProfileResponse>()
                    .withOperationName("PutSigningProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putSigningProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutSigningProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes cross-account permissions from a signing profile.
     * </p>
     *
     * @param removeProfilePermissionRequest
     * @return Result of the RemoveProfilePermission operation returned by the service.
     * @throws ValidationException
     *         You signing certificate could not be validated.
     * @throws ResourceNotFoundException
     *         A specified resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The resource encountered a conflicting state.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.RemoveProfilePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/RemoveProfilePermission"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemoveProfilePermissionResponse removeProfilePermission(RemoveProfilePermissionRequest removeProfilePermissionRequest)
            throws ValidationException, ResourceNotFoundException, AccessDeniedException, ConflictException,
            TooManyRequestsException, InternalServiceErrorException, AwsServiceException, SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveProfilePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveProfilePermissionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(removeProfilePermissionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeProfilePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveProfilePermission");

            return clientHandler
                    .execute(new ClientExecutionParams<RemoveProfilePermissionRequest, RemoveProfilePermissionResponse>()
                            .withOperationName("RemoveProfilePermission").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(removeProfilePermissionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RemoveProfilePermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Changes the state of a signing job to REVOKED. This indicates that the signature is no longer valid.
     * </p>
     *
     * @param revokeSignatureRequest
     * @return Result of the RevokeSignature operation returned by the service.
     * @throws ValidationException
     *         You signing certificate could not be validated.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A specified resource could not be found.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.RevokeSignature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/RevokeSignature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RevokeSignatureResponse revokeSignature(RevokeSignatureRequest revokeSignatureRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, TooManyRequestsException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RevokeSignatureResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RevokeSignatureResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(revokeSignatureRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, revokeSignatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RevokeSignature");

            return clientHandler.execute(new ClientExecutionParams<RevokeSignatureRequest, RevokeSignatureResponse>()
                    .withOperationName("RevokeSignature").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(revokeSignatureRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RevokeSignatureRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Changes the state of a signing profile to REVOKED. This indicates that signatures generated using the signing
     * profile after an effective start date are no longer valid.
     * </p>
     *
     * @param revokeSigningProfileRequest
     * @return Result of the RevokeSigningProfile operation returned by the service.
     * @throws ValidationException
     *         You signing certificate could not be validated.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A specified resource could not be found.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.RevokeSigningProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/RevokeSigningProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RevokeSigningProfileResponse revokeSigningProfile(RevokeSigningProfileRequest revokeSigningProfileRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, TooManyRequestsException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RevokeSigningProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RevokeSigningProfileResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(revokeSigningProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, revokeSigningProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RevokeSigningProfile");

            return clientHandler.execute(new ClientExecutionParams<RevokeSigningProfileRequest, RevokeSigningProfileResponse>()
                    .withOperationName("RevokeSigningProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(revokeSigningProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RevokeSigningProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Signs a binary payload and returns a signature envelope.
     * </p>
     *
     * @param signPayloadRequest
     * @return Result of the SignPayload operation returned by the service.
     * @throws ValidationException
     *         You signing certificate could not be validated.
     * @throws ResourceNotFoundException
     *         A specified resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.SignPayload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/SignPayload" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SignPayloadResponse signPayload(SignPayloadRequest signPayloadRequest) throws ValidationException,
            ResourceNotFoundException, AccessDeniedException, TooManyRequestsException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SignPayloadResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SignPayloadResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(signPayloadRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, signPayloadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SignPayload");

            return clientHandler.execute(new ClientExecutionParams<SignPayloadRequest, SignPayloadResponse>()
                    .withOperationName("SignPayload").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(signPayloadRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SignPayloadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Initiates a signing job to be performed on the code provided. Signing jobs are viewable by the
     * <code>ListSigningJobs</code> operation for two years after they are performed. Note the following requirements:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must create an Amazon S3 source bucket. For more information, see <a
     * href="http://docs.aws.amazon.com/AmazonS3/latest/gsg/CreatingABucket.html">Creating a Bucket</a> in the <i>Amazon
     * S3 Getting Started Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Your S3 source bucket must be version enabled.
     * </p>
     * </li>
     * <li>
     * <p>
     * You must create an S3 destination bucket. AWS Signer uses your S3 destination bucket to write your signed code.
     * </p>
     * </li>
     * <li>
     * <p>
     * You specify the name of the source and destination buckets when calling the <code>StartSigningJob</code>
     * operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * You must ensure the S3 buckets are from the same Region as the signing profile. Cross-Region signing isn't
     * supported.
     * </p>
     * </li>
     * <li>
     * <p>
     * You must also specify a request token that identifies your request to Signer.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can call the <a>DescribeSigningJob</a> and the <a>ListSigningJobs</a> actions after you call
     * <code>StartSigningJob</code>.
     * </p>
     * <p>
     * For a Java example that shows how to use this action, see <a
     * href="https://docs.aws.amazon.com/signer/latest/developerguide/api-startsigningjob.html">StartSigningJob</a>.
     * </p>
     *
     * @param startSigningJobRequest
     * @return Result of the StartSigningJob operation returned by the service.
     * @throws ValidationException
     *         You signing certificate could not be validated.
     * @throws ResourceNotFoundException
     *         A specified resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.</p>
     *         <p>
     *         Instead of this error, <code>TooManyRequestsException</code> should be used.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.
     *         </p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.StartSigningJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/StartSigningJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartSigningJobResponse startSigningJob(StartSigningJobRequest startSigningJobRequest) throws ValidationException,
            ResourceNotFoundException, AccessDeniedException, ThrottlingException, TooManyRequestsException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartSigningJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartSigningJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startSigningJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSigningJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSigningJob");

            return clientHandler.execute(new ClientExecutionParams<StartSigningJobRequest, StartSigningJobResponse>()
                    .withOperationName("StartSigningJob").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startSigningJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartSigningJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more tags to a signing profile. Tags are labels that you can use to identify and organize your AWS
     * resources. Each tag consists of a key and an optional value. To specify the signing profile, use its Amazon
     * Resource Name (ARN). To specify the tag, use a key-value pair.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws BadRequestException
     *         The request contains invalid parameters for the ARN or tags. This exception also occurs when you call a
     *         tagging API on a cancelled signing profile.
     * @throws NotFoundException
     *         The signing profile was not found.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServiceErrorException,
            BadRequestException, NotFoundException, TooManyRequestsException, AwsServiceException, SdkClientException,
            SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from a signing profile. To remove the tags, specify a list of tag keys.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal error occurred.
     * @throws BadRequestException
     *         The request contains invalid parameters for the ARN or tags. This exception also occurs when you call a
     *         tagging API on a cancelled signing profile.
     * @throws NotFoundException
     *         The signing profile was not found.
     * @throws TooManyRequestsException
     *         The allowed number of job-signing requests has been exceeded.</p>
     *         <p>
     *         This error supersedes the error <code>ThrottlingException</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SignerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SignerClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServiceErrorException,
            BadRequestException, NotFoundException, TooManyRequestsException, AwsServiceException, SdkClientException,
            SignerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "InternalServiceErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "signer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create an instance of {@link SignerWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link SignerWaiter}
     */
    @Override
    public SignerWaiter waiter() {
        return SignerWaiter.builder().client(this).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        SignerServiceClientConfigurationBuilder serviceConfigBuilder = new SignerServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(SignerException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final SignerServiceClientConfiguration serviceClientConfiguration() {
        return new SignerServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
