/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sfn.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The <code>LoggingConfiguration</code> data type is used to set CloudWatch Logs options.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LoggingConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<LoggingConfiguration.Builder, LoggingConfiguration> {
    private static final SdkField<String> LEVEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LoggingConfiguration::levelAsString)).setter(setter(Builder::level))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("level").build()).build();

    private static final SdkField<Boolean> INCLUDE_EXECUTION_DATA_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(LoggingConfiguration::includeExecutionData)).setter(setter(Builder::includeExecutionData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("includeExecutionData").build())
            .build();

    private static final SdkField<List<LogDestination>> DESTINATIONS_FIELD = SdkField
            .<List<LogDestination>> builder(MarshallingType.LIST)
            .getter(getter(LoggingConfiguration::destinations))
            .setter(setter(Builder::destinations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("destinations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LogDestination> builder(MarshallingType.SDK_POJO)
                                            .constructor(LogDestination::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LEVEL_FIELD,
            INCLUDE_EXECUTION_DATA_FIELD, DESTINATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String level;

    private final Boolean includeExecutionData;

    private final List<LogDestination> destinations;

    private LoggingConfiguration(BuilderImpl builder) {
        this.level = builder.level;
        this.includeExecutionData = builder.includeExecutionData;
        this.destinations = builder.destinations;
    }

    /**
     * <p>
     * Defines which category of execution history events are logged.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #level} will return
     * {@link LogLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #levelAsString}.
     * </p>
     * 
     * @return Defines which category of execution history events are logged.
     * @see LogLevel
     */
    public LogLevel level() {
        return LogLevel.fromValue(level);
    }

    /**
     * <p>
     * Defines which category of execution history events are logged.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #level} will return
     * {@link LogLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #levelAsString}.
     * </p>
     * 
     * @return Defines which category of execution history events are logged.
     * @see LogLevel
     */
    public String levelAsString() {
        return level;
    }

    /**
     * <p>
     * Determines whether execution data is included in your log. When set to <code>FALSE</code>, data is excluded.
     * </p>
     * 
     * @return Determines whether execution data is included in your log. When set to <code>FALSE</code>, data is
     *         excluded.
     */
    public Boolean includeExecutionData() {
        return includeExecutionData;
    }

    /**
     * Returns true if the Destinations property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDestinations() {
        return destinations != null && !(destinations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describes where your execution history events will be logged. Limited to size 1.
     * Required, if your log level is not set to <code>OFF</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDestinations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of objects that describes where your execution history events will be logged. Limited to size 1.
     *         Required, if your log level is not set to <code>OFF</code>.
     */
    public List<LogDestination> destinations() {
        return destinations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(levelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(includeExecutionData());
        hashCode = 31 * hashCode + Objects.hashCode(destinations());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LoggingConfiguration)) {
            return false;
        }
        LoggingConfiguration other = (LoggingConfiguration) obj;
        return Objects.equals(levelAsString(), other.levelAsString())
                && Objects.equals(includeExecutionData(), other.includeExecutionData())
                && Objects.equals(destinations(), other.destinations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("LoggingConfiguration").add("Level", levelAsString())
                .add("IncludeExecutionData", includeExecutionData()).add("Destinations", destinations()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "level":
            return Optional.ofNullable(clazz.cast(levelAsString()));
        case "includeExecutionData":
            return Optional.ofNullable(clazz.cast(includeExecutionData()));
        case "destinations":
            return Optional.ofNullable(clazz.cast(destinations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LoggingConfiguration, T> g) {
        return obj -> g.apply((LoggingConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LoggingConfiguration> {
        /**
         * <p>
         * Defines which category of execution history events are logged.
         * </p>
         * 
         * @param level
         *        Defines which category of execution history events are logged.
         * @see LogLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogLevel
         */
        Builder level(String level);

        /**
         * <p>
         * Defines which category of execution history events are logged.
         * </p>
         * 
         * @param level
         *        Defines which category of execution history events are logged.
         * @see LogLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogLevel
         */
        Builder level(LogLevel level);

        /**
         * <p>
         * Determines whether execution data is included in your log. When set to <code>FALSE</code>, data is excluded.
         * </p>
         * 
         * @param includeExecutionData
         *        Determines whether execution data is included in your log. When set to <code>FALSE</code>, data is
         *        excluded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeExecutionData(Boolean includeExecutionData);

        /**
         * <p>
         * An array of objects that describes where your execution history events will be logged. Limited to size 1.
         * Required, if your log level is not set to <code>OFF</code>.
         * </p>
         * 
         * @param destinations
         *        An array of objects that describes where your execution history events will be logged. Limited to size
         *        1. Required, if your log level is not set to <code>OFF</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(Collection<LogDestination> destinations);

        /**
         * <p>
         * An array of objects that describes where your execution history events will be logged. Limited to size 1.
         * Required, if your log level is not set to <code>OFF</code>.
         * </p>
         * 
         * @param destinations
         *        An array of objects that describes where your execution history events will be logged. Limited to size
         *        1. Required, if your log level is not set to <code>OFF</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(LogDestination... destinations);

        /**
         * <p>
         * An array of objects that describes where your execution history events will be logged. Limited to size 1.
         * Required, if your log level is not set to <code>OFF</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<LogDestination>.Builder} avoiding the need
         * to create one manually via {@link List<LogDestination>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<LogDestination>.Builder#build()} is called immediately and
         * its result is passed to {@link #destinations(List<LogDestination>)}.
         * 
         * @param destinations
         *        a consumer that will call methods on {@link List<LogDestination>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinations(List<LogDestination>)
         */
        Builder destinations(Consumer<LogDestination.Builder>... destinations);
    }

    static final class BuilderImpl implements Builder {
        private String level;

        private Boolean includeExecutionData;

        private List<LogDestination> destinations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LoggingConfiguration model) {
            level(model.level);
            includeExecutionData(model.includeExecutionData);
            destinations(model.destinations);
        }

        public final String getLevel() {
            return level;
        }

        @Override
        public final Builder level(String level) {
            this.level = level;
            return this;
        }

        @Override
        public final Builder level(LogLevel level) {
            this.level(level == null ? null : level.toString());
            return this;
        }

        public final void setLevel(String level) {
            this.level = level;
        }

        public final Boolean getIncludeExecutionData() {
            return includeExecutionData;
        }

        @Override
        public final Builder includeExecutionData(Boolean includeExecutionData) {
            this.includeExecutionData = includeExecutionData;
            return this;
        }

        public final void setIncludeExecutionData(Boolean includeExecutionData) {
            this.includeExecutionData = includeExecutionData;
        }

        public final Collection<LogDestination.Builder> getDestinations() {
            return destinations != null ? destinations.stream().map(LogDestination::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder destinations(Collection<LogDestination> destinations) {
            this.destinations = LogDestinationListCopier.copy(destinations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(LogDestination... destinations) {
            destinations(Arrays.asList(destinations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(Consumer<LogDestination.Builder>... destinations) {
            destinations(Stream.of(destinations).map(c -> LogDestination.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDestinations(Collection<LogDestination.BuilderImpl> destinations) {
            this.destinations = LogDestinationListCopier.copyFromBuilder(destinations);
        }

        @Override
        public LoggingConfiguration build() {
            return new LoggingConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
