/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to send a single raw email using Amazon SES. For more information, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/dg/send-email-raw.html">Amazon SES Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SendRawEmailRequest extends SesRequest implements
        ToCopyableBuilder<SendRawEmailRequest.Builder, SendRawEmailRequest> {
    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Source")
            .getter(getter(SendRawEmailRequest::source)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Source").build()).build();

    private static final SdkField<List<String>> DESTINATIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Destinations")
            .getter(getter(SendRawEmailRequest::destinations))
            .setter(setter(Builder::destinations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Destinations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<RawMessage> RAW_MESSAGE_FIELD = SdkField.<RawMessage> builder(MarshallingType.SDK_POJO)
            .memberName("RawMessage").getter(getter(SendRawEmailRequest::rawMessage)).setter(setter(Builder::rawMessage))
            .constructor(RawMessage::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RawMessage").build()).build();

    private static final SdkField<String> FROM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FromArn").getter(getter(SendRawEmailRequest::fromArn)).setter(setter(Builder::fromArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FromArn").build()).build();

    private static final SdkField<String> SOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceArn").getter(getter(SendRawEmailRequest::sourceArn)).setter(setter(Builder::sourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceArn").build()).build();

    private static final SdkField<String> RETURN_PATH_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReturnPathArn").getter(getter(SendRawEmailRequest::returnPathArn))
            .setter(setter(Builder::returnPathArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReturnPathArn").build()).build();

    private static final SdkField<List<MessageTag>> TAGS_FIELD = SdkField
            .<List<MessageTag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(SendRawEmailRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MessageTag> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageTag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CONFIGURATION_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigurationSetName").getter(getter(SendRawEmailRequest::configurationSetName))
            .setter(setter(Builder::configurationSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationSetName").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_FIELD,
            DESTINATIONS_FIELD, RAW_MESSAGE_FIELD, FROM_ARN_FIELD, SOURCE_ARN_FIELD, RETURN_PATH_ARN_FIELD, TAGS_FIELD,
            CONFIGURATION_SET_NAME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Source", SOURCE_FIELD);
                    put("Destinations", DESTINATIONS_FIELD);
                    put("RawMessage", RAW_MESSAGE_FIELD);
                    put("FromArn", FROM_ARN_FIELD);
                    put("SourceArn", SOURCE_ARN_FIELD);
                    put("ReturnPathArn", RETURN_PATH_ARN_FIELD);
                    put("Tags", TAGS_FIELD);
                    put("ConfigurationSetName", CONFIGURATION_SET_NAME_FIELD);
                }
            });

    private final String source;

    private final List<String> destinations;

    private final RawMessage rawMessage;

    private final String fromArn;

    private final String sourceArn;

    private final String returnPathArn;

    private final List<MessageTag> tags;

    private final String configurationSetName;

    private SendRawEmailRequest(BuilderImpl builder) {
        super(builder);
        this.source = builder.source;
        this.destinations = builder.destinations;
        this.rawMessage = builder.rawMessage;
        this.fromArn = builder.fromArn;
        this.sourceArn = builder.sourceArn;
        this.returnPathArn = builder.returnPathArn;
        this.tags = builder.tags;
        this.configurationSetName = builder.configurationSetName;
    }

    /**
     * <p>
     * The identity's email address. If you do not provide a value for this parameter, you must specify a "From" address
     * in the raw text of the message. (You can also specify both.)
     * </p>
     * <note>
     * <p>
     * Amazon SES does not support the SMTPUTF8 extension, as described in<a
     * href="https://tools.ietf.org/html/rfc6531">RFC6531</a>. For this reason, the email address string must be 7-bit
     * ASCII. If you want to send to or from email addresses that contain Unicode characters in the domain part of an
     * address, you must encode the domain using Punycode. Punycode is not permitted in the local part of the email
     * address (the part before the @ sign) nor in the "friendly from" name. If you want to use Unicode characters in
     * the "friendly from" name, you must encode the "friendly from" name using MIME encoded-word syntax, as described
     * in <a href="https://docs.aws.amazon.com/ses/latest/dg/send-email-raw.html">Sending raw email using the Amazon SES
     * API</a>. For more information about Punycode, see <a href="http://tools.ietf.org/html/rfc3492">RFC 3492</a>.
     * </p>
     * </note>
     * <p>
     * If you specify the <code>Source</code> parameter and have feedback forwarding enabled, then bounces and
     * complaints are sent to this email address. This takes precedence over any Return-Path header that you might
     * include in the raw text of the message.
     * </p>
     * 
     * @return The identity's email address. If you do not provide a value for this parameter, you must specify a "From"
     *         address in the raw text of the message. (You can also specify both.)</p> <note>
     *         <p>
     *         Amazon SES does not support the SMTPUTF8 extension, as described in<a
     *         href="https://tools.ietf.org/html/rfc6531">RFC6531</a>. For this reason, the email address string must be
     *         7-bit ASCII. If you want to send to or from email addresses that contain Unicode characters in the domain
     *         part of an address, you must encode the domain using Punycode. Punycode is not permitted in the local
     *         part of the email address (the part before the @ sign) nor in the "friendly from" name. If you want to
     *         use Unicode characters in the "friendly from" name, you must encode the "friendly from" name using MIME
     *         encoded-word syntax, as described in <a
     *         href="https://docs.aws.amazon.com/ses/latest/dg/send-email-raw.html">Sending raw email using the Amazon
     *         SES API</a>. For more information about Punycode, see <a href="http://tools.ietf.org/html/rfc3492">RFC
     *         3492</a>.
     *         </p>
     *         </note>
     *         <p>
     *         If you specify the <code>Source</code> parameter and have feedback forwarding enabled, then bounces and
     *         complaints are sent to this email address. This takes precedence over any Return-Path header that you
     *         might include in the raw text of the message.
     */
    public final String source() {
        return source;
    }

    /**
     * For responses, this returns true if the service returned a value for the Destinations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDestinations() {
        return destinations != null && !(destinations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDestinations} method.
     * </p>
     * 
     * @return A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
     */
    public final List<String> destinations() {
        return destinations;
    }

    /**
     * <p>
     * The raw email message itself. The message has to meet the following criteria:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The message has to contain a header and a body, separated by a blank line.
     * </p>
     * </li>
     * <li>
     * <p>
     * All of the required header fields must be present in the message.
     * </p>
     * </li>
     * <li>
     * <p>
     * Each part of a multipart MIME message must be formatted properly.
     * </p>
     * </li>
     * <li>
     * <p>
     * Attachments must be of a content type that Amazon SES supports. For a list on unsupported content types, see <a
     * href="https://docs.aws.amazon.com/ses/latest/dg/mime-types.html">Unsupported Attachment Types</a> in the
     * <i>Amazon SES Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The entire message must be base64-encoded.
     * </p>
     * </li>
     * <li>
     * <p>
     * If any of the MIME parts in your message contain content that is outside of the 7-bit ASCII character range, we
     * highly recommend that you encode that content. For more information, see <a
     * href="https://docs.aws.amazon.com/ses/latest/dg/send-email-raw.html">Sending Raw Email</a> in the <i>Amazon SES
     * Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Per <a href="https://tools.ietf.org/html/rfc5321#section-4.5.3.1.6">RFC 5321</a>, the maximum length of each line
     * of text, including the &lt;CRLF&gt;, must not exceed 1,000 characters.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The raw email message itself. The message has to meet the following criteria:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The message has to contain a header and a body, separated by a blank line.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         All of the required header fields must be present in the message.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Each part of a multipart MIME message must be formatted properly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Attachments must be of a content type that Amazon SES supports. For a list on unsupported content types,
     *         see <a href="https://docs.aws.amazon.com/ses/latest/dg/mime-types.html">Unsupported Attachment Types</a>
     *         in the <i>Amazon SES Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The entire message must be base64-encoded.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If any of the MIME parts in your message contain content that is outside of the 7-bit ASCII character
     *         range, we highly recommend that you encode that content. For more information, see <a
     *         href="https://docs.aws.amazon.com/ses/latest/dg/send-email-raw.html">Sending Raw Email</a> in the
     *         <i>Amazon SES Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Per <a href="https://tools.ietf.org/html/rfc5321#section-4.5.3.1.6">RFC 5321</a>, the maximum length of
     *         each line of text, including the &lt;CRLF&gt;, must not exceed 1,000 characters.
     *         </p>
     *         </li>
     */
    public final RawMessage rawMessage() {
        return rawMessage;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to specify a particular "From" address in the header of the raw
     * email.
     * </p>
     * <p>
     * Instead of using this parameter, you can use the X-header <code>X-SES-FROM-ARN</code> in the raw message of the
     * email. If you use both the <code>FromArn</code> parameter and the corresponding X-header, Amazon SES uses the
     * value of the <code>FromArn</code> parameter.
     * </p>
     * <note>
     * <p>
     * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this guide,
     * or see the <a
     * href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization-delegate-sender-tasks-email.html">Amazon
     * SES Developer Guide</a>.
     * </p>
     * </note>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to specify a particular "From" address in the
     *         header of the raw email.</p>
     *         <p>
     *         Instead of using this parameter, you can use the X-header <code>X-SES-FROM-ARN</code> in the raw message
     *         of the email. If you use both the <code>FromArn</code> parameter and the corresponding X-header, Amazon
     *         SES uses the value of the <code>FromArn</code> parameter.
     *         </p>
     *         <note>
     *         <p>
     *         For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
     *         this guide, or see the <a
     *         href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization-delegate-sender-tasks-email.html"
     *         >Amazon SES Developer Guide</a>.
     *         </p>
     */
    public final String fromArn() {
        return fromArn;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to send for the email address specified in the <code>Source</code>
     * parameter.
     * </p>
     * <p>
     * For example, if the owner of <code>example.com</code> (which has ARN
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes you
     * to send from <code>user@example.com</code>, then you would specify the <code>SourceArn</code> to be
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>Source</code> to be
     * <code>user@example.com</code>.
     * </p>
     * <p>
     * Instead of using this parameter, you can use the X-header <code>X-SES-SOURCE-ARN</code> in the raw message of the
     * email. If you use both the <code>SourceArn</code> parameter and the corresponding X-header, Amazon SES uses the
     * value of the <code>SourceArn</code> parameter.
     * </p>
     * <note>
     * <p>
     * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this guide,
     * or see the <a
     * href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization-delegate-sender-tasks-email.html">Amazon
     * SES Developer Guide</a>.
     * </p>
     * </note>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to send for the email address specified in the
     *         <code>Source</code> parameter.</p>
     *         <p>
     *         For example, if the owner of <code>example.com</code> (which has ARN
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
     *         authorizes you to send from <code>user@example.com</code>, then you would specify the
     *         <code>SourceArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and
     *         the <code>Source</code> to be <code>user@example.com</code>.
     *         </p>
     *         <p>
     *         Instead of using this parameter, you can use the X-header <code>X-SES-SOURCE-ARN</code> in the raw
     *         message of the email. If you use both the <code>SourceArn</code> parameter and the corresponding
     *         X-header, Amazon SES uses the value of the <code>SourceArn</code> parameter.
     *         </p>
     *         <note>
     *         <p>
     *         For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
     *         this guide, or see the <a
     *         href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization-delegate-sender-tasks-email.html"
     *         >Amazon SES Developer Guide</a>.
     *         </p>
     */
    public final String sourceArn() {
        return sourceArn;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to use the email address specified in the <code>ReturnPath</code>
     * parameter.
     * </p>
     * <p>
     * For example, if the owner of <code>example.com</code> (which has ARN
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes you
     * to use <code>feedback@example.com</code>, then you would specify the <code>ReturnPathArn</code> to be
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>ReturnPath</code> to be
     * <code>feedback@example.com</code>.
     * </p>
     * <p>
     * Instead of using this parameter, you can use the X-header <code>X-SES-RETURN-PATH-ARN</code> in the raw message
     * of the email. If you use both the <code>ReturnPathArn</code> parameter and the corresponding X-header, Amazon SES
     * uses the value of the <code>ReturnPathArn</code> parameter.
     * </p>
     * <note>
     * <p>
     * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this guide,
     * or see the <a
     * href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization-delegate-sender-tasks-email.html">Amazon
     * SES Developer Guide</a>.
     * </p>
     * </note>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to use the email address specified in the
     *         <code>ReturnPath</code> parameter.</p>
     *         <p>
     *         For example, if the owner of <code>example.com</code> (which has ARN
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
     *         authorizes you to use <code>feedback@example.com</code>, then you would specify the
     *         <code>ReturnPathArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>,
     *         and the <code>ReturnPath</code> to be <code>feedback@example.com</code>.
     *         </p>
     *         <p>
     *         Instead of using this parameter, you can use the X-header <code>X-SES-RETURN-PATH-ARN</code> in the raw
     *         message of the email. If you use both the <code>ReturnPathArn</code> parameter and the corresponding
     *         X-header, Amazon SES uses the value of the <code>ReturnPathArn</code> parameter.
     *         </p>
     *         <note>
     *         <p>
     *         For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
     *         this guide, or see the <a
     *         href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization-delegate-sender-tasks-email.html"
     *         >Amazon SES Developer Guide</a>.
     *         </p>
     */
    public final String returnPathArn() {
        return returnPathArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags, in the form of name/value pairs, to apply to an email that you send using
     * <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
     * publish email sending events.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags, in the form of name/value pairs, to apply to an email that you send using
     *         <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that you
     *         can publish email sending events.
     */
    public final List<MessageTag> tags() {
        return tags;
    }

    /**
     * <p>
     * The name of the configuration set to use when you send an email using <code>SendRawEmail</code>.
     * </p>
     * 
     * @return The name of the configuration set to use when you send an email using <code>SendRawEmail</code>.
     */
    public final String configurationSetName() {
        return configurationSetName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(hasDestinations() ? destinations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(rawMessage());
        hashCode = 31 * hashCode + Objects.hashCode(fromArn());
        hashCode = 31 * hashCode + Objects.hashCode(sourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(returnPathArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(configurationSetName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SendRawEmailRequest)) {
            return false;
        }
        SendRawEmailRequest other = (SendRawEmailRequest) obj;
        return Objects.equals(source(), other.source()) && hasDestinations() == other.hasDestinations()
                && Objects.equals(destinations(), other.destinations()) && Objects.equals(rawMessage(), other.rawMessage())
                && Objects.equals(fromArn(), other.fromArn()) && Objects.equals(sourceArn(), other.sourceArn())
                && Objects.equals(returnPathArn(), other.returnPathArn()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(configurationSetName(), other.configurationSetName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SendRawEmailRequest").add("Source", source())
                .add("Destinations", hasDestinations() ? destinations() : null).add("RawMessage", rawMessage())
                .add("FromArn", fromArn()).add("SourceArn", sourceArn()).add("ReturnPathArn", returnPathArn())
                .add("Tags", hasTags() ? tags() : null).add("ConfigurationSetName", configurationSetName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Source":
            return Optional.ofNullable(clazz.cast(source()));
        case "Destinations":
            return Optional.ofNullable(clazz.cast(destinations()));
        case "RawMessage":
            return Optional.ofNullable(clazz.cast(rawMessage()));
        case "FromArn":
            return Optional.ofNullable(clazz.cast(fromArn()));
        case "SourceArn":
            return Optional.ofNullable(clazz.cast(sourceArn()));
        case "ReturnPathArn":
            return Optional.ofNullable(clazz.cast(returnPathArn()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "ConfigurationSetName":
            return Optional.ofNullable(clazz.cast(configurationSetName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<SendRawEmailRequest, T> g) {
        return obj -> g.apply((SendRawEmailRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SesRequest.Builder, SdkPojo, CopyableBuilder<Builder, SendRawEmailRequest> {
        /**
         * <p>
         * The identity's email address. If you do not provide a value for this parameter, you must specify a "From"
         * address in the raw text of the message. (You can also specify both.)
         * </p>
         * <note>
         * <p>
         * Amazon SES does not support the SMTPUTF8 extension, as described in<a
         * href="https://tools.ietf.org/html/rfc6531">RFC6531</a>. For this reason, the email address string must be
         * 7-bit ASCII. If you want to send to or from email addresses that contain Unicode characters in the domain
         * part of an address, you must encode the domain using Punycode. Punycode is not permitted in the local part of
         * the email address (the part before the @ sign) nor in the "friendly from" name. If you want to use Unicode
         * characters in the "friendly from" name, you must encode the "friendly from" name using MIME encoded-word
         * syntax, as described in <a href="https://docs.aws.amazon.com/ses/latest/dg/send-email-raw.html">Sending raw
         * email using the Amazon SES API</a>. For more information about Punycode, see <a
         * href="http://tools.ietf.org/html/rfc3492">RFC 3492</a>.
         * </p>
         * </note>
         * <p>
         * If you specify the <code>Source</code> parameter and have feedback forwarding enabled, then bounces and
         * complaints are sent to this email address. This takes precedence over any Return-Path header that you might
         * include in the raw text of the message.
         * </p>
         * 
         * @param source
         *        The identity's email address. If you do not provide a value for this parameter, you must specify a
         *        "From" address in the raw text of the message. (You can also specify both.)</p> <note>
         *        <p>
         *        Amazon SES does not support the SMTPUTF8 extension, as described in<a
         *        href="https://tools.ietf.org/html/rfc6531">RFC6531</a>. For this reason, the email address string must
         *        be 7-bit ASCII. If you want to send to or from email addresses that contain Unicode characters in the
         *        domain part of an address, you must encode the domain using Punycode. Punycode is not permitted in the
         *        local part of the email address (the part before the @ sign) nor in the "friendly from" name. If you
         *        want to use Unicode characters in the "friendly from" name, you must encode the "friendly from" name
         *        using MIME encoded-word syntax, as described in <a
         *        href="https://docs.aws.amazon.com/ses/latest/dg/send-email-raw.html">Sending raw email using the
         *        Amazon SES API</a>. For more information about Punycode, see <a
         *        href="http://tools.ietf.org/html/rfc3492">RFC 3492</a>.
         *        </p>
         *        </note>
         *        <p>
         *        If you specify the <code>Source</code> parameter and have feedback forwarding enabled, then bounces
         *        and complaints are sent to this email address. This takes precedence over any Return-Path header that
         *        you might include in the raw text of the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
         * </p>
         * 
         * @param destinations
         *        A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(Collection<String> destinations);

        /**
         * <p>
         * A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
         * </p>
         * 
         * @param destinations
         *        A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(String... destinations);

        /**
         * <p>
         * The raw email message itself. The message has to meet the following criteria:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The message has to contain a header and a body, separated by a blank line.
         * </p>
         * </li>
         * <li>
         * <p>
         * All of the required header fields must be present in the message.
         * </p>
         * </li>
         * <li>
         * <p>
         * Each part of a multipart MIME message must be formatted properly.
         * </p>
         * </li>
         * <li>
         * <p>
         * Attachments must be of a content type that Amazon SES supports. For a list on unsupported content types, see
         * <a href="https://docs.aws.amazon.com/ses/latest/dg/mime-types.html">Unsupported Attachment Types</a> in the
         * <i>Amazon SES Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * The entire message must be base64-encoded.
         * </p>
         * </li>
         * <li>
         * <p>
         * If any of the MIME parts in your message contain content that is outside of the 7-bit ASCII character range,
         * we highly recommend that you encode that content. For more information, see <a
         * href="https://docs.aws.amazon.com/ses/latest/dg/send-email-raw.html">Sending Raw Email</a> in the <i>Amazon
         * SES Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Per <a href="https://tools.ietf.org/html/rfc5321#section-4.5.3.1.6">RFC 5321</a>, the maximum length of each
         * line of text, including the &lt;CRLF&gt;, must not exceed 1,000 characters.
         * </p>
         * </li>
         * </ul>
         * 
         * @param rawMessage
         *        The raw email message itself. The message has to meet the following criteria:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The message has to contain a header and a body, separated by a blank line.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        All of the required header fields must be present in the message.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Each part of a multipart MIME message must be formatted properly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Attachments must be of a content type that Amazon SES supports. For a list on unsupported content
         *        types, see <a href="https://docs.aws.amazon.com/ses/latest/dg/mime-types.html">Unsupported Attachment
         *        Types</a> in the <i>Amazon SES Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The entire message must be base64-encoded.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If any of the MIME parts in your message contain content that is outside of the 7-bit ASCII character
         *        range, we highly recommend that you encode that content. For more information, see <a
         *        href="https://docs.aws.amazon.com/ses/latest/dg/send-email-raw.html">Sending Raw Email</a> in the
         *        <i>Amazon SES Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Per <a href="https://tools.ietf.org/html/rfc5321#section-4.5.3.1.6">RFC 5321</a>, the maximum length
         *        of each line of text, including the &lt;CRLF&gt;, must not exceed 1,000 characters.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rawMessage(RawMessage rawMessage);

        /**
         * <p>
         * The raw email message itself. The message has to meet the following criteria:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The message has to contain a header and a body, separated by a blank line.
         * </p>
         * </li>
         * <li>
         * <p>
         * All of the required header fields must be present in the message.
         * </p>
         * </li>
         * <li>
         * <p>
         * Each part of a multipart MIME message must be formatted properly.
         * </p>
         * </li>
         * <li>
         * <p>
         * Attachments must be of a content type that Amazon SES supports. For a list on unsupported content types, see
         * <a href="https://docs.aws.amazon.com/ses/latest/dg/mime-types.html">Unsupported Attachment Types</a> in the
         * <i>Amazon SES Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * The entire message must be base64-encoded.
         * </p>
         * </li>
         * <li>
         * <p>
         * If any of the MIME parts in your message contain content that is outside of the 7-bit ASCII character range,
         * we highly recommend that you encode that content. For more information, see <a
         * href="https://docs.aws.amazon.com/ses/latest/dg/send-email-raw.html">Sending Raw Email</a> in the <i>Amazon
         * SES Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Per <a href="https://tools.ietf.org/html/rfc5321#section-4.5.3.1.6">RFC 5321</a>, the maximum length of each
         * line of text, including the &lt;CRLF&gt;, must not exceed 1,000 characters.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the {@link RawMessage.Builder} avoiding the need to
         * create one manually via {@link RawMessage#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RawMessage.Builder#build()} is called immediately and its result
         * is passed to {@link #rawMessage(RawMessage)}.
         * 
         * @param rawMessage
         *        a consumer that will call methods on {@link RawMessage.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rawMessage(RawMessage)
         */
        default Builder rawMessage(Consumer<RawMessage.Builder> rawMessage) {
            return rawMessage(RawMessage.builder().applyMutation(rawMessage).build());
        }

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to specify a particular "From" address in the header of the
         * raw email.
         * </p>
         * <p>
         * Instead of using this parameter, you can use the X-header <code>X-SES-FROM-ARN</code> in the raw message of
         * the email. If you use both the <code>FromArn</code> parameter and the corresponding X-header, Amazon SES uses
         * the value of the <code>FromArn</code> parameter.
         * </p>
         * <note>
         * <p>
         * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this
         * guide, or see the <a
         * href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization-delegate-sender-tasks-email.html"
         * >Amazon SES Developer Guide</a>.
         * </p>
         * </note>
         * 
         * @param fromArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to specify a particular "From"
         *        address in the header of the raw email.</p>
         *        <p>
         *        Instead of using this parameter, you can use the X-header <code>X-SES-FROM-ARN</code> in the raw
         *        message of the email. If you use both the <code>FromArn</code> parameter and the corresponding
         *        X-header, Amazon SES uses the value of the <code>FromArn</code> parameter.
         *        </p>
         *        <note>
         *        <p>
         *        For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
         *        this guide, or see the <a href=
         *        "https://docs.aws.amazon.com/ses/latest/dg/sending-authorization-delegate-sender-tasks-email.html"
         *        >Amazon SES Developer Guide</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromArn(String fromArn);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to send for the email address specified in the
         * <code>Source</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of <code>example.com</code> (which has ARN
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes
         * you to send from <code>user@example.com</code>, then you would specify the <code>SourceArn</code> to be
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>Source</code> to be
         * <code>user@example.com</code>.
         * </p>
         * <p>
         * Instead of using this parameter, you can use the X-header <code>X-SES-SOURCE-ARN</code> in the raw message of
         * the email. If you use both the <code>SourceArn</code> parameter and the corresponding X-header, Amazon SES
         * uses the value of the <code>SourceArn</code> parameter.
         * </p>
         * <note>
         * <p>
         * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this
         * guide, or see the <a
         * href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization-delegate-sender-tasks-email.html"
         * >Amazon SES Developer Guide</a>.
         * </p>
         * </note>
         * 
         * @param sourceArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to send for the email address
         *        specified in the <code>Source</code> parameter.</p>
         *        <p>
         *        For example, if the owner of <code>example.com</code> (which has ARN
         *        <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
         *        authorizes you to send from <code>user@example.com</code>, then you would specify the
         *        <code>SourceArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and
         *        the <code>Source</code> to be <code>user@example.com</code>.
         *        </p>
         *        <p>
         *        Instead of using this parameter, you can use the X-header <code>X-SES-SOURCE-ARN</code> in the raw
         *        message of the email. If you use both the <code>SourceArn</code> parameter and the corresponding
         *        X-header, Amazon SES uses the value of the <code>SourceArn</code> parameter.
         *        </p>
         *        <note>
         *        <p>
         *        For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
         *        this guide, or see the <a href=
         *        "https://docs.aws.amazon.com/ses/latest/dg/sending-authorization-delegate-sender-tasks-email.html"
         *        >Amazon SES Developer Guide</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceArn(String sourceArn);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to use the email address specified in the
         * <code>ReturnPath</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of <code>example.com</code> (which has ARN
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes
         * you to use <code>feedback@example.com</code>, then you would specify the <code>ReturnPathArn</code> to be
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>ReturnPath</code> to be
         * <code>feedback@example.com</code>.
         * </p>
         * <p>
         * Instead of using this parameter, you can use the X-header <code>X-SES-RETURN-PATH-ARN</code> in the raw
         * message of the email. If you use both the <code>ReturnPathArn</code> parameter and the corresponding
         * X-header, Amazon SES uses the value of the <code>ReturnPathArn</code> parameter.
         * </p>
         * <note>
         * <p>
         * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this
         * guide, or see the <a
         * href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization-delegate-sender-tasks-email.html"
         * >Amazon SES Developer Guide</a>.
         * </p>
         * </note>
         * 
         * @param returnPathArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to use the email address specified
         *        in the <code>ReturnPath</code> parameter.</p>
         *        <p>
         *        For example, if the owner of <code>example.com</code> (which has ARN
         *        <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
         *        authorizes you to use <code>feedback@example.com</code>, then you would specify the
         *        <code>ReturnPathArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>,
         *        and the <code>ReturnPath</code> to be <code>feedback@example.com</code>.
         *        </p>
         *        <p>
         *        Instead of using this parameter, you can use the X-header <code>X-SES-RETURN-PATH-ARN</code> in the
         *        raw message of the email. If you use both the <code>ReturnPathArn</code> parameter and the
         *        corresponding X-header, Amazon SES uses the value of the <code>ReturnPathArn</code> parameter.
         *        </p>
         *        <note>
         *        <p>
         *        For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
         *        this guide, or see the <a href=
         *        "https://docs.aws.amazon.com/ses/latest/dg/sending-authorization-delegate-sender-tasks-email.html"
         *        >Amazon SES Developer Guide</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder returnPathArn(String returnPathArn);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using
         * <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
         * publish email sending events.
         * </p>
         * 
         * @param tags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using
         *        <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that
         *        you can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<MessageTag> tags);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using
         * <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
         * publish email sending events.
         * </p>
         * 
         * @param tags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using
         *        <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that
         *        you can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(MessageTag... tags);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using
         * <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
         * publish email sending events.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ses.model.MessageTag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.ses.model.MessageTag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ses.model.MessageTag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<MessageTag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ses.model.MessageTag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<MessageTag>)
         */
        Builder tags(Consumer<MessageTag.Builder>... tags);

        /**
         * <p>
         * The name of the configuration set to use when you send an email using <code>SendRawEmail</code>.
         * </p>
         * 
         * @param configurationSetName
         *        The name of the configuration set to use when you send an email using <code>SendRawEmail</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSetName(String configurationSetName);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesRequest.BuilderImpl implements Builder {
        private String source;

        private List<String> destinations = DefaultSdkAutoConstructList.getInstance();

        private RawMessage rawMessage;

        private String fromArn;

        private String sourceArn;

        private String returnPathArn;

        private List<MessageTag> tags = DefaultSdkAutoConstructList.getInstance();

        private String configurationSetName;

        private BuilderImpl() {
        }

        private BuilderImpl(SendRawEmailRequest model) {
            super(model);
            source(model.source);
            destinations(model.destinations);
            rawMessage(model.rawMessage);
            fromArn(model.fromArn);
            sourceArn(model.sourceArn);
            returnPathArn(model.returnPathArn);
            tags(model.tags);
            configurationSetName(model.configurationSetName);
        }

        public final String getSource() {
            return source;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final Collection<String> getDestinations() {
            if (destinations instanceof SdkAutoConstructList) {
                return null;
            }
            return destinations;
        }

        public final void setDestinations(Collection<String> destinations) {
            this.destinations = AddressListCopier.copy(destinations);
        }

        @Override
        public final Builder destinations(Collection<String> destinations) {
            this.destinations = AddressListCopier.copy(destinations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(String... destinations) {
            destinations(Arrays.asList(destinations));
            return this;
        }

        public final RawMessage.Builder getRawMessage() {
            return rawMessage != null ? rawMessage.toBuilder() : null;
        }

        public final void setRawMessage(RawMessage.BuilderImpl rawMessage) {
            this.rawMessage = rawMessage != null ? rawMessage.build() : null;
        }

        @Override
        public final Builder rawMessage(RawMessage rawMessage) {
            this.rawMessage = rawMessage;
            return this;
        }

        public final String getFromArn() {
            return fromArn;
        }

        public final void setFromArn(String fromArn) {
            this.fromArn = fromArn;
        }

        @Override
        public final Builder fromArn(String fromArn) {
            this.fromArn = fromArn;
            return this;
        }

        public final String getSourceArn() {
            return sourceArn;
        }

        public final void setSourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
        }

        @Override
        public final Builder sourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
            return this;
        }

        public final String getReturnPathArn() {
            return returnPathArn;
        }

        public final void setReturnPathArn(String returnPathArn) {
            this.returnPathArn = returnPathArn;
        }

        @Override
        public final Builder returnPathArn(String returnPathArn) {
            this.returnPathArn = returnPathArn;
            return this;
        }

        public final List<MessageTag.Builder> getTags() {
            List<MessageTag.Builder> result = MessageTagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<MessageTag.BuilderImpl> tags) {
            this.tags = MessageTagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<MessageTag> tags) {
            this.tags = MessageTagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(MessageTag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<MessageTag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> MessageTag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getConfigurationSetName() {
            return configurationSetName;
        }

        public final void setConfigurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
        }

        @Override
        public final Builder configurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SendRawEmailRequest build() {
            return new SendRawEmailRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
