/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicediscovery.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class RegisterInstanceRequest extends ServiceDiscoveryRequest implements
        ToCopyableBuilder<RegisterInstanceRequest.Builder, RegisterInstanceRequest> {
    private static final SdkField<String> SERVICE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceId").getter(getter(RegisterInstanceRequest::serviceId)).setter(setter(Builder::serviceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceId").build()).build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceId").getter(getter(RegisterInstanceRequest::instanceId)).setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId").build()).build();

    private static final SdkField<String> CREATOR_REQUEST_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CreatorRequestId")
            .getter(getter(RegisterInstanceRequest::creatorRequestId))
            .setter(setter(Builder::creatorRequestId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatorRequestId").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<Map<String, String>> ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Attributes")
            .getter(getter(RegisterInstanceRequest::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SERVICE_ID_FIELD,
            INSTANCE_ID_FIELD, CREATOR_REQUEST_ID_FIELD, ATTRIBUTES_FIELD));

    private final String serviceId;

    private final String instanceId;

    private final String creatorRequestId;

    private final Map<String, String> attributes;

    private RegisterInstanceRequest(BuilderImpl builder) {
        super(builder);
        this.serviceId = builder.serviceId;
        this.instanceId = builder.instanceId;
        this.creatorRequestId = builder.creatorRequestId;
        this.attributes = builder.attributes;
    }

    /**
     * <p>
     * The ID of the service that you want to use for settings for the instance.
     * </p>
     * 
     * @return The ID of the service that you want to use for settings for the instance.
     */
    public final String serviceId() {
        return serviceId;
    }

    /**
     * <p>
     * An identifier that you want to associate with the instance. Note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the service that is specified by <code>ServiceId</code> includes settings for an <code>SRV</code> record, the
     * value of <code>InstanceId</code> is automatically included as part of the value for the <code>SRV</code> record.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_DnsRecord.html#cloudmap-Type-DnsRecord-Type">DnsRecord
     * &gt; Type</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can use this value to update an existing instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * To register a new instance, you must specify a value that is unique among instances that you register by using
     * the same service.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify an existing <code>InstanceId</code> and <code>ServiceId</code>, AWS Cloud Map updates the existing
     * DNS records, if any. If there's also an existing health check, AWS Cloud Map deletes the old health check and
     * creates a new one.
     * </p>
     * <note>
     * <p>
     * The health check isn't deleted immediately, so it will still appear for a while if you submit a
     * <code>ListHealthChecks</code> request, for example.
     * </p>
     * </note></li>
     * </ul>
     * 
     * @return An identifier that you want to associate with the instance. Note the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the service that is specified by <code>ServiceId</code> includes settings for an <code>SRV</code>
     *         record, the value of <code>InstanceId</code> is automatically included as part of the value for the
     *         <code>SRV</code> record. For more information, see <a
     *         href="https://docs.aws.amazon.com/cloud-map/latest/api/API_DnsRecord.html#cloudmap-Type-DnsRecord-Type"
     *         >DnsRecord &gt; Type</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can use this value to update an existing instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To register a new instance, you must specify a value that is unique among instances that you register by
     *         using the same service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you specify an existing <code>InstanceId</code> and <code>ServiceId</code>, AWS Cloud Map updates the
     *         existing DNS records, if any. If there's also an existing health check, AWS Cloud Map deletes the old
     *         health check and creates a new one.
     *         </p>
     *         <note>
     *         <p>
     *         The health check isn't deleted immediately, so it will still appear for a while if you submit a
     *         <code>ListHealthChecks</code> request, for example.
     *         </p>
     *         </note></li>
     */
    public final String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * A unique string that identifies the request and that allows failed <code>RegisterInstance</code> requests to be
     * retried without the risk of executing the operation twice. You must use a unique <code>CreatorRequestId</code>
     * string every time you submit a <code>RegisterInstance</code> request if you're registering additional instances
     * for the same namespace and service. <code>CreatorRequestId</code> can be any unique string, for example, a
     * date/time stamp.
     * </p>
     * 
     * @return A unique string that identifies the request and that allows failed <code>RegisterInstance</code> requests
     *         to be retried without the risk of executing the operation twice. You must use a unique
     *         <code>CreatorRequestId</code> string every time you submit a <code>RegisterInstance</code> request if
     *         you're registering additional instances for the same namespace and service. <code>CreatorRequestId</code>
     *         can be any unique string, for example, a date/time stamp.
     */
    public final String creatorRequestId() {
        return creatorRequestId;
    }

    /**
     * Returns true if the Attributes property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A string map that contains the following information for the service that you specify in <code>ServiceId</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The attributes that apply to the records that are defined in the service.
     * </p>
     * </li>
     * <li>
     * <p>
     * For each attribute, the applicable value.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Supported attribute keys include the following:
     * </p>
     * <p>
     * <b>AWS_ALIAS_DNS_NAME</b>
     * </p>
     * <p>
     * If you want AWS Cloud Map to create an Amazon Route 53 alias record that routes traffic to an Elastic Load
     * Balancing load balancer, specify the DNS name that is associated with the load balancer. For information about
     * how to get the DNS name, see "DNSName" in the topic <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_AliasTarget.html">AliasTarget</a> in the
     * <i>Route 53 API Reference</i>.
     * </p>
     * <p>
     * Note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The configuration for the service that is specified by <code>ServiceId</code> must include settings for an
     * <code>A</code> record, an <code>AAAA</code> record, or both.
     * </p>
     * </li>
     * <li>
     * <p>
     * In the service that is specified by <code>ServiceId</code>, the value of <code>RoutingPolicy</code> must be
     * <code>WEIGHTED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the service that is specified by <code>ServiceId</code> includes <code>HealthCheckConfig</code> settings, AWS
     * Cloud Map will create the Route 53 health check, but it won't associate the health check with the alias record.
     * </p>
     * </li>
     * <li>
     * <p>
     * Auto naming currently doesn't support creating alias records that route traffic to AWS resources other than
     * Elastic Load Balancing load balancers.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify a value for <code>AWS_ALIAS_DNS_NAME</code>, don't specify values for any of the
     * <code>AWS_INSTANCE</code> attributes.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>AWS_EC2_INSTANCE_ID</b>
     * </p>
     * <p>
     * <i>HTTP namespaces only.</i> The Amazon EC2 instance ID for the instance. If the <code>AWS_EC2_INSTANCE_ID</code>
     * attribute is specified, then the only other attribute that can be specified is
     * <code>AWS_INIT_HEALTH_STATUS</code>. When the <code>AWS_EC2_INSTANCE_ID</code> attribute is specified, then the
     * <code>AWS_INSTANCE_IPV4</code> attribute will be filled out with the primary private IPv4 address.
     * </p>
     * <p>
     * <b>AWS_INIT_HEALTH_STATUS</b>
     * </p>
     * <p>
     * If the service configuration includes <code>HealthCheckCustomConfig</code>, you can optionally use
     * <code>AWS_INIT_HEALTH_STATUS</code> to specify the initial status of the custom health check,
     * <code>HEALTHY</code> or <code>UNHEALTHY</code>. If you don't specify a value for
     * <code>AWS_INIT_HEALTH_STATUS</code>, the initial status is <code>HEALTHY</code>.
     * </p>
     * <p>
     * <b>AWS_INSTANCE_CNAME</b>
     * </p>
     * <p>
     * If the service configuration includes a <code>CNAME</code> record, the domain name that you want Route 53 to
     * return in response to DNS queries, for example, <code>example.com</code>.
     * </p>
     * <p>
     * This value is required if the service specified by <code>ServiceId</code> includes settings for an
     * <code>CNAME</code> record.
     * </p>
     * <p>
     * <b>AWS_INSTANCE_IPV4</b>
     * </p>
     * <p>
     * If the service configuration includes an <code>A</code> record, the IPv4 address that you want Route 53 to return
     * in response to DNS queries, for example, <code>192.0.2.44</code>.
     * </p>
     * <p>
     * This value is required if the service specified by <code>ServiceId</code> includes settings for an <code>A</code>
     * record. If the service includes settings for an <code>SRV</code> record, you must specify a value for
     * <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both.
     * </p>
     * <p>
     * <b>AWS_INSTANCE_IPV6</b>
     * </p>
     * <p>
     * If the service configuration includes an <code>AAAA</code> record, the IPv6 address that you want Route 53 to
     * return in response to DNS queries, for example, <code>2001:0db8:85a3:0000:0000:abcd:0001:2345</code>.
     * </p>
     * <p>
     * This value is required if the service specified by <code>ServiceId</code> includes settings for an
     * <code>AAAA</code> record. If the service includes settings for an <code>SRV</code> record, you must specify a
     * value for <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both.
     * </p>
     * <p>
     * <b>AWS_INSTANCE_PORT</b>
     * </p>
     * <p>
     * If the service includes an <code>SRV</code> record, the value that you want Route 53 to return for the port.
     * </p>
     * <p>
     * If the service includes <code>HealthCheckConfig</code>, the port on the endpoint that you want Route 53 to send
     * requests to.
     * </p>
     * <p>
     * This value is required if you specified settings for an <code>SRV</code> record or a Route 53 health check when
     * you created the service.
     * </p>
     * <p>
     * <b>Custom attributes</b>
     * </p>
     * <p>
     * You can add up to 30 custom attributes. For each key-value pair, the maximum length of the attribute name is 255
     * characters, and the maximum length of the attribute value is 1,024 characters. The total size of all provided
     * attributes (sum of all keys and values) must not exceed 5,000 characters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAttributes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A string map that contains the following information for the service that you specify in
     *         <code>ServiceId</code>:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The attributes that apply to the records that are defined in the service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For each attribute, the applicable value.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Supported attribute keys include the following:
     *         </p>
     *         <p>
     *         <b>AWS_ALIAS_DNS_NAME</b>
     *         </p>
     *         <p>
     *         If you want AWS Cloud Map to create an Amazon Route 53 alias record that routes traffic to an Elastic
     *         Load Balancing load balancer, specify the DNS name that is associated with the load balancer. For
     *         information about how to get the DNS name, see "DNSName" in the topic <a
     *         href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_AliasTarget.html">AliasTarget</a> in
     *         the <i>Route 53 API Reference</i>.
     *         </p>
     *         <p>
     *         Note the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The configuration for the service that is specified by <code>ServiceId</code> must include settings for
     *         an <code>A</code> record, an <code>AAAA</code> record, or both.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In the service that is specified by <code>ServiceId</code>, the value of <code>RoutingPolicy</code> must
     *         be <code>WEIGHTED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the service that is specified by <code>ServiceId</code> includes <code>HealthCheckConfig</code>
     *         settings, AWS Cloud Map will create the Route 53 health check, but it won't associate the health check
     *         with the alias record.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Auto naming currently doesn't support creating alias records that route traffic to AWS resources other
     *         than Elastic Load Balancing load balancers.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you specify a value for <code>AWS_ALIAS_DNS_NAME</code>, don't specify values for any of the
     *         <code>AWS_INSTANCE</code> attributes.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>AWS_EC2_INSTANCE_ID</b>
     *         </p>
     *         <p>
     *         <i>HTTP namespaces only.</i> The Amazon EC2 instance ID for the instance. If the
     *         <code>AWS_EC2_INSTANCE_ID</code> attribute is specified, then the only other attribute that can be
     *         specified is <code>AWS_INIT_HEALTH_STATUS</code>. When the <code>AWS_EC2_INSTANCE_ID</code> attribute is
     *         specified, then the <code>AWS_INSTANCE_IPV4</code> attribute will be filled out with the primary private
     *         IPv4 address.
     *         </p>
     *         <p>
     *         <b>AWS_INIT_HEALTH_STATUS</b>
     *         </p>
     *         <p>
     *         If the service configuration includes <code>HealthCheckCustomConfig</code>, you can optionally use
     *         <code>AWS_INIT_HEALTH_STATUS</code> to specify the initial status of the custom health check,
     *         <code>HEALTHY</code> or <code>UNHEALTHY</code>. If you don't specify a value for
     *         <code>AWS_INIT_HEALTH_STATUS</code>, the initial status is <code>HEALTHY</code>.
     *         </p>
     *         <p>
     *         <b>AWS_INSTANCE_CNAME</b>
     *         </p>
     *         <p>
     *         If the service configuration includes a <code>CNAME</code> record, the domain name that you want Route 53
     *         to return in response to DNS queries, for example, <code>example.com</code>.
     *         </p>
     *         <p>
     *         This value is required if the service specified by <code>ServiceId</code> includes settings for an
     *         <code>CNAME</code> record.
     *         </p>
     *         <p>
     *         <b>AWS_INSTANCE_IPV4</b>
     *         </p>
     *         <p>
     *         If the service configuration includes an <code>A</code> record, the IPv4 address that you want Route 53
     *         to return in response to DNS queries, for example, <code>192.0.2.44</code>.
     *         </p>
     *         <p>
     *         This value is required if the service specified by <code>ServiceId</code> includes settings for an
     *         <code>A</code> record. If the service includes settings for an <code>SRV</code> record, you must specify
     *         a value for <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both.
     *         </p>
     *         <p>
     *         <b>AWS_INSTANCE_IPV6</b>
     *         </p>
     *         <p>
     *         If the service configuration includes an <code>AAAA</code> record, the IPv6 address that you want
     *         Route 53 to return in response to DNS queries, for example,
     *         <code>2001:0db8:85a3:0000:0000:abcd:0001:2345</code>.
     *         </p>
     *         <p>
     *         This value is required if the service specified by <code>ServiceId</code> includes settings for an
     *         <code>AAAA</code> record. If the service includes settings for an <code>SRV</code> record, you must
     *         specify a value for <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both.
     *         </p>
     *         <p>
     *         <b>AWS_INSTANCE_PORT</b>
     *         </p>
     *         <p>
     *         If the service includes an <code>SRV</code> record, the value that you want Route 53 to return for the
     *         port.
     *         </p>
     *         <p>
     *         If the service includes <code>HealthCheckConfig</code>, the port on the endpoint that you want Route 53
     *         to send requests to.
     *         </p>
     *         <p>
     *         This value is required if you specified settings for an <code>SRV</code> record or a Route 53 health
     *         check when you created the service.
     *         </p>
     *         <p>
     *         <b>Custom attributes</b>
     *         </p>
     *         <p>
     *         You can add up to 30 custom attributes. For each key-value pair, the maximum length of the attribute name
     *         is 255 characters, and the maximum length of the attribute value is 1,024 characters. The total size of
     *         all provided attributes (sum of all keys and values) must not exceed 5,000 characters.
     */
    public final Map<String, String> attributes() {
        return attributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(serviceId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(creatorRequestId());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributes() ? attributes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RegisterInstanceRequest)) {
            return false;
        }
        RegisterInstanceRequest other = (RegisterInstanceRequest) obj;
        return Objects.equals(serviceId(), other.serviceId()) && Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(creatorRequestId(), other.creatorRequestId()) && hasAttributes() == other.hasAttributes()
                && Objects.equals(attributes(), other.attributes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RegisterInstanceRequest").add("ServiceId", serviceId()).add("InstanceId", instanceId())
                .add("CreatorRequestId", creatorRequestId()).add("Attributes", hasAttributes() ? attributes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ServiceId":
            return Optional.ofNullable(clazz.cast(serviceId()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "CreatorRequestId":
            return Optional.ofNullable(clazz.cast(creatorRequestId()));
        case "Attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RegisterInstanceRequest, T> g) {
        return obj -> g.apply((RegisterInstanceRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ServiceDiscoveryRequest.Builder, SdkPojo, CopyableBuilder<Builder, RegisterInstanceRequest> {
        /**
         * <p>
         * The ID of the service that you want to use for settings for the instance.
         * </p>
         * 
         * @param serviceId
         *        The ID of the service that you want to use for settings for the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceId(String serviceId);

        /**
         * <p>
         * An identifier that you want to associate with the instance. Note the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the service that is specified by <code>ServiceId</code> includes settings for an <code>SRV</code> record,
         * the value of <code>InstanceId</code> is automatically included as part of the value for the <code>SRV</code>
         * record. For more information, see <a
         * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_DnsRecord.html#cloudmap-Type-DnsRecord-Type"
         * >DnsRecord &gt; Type</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * You can use this value to update an existing instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * To register a new instance, you must specify a value that is unique among instances that you register by
         * using the same service.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify an existing <code>InstanceId</code> and <code>ServiceId</code>, AWS Cloud Map updates the
         * existing DNS records, if any. If there's also an existing health check, AWS Cloud Map deletes the old health
         * check and creates a new one.
         * </p>
         * <note>
         * <p>
         * The health check isn't deleted immediately, so it will still appear for a while if you submit a
         * <code>ListHealthChecks</code> request, for example.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param instanceId
         *        An identifier that you want to associate with the instance. Note the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the service that is specified by <code>ServiceId</code> includes settings for an <code>SRV</code>
         *        record, the value of <code>InstanceId</code> is automatically included as part of the value for the
         *        <code>SRV</code> record. For more information, see <a
         *        href="https://docs.aws.amazon.com/cloud-map/latest/api/API_DnsRecord.html#cloudmap-Type-DnsRecord-Type"
         *        >DnsRecord &gt; Type</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You can use this value to update an existing instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To register a new instance, you must specify a value that is unique among instances that you register
         *        by using the same service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify an existing <code>InstanceId</code> and <code>ServiceId</code>, AWS Cloud Map updates
         *        the existing DNS records, if any. If there's also an existing health check, AWS Cloud Map deletes the
         *        old health check and creates a new one.
         *        </p>
         *        <note>
         *        <p>
         *        The health check isn't deleted immediately, so it will still appear for a while if you submit a
         *        <code>ListHealthChecks</code> request, for example.
         *        </p>
         *        </note></li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * A unique string that identifies the request and that allows failed <code>RegisterInstance</code> requests to
         * be retried without the risk of executing the operation twice. You must use a unique
         * <code>CreatorRequestId</code> string every time you submit a <code>RegisterInstance</code> request if you're
         * registering additional instances for the same namespace and service. <code>CreatorRequestId</code> can be any
         * unique string, for example, a date/time stamp.
         * </p>
         * 
         * @param creatorRequestId
         *        A unique string that identifies the request and that allows failed <code>RegisterInstance</code>
         *        requests to be retried without the risk of executing the operation twice. You must use a unique
         *        <code>CreatorRequestId</code> string every time you submit a <code>RegisterInstance</code> request if
         *        you're registering additional instances for the same namespace and service.
         *        <code>CreatorRequestId</code> can be any unique string, for example, a date/time stamp.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creatorRequestId(String creatorRequestId);

        /**
         * <p>
         * A string map that contains the following information for the service that you specify in
         * <code>ServiceId</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The attributes that apply to the records that are defined in the service.
         * </p>
         * </li>
         * <li>
         * <p>
         * For each attribute, the applicable value.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Supported attribute keys include the following:
         * </p>
         * <p>
         * <b>AWS_ALIAS_DNS_NAME</b>
         * </p>
         * <p>
         * If you want AWS Cloud Map to create an Amazon Route 53 alias record that routes traffic to an Elastic Load
         * Balancing load balancer, specify the DNS name that is associated with the load balancer. For information
         * about how to get the DNS name, see "DNSName" in the topic <a
         * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_AliasTarget.html">AliasTarget</a> in the
         * <i>Route 53 API Reference</i>.
         * </p>
         * <p>
         * Note the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The configuration for the service that is specified by <code>ServiceId</code> must include settings for an
         * <code>A</code> record, an <code>AAAA</code> record, or both.
         * </p>
         * </li>
         * <li>
         * <p>
         * In the service that is specified by <code>ServiceId</code>, the value of <code>RoutingPolicy</code> must be
         * <code>WEIGHTED</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the service that is specified by <code>ServiceId</code> includes <code>HealthCheckConfig</code> settings,
         * AWS Cloud Map will create the Route 53 health check, but it won't associate the health check with the alias
         * record.
         * </p>
         * </li>
         * <li>
         * <p>
         * Auto naming currently doesn't support creating alias records that route traffic to AWS resources other than
         * Elastic Load Balancing load balancers.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify a value for <code>AWS_ALIAS_DNS_NAME</code>, don't specify values for any of the
         * <code>AWS_INSTANCE</code> attributes.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>AWS_EC2_INSTANCE_ID</b>
         * </p>
         * <p>
         * <i>HTTP namespaces only.</i> The Amazon EC2 instance ID for the instance. If the
         * <code>AWS_EC2_INSTANCE_ID</code> attribute is specified, then the only other attribute that can be specified
         * is <code>AWS_INIT_HEALTH_STATUS</code>. When the <code>AWS_EC2_INSTANCE_ID</code> attribute is specified,
         * then the <code>AWS_INSTANCE_IPV4</code> attribute will be filled out with the primary private IPv4 address.
         * </p>
         * <p>
         * <b>AWS_INIT_HEALTH_STATUS</b>
         * </p>
         * <p>
         * If the service configuration includes <code>HealthCheckCustomConfig</code>, you can optionally use
         * <code>AWS_INIT_HEALTH_STATUS</code> to specify the initial status of the custom health check,
         * <code>HEALTHY</code> or <code>UNHEALTHY</code>. If you don't specify a value for
         * <code>AWS_INIT_HEALTH_STATUS</code>, the initial status is <code>HEALTHY</code>.
         * </p>
         * <p>
         * <b>AWS_INSTANCE_CNAME</b>
         * </p>
         * <p>
         * If the service configuration includes a <code>CNAME</code> record, the domain name that you want Route 53 to
         * return in response to DNS queries, for example, <code>example.com</code>.
         * </p>
         * <p>
         * This value is required if the service specified by <code>ServiceId</code> includes settings for an
         * <code>CNAME</code> record.
         * </p>
         * <p>
         * <b>AWS_INSTANCE_IPV4</b>
         * </p>
         * <p>
         * If the service configuration includes an <code>A</code> record, the IPv4 address that you want Route 53 to
         * return in response to DNS queries, for example, <code>192.0.2.44</code>.
         * </p>
         * <p>
         * This value is required if the service specified by <code>ServiceId</code> includes settings for an
         * <code>A</code> record. If the service includes settings for an <code>SRV</code> record, you must specify a
         * value for <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both.
         * </p>
         * <p>
         * <b>AWS_INSTANCE_IPV6</b>
         * </p>
         * <p>
         * If the service configuration includes an <code>AAAA</code> record, the IPv6 address that you want Route 53 to
         * return in response to DNS queries, for example, <code>2001:0db8:85a3:0000:0000:abcd:0001:2345</code>.
         * </p>
         * <p>
         * This value is required if the service specified by <code>ServiceId</code> includes settings for an
         * <code>AAAA</code> record. If the service includes settings for an <code>SRV</code> record, you must specify a
         * value for <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both.
         * </p>
         * <p>
         * <b>AWS_INSTANCE_PORT</b>
         * </p>
         * <p>
         * If the service includes an <code>SRV</code> record, the value that you want Route 53 to return for the port.
         * </p>
         * <p>
         * If the service includes <code>HealthCheckConfig</code>, the port on the endpoint that you want Route 53 to
         * send requests to.
         * </p>
         * <p>
         * This value is required if you specified settings for an <code>SRV</code> record or a Route 53 health check
         * when you created the service.
         * </p>
         * <p>
         * <b>Custom attributes</b>
         * </p>
         * <p>
         * You can add up to 30 custom attributes. For each key-value pair, the maximum length of the attribute name is
         * 255 characters, and the maximum length of the attribute value is 1,024 characters. The total size of all
         * provided attributes (sum of all keys and values) must not exceed 5,000 characters.
         * </p>
         * 
         * @param attributes
         *        A string map that contains the following information for the service that you specify in
         *        <code>ServiceId</code>:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The attributes that apply to the records that are defined in the service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For each attribute, the applicable value.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Supported attribute keys include the following:
         *        </p>
         *        <p>
         *        <b>AWS_ALIAS_DNS_NAME</b>
         *        </p>
         *        <p>
         *        If you want AWS Cloud Map to create an Amazon Route 53 alias record that routes traffic to an Elastic
         *        Load Balancing load balancer, specify the DNS name that is associated with the load balancer. For
         *        information about how to get the DNS name, see "DNSName" in the topic <a
         *        href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_AliasTarget.html">AliasTarget</a> in
         *        the <i>Route 53 API Reference</i>.
         *        </p>
         *        <p>
         *        Note the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The configuration for the service that is specified by <code>ServiceId</code> must include settings
         *        for an <code>A</code> record, an <code>AAAA</code> record, or both.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In the service that is specified by <code>ServiceId</code>, the value of <code>RoutingPolicy</code>
         *        must be <code>WEIGHTED</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the service that is specified by <code>ServiceId</code> includes <code>HealthCheckConfig</code>
         *        settings, AWS Cloud Map will create the Route 53 health check, but it won't associate the health check
         *        with the alias record.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Auto naming currently doesn't support creating alias records that route traffic to AWS resources other
         *        than Elastic Load Balancing load balancers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify a value for <code>AWS_ALIAS_DNS_NAME</code>, don't specify values for any of the
         *        <code>AWS_INSTANCE</code> attributes.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>AWS_EC2_INSTANCE_ID</b>
         *        </p>
         *        <p>
         *        <i>HTTP namespaces only.</i> The Amazon EC2 instance ID for the instance. If the
         *        <code>AWS_EC2_INSTANCE_ID</code> attribute is specified, then the only other attribute that can be
         *        specified is <code>AWS_INIT_HEALTH_STATUS</code>. When the <code>AWS_EC2_INSTANCE_ID</code> attribute
         *        is specified, then the <code>AWS_INSTANCE_IPV4</code> attribute will be filled out with the primary
         *        private IPv4 address.
         *        </p>
         *        <p>
         *        <b>AWS_INIT_HEALTH_STATUS</b>
         *        </p>
         *        <p>
         *        If the service configuration includes <code>HealthCheckCustomConfig</code>, you can optionally use
         *        <code>AWS_INIT_HEALTH_STATUS</code> to specify the initial status of the custom health check,
         *        <code>HEALTHY</code> or <code>UNHEALTHY</code>. If you don't specify a value for
         *        <code>AWS_INIT_HEALTH_STATUS</code>, the initial status is <code>HEALTHY</code>.
         *        </p>
         *        <p>
         *        <b>AWS_INSTANCE_CNAME</b>
         *        </p>
         *        <p>
         *        If the service configuration includes a <code>CNAME</code> record, the domain name that you want
         *        Route 53 to return in response to DNS queries, for example, <code>example.com</code>.
         *        </p>
         *        <p>
         *        This value is required if the service specified by <code>ServiceId</code> includes settings for an
         *        <code>CNAME</code> record.
         *        </p>
         *        <p>
         *        <b>AWS_INSTANCE_IPV4</b>
         *        </p>
         *        <p>
         *        If the service configuration includes an <code>A</code> record, the IPv4 address that you want
         *        Route 53 to return in response to DNS queries, for example, <code>192.0.2.44</code>.
         *        </p>
         *        <p>
         *        This value is required if the service specified by <code>ServiceId</code> includes settings for an
         *        <code>A</code> record. If the service includes settings for an <code>SRV</code> record, you must
         *        specify a value for <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both.
         *        </p>
         *        <p>
         *        <b>AWS_INSTANCE_IPV6</b>
         *        </p>
         *        <p>
         *        If the service configuration includes an <code>AAAA</code> record, the IPv6 address that you want
         *        Route 53 to return in response to DNS queries, for example,
         *        <code>2001:0db8:85a3:0000:0000:abcd:0001:2345</code>.
         *        </p>
         *        <p>
         *        This value is required if the service specified by <code>ServiceId</code> includes settings for an
         *        <code>AAAA</code> record. If the service includes settings for an <code>SRV</code> record, you must
         *        specify a value for <code>AWS_INSTANCE_IPV4</code>, <code>AWS_INSTANCE_IPV6</code>, or both.
         *        </p>
         *        <p>
         *        <b>AWS_INSTANCE_PORT</b>
         *        </p>
         *        <p>
         *        If the service includes an <code>SRV</code> record, the value that you want Route 53 to return for the
         *        port.
         *        </p>
         *        <p>
         *        If the service includes <code>HealthCheckConfig</code>, the port on the endpoint that you want
         *        Route 53 to send requests to.
         *        </p>
         *        <p>
         *        This value is required if you specified settings for an <code>SRV</code> record or a Route 53 health
         *        check when you created the service.
         *        </p>
         *        <p>
         *        <b>Custom attributes</b>
         *        </p>
         *        <p>
         *        You can add up to 30 custom attributes. For each key-value pair, the maximum length of the attribute
         *        name is 255 characters, and the maximum length of the attribute value is 1,024 characters. The total
         *        size of all provided attributes (sum of all keys and values) must not exceed 5,000 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Map<String, String> attributes);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ServiceDiscoveryRequest.BuilderImpl implements Builder {
        private String serviceId;

        private String instanceId;

        private String creatorRequestId;

        private Map<String, String> attributes = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RegisterInstanceRequest model) {
            super(model);
            serviceId(model.serviceId);
            instanceId(model.instanceId);
            creatorRequestId(model.creatorRequestId);
            attributes(model.attributes);
        }

        public final String getServiceId() {
            return serviceId;
        }

        @Override
        public final Builder serviceId(String serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public final void setServiceId(String serviceId) {
            this.serviceId = serviceId;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        public final String getCreatorRequestId() {
            return creatorRequestId;
        }

        @Override
        public final Builder creatorRequestId(String creatorRequestId) {
            this.creatorRequestId = creatorRequestId;
            return this;
        }

        public final void setCreatorRequestId(String creatorRequestId) {
            this.creatorRequestId = creatorRequestId;
        }

        public final Map<String, String> getAttributes() {
            if (attributes instanceof SdkAutoConstructMap) {
                return null;
            }
            return attributes;
        }

        @Override
        public final Builder attributes(Map<String, String> attributes) {
            this.attributes = AttributesCopier.copy(attributes);
            return this;
        }

        public final void setAttributes(Map<String, String> attributes) {
            this.attributes = AttributesCopier.copy(attributes);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public RegisterInstanceRequest build() {
            return new RegisterInstanceRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
