/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicecatalog.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An AWS CloudFormation stack, in a specific account and region, that's part of a stack set operation. A stack instance
 * is a reference to an attempted or actual stack in a given account within a given region. A stack instance can exist
 * without a stack—for example, if the stack couldn't be created for some reason. A stack instance is associated with
 * only one stack set. Each stack instance contains the ID of its associated stack set, as well as the ID of the actual
 * stack and the stack status.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackInstance implements SdkPojo, Serializable, ToCopyableBuilder<StackInstance.Builder, StackInstance> {
    private static final SdkField<String> ACCOUNT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Account")
            .getter(getter(StackInstance::account)).setter(setter(Builder::account))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Account").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Region")
            .getter(getter(StackInstance::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Region").build()).build();

    private static final SdkField<String> STACK_INSTANCE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StackInstanceStatus").getter(getter(StackInstance::stackInstanceStatusAsString))
            .setter(setter(Builder::stackInstanceStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackInstanceStatus").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_FIELD, REGION_FIELD,
            STACK_INSTANCE_STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String account;

    private final String region;

    private final String stackInstanceStatus;

    private StackInstance(BuilderImpl builder) {
        this.account = builder.account;
        this.region = builder.region;
        this.stackInstanceStatus = builder.stackInstanceStatus;
    }

    /**
     * <p>
     * The name of the AWS account that the stack instance is associated with.
     * </p>
     * 
     * @return The name of the AWS account that the stack instance is associated with.
     */
    public final String account() {
        return account;
    }

    /**
     * <p>
     * The name of the AWS region that the stack instance is associated with.
     * </p>
     * 
     * @return The name of the AWS region that the stack instance is associated with.
     */
    public final String region() {
        return region;
    }

    /**
     * <p>
     * The status of the stack instance, in terms of its synchronization with its associated stack set.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack in an
     * unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code> operations. You might
     * need to perform a <code>DeleteStackInstances</code> operation, with <code>RetainStacks</code> set to true, to
     * delete the stack instance, and then delete the stack manually.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because either the associated
     * stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation, or the stack was part
     * of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that failed or was stopped before the
     * stack was created or updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CURRENT</code>: The stack is currently up to date with the stack set.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #stackInstanceStatus} will return {@link StackInstanceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #stackInstanceStatusAsString}.
     * </p>
     * 
     * @return The status of the stack instance, in terms of its synchronization with its associated stack set. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack in
     *         an unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code> operations.
     *         You might need to perform a <code>DeleteStackInstances</code> operation, with <code>RetainStacks</code>
     *         set to true, to delete the stack instance, and then delete the stack manually.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because either the
     *         associated stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation, or
     *         the stack was part of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that failed
     *         or was stopped before the stack was created or updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CURRENT</code>: The stack is currently up to date with the stack set.
     *         </p>
     *         </li>
     * @see StackInstanceStatus
     */
    public final StackInstanceStatus stackInstanceStatus() {
        return StackInstanceStatus.fromValue(stackInstanceStatus);
    }

    /**
     * <p>
     * The status of the stack instance, in terms of its synchronization with its associated stack set.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack in an
     * unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code> operations. You might
     * need to perform a <code>DeleteStackInstances</code> operation, with <code>RetainStacks</code> set to true, to
     * delete the stack instance, and then delete the stack manually.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because either the associated
     * stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation, or the stack was part
     * of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that failed or was stopped before the
     * stack was created or updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CURRENT</code>: The stack is currently up to date with the stack set.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #stackInstanceStatus} will return {@link StackInstanceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #stackInstanceStatusAsString}.
     * </p>
     * 
     * @return The status of the stack instance, in terms of its synchronization with its associated stack set. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack in
     *         an unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code> operations.
     *         You might need to perform a <code>DeleteStackInstances</code> operation, with <code>RetainStacks</code>
     *         set to true, to delete the stack instance, and then delete the stack manually.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because either the
     *         associated stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation, or
     *         the stack was part of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that failed
     *         or was stopped before the stack was created or updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CURRENT</code>: The stack is currently up to date with the stack set.
     *         </p>
     *         </li>
     * @see StackInstanceStatus
     */
    public final String stackInstanceStatusAsString() {
        return stackInstanceStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(account());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(stackInstanceStatusAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackInstance)) {
            return false;
        }
        StackInstance other = (StackInstance) obj;
        return Objects.equals(account(), other.account()) && Objects.equals(region(), other.region())
                && Objects.equals(stackInstanceStatusAsString(), other.stackInstanceStatusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StackInstance").add("Account", account()).add("Region", region())
                .add("StackInstanceStatus", stackInstanceStatusAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Account":
            return Optional.ofNullable(clazz.cast(account()));
        case "Region":
            return Optional.ofNullable(clazz.cast(region()));
        case "StackInstanceStatus":
            return Optional.ofNullable(clazz.cast(stackInstanceStatusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StackInstance, T> g) {
        return obj -> g.apply((StackInstance) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StackInstance> {
        /**
         * <p>
         * The name of the AWS account that the stack instance is associated with.
         * </p>
         * 
         * @param account
         *        The name of the AWS account that the stack instance is associated with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder account(String account);

        /**
         * <p>
         * The name of the AWS region that the stack instance is associated with.
         * </p>
         * 
         * @param region
         *        The name of the AWS region that the stack instance is associated with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The status of the stack instance, in terms of its synchronization with its associated stack set.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack in an
         * unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code> operations. You
         * might need to perform a <code>DeleteStackInstances</code> operation, with <code>RetainStacks</code> set to
         * true, to delete the stack instance, and then delete the stack manually.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because either the associated
         * stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation, or the stack was
         * part of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that failed or was stopped
         * before the stack was created or updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CURRENT</code>: The stack is currently up to date with the stack set.
         * </p>
         * </li>
         * </ul>
         * 
         * @param stackInstanceStatus
         *        The status of the stack instance, in terms of its synchronization with its associated stack set. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack
         *        in an unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code>
         *        operations. You might need to perform a <code>DeleteStackInstances</code> operation, with
         *        <code>RetainStacks</code> set to true, to delete the stack instance, and then delete the stack
         *        manually.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because either the
         *        associated stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation,
         *        or the stack was part of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that
         *        failed or was stopped before the stack was created or updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CURRENT</code>: The stack is currently up to date with the stack set.
         *        </p>
         *        </li>
         * @see StackInstanceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackInstanceStatus
         */
        Builder stackInstanceStatus(String stackInstanceStatus);

        /**
         * <p>
         * The status of the stack instance, in terms of its synchronization with its associated stack set.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack in an
         * unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code> operations. You
         * might need to perform a <code>DeleteStackInstances</code> operation, with <code>RetainStacks</code> set to
         * true, to delete the stack instance, and then delete the stack manually.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because either the associated
         * stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation, or the stack was
         * part of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that failed or was stopped
         * before the stack was created or updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CURRENT</code>: The stack is currently up to date with the stack set.
         * </p>
         * </li>
         * </ul>
         * 
         * @param stackInstanceStatus
         *        The status of the stack instance, in terms of its synchronization with its associated stack set. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack
         *        in an unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code>
         *        operations. You might need to perform a <code>DeleteStackInstances</code> operation, with
         *        <code>RetainStacks</code> set to true, to delete the stack instance, and then delete the stack
         *        manually.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because either the
         *        associated stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation,
         *        or the stack was part of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that
         *        failed or was stopped before the stack was created or updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CURRENT</code>: The stack is currently up to date with the stack set.
         *        </p>
         *        </li>
         * @see StackInstanceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackInstanceStatus
         */
        Builder stackInstanceStatus(StackInstanceStatus stackInstanceStatus);
    }

    static final class BuilderImpl implements Builder {
        private String account;

        private String region;

        private String stackInstanceStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(StackInstance model) {
            account(model.account);
            region(model.region);
            stackInstanceStatus(model.stackInstanceStatus);
        }

        public final String getAccount() {
            return account;
        }

        @Override
        public final Builder account(String account) {
            this.account = account;
            return this;
        }

        public final void setAccount(String account) {
            this.account = account;
        }

        public final String getRegion() {
            return region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        public final String getStackInstanceStatus() {
            return stackInstanceStatus;
        }

        @Override
        public final Builder stackInstanceStatus(String stackInstanceStatus) {
            this.stackInstanceStatus = stackInstanceStatus;
            return this;
        }

        @Override
        public final Builder stackInstanceStatus(StackInstanceStatus stackInstanceStatus) {
            this.stackInstanceStatus(stackInstanceStatus == null ? null : stackInstanceStatus.toString());
            return this;
        }

        public final void setStackInstanceStatus(String stackInstanceStatus) {
            this.stackInstanceStatus = stackInstanceStatus;
        }

        @Override
        public StackInstance build() {
            return new StackInstance(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
