/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.paginators;

import java.util.Iterator;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.securityhub.SecurityHubClient;
import software.amazon.awssdk.services.securityhub.model.GetInsightsRequest;
import software.amazon.awssdk.services.securityhub.model.GetInsightsResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.securityhub.SecurityHubClient#getInsightsPaginator(software.amazon.awssdk.services.securityhub.model.GetInsightsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.securityhub.model.GetInsightsResponse} that can be used to iterate through all
 * the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.securityhub.paginators.GetInsightsIterable responses = client.getInsightsPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.securityhub.paginators.GetInsightsIterable responses = client.getInsightsPaginator(request);
 *     for (software.amazon.awssdk.services.securityhub.model.GetInsightsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.securityhub.paginators.GetInsightsIterable responses = client.getInsightsPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #getInsights(software.amazon.awssdk.services.securityhub.model.GetInsightsRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class GetInsightsIterable implements SdkIterable<GetInsightsResponse> {
    private final SecurityHubClient client;

    private final GetInsightsRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public GetInsightsIterable(SecurityHubClient client, GetInsightsRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new GetInsightsResponseFetcher();
    }

    @Override
    public Iterator<GetInsightsResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    private class GetInsightsResponseFetcher implements SyncPageFetcher<GetInsightsResponse> {
        @Override
        public boolean hasNextPage(GetInsightsResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public GetInsightsResponse nextPage(GetInsightsResponse previousPage) {
            if (previousPage == null) {
                return client.getInsights(firstRequest);
            }
            return client.getInsights(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
