/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemakeredge.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a model deployed on an edge device that is registered with SageMaker Edge Manager.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Model implements SdkPojo, Serializable, ToCopyableBuilder<Model.Builder, Model> {
    private static final SdkField<String> MODEL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelName").getter(getter(Model::modelName)).setter(setter(Builder::modelName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelName").build()).build();

    private static final SdkField<String> MODEL_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelVersion").getter(getter(Model::modelVersion)).setter(setter(Builder::modelVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelVersion").build()).build();

    private static final SdkField<Instant> LATEST_SAMPLE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LatestSampleTime").getter(getter(Model::latestSampleTime)).setter(setter(Builder::latestSampleTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LatestSampleTime").build()).build();

    private static final SdkField<Instant> LATEST_INFERENCE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LatestInference").getter(getter(Model::latestInference)).setter(setter(Builder::latestInference))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LatestInference").build()).build();

    private static final SdkField<List<EdgeMetric>> MODEL_METRICS_FIELD = SdkField
            .<List<EdgeMetric>> builder(MarshallingType.LIST)
            .memberName("ModelMetrics")
            .getter(getter(Model::modelMetrics))
            .setter(setter(Builder::modelMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelMetrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EdgeMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(EdgeMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MODEL_NAME_FIELD,
            MODEL_VERSION_FIELD, LATEST_SAMPLE_TIME_FIELD, LATEST_INFERENCE_FIELD, MODEL_METRICS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String modelName;

    private final String modelVersion;

    private final Instant latestSampleTime;

    private final Instant latestInference;

    private final List<EdgeMetric> modelMetrics;

    private Model(BuilderImpl builder) {
        this.modelName = builder.modelName;
        this.modelVersion = builder.modelVersion;
        this.latestSampleTime = builder.latestSampleTime;
        this.latestInference = builder.latestInference;
        this.modelMetrics = builder.modelMetrics;
    }

    /**
     * <p>
     * The name of the model.
     * </p>
     * 
     * @return The name of the model.
     */
    public final String modelName() {
        return modelName;
    }

    /**
     * <p>
     * The version of the model.
     * </p>
     * 
     * @return The version of the model.
     */
    public final String modelVersion() {
        return modelVersion;
    }

    /**
     * <p>
     * The timestamp of the last data sample taken.
     * </p>
     * 
     * @return The timestamp of the last data sample taken.
     */
    public final Instant latestSampleTime() {
        return latestSampleTime;
    }

    /**
     * <p>
     * The timestamp of the last inference that was made.
     * </p>
     * 
     * @return The timestamp of the last inference that was made.
     */
    public final Instant latestInference() {
        return latestInference;
    }

    /**
     * Returns true if the ModelMetrics property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasModelMetrics() {
        return modelMetrics != null && !(modelMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information required for model metrics.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasModelMetrics()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Information required for model metrics.
     */
    public final List<EdgeMetric> modelMetrics() {
        return modelMetrics;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(modelName());
        hashCode = 31 * hashCode + Objects.hashCode(modelVersion());
        hashCode = 31 * hashCode + Objects.hashCode(latestSampleTime());
        hashCode = 31 * hashCode + Objects.hashCode(latestInference());
        hashCode = 31 * hashCode + Objects.hashCode(hasModelMetrics() ? modelMetrics() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Model)) {
            return false;
        }
        Model other = (Model) obj;
        return Objects.equals(modelName(), other.modelName()) && Objects.equals(modelVersion(), other.modelVersion())
                && Objects.equals(latestSampleTime(), other.latestSampleTime())
                && Objects.equals(latestInference(), other.latestInference()) && hasModelMetrics() == other.hasModelMetrics()
                && Objects.equals(modelMetrics(), other.modelMetrics());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Model").add("ModelName", modelName()).add("ModelVersion", modelVersion())
                .add("LatestSampleTime", latestSampleTime()).add("LatestInference", latestInference())
                .add("ModelMetrics", hasModelMetrics() ? modelMetrics() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ModelName":
            return Optional.ofNullable(clazz.cast(modelName()));
        case "ModelVersion":
            return Optional.ofNullable(clazz.cast(modelVersion()));
        case "LatestSampleTime":
            return Optional.ofNullable(clazz.cast(latestSampleTime()));
        case "LatestInference":
            return Optional.ofNullable(clazz.cast(latestInference()));
        case "ModelMetrics":
            return Optional.ofNullable(clazz.cast(modelMetrics()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Model, T> g) {
        return obj -> g.apply((Model) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Model> {
        /**
         * <p>
         * The name of the model.
         * </p>
         * 
         * @param modelName
         *        The name of the model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelName(String modelName);

        /**
         * <p>
         * The version of the model.
         * </p>
         * 
         * @param modelVersion
         *        The version of the model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelVersion(String modelVersion);

        /**
         * <p>
         * The timestamp of the last data sample taken.
         * </p>
         * 
         * @param latestSampleTime
         *        The timestamp of the last data sample taken.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latestSampleTime(Instant latestSampleTime);

        /**
         * <p>
         * The timestamp of the last inference that was made.
         * </p>
         * 
         * @param latestInference
         *        The timestamp of the last inference that was made.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latestInference(Instant latestInference);

        /**
         * <p>
         * Information required for model metrics.
         * </p>
         * 
         * @param modelMetrics
         *        Information required for model metrics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelMetrics(Collection<EdgeMetric> modelMetrics);

        /**
         * <p>
         * Information required for model metrics.
         * </p>
         * 
         * @param modelMetrics
         *        Information required for model metrics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelMetrics(EdgeMetric... modelMetrics);

        /**
         * <p>
         * Information required for model metrics.
         * </p>
         * This is a convenience that creates an instance of the {@link List<EdgeMetric>.Builder} avoiding the need to
         * create one manually via {@link List<EdgeMetric>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<EdgeMetric>.Builder#build()} is called immediately and its
         * result is passed to {@link #modelMetrics(List<EdgeMetric>)}.
         * 
         * @param modelMetrics
         *        a consumer that will call methods on {@link List<EdgeMetric>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #modelMetrics(List<EdgeMetric>)
         */
        Builder modelMetrics(Consumer<EdgeMetric.Builder>... modelMetrics);
    }

    static final class BuilderImpl implements Builder {
        private String modelName;

        private String modelVersion;

        private Instant latestSampleTime;

        private Instant latestInference;

        private List<EdgeMetric> modelMetrics = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Model model) {
            modelName(model.modelName);
            modelVersion(model.modelVersion);
            latestSampleTime(model.latestSampleTime);
            latestInference(model.latestInference);
            modelMetrics(model.modelMetrics);
        }

        public final String getModelName() {
            return modelName;
        }

        @Override
        public final Builder modelName(String modelName) {
            this.modelName = modelName;
            return this;
        }

        public final void setModelName(String modelName) {
            this.modelName = modelName;
        }

        public final String getModelVersion() {
            return modelVersion;
        }

        @Override
        public final Builder modelVersion(String modelVersion) {
            this.modelVersion = modelVersion;
            return this;
        }

        public final void setModelVersion(String modelVersion) {
            this.modelVersion = modelVersion;
        }

        public final Instant getLatestSampleTime() {
            return latestSampleTime;
        }

        @Override
        public final Builder latestSampleTime(Instant latestSampleTime) {
            this.latestSampleTime = latestSampleTime;
            return this;
        }

        public final void setLatestSampleTime(Instant latestSampleTime) {
            this.latestSampleTime = latestSampleTime;
        }

        public final Instant getLatestInference() {
            return latestInference;
        }

        @Override
        public final Builder latestInference(Instant latestInference) {
            this.latestInference = latestInference;
            return this;
        }

        public final void setLatestInference(Instant latestInference) {
            this.latestInference = latestInference;
        }

        public final Collection<EdgeMetric.Builder> getModelMetrics() {
            if (modelMetrics instanceof SdkAutoConstructList) {
                return null;
            }
            return modelMetrics != null ? modelMetrics.stream().map(EdgeMetric::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder modelMetrics(Collection<EdgeMetric> modelMetrics) {
            this.modelMetrics = EdgeMetricsCopier.copy(modelMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder modelMetrics(EdgeMetric... modelMetrics) {
            modelMetrics(Arrays.asList(modelMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder modelMetrics(Consumer<EdgeMetric.Builder>... modelMetrics) {
            modelMetrics(Stream.of(modelMetrics).map(c -> EdgeMetric.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setModelMetrics(Collection<EdgeMetric.BuilderImpl> modelMetrics) {
            this.modelMetrics = EdgeMetricsCopier.copyFromBuilder(modelMetrics);
        }

        @Override
        public Model build() {
            return new Model(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
