/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A channel is a named input source that training algorithms can consume.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Channel implements SdkPojo, Serializable, ToCopyableBuilder<Channel.Builder, Channel> {
    private static final SdkField<String> CHANNEL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Channel::channelName)).setter(setter(Builder::channelName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChannelName").build()).build();

    private static final SdkField<DataSource> DATA_SOURCE_FIELD = SdkField.<DataSource> builder(MarshallingType.SDK_POJO)
            .getter(getter(Channel::dataSource)).setter(setter(Builder::dataSource)).constructor(DataSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataSource").build()).build();

    private static final SdkField<String> CONTENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Channel::contentType)).setter(setter(Builder::contentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentType").build()).build();

    private static final SdkField<String> COMPRESSION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Channel::compressionTypeAsString)).setter(setter(Builder::compressionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompressionType").build()).build();

    private static final SdkField<String> RECORD_WRAPPER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Channel::recordWrapperTypeAsString)).setter(setter(Builder::recordWrapperType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecordWrapperType").build()).build();

    private static final SdkField<String> INPUT_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Channel::inputModeAsString)).setter(setter(Builder::inputMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputMode").build()).build();

    private static final SdkField<ShuffleConfig> SHUFFLE_CONFIG_FIELD = SdkField
            .<ShuffleConfig> builder(MarshallingType.SDK_POJO).getter(getter(Channel::shuffleConfig))
            .setter(setter(Builder::shuffleConfig)).constructor(ShuffleConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShuffleConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CHANNEL_NAME_FIELD,
            DATA_SOURCE_FIELD, CONTENT_TYPE_FIELD, COMPRESSION_TYPE_FIELD, RECORD_WRAPPER_TYPE_FIELD, INPUT_MODE_FIELD,
            SHUFFLE_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String channelName;

    private final DataSource dataSource;

    private final String contentType;

    private final String compressionType;

    private final String recordWrapperType;

    private final String inputMode;

    private final ShuffleConfig shuffleConfig;

    private Channel(BuilderImpl builder) {
        this.channelName = builder.channelName;
        this.dataSource = builder.dataSource;
        this.contentType = builder.contentType;
        this.compressionType = builder.compressionType;
        this.recordWrapperType = builder.recordWrapperType;
        this.inputMode = builder.inputMode;
        this.shuffleConfig = builder.shuffleConfig;
    }

    /**
     * <p>
     * The name of the channel.
     * </p>
     * 
     * @return The name of the channel.
     */
    public String channelName() {
        return channelName;
    }

    /**
     * <p>
     * The location of the channel data.
     * </p>
     * 
     * @return The location of the channel data.
     */
    public DataSource dataSource() {
        return dataSource;
    }

    /**
     * <p>
     * The MIME type of the data.
     * </p>
     * 
     * @return The MIME type of the data.
     */
    public String contentType() {
        return contentType;
    }

    /**
     * <p>
     * If training data is compressed, the compression type. The default value is <code>None</code>.
     * <code>CompressionType</code> is used only in Pipe input mode. In File mode, leave this field unset or set it to
     * None.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #compressionType}
     * will return {@link CompressionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #compressionTypeAsString}.
     * </p>
     * 
     * @return If training data is compressed, the compression type. The default value is <code>None</code>.
     *         <code>CompressionType</code> is used only in Pipe input mode. In File mode, leave this field unset or set
     *         it to None.
     * @see CompressionType
     */
    public CompressionType compressionType() {
        return CompressionType.fromValue(compressionType);
    }

    /**
     * <p>
     * If training data is compressed, the compression type. The default value is <code>None</code>.
     * <code>CompressionType</code> is used only in Pipe input mode. In File mode, leave this field unset or set it to
     * None.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #compressionType}
     * will return {@link CompressionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #compressionTypeAsString}.
     * </p>
     * 
     * @return If training data is compressed, the compression type. The default value is <code>None</code>.
     *         <code>CompressionType</code> is used only in Pipe input mode. In File mode, leave this field unset or set
     *         it to None.
     * @see CompressionType
     */
    public String compressionTypeAsString() {
        return compressionType;
    }

    /**
     * <p/>
     * <p>
     * Specify RecordIO as the value when input data is in raw format but the training algorithm requires the RecordIO
     * format. In this case, Amazon SageMaker wraps each individual S3 object in a RecordIO record. If the input data is
     * already in RecordIO format, you don't need to set this attribute. For more information, see <a
     * href="https://mxnet.apache.org/api/architecture/note_data_loading#data-format">Create a Dataset Using
     * RecordIO</a>.
     * </p>
     * <p>
     * In File mode, leave this field unset or set it to None.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #recordWrapperType}
     * will return {@link RecordWrapper#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #recordWrapperTypeAsString}.
     * </p>
     * 
     * @return <p>
     *         Specify RecordIO as the value when input data is in raw format but the training algorithm requires the
     *         RecordIO format. In this case, Amazon SageMaker wraps each individual S3 object in a RecordIO record. If
     *         the input data is already in RecordIO format, you don't need to set this attribute. For more information,
     *         see <a href="https://mxnet.apache.org/api/architecture/note_data_loading#data-format">Create a Dataset
     *         Using RecordIO</a>.
     *         </p>
     *         <p>
     *         In File mode, leave this field unset or set it to None.
     * @see RecordWrapper
     */
    public RecordWrapper recordWrapperType() {
        return RecordWrapper.fromValue(recordWrapperType);
    }

    /**
     * <p/>
     * <p>
     * Specify RecordIO as the value when input data is in raw format but the training algorithm requires the RecordIO
     * format. In this case, Amazon SageMaker wraps each individual S3 object in a RecordIO record. If the input data is
     * already in RecordIO format, you don't need to set this attribute. For more information, see <a
     * href="https://mxnet.apache.org/api/architecture/note_data_loading#data-format">Create a Dataset Using
     * RecordIO</a>.
     * </p>
     * <p>
     * In File mode, leave this field unset or set it to None.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #recordWrapperType}
     * will return {@link RecordWrapper#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #recordWrapperTypeAsString}.
     * </p>
     * 
     * @return <p>
     *         Specify RecordIO as the value when input data is in raw format but the training algorithm requires the
     *         RecordIO format. In this case, Amazon SageMaker wraps each individual S3 object in a RecordIO record. If
     *         the input data is already in RecordIO format, you don't need to set this attribute. For more information,
     *         see <a href="https://mxnet.apache.org/api/architecture/note_data_loading#data-format">Create a Dataset
     *         Using RecordIO</a>.
     *         </p>
     *         <p>
     *         In File mode, leave this field unset or set it to None.
     * @see RecordWrapper
     */
    public String recordWrapperTypeAsString() {
        return recordWrapperType;
    }

    /**
     * <p>
     * (Optional) The input mode to use for the data channel in a training job. If you don't set a value for
     * <code>InputMode</code>, Amazon SageMaker uses the value set for <code>TrainingInputMode</code>. Use this
     * parameter to override the <code>TrainingInputMode</code> setting in a <a>AlgorithmSpecification</a> request when
     * you have a channel that needs a different input mode from the training job's general setting. To download the
     * data from Amazon Simple Storage Service (Amazon S3) to the provisioned ML storage volume, and mount the directory
     * to a Docker volume, use <code>File</code> input mode. To stream data directly from Amazon S3 to the container,
     * choose <code>Pipe</code> input mode.
     * </p>
     * <p>
     * To use a model for incremental training, choose <code>File</code> input model.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #inputMode} will
     * return {@link TrainingInputMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #inputModeAsString}.
     * </p>
     * 
     * @return (Optional) The input mode to use for the data channel in a training job. If you don't set a value for
     *         <code>InputMode</code>, Amazon SageMaker uses the value set for <code>TrainingInputMode</code>. Use this
     *         parameter to override the <code>TrainingInputMode</code> setting in a <a>AlgorithmSpecification</a>
     *         request when you have a channel that needs a different input mode from the training job's general
     *         setting. To download the data from Amazon Simple Storage Service (Amazon S3) to the provisioned ML
     *         storage volume, and mount the directory to a Docker volume, use <code>File</code> input mode. To stream
     *         data directly from Amazon S3 to the container, choose <code>Pipe</code> input mode.</p>
     *         <p>
     *         To use a model for incremental training, choose <code>File</code> input model.
     * @see TrainingInputMode
     */
    public TrainingInputMode inputMode() {
        return TrainingInputMode.fromValue(inputMode);
    }

    /**
     * <p>
     * (Optional) The input mode to use for the data channel in a training job. If you don't set a value for
     * <code>InputMode</code>, Amazon SageMaker uses the value set for <code>TrainingInputMode</code>. Use this
     * parameter to override the <code>TrainingInputMode</code> setting in a <a>AlgorithmSpecification</a> request when
     * you have a channel that needs a different input mode from the training job's general setting. To download the
     * data from Amazon Simple Storage Service (Amazon S3) to the provisioned ML storage volume, and mount the directory
     * to a Docker volume, use <code>File</code> input mode. To stream data directly from Amazon S3 to the container,
     * choose <code>Pipe</code> input mode.
     * </p>
     * <p>
     * To use a model for incremental training, choose <code>File</code> input model.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #inputMode} will
     * return {@link TrainingInputMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #inputModeAsString}.
     * </p>
     * 
     * @return (Optional) The input mode to use for the data channel in a training job. If you don't set a value for
     *         <code>InputMode</code>, Amazon SageMaker uses the value set for <code>TrainingInputMode</code>. Use this
     *         parameter to override the <code>TrainingInputMode</code> setting in a <a>AlgorithmSpecification</a>
     *         request when you have a channel that needs a different input mode from the training job's general
     *         setting. To download the data from Amazon Simple Storage Service (Amazon S3) to the provisioned ML
     *         storage volume, and mount the directory to a Docker volume, use <code>File</code> input mode. To stream
     *         data directly from Amazon S3 to the container, choose <code>Pipe</code> input mode.</p>
     *         <p>
     *         To use a model for incremental training, choose <code>File</code> input model.
     * @see TrainingInputMode
     */
    public String inputModeAsString() {
        return inputMode;
    }

    /**
     * <p>
     * A configuration for a shuffle option for input data in a channel. If you use <code>S3Prefix</code> for
     * <code>S3DataType</code>, this shuffles the results of the S3 key prefix matches. If you use
     * <code>ManifestFile</code>, the order of the S3 object references in the <code>ManifestFile</code> is shuffled. If
     * you use <code>AugmentedManifestFile</code>, the order of the JSON lines in the <code>AugmentedManifestFile</code>
     * is shuffled. The shuffling order is determined using the <code>Seed</code> value.
     * </p>
     * <p>
     * For Pipe input mode, shuffling is done at the start of every epoch. With large datasets this ensures that the
     * order of the training data is different for each epoch, it helps reduce bias and possible overfitting. In a
     * multi-node training job when ShuffleConfig is combined with <code>S3DataDistributionType</code> of
     * <code>ShardedByS3Key</code>, the data is shuffled across nodes so that the content sent to a particular node on
     * the first epoch might be sent to a different node on the second epoch.
     * </p>
     * 
     * @return A configuration for a shuffle option for input data in a channel. If you use <code>S3Prefix</code> for
     *         <code>S3DataType</code>, this shuffles the results of the S3 key prefix matches. If you use
     *         <code>ManifestFile</code>, the order of the S3 object references in the <code>ManifestFile</code> is
     *         shuffled. If you use <code>AugmentedManifestFile</code>, the order of the JSON lines in the
     *         <code>AugmentedManifestFile</code> is shuffled. The shuffling order is determined using the
     *         <code>Seed</code> value.</p>
     *         <p>
     *         For Pipe input mode, shuffling is done at the start of every epoch. With large datasets this ensures that
     *         the order of the training data is different for each epoch, it helps reduce bias and possible
     *         overfitting. In a multi-node training job when ShuffleConfig is combined with
     *         <code>S3DataDistributionType</code> of <code>ShardedByS3Key</code>, the data is shuffled across nodes so
     *         that the content sent to a particular node on the first epoch might be sent to a different node on the
     *         second epoch.
     */
    public ShuffleConfig shuffleConfig() {
        return shuffleConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(channelName());
        hashCode = 31 * hashCode + Objects.hashCode(dataSource());
        hashCode = 31 * hashCode + Objects.hashCode(contentType());
        hashCode = 31 * hashCode + Objects.hashCode(compressionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(recordWrapperTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(inputModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(shuffleConfig());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Channel)) {
            return false;
        }
        Channel other = (Channel) obj;
        return Objects.equals(channelName(), other.channelName()) && Objects.equals(dataSource(), other.dataSource())
                && Objects.equals(contentType(), other.contentType())
                && Objects.equals(compressionTypeAsString(), other.compressionTypeAsString())
                && Objects.equals(recordWrapperTypeAsString(), other.recordWrapperTypeAsString())
                && Objects.equals(inputModeAsString(), other.inputModeAsString())
                && Objects.equals(shuffleConfig(), other.shuffleConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Channel").add("ChannelName", channelName()).add("DataSource", dataSource())
                .add("ContentType", contentType()).add("CompressionType", compressionTypeAsString())
                .add("RecordWrapperType", recordWrapperTypeAsString()).add("InputMode", inputModeAsString())
                .add("ShuffleConfig", shuffleConfig()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ChannelName":
            return Optional.ofNullable(clazz.cast(channelName()));
        case "DataSource":
            return Optional.ofNullable(clazz.cast(dataSource()));
        case "ContentType":
            return Optional.ofNullable(clazz.cast(contentType()));
        case "CompressionType":
            return Optional.ofNullable(clazz.cast(compressionTypeAsString()));
        case "RecordWrapperType":
            return Optional.ofNullable(clazz.cast(recordWrapperTypeAsString()));
        case "InputMode":
            return Optional.ofNullable(clazz.cast(inputModeAsString()));
        case "ShuffleConfig":
            return Optional.ofNullable(clazz.cast(shuffleConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Channel, T> g) {
        return obj -> g.apply((Channel) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Channel> {
        /**
         * <p>
         * The name of the channel.
         * </p>
         * 
         * @param channelName
         *        The name of the channel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder channelName(String channelName);

        /**
         * <p>
         * The location of the channel data.
         * </p>
         * 
         * @param dataSource
         *        The location of the channel data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSource(DataSource dataSource);

        /**
         * <p>
         * The location of the channel data.
         * </p>
         * This is a convenience that creates an instance of the {@link DataSource.Builder} avoiding the need to create
         * one manually via {@link DataSource#builder()}.
         *
         * When the {@link Consumer} completes, {@link DataSource.Builder#build()} is called immediately and its result
         * is passed to {@link #dataSource(DataSource)}.
         * 
         * @param dataSource
         *        a consumer that will call methods on {@link DataSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataSource(DataSource)
         */
        default Builder dataSource(Consumer<DataSource.Builder> dataSource) {
            return dataSource(DataSource.builder().applyMutation(dataSource).build());
        }

        /**
         * <p>
         * The MIME type of the data.
         * </p>
         * 
         * @param contentType
         *        The MIME type of the data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentType(String contentType);

        /**
         * <p>
         * If training data is compressed, the compression type. The default value is <code>None</code>.
         * <code>CompressionType</code> is used only in Pipe input mode. In File mode, leave this field unset or set it
         * to None.
         * </p>
         * 
         * @param compressionType
         *        If training data is compressed, the compression type. The default value is <code>None</code>.
         *        <code>CompressionType</code> is used only in Pipe input mode. In File mode, leave this field unset or
         *        set it to None.
         * @see CompressionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CompressionType
         */
        Builder compressionType(String compressionType);

        /**
         * <p>
         * If training data is compressed, the compression type. The default value is <code>None</code>.
         * <code>CompressionType</code> is used only in Pipe input mode. In File mode, leave this field unset or set it
         * to None.
         * </p>
         * 
         * @param compressionType
         *        If training data is compressed, the compression type. The default value is <code>None</code>.
         *        <code>CompressionType</code> is used only in Pipe input mode. In File mode, leave this field unset or
         *        set it to None.
         * @see CompressionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CompressionType
         */
        Builder compressionType(CompressionType compressionType);

        /**
         * <p/>
         * <p>
         * Specify RecordIO as the value when input data is in raw format but the training algorithm requires the
         * RecordIO format. In this case, Amazon SageMaker wraps each individual S3 object in a RecordIO record. If the
         * input data is already in RecordIO format, you don't need to set this attribute. For more information, see <a
         * href="https://mxnet.apache.org/api/architecture/note_data_loading#data-format">Create a Dataset Using
         * RecordIO</a>.
         * </p>
         * <p>
         * In File mode, leave this field unset or set it to None.
         * </p>
         * 
         * @param recordWrapperType
         *        <p>
         *        Specify RecordIO as the value when input data is in raw format but the training algorithm requires the
         *        RecordIO format. In this case, Amazon SageMaker wraps each individual S3 object in a RecordIO record.
         *        If the input data is already in RecordIO format, you don't need to set this attribute. For more
         *        information, see <a
         *        href="https://mxnet.apache.org/api/architecture/note_data_loading#data-format">Create a Dataset Using
         *        RecordIO</a>.
         *        </p>
         *        <p>
         *        In File mode, leave this field unset or set it to None.
         * @see RecordWrapper
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecordWrapper
         */
        Builder recordWrapperType(String recordWrapperType);

        /**
         * <p/>
         * <p>
         * Specify RecordIO as the value when input data is in raw format but the training algorithm requires the
         * RecordIO format. In this case, Amazon SageMaker wraps each individual S3 object in a RecordIO record. If the
         * input data is already in RecordIO format, you don't need to set this attribute. For more information, see <a
         * href="https://mxnet.apache.org/api/architecture/note_data_loading#data-format">Create a Dataset Using
         * RecordIO</a>.
         * </p>
         * <p>
         * In File mode, leave this field unset or set it to None.
         * </p>
         * 
         * @param recordWrapperType
         *        <p>
         *        Specify RecordIO as the value when input data is in raw format but the training algorithm requires the
         *        RecordIO format. In this case, Amazon SageMaker wraps each individual S3 object in a RecordIO record.
         *        If the input data is already in RecordIO format, you don't need to set this attribute. For more
         *        information, see <a
         *        href="https://mxnet.apache.org/api/architecture/note_data_loading#data-format">Create a Dataset Using
         *        RecordIO</a>.
         *        </p>
         *        <p>
         *        In File mode, leave this field unset or set it to None.
         * @see RecordWrapper
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecordWrapper
         */
        Builder recordWrapperType(RecordWrapper recordWrapperType);

        /**
         * <p>
         * (Optional) The input mode to use for the data channel in a training job. If you don't set a value for
         * <code>InputMode</code>, Amazon SageMaker uses the value set for <code>TrainingInputMode</code>. Use this
         * parameter to override the <code>TrainingInputMode</code> setting in a <a>AlgorithmSpecification</a> request
         * when you have a channel that needs a different input mode from the training job's general setting. To
         * download the data from Amazon Simple Storage Service (Amazon S3) to the provisioned ML storage volume, and
         * mount the directory to a Docker volume, use <code>File</code> input mode. To stream data directly from Amazon
         * S3 to the container, choose <code>Pipe</code> input mode.
         * </p>
         * <p>
         * To use a model for incremental training, choose <code>File</code> input model.
         * </p>
         * 
         * @param inputMode
         *        (Optional) The input mode to use for the data channel in a training job. If you don't set a value for
         *        <code>InputMode</code>, Amazon SageMaker uses the value set for <code>TrainingInputMode</code>. Use
         *        this parameter to override the <code>TrainingInputMode</code> setting in a
         *        <a>AlgorithmSpecification</a> request when you have a channel that needs a different input mode from
         *        the training job's general setting. To download the data from Amazon Simple Storage Service (Amazon
         *        S3) to the provisioned ML storage volume, and mount the directory to a Docker volume, use
         *        <code>File</code> input mode. To stream data directly from Amazon S3 to the container, choose
         *        <code>Pipe</code> input mode.</p>
         *        <p>
         *        To use a model for incremental training, choose <code>File</code> input model.
         * @see TrainingInputMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingInputMode
         */
        Builder inputMode(String inputMode);

        /**
         * <p>
         * (Optional) The input mode to use for the data channel in a training job. If you don't set a value for
         * <code>InputMode</code>, Amazon SageMaker uses the value set for <code>TrainingInputMode</code>. Use this
         * parameter to override the <code>TrainingInputMode</code> setting in a <a>AlgorithmSpecification</a> request
         * when you have a channel that needs a different input mode from the training job's general setting. To
         * download the data from Amazon Simple Storage Service (Amazon S3) to the provisioned ML storage volume, and
         * mount the directory to a Docker volume, use <code>File</code> input mode. To stream data directly from Amazon
         * S3 to the container, choose <code>Pipe</code> input mode.
         * </p>
         * <p>
         * To use a model for incremental training, choose <code>File</code> input model.
         * </p>
         * 
         * @param inputMode
         *        (Optional) The input mode to use for the data channel in a training job. If you don't set a value for
         *        <code>InputMode</code>, Amazon SageMaker uses the value set for <code>TrainingInputMode</code>. Use
         *        this parameter to override the <code>TrainingInputMode</code> setting in a
         *        <a>AlgorithmSpecification</a> request when you have a channel that needs a different input mode from
         *        the training job's general setting. To download the data from Amazon Simple Storage Service (Amazon
         *        S3) to the provisioned ML storage volume, and mount the directory to a Docker volume, use
         *        <code>File</code> input mode. To stream data directly from Amazon S3 to the container, choose
         *        <code>Pipe</code> input mode.</p>
         *        <p>
         *        To use a model for incremental training, choose <code>File</code> input model.
         * @see TrainingInputMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingInputMode
         */
        Builder inputMode(TrainingInputMode inputMode);

        /**
         * <p>
         * A configuration for a shuffle option for input data in a channel. If you use <code>S3Prefix</code> for
         * <code>S3DataType</code>, this shuffles the results of the S3 key prefix matches. If you use
         * <code>ManifestFile</code>, the order of the S3 object references in the <code>ManifestFile</code> is
         * shuffled. If you use <code>AugmentedManifestFile</code>, the order of the JSON lines in the
         * <code>AugmentedManifestFile</code> is shuffled. The shuffling order is determined using the <code>Seed</code>
         * value.
         * </p>
         * <p>
         * For Pipe input mode, shuffling is done at the start of every epoch. With large datasets this ensures that the
         * order of the training data is different for each epoch, it helps reduce bias and possible overfitting. In a
         * multi-node training job when ShuffleConfig is combined with <code>S3DataDistributionType</code> of
         * <code>ShardedByS3Key</code>, the data is shuffled across nodes so that the content sent to a particular node
         * on the first epoch might be sent to a different node on the second epoch.
         * </p>
         * 
         * @param shuffleConfig
         *        A configuration for a shuffle option for input data in a channel. If you use <code>S3Prefix</code> for
         *        <code>S3DataType</code>, this shuffles the results of the S3 key prefix matches. If you use
         *        <code>ManifestFile</code>, the order of the S3 object references in the <code>ManifestFile</code> is
         *        shuffled. If you use <code>AugmentedManifestFile</code>, the order of the JSON lines in the
         *        <code>AugmentedManifestFile</code> is shuffled. The shuffling order is determined using the
         *        <code>Seed</code> value.</p>
         *        <p>
         *        For Pipe input mode, shuffling is done at the start of every epoch. With large datasets this ensures
         *        that the order of the training data is different for each epoch, it helps reduce bias and possible
         *        overfitting. In a multi-node training job when ShuffleConfig is combined with
         *        <code>S3DataDistributionType</code> of <code>ShardedByS3Key</code>, the data is shuffled across nodes
         *        so that the content sent to a particular node on the first epoch might be sent to a different node on
         *        the second epoch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shuffleConfig(ShuffleConfig shuffleConfig);

        /**
         * <p>
         * A configuration for a shuffle option for input data in a channel. If you use <code>S3Prefix</code> for
         * <code>S3DataType</code>, this shuffles the results of the S3 key prefix matches. If you use
         * <code>ManifestFile</code>, the order of the S3 object references in the <code>ManifestFile</code> is
         * shuffled. If you use <code>AugmentedManifestFile</code>, the order of the JSON lines in the
         * <code>AugmentedManifestFile</code> is shuffled. The shuffling order is determined using the <code>Seed</code>
         * value.
         * </p>
         * <p>
         * For Pipe input mode, shuffling is done at the start of every epoch. With large datasets this ensures that the
         * order of the training data is different for each epoch, it helps reduce bias and possible overfitting. In a
         * multi-node training job when ShuffleConfig is combined with <code>S3DataDistributionType</code> of
         * <code>ShardedByS3Key</code>, the data is shuffled across nodes so that the content sent to a particular node
         * on the first epoch might be sent to a different node on the second epoch.
         * </p>
         * This is a convenience that creates an instance of the {@link ShuffleConfig.Builder} avoiding the need to
         * create one manually via {@link ShuffleConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link ShuffleConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #shuffleConfig(ShuffleConfig)}.
         * 
         * @param shuffleConfig
         *        a consumer that will call methods on {@link ShuffleConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #shuffleConfig(ShuffleConfig)
         */
        default Builder shuffleConfig(Consumer<ShuffleConfig.Builder> shuffleConfig) {
            return shuffleConfig(ShuffleConfig.builder().applyMutation(shuffleConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String channelName;

        private DataSource dataSource;

        private String contentType;

        private String compressionType;

        private String recordWrapperType;

        private String inputMode;

        private ShuffleConfig shuffleConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(Channel model) {
            channelName(model.channelName);
            dataSource(model.dataSource);
            contentType(model.contentType);
            compressionType(model.compressionType);
            recordWrapperType(model.recordWrapperType);
            inputMode(model.inputMode);
            shuffleConfig(model.shuffleConfig);
        }

        public final String getChannelName() {
            return channelName;
        }

        @Override
        public final Builder channelName(String channelName) {
            this.channelName = channelName;
            return this;
        }

        public final void setChannelName(String channelName) {
            this.channelName = channelName;
        }

        public final DataSource.Builder getDataSource() {
            return dataSource != null ? dataSource.toBuilder() : null;
        }

        @Override
        public final Builder dataSource(DataSource dataSource) {
            this.dataSource = dataSource;
            return this;
        }

        public final void setDataSource(DataSource.BuilderImpl dataSource) {
            this.dataSource = dataSource != null ? dataSource.build() : null;
        }

        public final String getContentType() {
            return contentType;
        }

        @Override
        public final Builder contentType(String contentType) {
            this.contentType = contentType;
            return this;
        }

        public final void setContentType(String contentType) {
            this.contentType = contentType;
        }

        public final String getCompressionType() {
            return compressionType;
        }

        @Override
        public final Builder compressionType(String compressionType) {
            this.compressionType = compressionType;
            return this;
        }

        @Override
        public final Builder compressionType(CompressionType compressionType) {
            this.compressionType(compressionType == null ? null : compressionType.toString());
            return this;
        }

        public final void setCompressionType(String compressionType) {
            this.compressionType = compressionType;
        }

        public final String getRecordWrapperType() {
            return recordWrapperType;
        }

        @Override
        public final Builder recordWrapperType(String recordWrapperType) {
            this.recordWrapperType = recordWrapperType;
            return this;
        }

        @Override
        public final Builder recordWrapperType(RecordWrapper recordWrapperType) {
            this.recordWrapperType(recordWrapperType == null ? null : recordWrapperType.toString());
            return this;
        }

        public final void setRecordWrapperType(String recordWrapperType) {
            this.recordWrapperType = recordWrapperType;
        }

        public final String getInputMode() {
            return inputMode;
        }

        @Override
        public final Builder inputMode(String inputMode) {
            this.inputMode = inputMode;
            return this;
        }

        @Override
        public final Builder inputMode(TrainingInputMode inputMode) {
            this.inputMode(inputMode == null ? null : inputMode.toString());
            return this;
        }

        public final void setInputMode(String inputMode) {
            this.inputMode = inputMode;
        }

        public final ShuffleConfig.Builder getShuffleConfig() {
            return shuffleConfig != null ? shuffleConfig.toBuilder() : null;
        }

        @Override
        public final Builder shuffleConfig(ShuffleConfig shuffleConfig) {
            this.shuffleConfig = shuffleConfig;
            return this;
        }

        public final void setShuffleConfig(ShuffleConfig.BuilderImpl shuffleConfig) {
            this.shuffleConfig = shuffleConfig != null ? shuffleConfig.build() : null;
        }

        @Override
        public Channel build() {
            return new Channel(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
