/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3vectors.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The encryption configuration for a vector bucket or index. By default, if you don't specify, all new vectors in
 * Amazon S3 vector buckets use server-side encryption with Amazon S3 managed keys (SSE-S3), specifically
 * <code>AES256</code>. You can optionally override bucket level encryption settings, and set a specific encryption
 * configuration for a vector index at the time of index creation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EncryptionConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<EncryptionConfiguration.Builder, EncryptionConfiguration> {
    private static final SdkField<String> SSE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sseType").getter(getter(EncryptionConfiguration::sseTypeAsString)).setter(setter(Builder::sseType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sseType").build()).build();

    private static final SdkField<String> KMS_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("kmsKeyArn").getter(getter(EncryptionConfiguration::kmsKeyArn)).setter(setter(Builder::kmsKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kmsKeyArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SSE_TYPE_FIELD,
            KMS_KEY_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String sseType;

    private final String kmsKeyArn;

    private EncryptionConfiguration(BuilderImpl builder) {
        this.sseType = builder.sseType;
        this.kmsKeyArn = builder.kmsKeyArn;
    }

    /**
     * <p>
     * The server-side encryption type to use for the encryption configuration of the vector bucket. By default, if you
     * don't specify, all new vectors in Amazon S3 vector buckets use server-side encryption with Amazon S3 managed keys
     * (SSE-S3), specifically <code>AES256</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sseType} will
     * return {@link SseType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sseTypeAsString}.
     * </p>
     * 
     * @return The server-side encryption type to use for the encryption configuration of the vector bucket. By default,
     *         if you don't specify, all new vectors in Amazon S3 vector buckets use server-side encryption with Amazon
     *         S3 managed keys (SSE-S3), specifically <code>AES256</code>.
     * @see SseType
     */
    public final SseType sseType() {
        return SseType.fromValue(sseType);
    }

    /**
     * <p>
     * The server-side encryption type to use for the encryption configuration of the vector bucket. By default, if you
     * don't specify, all new vectors in Amazon S3 vector buckets use server-side encryption with Amazon S3 managed keys
     * (SSE-S3), specifically <code>AES256</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sseType} will
     * return {@link SseType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sseTypeAsString}.
     * </p>
     * 
     * @return The server-side encryption type to use for the encryption configuration of the vector bucket. By default,
     *         if you don't specify, all new vectors in Amazon S3 vector buckets use server-side encryption with Amazon
     *         S3 managed keys (SSE-S3), specifically <code>AES256</code>.
     * @see SseType
     */
    public final String sseTypeAsString() {
        return sseType;
    }

    /**
     * <p>
     * Amazon Web Services Key Management Service (KMS) customer managed key ID to use for the encryption configuration.
     * This parameter is allowed if and only if <code>sseType</code> is set to <code>aws:kms</code>.
     * </p>
     * <p>
     * To specify the KMS key, you must use the format of the KMS key Amazon Resource Name (ARN).
     * </p>
     * <p>
     * For example, specify Key ARN in the following format:
     * <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * 
     * @return Amazon Web Services Key Management Service (KMS) customer managed key ID to use for the encryption
     *         configuration. This parameter is allowed if and only if <code>sseType</code> is set to
     *         <code>aws:kms</code>.</p>
     *         <p>
     *         To specify the KMS key, you must use the format of the KMS key Amazon Resource Name (ARN).
     *         </p>
     *         <p>
     *         For example, specify Key ARN in the following format:
     *         <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     */
    public final String kmsKeyArn() {
        return kmsKeyArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sseTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EncryptionConfiguration)) {
            return false;
        }
        EncryptionConfiguration other = (EncryptionConfiguration) obj;
        return Objects.equals(sseTypeAsString(), other.sseTypeAsString()) && Objects.equals(kmsKeyArn(), other.kmsKeyArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EncryptionConfiguration").add("SseType", sseTypeAsString()).add("KmsKeyArn", kmsKeyArn())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "sseType":
            return Optional.ofNullable(clazz.cast(sseTypeAsString()));
        case "kmsKeyArn":
            return Optional.ofNullable(clazz.cast(kmsKeyArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("sseType", SSE_TYPE_FIELD);
        map.put("kmsKeyArn", KMS_KEY_ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EncryptionConfiguration, T> g) {
        return obj -> g.apply((EncryptionConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EncryptionConfiguration> {
        /**
         * <p>
         * The server-side encryption type to use for the encryption configuration of the vector bucket. By default, if
         * you don't specify, all new vectors in Amazon S3 vector buckets use server-side encryption with Amazon S3
         * managed keys (SSE-S3), specifically <code>AES256</code>.
         * </p>
         * 
         * @param sseType
         *        The server-side encryption type to use for the encryption configuration of the vector bucket. By
         *        default, if you don't specify, all new vectors in Amazon S3 vector buckets use server-side encryption
         *        with Amazon S3 managed keys (SSE-S3), specifically <code>AES256</code>.
         * @see SseType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SseType
         */
        Builder sseType(String sseType);

        /**
         * <p>
         * The server-side encryption type to use for the encryption configuration of the vector bucket. By default, if
         * you don't specify, all new vectors in Amazon S3 vector buckets use server-side encryption with Amazon S3
         * managed keys (SSE-S3), specifically <code>AES256</code>.
         * </p>
         * 
         * @param sseType
         *        The server-side encryption type to use for the encryption configuration of the vector bucket. By
         *        default, if you don't specify, all new vectors in Amazon S3 vector buckets use server-side encryption
         *        with Amazon S3 managed keys (SSE-S3), specifically <code>AES256</code>.
         * @see SseType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SseType
         */
        Builder sseType(SseType sseType);

        /**
         * <p>
         * Amazon Web Services Key Management Service (KMS) customer managed key ID to use for the encryption
         * configuration. This parameter is allowed if and only if <code>sseType</code> is set to <code>aws:kms</code>.
         * </p>
         * <p>
         * To specify the KMS key, you must use the format of the KMS key Amazon Resource Name (ARN).
         * </p>
         * <p>
         * For example, specify Key ARN in the following format:
         * <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * 
         * @param kmsKeyArn
         *        Amazon Web Services Key Management Service (KMS) customer managed key ID to use for the encryption
         *        configuration. This parameter is allowed if and only if <code>sseType</code> is set to
         *        <code>aws:kms</code>.</p>
         *        <p>
         *        To specify the KMS key, you must use the format of the KMS key Amazon Resource Name (ARN).
         *        </p>
         *        <p>
         *        For example, specify Key ARN in the following format:
         *        <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyArn(String kmsKeyArn);
    }

    static final class BuilderImpl implements Builder {
        private String sseType;

        private String kmsKeyArn;

        private BuilderImpl() {
        }

        private BuilderImpl(EncryptionConfiguration model) {
            sseType(model.sseType);
            kmsKeyArn(model.kmsKeyArn);
        }

        public final String getSseType() {
            return sseType;
        }

        public final void setSseType(String sseType) {
            this.sseType = sseType;
        }

        @Override
        public final Builder sseType(String sseType) {
            this.sseType = sseType;
            return this;
        }

        @Override
        public final Builder sseType(SseType sseType) {
            this.sseType(sseType == null ? null : sseType.toString());
            return this;
        }

        public final String getKmsKeyArn() {
            return kmsKeyArn;
        }

        public final void setKmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
        }

        @Override
        public final Builder kmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
            return this;
        }

        @Override
        public EncryptionConfiguration build() {
            return new EncryptionConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
