/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the S3 Object Lock retention mode to be applied to all objects in the S3 Batch Operations job. If you don't
 * provide <code>Mode</code> and <code>RetainUntilDate</code> data types in your operation, you will remove the
 * retention from your objects. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-retention-date.html">Using S3 Object Lock retention
 * with S3 Batch Operations</a> in the <i>Amazon S3 User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3Retention implements SdkPojo, Serializable, ToCopyableBuilder<S3Retention.Builder, S3Retention> {
    private static final SdkField<Instant> RETAIN_UNTIL_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("RetainUntilDate")
            .getter(getter(S3Retention::retainUntilDate))
            .setter(setter(Builder::retainUntilDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RetainUntilDate")
                    .unmarshallLocationName("RetainUntilDate").build()).build();

    private static final SdkField<String> MODE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Mode")
            .getter(getter(S3Retention::modeAsString))
            .setter(setter(Builder::mode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Mode")
                    .unmarshallLocationName("Mode").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RETAIN_UNTIL_DATE_FIELD,
            MODE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Instant retainUntilDate;

    private final String mode;

    private S3Retention(BuilderImpl builder) {
        this.retainUntilDate = builder.retainUntilDate;
        this.mode = builder.mode;
    }

    /**
     * <p>
     * The date when the applied Object Lock retention will expire on all objects set by the Batch Operations job.
     * </p>
     * 
     * @return The date when the applied Object Lock retention will expire on all objects set by the Batch Operations
     *         job.
     */
    public final Instant retainUntilDate() {
        return retainUntilDate;
    }

    /**
     * <p>
     * The Object Lock retention mode to be applied to all objects in the Batch Operations job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link S3ObjectLockRetentionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return The Object Lock retention mode to be applied to all objects in the Batch Operations job.
     * @see S3ObjectLockRetentionMode
     */
    public final S3ObjectLockRetentionMode mode() {
        return S3ObjectLockRetentionMode.fromValue(mode);
    }

    /**
     * <p>
     * The Object Lock retention mode to be applied to all objects in the Batch Operations job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link S3ObjectLockRetentionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return The Object Lock retention mode to be applied to all objects in the Batch Operations job.
     * @see S3ObjectLockRetentionMode
     */
    public final String modeAsString() {
        return mode;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(retainUntilDate());
        hashCode = 31 * hashCode + Objects.hashCode(modeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3Retention)) {
            return false;
        }
        S3Retention other = (S3Retention) obj;
        return Objects.equals(retainUntilDate(), other.retainUntilDate()) && Objects.equals(modeAsString(), other.modeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3Retention").add("RetainUntilDate", retainUntilDate()).add("Mode", modeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RetainUntilDate":
            return Optional.ofNullable(clazz.cast(retainUntilDate()));
        case "Mode":
            return Optional.ofNullable(clazz.cast(modeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3Retention, T> g) {
        return obj -> g.apply((S3Retention) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3Retention> {
        /**
         * <p>
         * The date when the applied Object Lock retention will expire on all objects set by the Batch Operations job.
         * </p>
         * 
         * @param retainUntilDate
         *        The date when the applied Object Lock retention will expire on all objects set by the Batch Operations
         *        job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retainUntilDate(Instant retainUntilDate);

        /**
         * <p>
         * The Object Lock retention mode to be applied to all objects in the Batch Operations job.
         * </p>
         * 
         * @param mode
         *        The Object Lock retention mode to be applied to all objects in the Batch Operations job.
         * @see S3ObjectLockRetentionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3ObjectLockRetentionMode
         */
        Builder mode(String mode);

        /**
         * <p>
         * The Object Lock retention mode to be applied to all objects in the Batch Operations job.
         * </p>
         * 
         * @param mode
         *        The Object Lock retention mode to be applied to all objects in the Batch Operations job.
         * @see S3ObjectLockRetentionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3ObjectLockRetentionMode
         */
        Builder mode(S3ObjectLockRetentionMode mode);
    }

    static final class BuilderImpl implements Builder {
        private Instant retainUntilDate;

        private String mode;

        private BuilderImpl() {
        }

        private BuilderImpl(S3Retention model) {
            retainUntilDate(model.retainUntilDate);
            mode(model.mode);
        }

        public final Instant getRetainUntilDate() {
            return retainUntilDate;
        }

        public final void setRetainUntilDate(Instant retainUntilDate) {
            this.retainUntilDate = retainUntilDate;
        }

        @Override
        public final Builder retainUntilDate(Instant retainUntilDate) {
            this.retainUntilDate = retainUntilDate;
            return this;
        }

        public final String getMode() {
            return mode;
        }

        public final void setMode(String mode) {
            this.mode = mode;
        }

        @Override
        public final Builder mode(String mode) {
            this.mode = mode;
            return this;
        }

        @Override
        public final Builder mode(S3ObjectLockRetentionMode mode) {
            this.mode(mode == null ? null : mode.toString());
            return this;
        }

        @Override
        public S3Retention build() {
            return new S3Retention(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
