/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The filter used to describe a set of objects for the job's manifest.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobManifestGeneratorFilter implements SdkPojo, Serializable,
        ToCopyableBuilder<JobManifestGeneratorFilter.Builder, JobManifestGeneratorFilter> {
    private static final SdkField<Boolean> ELIGIBLE_FOR_REPLICATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EligibleForReplication")
            .getter(getter(JobManifestGeneratorFilter::eligibleForReplication))
            .setter(setter(Builder::eligibleForReplication))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EligibleForReplication")
                    .unmarshallLocationName("EligibleForReplication").build()).build();

    private static final SdkField<Instant> CREATED_AFTER_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAfter")
            .getter(getter(JobManifestGeneratorFilter::createdAfter))
            .setter(setter(Builder::createdAfter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAfter")
                    .unmarshallLocationName("CreatedAfter").build()).build();

    private static final SdkField<Instant> CREATED_BEFORE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedBefore")
            .getter(getter(JobManifestGeneratorFilter::createdBefore))
            .setter(setter(Builder::createdBefore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedBefore")
                    .unmarshallLocationName("CreatedBefore").build()).build();

    private static final SdkField<List<String>> OBJECT_REPLICATION_STATUSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ObjectReplicationStatuses")
            .getter(getter(JobManifestGeneratorFilter::objectReplicationStatusesAsStrings))
            .setter(setter(Builder::objectReplicationStatusesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectReplicationStatuses")
                    .unmarshallLocationName("ObjectReplicationStatuses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ELIGIBLE_FOR_REPLICATION_FIELD, CREATED_AFTER_FIELD, CREATED_BEFORE_FIELD, OBJECT_REPLICATION_STATUSES_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean eligibleForReplication;

    private final Instant createdAfter;

    private final Instant createdBefore;

    private final List<String> objectReplicationStatuses;

    private JobManifestGeneratorFilter(BuilderImpl builder) {
        this.eligibleForReplication = builder.eligibleForReplication;
        this.createdAfter = builder.createdAfter;
        this.createdBefore = builder.createdBefore;
        this.objectReplicationStatuses = builder.objectReplicationStatuses;
    }

    /**
     * <p>
     * Include objects in the generated manifest only if they are eligible for replication according to the Replication
     * configuration on the source bucket.
     * </p>
     * 
     * @return Include objects in the generated manifest only if they are eligible for replication according to the
     *         Replication configuration on the source bucket.
     */
    public final Boolean eligibleForReplication() {
        return eligibleForReplication;
    }

    /**
     * <p>
     * If provided, the generated manifest should include only source bucket objects that were created after this time.
     * </p>
     * 
     * @return If provided, the generated manifest should include only source bucket objects that were created after
     *         this time.
     */
    public final Instant createdAfter() {
        return createdAfter;
    }

    /**
     * <p>
     * If provided, the generated manifest should include only source bucket objects that were created before this time.
     * </p>
     * 
     * @return If provided, the generated manifest should include only source bucket objects that were created before
     *         this time.
     */
    public final Instant createdBefore() {
        return createdBefore;
    }

    /**
     * <p>
     * If provided, the generated manifest should include only source bucket objects that have one of the specified
     * Replication statuses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasObjectReplicationStatuses} method.
     * </p>
     * 
     * @return If provided, the generated manifest should include only source bucket objects that have one of the
     *         specified Replication statuses.
     */
    public final List<ReplicationStatus> objectReplicationStatuses() {
        return ReplicationStatusFilterListCopier.copyStringToEnum(objectReplicationStatuses);
    }

    /**
     * For responses, this returns true if the service returned a value for the ObjectReplicationStatuses property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasObjectReplicationStatuses() {
        return objectReplicationStatuses != null && !(objectReplicationStatuses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If provided, the generated manifest should include only source bucket objects that have one of the specified
     * Replication statuses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasObjectReplicationStatuses} method.
     * </p>
     * 
     * @return If provided, the generated manifest should include only source bucket objects that have one of the
     *         specified Replication statuses.
     */
    public final List<String> objectReplicationStatusesAsStrings() {
        return objectReplicationStatuses;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eligibleForReplication());
        hashCode = 31 * hashCode + Objects.hashCode(createdAfter());
        hashCode = 31 * hashCode + Objects.hashCode(createdBefore());
        hashCode = 31 * hashCode + Objects.hashCode(hasObjectReplicationStatuses() ? objectReplicationStatusesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobManifestGeneratorFilter)) {
            return false;
        }
        JobManifestGeneratorFilter other = (JobManifestGeneratorFilter) obj;
        return Objects.equals(eligibleForReplication(), other.eligibleForReplication())
                && Objects.equals(createdAfter(), other.createdAfter()) && Objects.equals(createdBefore(), other.createdBefore())
                && hasObjectReplicationStatuses() == other.hasObjectReplicationStatuses()
                && Objects.equals(objectReplicationStatusesAsStrings(), other.objectReplicationStatusesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JobManifestGeneratorFilter").add("EligibleForReplication", eligibleForReplication())
                .add("CreatedAfter", createdAfter()).add("CreatedBefore", createdBefore())
                .add("ObjectReplicationStatuses", hasObjectReplicationStatuses() ? objectReplicationStatusesAsStrings() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EligibleForReplication":
            return Optional.ofNullable(clazz.cast(eligibleForReplication()));
        case "CreatedAfter":
            return Optional.ofNullable(clazz.cast(createdAfter()));
        case "CreatedBefore":
            return Optional.ofNullable(clazz.cast(createdBefore()));
        case "ObjectReplicationStatuses":
            return Optional.ofNullable(clazz.cast(objectReplicationStatusesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<JobManifestGeneratorFilter, T> g) {
        return obj -> g.apply((JobManifestGeneratorFilter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobManifestGeneratorFilter> {
        /**
         * <p>
         * Include objects in the generated manifest only if they are eligible for replication according to the
         * Replication configuration on the source bucket.
         * </p>
         * 
         * @param eligibleForReplication
         *        Include objects in the generated manifest only if they are eligible for replication according to the
         *        Replication configuration on the source bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eligibleForReplication(Boolean eligibleForReplication);

        /**
         * <p>
         * If provided, the generated manifest should include only source bucket objects that were created after this
         * time.
         * </p>
         * 
         * @param createdAfter
         *        If provided, the generated manifest should include only source bucket objects that were created after
         *        this time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAfter(Instant createdAfter);

        /**
         * <p>
         * If provided, the generated manifest should include only source bucket objects that were created before this
         * time.
         * </p>
         * 
         * @param createdBefore
         *        If provided, the generated manifest should include only source bucket objects that were created before
         *        this time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBefore(Instant createdBefore);

        /**
         * <p>
         * If provided, the generated manifest should include only source bucket objects that have one of the specified
         * Replication statuses.
         * </p>
         * 
         * @param objectReplicationStatuses
         *        If provided, the generated manifest should include only source bucket objects that have one of the
         *        specified Replication statuses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectReplicationStatusesWithStrings(Collection<String> objectReplicationStatuses);

        /**
         * <p>
         * If provided, the generated manifest should include only source bucket objects that have one of the specified
         * Replication statuses.
         * </p>
         * 
         * @param objectReplicationStatuses
         *        If provided, the generated manifest should include only source bucket objects that have one of the
         *        specified Replication statuses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectReplicationStatusesWithStrings(String... objectReplicationStatuses);

        /**
         * <p>
         * If provided, the generated manifest should include only source bucket objects that have one of the specified
         * Replication statuses.
         * </p>
         * 
         * @param objectReplicationStatuses
         *        If provided, the generated manifest should include only source bucket objects that have one of the
         *        specified Replication statuses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectReplicationStatuses(Collection<ReplicationStatus> objectReplicationStatuses);

        /**
         * <p>
         * If provided, the generated manifest should include only source bucket objects that have one of the specified
         * Replication statuses.
         * </p>
         * 
         * @param objectReplicationStatuses
         *        If provided, the generated manifest should include only source bucket objects that have one of the
         *        specified Replication statuses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectReplicationStatuses(ReplicationStatus... objectReplicationStatuses);
    }

    static final class BuilderImpl implements Builder {
        private Boolean eligibleForReplication;

        private Instant createdAfter;

        private Instant createdBefore;

        private List<String> objectReplicationStatuses = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(JobManifestGeneratorFilter model) {
            eligibleForReplication(model.eligibleForReplication);
            createdAfter(model.createdAfter);
            createdBefore(model.createdBefore);
            objectReplicationStatusesWithStrings(model.objectReplicationStatuses);
        }

        public final Boolean getEligibleForReplication() {
            return eligibleForReplication;
        }

        public final void setEligibleForReplication(Boolean eligibleForReplication) {
            this.eligibleForReplication = eligibleForReplication;
        }

        @Override
        public final Builder eligibleForReplication(Boolean eligibleForReplication) {
            this.eligibleForReplication = eligibleForReplication;
            return this;
        }

        public final Instant getCreatedAfter() {
            return createdAfter;
        }

        public final void setCreatedAfter(Instant createdAfter) {
            this.createdAfter = createdAfter;
        }

        @Override
        public final Builder createdAfter(Instant createdAfter) {
            this.createdAfter = createdAfter;
            return this;
        }

        public final Instant getCreatedBefore() {
            return createdBefore;
        }

        public final void setCreatedBefore(Instant createdBefore) {
            this.createdBefore = createdBefore;
        }

        @Override
        public final Builder createdBefore(Instant createdBefore) {
            this.createdBefore = createdBefore;
            return this;
        }

        public final Collection<String> getObjectReplicationStatuses() {
            if (objectReplicationStatuses instanceof SdkAutoConstructList) {
                return null;
            }
            return objectReplicationStatuses;
        }

        public final void setObjectReplicationStatuses(Collection<String> objectReplicationStatuses) {
            this.objectReplicationStatuses = ReplicationStatusFilterListCopier.copy(objectReplicationStatuses);
        }

        @Override
        public final Builder objectReplicationStatusesWithStrings(Collection<String> objectReplicationStatuses) {
            this.objectReplicationStatuses = ReplicationStatusFilterListCopier.copy(objectReplicationStatuses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder objectReplicationStatusesWithStrings(String... objectReplicationStatuses) {
            objectReplicationStatusesWithStrings(Arrays.asList(objectReplicationStatuses));
            return this;
        }

        @Override
        public final Builder objectReplicationStatuses(Collection<ReplicationStatus> objectReplicationStatuses) {
            this.objectReplicationStatuses = ReplicationStatusFilterListCopier.copyEnumToString(objectReplicationStatuses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder objectReplicationStatuses(ReplicationStatus... objectReplicationStatuses) {
            objectReplicationStatuses(Arrays.asList(objectReplicationStatuses));
            return this;
        }

        @Override
        public JobManifestGeneratorFilter build() {
            return new JobManifestGeneratorFilter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
