/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A container for the information associated with a <a
 * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateMultiRegionAccessPoint.html"
 * >CreateMultiRegionAccessPoint</a> request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateMultiRegionAccessPointInput implements SdkPojo, Serializable,
        ToCopyableBuilder<CreateMultiRegionAccessPointInput.Builder, CreateMultiRegionAccessPointInput> {
    private static final SdkField<String> NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Name")
            .getter(getter(CreateMultiRegionAccessPointInput::name))
            .setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name")
                    .unmarshallLocationName("Name").build()).build();

    private static final SdkField<PublicAccessBlockConfiguration> PUBLIC_ACCESS_BLOCK_FIELD = SdkField
            .<PublicAccessBlockConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("PublicAccessBlock")
            .getter(getter(CreateMultiRegionAccessPointInput::publicAccessBlock))
            .setter(setter(Builder::publicAccessBlock))
            .constructor(PublicAccessBlockConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublicAccessBlock")
                    .unmarshallLocationName("PublicAccessBlock").build()).build();

    private static final SdkField<List<Region>> REGIONS_FIELD = SdkField
            .<List<Region>> builder(MarshallingType.LIST)
            .memberName("Regions")
            .getter(getter(CreateMultiRegionAccessPointInput::regions))
            .setter(setter(Builder::regions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Regions")
                    .unmarshallLocationName("Regions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Region")
                            .memberFieldInfo(
                                    SdkField.<Region> builder(MarshallingType.SDK_POJO)
                                            .constructor(Region::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Region").unmarshallLocationName("Region").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            PUBLIC_ACCESS_BLOCK_FIELD, REGIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final PublicAccessBlockConfiguration publicAccessBlock;

    private final List<Region> regions;

    private CreateMultiRegionAccessPointInput(BuilderImpl builder) {
        this.name = builder.name;
        this.publicAccessBlock = builder.publicAccessBlock;
        this.regions = builder.regions;
    }

    /**
     * <p>
     * The name of the Multi-Region Access Point associated with this request.
     * </p>
     * 
     * @return The name of the Multi-Region Access Point associated with this request.
     */
    public final String name() {
        return name;
    }

    /**
     * Returns the value of the PublicAccessBlock property for this object.
     * 
     * @return The value of the PublicAccessBlock property for this object.
     */
    public final PublicAccessBlockConfiguration publicAccessBlock() {
        return publicAccessBlock;
    }

    /**
     * For responses, this returns true if the service returned a value for the Regions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasRegions() {
        return regions != null && !(regions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The buckets in different Regions that are associated with the Multi-Region Access Point.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRegions} method.
     * </p>
     * 
     * @return The buckets in different Regions that are associated with the Multi-Region Access Point.
     */
    public final List<Region> regions() {
        return regions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(publicAccessBlock());
        hashCode = 31 * hashCode + Objects.hashCode(hasRegions() ? regions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateMultiRegionAccessPointInput)) {
            return false;
        }
        CreateMultiRegionAccessPointInput other = (CreateMultiRegionAccessPointInput) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(publicAccessBlock(), other.publicAccessBlock())
                && hasRegions() == other.hasRegions() && Objects.equals(regions(), other.regions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateMultiRegionAccessPointInput").add("Name", name())
                .add("PublicAccessBlock", publicAccessBlock()).add("Regions", hasRegions() ? regions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "PublicAccessBlock":
            return Optional.ofNullable(clazz.cast(publicAccessBlock()));
        case "Regions":
            return Optional.ofNullable(clazz.cast(regions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateMultiRegionAccessPointInput, T> g) {
        return obj -> g.apply((CreateMultiRegionAccessPointInput) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CreateMultiRegionAccessPointInput> {
        /**
         * <p>
         * The name of the Multi-Region Access Point associated with this request.
         * </p>
         * 
         * @param name
         *        The name of the Multi-Region Access Point associated with this request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * Sets the value of the PublicAccessBlock property for this object.
         *
         * @param publicAccessBlock
         *        The new value for the PublicAccessBlock property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicAccessBlock(PublicAccessBlockConfiguration publicAccessBlock);

        /**
         * Sets the value of the PublicAccessBlock property for this object.
         *
         * This is a convenience method that creates an instance of the {@link PublicAccessBlockConfiguration.Builder}
         * avoiding the need to create one manually via {@link PublicAccessBlockConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PublicAccessBlockConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #publicAccessBlock(PublicAccessBlockConfiguration)}.
         * 
         * @param publicAccessBlock
         *        a consumer that will call methods on {@link PublicAccessBlockConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #publicAccessBlock(PublicAccessBlockConfiguration)
         */
        default Builder publicAccessBlock(Consumer<PublicAccessBlockConfiguration.Builder> publicAccessBlock) {
            return publicAccessBlock(PublicAccessBlockConfiguration.builder().applyMutation(publicAccessBlock).build());
        }

        /**
         * <p>
         * The buckets in different Regions that are associated with the Multi-Region Access Point.
         * </p>
         * 
         * @param regions
         *        The buckets in different Regions that are associated with the Multi-Region Access Point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(Collection<Region> regions);

        /**
         * <p>
         * The buckets in different Regions that are associated with the Multi-Region Access Point.
         * </p>
         * 
         * @param regions
         *        The buckets in different Regions that are associated with the Multi-Region Access Point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(Region... regions);

        /**
         * <p>
         * The buckets in different Regions that are associated with the Multi-Region Access Point.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.s3control.model.Region.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.s3control.model.Region#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.s3control.model.Region.Builder#build()} is called immediately and its
         * result is passed to {@link #regions(List<Region>)}.
         * 
         * @param regions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.s3control.model.Region.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #regions(java.util.Collection<Region>)
         */
        Builder regions(Consumer<Region.Builder>... regions);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private PublicAccessBlockConfiguration publicAccessBlock;

        private List<Region> regions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateMultiRegionAccessPointInput model) {
            name(model.name);
            publicAccessBlock(model.publicAccessBlock);
            regions(model.regions);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final PublicAccessBlockConfiguration.Builder getPublicAccessBlock() {
            return publicAccessBlock != null ? publicAccessBlock.toBuilder() : null;
        }

        public final void setPublicAccessBlock(PublicAccessBlockConfiguration.BuilderImpl publicAccessBlock) {
            this.publicAccessBlock = publicAccessBlock != null ? publicAccessBlock.build() : null;
        }

        @Override
        public final Builder publicAccessBlock(PublicAccessBlockConfiguration publicAccessBlock) {
            this.publicAccessBlock = publicAccessBlock;
            return this;
        }

        public final List<Region.Builder> getRegions() {
            List<Region.Builder> result = RegionCreationListCopier.copyToBuilder(this.regions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRegions(Collection<Region.BuilderImpl> regions) {
            this.regions = RegionCreationListCopier.copyFromBuilder(regions);
        }

        @Override
        public final Builder regions(Collection<Region> regions) {
            this.regions = RegionCreationListCopier.copy(regions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder regions(Region... regions) {
            regions(Arrays.asList(regions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder regions(Consumer<Region.Builder>... regions) {
            regions(Stream.of(regions).map(c -> Region.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public CreateMultiRegionAccessPointInput build() {
            return new CreateMultiRegionAccessPointInput(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
