/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the configuration parameters for a Set Object Retention operation. Amazon S3 Batch Operations passes each
 * value through to the underlying PUT Object Retention API. For more information about the parameters for this
 * operation, see <a
 * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-retention-modes">PUT
 * Object Retention</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3SetObjectRetentionOperation implements SdkPojo, Serializable,
        ToCopyableBuilder<S3SetObjectRetentionOperation.Builder, S3SetObjectRetentionOperation> {
    private static final SdkField<Boolean> BYPASS_GOVERNANCE_RETENTION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(S3SetObjectRetentionOperation::bypassGovernanceRetention))
            .setter(setter(Builder::bypassGovernanceRetention))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BypassGovernanceRetention")
                    .unmarshallLocationName("BypassGovernanceRetention").build()).build();

    private static final SdkField<S3Retention> RETENTION_FIELD = SdkField
            .<S3Retention> builder(MarshallingType.SDK_POJO)
            .getter(getter(S3SetObjectRetentionOperation::retention))
            .setter(setter(Builder::retention))
            .constructor(S3Retention::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Retention")
                    .unmarshallLocationName("Retention").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            BYPASS_GOVERNANCE_RETENTION_FIELD, RETENTION_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean bypassGovernanceRetention;

    private final S3Retention retention;

    private S3SetObjectRetentionOperation(BuilderImpl builder) {
        this.bypassGovernanceRetention = builder.bypassGovernanceRetention;
        this.retention = builder.retention;
    }

    /**
     * <p>
     * Indicates if the operation should be applied to objects in the Batch Operations job even if they have
     * Governance-type Object Lock in place.
     * </p>
     * 
     * @return Indicates if the operation should be applied to objects in the Batch Operations job even if they have
     *         Governance-type Object Lock in place.
     */
    public Boolean bypassGovernanceRetention() {
        return bypassGovernanceRetention;
    }

    /**
     * <p>
     * Amazon S3 object lock Retention contains the retention mode to be applied to all objects in the Batch Operations
     * job.
     * </p>
     * 
     * @return Amazon S3 object lock Retention contains the retention mode to be applied to all objects in the Batch
     *         Operations job.
     */
    public S3Retention retention() {
        return retention;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bypassGovernanceRetention());
        hashCode = 31 * hashCode + Objects.hashCode(retention());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3SetObjectRetentionOperation)) {
            return false;
        }
        S3SetObjectRetentionOperation other = (S3SetObjectRetentionOperation) obj;
        return Objects.equals(bypassGovernanceRetention(), other.bypassGovernanceRetention())
                && Objects.equals(retention(), other.retention());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("S3SetObjectRetentionOperation").add("BypassGovernanceRetention", bypassGovernanceRetention())
                .add("Retention", retention()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BypassGovernanceRetention":
            return Optional.ofNullable(clazz.cast(bypassGovernanceRetention()));
        case "Retention":
            return Optional.ofNullable(clazz.cast(retention()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3SetObjectRetentionOperation, T> g) {
        return obj -> g.apply((S3SetObjectRetentionOperation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3SetObjectRetentionOperation> {
        /**
         * <p>
         * Indicates if the operation should be applied to objects in the Batch Operations job even if they have
         * Governance-type Object Lock in place.
         * </p>
         * 
         * @param bypassGovernanceRetention
         *        Indicates if the operation should be applied to objects in the Batch Operations job even if they have
         *        Governance-type Object Lock in place.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bypassGovernanceRetention(Boolean bypassGovernanceRetention);

        /**
         * <p>
         * Amazon S3 object lock Retention contains the retention mode to be applied to all objects in the Batch
         * Operations job.
         * </p>
         * 
         * @param retention
         *        Amazon S3 object lock Retention contains the retention mode to be applied to all objects in the Batch
         *        Operations job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retention(S3Retention retention);

        /**
         * <p>
         * Amazon S3 object lock Retention contains the retention mode to be applied to all objects in the Batch
         * Operations job.
         * </p>
         * This is a convenience that creates an instance of the {@link S3Retention.Builder} avoiding the need to create
         * one manually via {@link S3Retention#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3Retention.Builder#build()} is called immediately and its result
         * is passed to {@link #retention(S3Retention)}.
         * 
         * @param retention
         *        a consumer that will call methods on {@link S3Retention.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #retention(S3Retention)
         */
        default Builder retention(Consumer<S3Retention.Builder> retention) {
            return retention(S3Retention.builder().applyMutation(retention).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Boolean bypassGovernanceRetention;

        private S3Retention retention;

        private BuilderImpl() {
        }

        private BuilderImpl(S3SetObjectRetentionOperation model) {
            bypassGovernanceRetention(model.bypassGovernanceRetention);
            retention(model.retention);
        }

        public final Boolean getBypassGovernanceRetention() {
            return bypassGovernanceRetention;
        }

        @Override
        public final Builder bypassGovernanceRetention(Boolean bypassGovernanceRetention) {
            this.bypassGovernanceRetention = bypassGovernanceRetention;
            return this;
        }

        public final void setBypassGovernanceRetention(Boolean bypassGovernanceRetention) {
            this.bypassGovernanceRetention = bypassGovernanceRetention;
        }

        public final S3Retention.Builder getRetention() {
            return retention != null ? retention.toBuilder() : null;
        }

        @Override
        public final Builder retention(S3Retention retention) {
            this.retention = retention;
            return this;
        }

        public final void setRetention(S3Retention.BuilderImpl retention) {
            this.retention = retention != null ? retention.build() : null;
        }

        @Override
        public S3SetObjectRetentionOperation build() {
            return new S3SetObjectRetentionOperation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
