/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.PayloadTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class RestoreObjectRequest extends S3Request implements
        ToCopyableBuilder<RestoreObjectRequest.Builder, RestoreObjectRequest> {
    private static final SdkField<String> BUCKET_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Bucket")
            .getter(getter(RestoreObjectRequest::bucket))
            .setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("Bucket")
                    .unmarshallLocationName("Bucket").build()).build();

    private static final SdkField<String> KEY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Key")
            .getter(getter(RestoreObjectRequest::key))
            .setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.GREEDY_PATH).locationName("Key")
                    .unmarshallLocationName("Key").build()).build();

    private static final SdkField<String> VERSION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VersionId")
            .getter(getter(RestoreObjectRequest::versionId))
            .setter(setter(Builder::versionId))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("versionId")
                    .unmarshallLocationName("versionId").build()).build();

    private static final SdkField<RestoreRequest> RESTORE_REQUEST_FIELD = SdkField
            .<RestoreRequest> builder(MarshallingType.SDK_POJO)
            .memberName("RestoreRequest")
            .getter(getter(RestoreObjectRequest::restoreRequest))
            .setter(setter(Builder::restoreRequest))
            .constructor(RestoreRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RestoreRequest")
                    .unmarshallLocationName("RestoreRequest").build(), PayloadTrait.create()).build();

    private static final SdkField<String> REQUEST_PAYER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RequestPayer")
            .getter(getter(RestoreObjectRequest::requestPayerAsString))
            .setter(setter(Builder::requestPayer))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-request-payer")
                    .unmarshallLocationName("x-amz-request-payer").build()).build();

    private static final SdkField<String> EXPECTED_BUCKET_OWNER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ExpectedBucketOwner")
            .getter(getter(RestoreObjectRequest::expectedBucketOwner))
            .setter(setter(Builder::expectedBucketOwner))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-expected-bucket-owner")
                    .unmarshallLocationName("x-amz-expected-bucket-owner").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_FIELD, KEY_FIELD,
            VERSION_ID_FIELD, RESTORE_REQUEST_FIELD, REQUEST_PAYER_FIELD, EXPECTED_BUCKET_OWNER_FIELD));

    private final String bucket;

    private final String key;

    private final String versionId;

    private final RestoreRequest restoreRequest;

    private final String requestPayer;

    private final String expectedBucketOwner;

    private RestoreObjectRequest(BuilderImpl builder) {
        super(builder);
        this.bucket = builder.bucket;
        this.key = builder.key;
        this.versionId = builder.versionId;
        this.restoreRequest = builder.restoreRequest;
        this.requestPayer = builder.requestPayer;
        this.expectedBucketOwner = builder.expectedBucketOwner;
    }

    /**
     * <p>
     * The bucket name or containing the object to restore.
     * </p>
     * <p>
     * When using this API with an access point, you must direct requests to the access point hostname. The access point
     * hostname takes the form <i>AccessPointName</i>-<i>AccountId</i>.s3-accesspoint.<i>Region</i>.amazonaws.com. When
     * using this operation using an access point through the AWS SDKs, you provide the access point ARN in place of the
     * bucket name. For more information about access point ARNs, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/using-access-points.html">Using Access Points</a> in the
     * <i>Amazon Simple Storage Service Developer Guide</i>.
     * </p>
     * <p>
     * When using this API with Amazon S3 on Outposts, you must direct requests to the S3 on Outposts hostname. The S3
     * on Outposts hostname takes the form
     * <i>AccessPointName</i>-<i>AccountId</i>.<i>outpostID</i>.s3-outposts.<i>Region</i>.amazonaws.com. When using this
     * operation using S3 on Outposts through the AWS SDKs, you provide the Outposts bucket ARN in place of the bucket
     * name. For more information about S3 on Outposts ARNs, see <a href="https://docs.aws.amazon.com/">Using S3 on
     * Outposts</a> in the <i>Amazon Simple Storage Service Developer Guide</i>.
     * </p>
     * 
     * @return The bucket name or containing the object to restore. </p>
     *         <p>
     *         When using this API with an access point, you must direct requests to the access point hostname. The
     *         access point hostname takes the form
     *         <i>AccessPointName</i>-<i>AccountId</i>.s3-accesspoint.<i>Region</i>.amazonaws.com. When using this
     *         operation using an access point through the AWS SDKs, you provide the access point ARN in place of the
     *         bucket name. For more information about access point ARNs, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/using-access-points.html">Using Access Points</a>
     *         in the <i>Amazon Simple Storage Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         When using this API with Amazon S3 on Outposts, you must direct requests to the S3 on Outposts hostname.
     *         The S3 on Outposts hostname takes the form
     *         <i>AccessPointName</i>-<i>AccountId</i>.<i>outpostID</i>.s3-outposts.<i>Region</i>.amazonaws.com. When
     *         using this operation using S3 on Outposts through the AWS SDKs, you provide the Outposts bucket ARN in
     *         place of the bucket name. For more information about S3 on Outposts ARNs, see <a
     *         href="https://docs.aws.amazon.com/">Using S3 on Outposts</a> in the <i>Amazon Simple Storage Service
     *         Developer Guide</i>.
     */
    public String bucket() {
        return bucket;
    }

    /**
     * <p>
     * Object key for which the operation was initiated.
     * </p>
     * 
     * @return Object key for which the operation was initiated.
     */
    public String key() {
        return key;
    }

    /**
     * <p>
     * VersionId used to reference a specific version of the object.
     * </p>
     * 
     * @return VersionId used to reference a specific version of the object.
     */
    public String versionId() {
        return versionId;
    }

    /**
     * Returns the value of the RestoreRequest property for this object.
     * 
     * @return The value of the RestoreRequest property for this object.
     */
    public RestoreRequest restoreRequest() {
        return restoreRequest;
    }

    /**
     * Returns the value of the RequestPayer property for this object.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #requestPayer} will
     * return {@link RequestPayer#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #requestPayerAsString}.
     * </p>
     * 
     * @return The value of the RequestPayer property for this object.
     * @see RequestPayer
     */
    public RequestPayer requestPayer() {
        return RequestPayer.fromValue(requestPayer);
    }

    /**
     * Returns the value of the RequestPayer property for this object.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #requestPayer} will
     * return {@link RequestPayer#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #requestPayerAsString}.
     * </p>
     * 
     * @return The value of the RequestPayer property for this object.
     * @see RequestPayer
     */
    public String requestPayerAsString() {
        return requestPayer;
    }

    /**
     * <p>
     * The account id of the expected bucket owner. If the bucket is owned by a different account, the request will fail
     * with an HTTP <code>403 (Access Denied)</code> error.
     * </p>
     * 
     * @return The account id of the expected bucket owner. If the bucket is owned by a different account, the request
     *         will fail with an HTTP <code>403 (Access Denied)</code> error.
     */
    public String expectedBucketOwner() {
        return expectedBucketOwner;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(versionId());
        hashCode = 31 * hashCode + Objects.hashCode(restoreRequest());
        hashCode = 31 * hashCode + Objects.hashCode(requestPayerAsString());
        hashCode = 31 * hashCode + Objects.hashCode(expectedBucketOwner());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RestoreObjectRequest)) {
            return false;
        }
        RestoreObjectRequest other = (RestoreObjectRequest) obj;
        return Objects.equals(bucket(), other.bucket()) && Objects.equals(key(), other.key())
                && Objects.equals(versionId(), other.versionId()) && Objects.equals(restoreRequest(), other.restoreRequest())
                && Objects.equals(requestPayerAsString(), other.requestPayerAsString())
                && Objects.equals(expectedBucketOwner(), other.expectedBucketOwner());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("RestoreObjectRequest").add("Bucket", bucket()).add("Key", key()).add("VersionId", versionId())
                .add("RestoreRequest", restoreRequest()).add("RequestPayer", requestPayerAsString())
                .add("ExpectedBucketOwner", expectedBucketOwner()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "Key":
            return Optional.ofNullable(clazz.cast(key()));
        case "VersionId":
            return Optional.ofNullable(clazz.cast(versionId()));
        case "RestoreRequest":
            return Optional.ofNullable(clazz.cast(restoreRequest()));
        case "RequestPayer":
            return Optional.ofNullable(clazz.cast(requestPayerAsString()));
        case "ExpectedBucketOwner":
            return Optional.ofNullable(clazz.cast(expectedBucketOwner()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RestoreObjectRequest, T> g) {
        return obj -> g.apply((RestoreObjectRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends S3Request.Builder, SdkPojo, CopyableBuilder<Builder, RestoreObjectRequest> {
        /**
         * <p>
         * The bucket name or containing the object to restore.
         * </p>
         * <p>
         * When using this API with an access point, you must direct requests to the access point hostname. The access
         * point hostname takes the form
         * <i>AccessPointName</i>-<i>AccountId</i>.s3-accesspoint.<i>Region</i>.amazonaws.com. When using this operation
         * using an access point through the AWS SDKs, you provide the access point ARN in place of the bucket name. For
         * more information about access point ARNs, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/using-access-points.html">Using Access Points</a> in
         * the <i>Amazon Simple Storage Service Developer Guide</i>.
         * </p>
         * <p>
         * When using this API with Amazon S3 on Outposts, you must direct requests to the S3 on Outposts hostname. The
         * S3 on Outposts hostname takes the form
         * <i>AccessPointName</i>-<i>AccountId</i>.<i>outpostID</i>.s3-outposts.<i>Region</i>.amazonaws.com. When using
         * this operation using S3 on Outposts through the AWS SDKs, you provide the Outposts bucket ARN in place of the
         * bucket name. For more information about S3 on Outposts ARNs, see <a href="https://docs.aws.amazon.com/">Using
         * S3 on Outposts</a> in the <i>Amazon Simple Storage Service Developer Guide</i>.
         * </p>
         * 
         * @param bucket
         *        The bucket name or containing the object to restore. </p>
         *        <p>
         *        When using this API with an access point, you must direct requests to the access point hostname. The
         *        access point hostname takes the form
         *        <i>AccessPointName</i>-<i>AccountId</i>.s3-accesspoint.<i>Region</i>.amazonaws.com. When using this
         *        operation using an access point through the AWS SDKs, you provide the access point ARN in place of the
         *        bucket name. For more information about access point ARNs, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/using-access-points.html">Using Access
         *        Points</a> in the <i>Amazon Simple Storage Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        When using this API with Amazon S3 on Outposts, you must direct requests to the S3 on Outposts
         *        hostname. The S3 on Outposts hostname takes the form
         *        <i>AccessPointName</i>-<i>AccountId</i>.<i>outpostID</i>.s3-outposts.<i>Region</i>.amazonaws.com. When
         *        using this operation using S3 on Outposts through the AWS SDKs, you provide the Outposts bucket ARN in
         *        place of the bucket name. For more information about S3 on Outposts ARNs, see <a
         *        href="https://docs.aws.amazon.com/">Using S3 on Outposts</a> in the <i>Amazon Simple Storage Service
         *        Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * Object key for which the operation was initiated.
         * </p>
         * 
         * @param key
         *        Object key for which the operation was initiated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * VersionId used to reference a specific version of the object.
         * </p>
         * 
         * @param versionId
         *        VersionId used to reference a specific version of the object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionId(String versionId);

        /**
         * Sets the value of the RestoreRequest property for this object.
         *
         * @param restoreRequest
         *        The new value for the RestoreRequest property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder restoreRequest(RestoreRequest restoreRequest);

        /**
         * Sets the value of the RestoreRequest property for this object.
         *
         * This is a convenience that creates an instance of the {@link RestoreRequest.Builder} avoiding the need to
         * create one manually via {@link RestoreRequest#builder()}.
         *
         * When the {@link Consumer} completes, {@link RestoreRequest.Builder#build()} is called immediately and its
         * result is passed to {@link #restoreRequest(RestoreRequest)}.
         * 
         * @param restoreRequest
         *        a consumer that will call methods on {@link RestoreRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #restoreRequest(RestoreRequest)
         */
        default Builder restoreRequest(Consumer<RestoreRequest.Builder> restoreRequest) {
            return restoreRequest(RestoreRequest.builder().applyMutation(restoreRequest).build());
        }

        /**
         * Sets the value of the RequestPayer property for this object.
         *
         * @param requestPayer
         *        The new value for the RequestPayer property for this object.
         * @see RequestPayer
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RequestPayer
         */
        Builder requestPayer(String requestPayer);

        /**
         * Sets the value of the RequestPayer property for this object.
         *
         * @param requestPayer
         *        The new value for the RequestPayer property for this object.
         * @see RequestPayer
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RequestPayer
         */
        Builder requestPayer(RequestPayer requestPayer);

        /**
         * <p>
         * The account id of the expected bucket owner. If the bucket is owned by a different account, the request will
         * fail with an HTTP <code>403 (Access Denied)</code> error.
         * </p>
         * 
         * @param expectedBucketOwner
         *        The account id of the expected bucket owner. If the bucket is owned by a different account, the
         *        request will fail with an HTTP <code>403 (Access Denied)</code> error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expectedBucketOwner(String expectedBucketOwner);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends S3Request.BuilderImpl implements Builder {
        private String bucket;

        private String key;

        private String versionId;

        private RestoreRequest restoreRequest;

        private String requestPayer;

        private String expectedBucketOwner;

        private BuilderImpl() {
        }

        private BuilderImpl(RestoreObjectRequest model) {
            super(model);
            bucket(model.bucket);
            key(model.key);
            versionId(model.versionId);
            restoreRequest(model.restoreRequest);
            requestPayer(model.requestPayer);
            expectedBucketOwner(model.expectedBucketOwner);
        }

        public final String getBucket() {
            return bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        public final String getKey() {
            return key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        public final String getVersionId() {
            return versionId;
        }

        @Override
        public final Builder versionId(String versionId) {
            this.versionId = versionId;
            return this;
        }

        public final void setVersionId(String versionId) {
            this.versionId = versionId;
        }

        public final RestoreRequest.Builder getRestoreRequest() {
            return restoreRequest != null ? restoreRequest.toBuilder() : null;
        }

        @Override
        public final Builder restoreRequest(RestoreRequest restoreRequest) {
            this.restoreRequest = restoreRequest;
            return this;
        }

        public final void setRestoreRequest(RestoreRequest.BuilderImpl restoreRequest) {
            this.restoreRequest = restoreRequest != null ? restoreRequest.build() : null;
        }

        public final String getRequestPayer() {
            return requestPayer;
        }

        @Override
        public final Builder requestPayer(String requestPayer) {
            this.requestPayer = requestPayer;
            return this;
        }

        @Override
        public final Builder requestPayer(RequestPayer requestPayer) {
            this.requestPayer(requestPayer == null ? null : requestPayer.toString());
            return this;
        }

        public final void setRequestPayer(String requestPayer) {
            this.requestPayer = requestPayer;
        }

        public final String getExpectedBucketOwner() {
            return expectedBucketOwner;
        }

        @Override
        public final Builder expectedBucketOwner(String expectedBucketOwner) {
            this.expectedBucketOwner = expectedBucketOwner;
            return this;
        }

        public final void setExpectedBucketOwner(String expectedBucketOwner) {
            this.expectedBucketOwner = expectedBucketOwner;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public RestoreObjectRequest build() {
            return new RestoreObjectRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
