/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3.transform;

import static software.amazon.awssdk.util.StringUtils.UTF8;

import java.io.StringWriter;

import javax.annotation.Generated;

import software.amazon.awssdk.SdkClientException;
import software.amazon.awssdk.Request;
import software.amazon.awssdk.DefaultRequest;
import software.amazon.awssdk.http.HttpMethodName;
import software.amazon.awssdk.services.s3.model.*;
import software.amazon.awssdk.runtime.transform.Marshaller;

import software.amazon.awssdk.util.StringInputStream;
import software.amazon.awssdk.util.StringUtils;

import software.amazon.awssdk.util.Md5Utils;
import software.amazon.awssdk.util.XmlWriter;

/**
 * RestoreObjectRequest Marshaller
 */

@Generated("software.amazon.awssdk:aws-java-sdk-code-generator")
public class RestoreObjectRequestMarshaller implements Marshaller<Request<RestoreObjectRequest>, RestoreObjectRequest> {

    public Request<RestoreObjectRequest> marshall(RestoreObjectRequest restoreObjectRequest) {

        if (restoreObjectRequest == null) {
            throw new SdkClientException("Invalid argument passed to marshall(...)");
        }

        Request<RestoreObjectRequest> request = new DefaultRequest<RestoreObjectRequest>(restoreObjectRequest, "S3Client");

        request.setHttpMethod(HttpMethodName.POST);

        if (restoreObjectRequest.requestPayer() != null) {
            request.addHeader("x-amz-request-payer", StringUtils.fromString(restoreObjectRequest.requestPayer()));
        }

        String uriResourcePath = "/{Bucket}/{Key+}?restore";

        uriResourcePath = software.amazon.awssdk.util.UriResourcePathUtils.addStaticQueryParamtersToRequest(request,
                uriResourcePath);

        uriResourcePath = software.amazon.awssdk.runtime.transform.PathMarshallers.NON_GREEDY.marshall(uriResourcePath, "Bucket",
                restoreObjectRequest.bucket());
        uriResourcePath = software.amazon.awssdk.runtime.transform.PathMarshallers.GREEDY.marshall(uriResourcePath, "Key",
                restoreObjectRequest.key());
        request.setResourcePath(uriResourcePath);

        if (restoreObjectRequest.versionId() != null) {
            request.addParameter("versionId", StringUtils.fromString(restoreObjectRequest.versionId()));
        }

        try {
            StringWriter stringWriter = null;
            RestoreRequest restoreRequest = restoreObjectRequest.restoreRequest();
            if (restoreRequest != null) {
                stringWriter = new StringWriter();
                XmlWriter xmlWriter = new XmlWriter(stringWriter, "http://s3.amazonaws.com/doc/2006-03-01/");
                xmlWriter.startElement("RestoreRequest");

                if (restoreRequest.days() != null) {
                    xmlWriter.startElement("Days").value(restoreRequest.days()).endElement();
                }

                GlacierJobParameters glacierJobParameters = restoreRequest.glacierJobParameters();
                if (glacierJobParameters != null) {
                    xmlWriter.startElement("GlacierJobParameters");

                    if (glacierJobParameters.tier() != null) {
                        xmlWriter.startElement("Tier").value(glacierJobParameters.tier()).endElement();
                    }
                    xmlWriter.endElement();
                }
                xmlWriter.endElement();
            }

            if (stringWriter != null) {
                if (!request.getHeaders().containsKey("Content-MD5")) {
                    request.addHeader("Content-MD5", Md5Utils.md5AsBase64(stringWriter.getBuffer().toString().getBytes(UTF8)));
                }
                request.setContent(new StringInputStream(stringWriter.getBuffer().toString()));
                request.addHeader("Content-Length", Integer.toString(stringWriter.getBuffer().toString().getBytes(UTF8).length));
            }
            if (!request.getHeaders().containsKey("Content-Type")) {
                request.addHeader("Content-Type", "application/xml");
            }
        } catch (Throwable t) {
            throw new SdkClientException("Unable to marshall request to XML: " + t.getMessage(), t);
        }

        return request;
    }

}