/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.transfer.s3.internal;

import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.function.Function;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.async.SdkPublisher;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.transfer.s3.S3TransferManager;
import software.amazon.awssdk.transfer.s3.internal.AsyncBufferingSubscriber;
import software.amazon.awssdk.transfer.s3.internal.TransferConfigurationOption;
import software.amazon.awssdk.transfer.s3.internal.TransferManagerConfiguration;
import software.amazon.awssdk.transfer.s3.internal.model.DefaultDirectoryUpload;
import software.amazon.awssdk.transfer.s3.model.CompletedDirectoryUpload;
import software.amazon.awssdk.transfer.s3.model.CompletedFileUpload;
import software.amazon.awssdk.transfer.s3.model.DirectoryUpload;
import software.amazon.awssdk.transfer.s3.model.FailedFileUpload;
import software.amazon.awssdk.transfer.s3.model.FileUpload;
import software.amazon.awssdk.transfer.s3.model.UploadDirectoryRequest;
import software.amazon.awssdk.transfer.s3.model.UploadFileRequest;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Logger;
import software.amazon.awssdk.utils.StringUtils;
import software.amazon.awssdk.utils.Validate;

@SdkInternalApi
public class UploadDirectoryHelper {
    private static final Logger log = Logger.loggerFor(S3TransferManager.class);
    private final TransferManagerConfiguration transferConfiguration;
    private final Function<UploadFileRequest, FileUpload> uploadFunction;

    public UploadDirectoryHelper(TransferManagerConfiguration transferConfiguration, Function<UploadFileRequest, FileUpload> uploadFunction) {
        this.transferConfiguration = transferConfiguration;
        this.uploadFunction = uploadFunction;
    }

    public DirectoryUpload uploadDirectory(UploadDirectoryRequest uploadDirectoryRequest) {
        CompletableFuture<CompletedDirectoryUpload> returnFuture = new CompletableFuture<CompletedDirectoryUpload>();
        CompletableFuture.runAsync(() -> this.doUploadDirectory(returnFuture, uploadDirectoryRequest), this.transferConfiguration.option(TransferConfigurationOption.EXECUTOR)).whenComplete((r, t) -> {
            if (t != null) {
                returnFuture.completeExceptionally((Throwable)t);
            }
        });
        return new DefaultDirectoryUpload(returnFuture);
    }

    private void doUploadDirectory(CompletableFuture<CompletedDirectoryUpload> returnFuture, UploadDirectoryRequest uploadDirectoryRequest) {
        Path directory = uploadDirectoryRequest.source();
        this.validateDirectory(uploadDirectoryRequest);
        ConcurrentLinkedQueue failedFileUploads = new ConcurrentLinkedQueue();
        Stream<Path> stream = this.listFiles(directory, uploadDirectoryRequest);
        SdkPublisher iterablePublisher = SdkPublisher.fromIterable(() -> stream.iterator()).doAfterOnCancel(() -> stream.close()).doAfterOnError(t -> stream.close()).doAfterOnComplete(() -> stream.close());
        CompletableFuture<Void> allOfFutures = new CompletableFuture<Void>();
        AsyncBufferingSubscriber<Path> bufferingSubscriber = new AsyncBufferingSubscriber<Path>(path -> this.uploadSingleFile(uploadDirectoryRequest, failedFileUploads, (Path)path), allOfFutures, this.transferConfiguration.option(TransferConfigurationOption.DIRECTORY_TRANSFER_MAX_CONCURRENCY));
        iterablePublisher.subscribe(bufferingSubscriber);
        CompletableFutureUtils.forwardExceptionTo(returnFuture, allOfFutures);
        allOfFutures.whenComplete((r, t) -> {
            if (t != null) {
                returnFuture.completeExceptionally((Throwable)SdkClientException.create((String)"Failed to send request", (Throwable)t));
                return;
            }
            returnFuture.complete(CompletedDirectoryUpload.builder().failedTransfers(failedFileUploads).build());
        });
    }

    private void validateDirectory(UploadDirectoryRequest uploadDirectoryRequest) {
        Path directory = uploadDirectoryRequest.source();
        Validate.isTrue((boolean)Files.exists(directory, new LinkOption[0]), (String)"The source directory provided (%s) does not exist", (Object[])new Object[]{directory});
        boolean followSymbolicLinks = this.transferConfiguration.resolveUploadDirectoryFollowSymbolicLinks(uploadDirectoryRequest);
        if (followSymbolicLinks) {
            Validate.isTrue((boolean)Files.isDirectory(directory, new LinkOption[0]), (String)"The source directory provided (%s) is not a directory", (Object[])new Object[]{directory});
        } else {
            Validate.isTrue((boolean)Files.isDirectory(directory, LinkOption.NOFOLLOW_LINKS), (String)"The source directory provided (%s) is not a directory", (Object[])new Object[]{directory});
        }
    }

    private CompletableFuture<CompletedFileUpload> uploadSingleFile(UploadDirectoryRequest uploadDirectoryRequest, Collection<FailedFileUpload> failedFileUploads, Path path) {
        int nameCount = uploadDirectoryRequest.source().getNameCount();
        UploadFileRequest uploadFileRequest = this.constructUploadRequest(uploadDirectoryRequest, nameCount, path);
        log.debug(() -> String.format("Sending upload request (%s) for path (%s)", uploadFileRequest, path));
        CompletableFuture<CompletedFileUpload> executionFuture = this.uploadFunction.apply(uploadFileRequest).completionFuture();
        CompletionStage future = executionFuture.whenComplete((r, t) -> {
            if (t != null) {
                failedFileUploads.add((FailedFileUpload)FailedFileUpload.builder().exception(t instanceof CompletionException ? t.getCause() : t).request(uploadFileRequest).build());
            }
        });
        CompletableFutureUtils.forwardExceptionTo((CompletableFuture)future, executionFuture);
        return future;
    }

    private Stream<Path> listFiles(Path directory, UploadDirectoryRequest request) {
        try {
            boolean followSymbolicLinks = this.transferConfiguration.resolveUploadDirectoryFollowSymbolicLinks(request);
            int maxDepth = this.transferConfiguration.resolveUploadDirectoryMaxDepth(request);
            if (followSymbolicLinks) {
                return Files.walk(directory, maxDepth, FileVisitOption.FOLLOW_LINKS).filter(path -> this.isRegularFile((Path)path, true));
            }
            return Files.walk(directory, maxDepth, new FileVisitOption[0]).filter(path -> this.isRegularFile((Path)path, false));
        }
        catch (IOException e) {
            throw SdkClientException.create((String)("Failed to list files within the provided directory: " + directory), (Throwable)e);
        }
    }

    private boolean isRegularFile(Path path, boolean followSymlinks) {
        if (followSymlinks) {
            return Files.isRegularFile(path, new LinkOption[0]);
        }
        return Files.isRegularFile(path, LinkOption.NOFOLLOW_LINKS);
    }

    private static String normalizePrefix(String prefix, String delimiter) {
        if (StringUtils.isEmpty((CharSequence)prefix)) {
            return "";
        }
        return prefix.endsWith(delimiter) ? prefix : prefix + delimiter;
    }

    private String getRelativePathName(Path source, int directoryNameCount, Path path, String delimiter) {
        String relativePathName = path.subpath(directoryNameCount, path.getNameCount()).toString();
        String separator = source.getFileSystem().getSeparator();
        if (delimiter.equals(separator)) {
            return relativePathName;
        }
        return StringUtils.replace((String)relativePathName, (String)separator, (String)delimiter);
    }

    private UploadFileRequest constructUploadRequest(UploadDirectoryRequest uploadDirectoryRequest, int directoryNameCount, Path path) {
        String delimiter = uploadDirectoryRequest.s3Delimiter().filter(s -> !s.isEmpty()).orElse("/");
        String prefix = uploadDirectoryRequest.s3Prefix().map(s -> UploadDirectoryHelper.normalizePrefix(s, delimiter)).orElse("");
        String relativePathName = this.getRelativePathName(uploadDirectoryRequest.source(), directoryNameCount, path, delimiter);
        String key = prefix + relativePathName;
        PutObjectRequest putObjectRequest = (PutObjectRequest)PutObjectRequest.builder().bucket(uploadDirectoryRequest.bucket()).key(key).build();
        UploadFileRequest.Builder requestBuilder = UploadFileRequest.builder().source(path).putObjectRequest(putObjectRequest);
        uploadDirectoryRequest.uploadFileRequestTransformer().accept(requestBuilder);
        return (UploadFileRequest)requestBuilder.build();
    }
}

