/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.transfer.s3.internal;

import java.io.IOException;
import java.nio.file.FileSystem;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.function.Function;
import java.util.function.Predicate;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Request;
import software.amazon.awssdk.services.s3.model.S3Object;
import software.amazon.awssdk.transfer.s3.CompletedDirectoryDownload;
import software.amazon.awssdk.transfer.s3.CompletedFileDownload;
import software.amazon.awssdk.transfer.s3.DirectoryDownload;
import software.amazon.awssdk.transfer.s3.DownloadDirectoryRequest;
import software.amazon.awssdk.transfer.s3.DownloadFileContext;
import software.amazon.awssdk.transfer.s3.DownloadFileRequest;
import software.amazon.awssdk.transfer.s3.FailedFileDownload;
import software.amazon.awssdk.transfer.s3.FileDownload;
import software.amazon.awssdk.transfer.s3.S3TransferManager;
import software.amazon.awssdk.transfer.s3.internal.AsyncBufferingSubscriber;
import software.amazon.awssdk.transfer.s3.internal.DefaultDirectoryDownload;
import software.amazon.awssdk.transfer.s3.internal.DefaultDownloadFileContext;
import software.amazon.awssdk.transfer.s3.internal.ListObjectsHelper;
import software.amazon.awssdk.transfer.s3.internal.TransferConfigurationOption;
import software.amazon.awssdk.transfer.s3.internal.TransferManagerConfiguration;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Logger;
import software.amazon.awssdk.utils.StringUtils;
import software.amazon.awssdk.utils.Validate;

@SdkInternalApi
public class DownloadDirectoryHelper {
    private static final Logger log = Logger.loggerFor(S3TransferManager.class);
    private final TransferManagerConfiguration transferConfiguration;
    private final Function<DownloadFileRequest, FileDownload> downloadFileFunction;
    private final ListObjectsHelper listObjectsHelper;

    public DownloadDirectoryHelper(TransferManagerConfiguration transferConfiguration, ListObjectsHelper listObjectsHelper, Function<DownloadFileRequest, FileDownload> downloadFileFunction) {
        this.transferConfiguration = transferConfiguration;
        this.downloadFileFunction = downloadFileFunction;
        this.listObjectsHelper = listObjectsHelper;
    }

    public DirectoryDownload downloadDirectory(DownloadDirectoryRequest downloadDirectoryRequest) {
        CompletableFuture<CompletedDirectoryDownload> returnFuture = new CompletableFuture<CompletedDirectoryDownload>();
        CompletableFuture.runAsync(() -> this.doDownloadDirectory(returnFuture, downloadDirectoryRequest), this.transferConfiguration.option(TransferConfigurationOption.EXECUTOR)).whenComplete((r, t) -> {
            if (t != null) {
                returnFuture.completeExceptionally((Throwable)t);
            }
        });
        return new DefaultDirectoryDownload(returnFuture);
    }

    private static void validateDirectoryIfExists(Path directory) {
        if (Files.exists(directory, new LinkOption[0])) {
            Validate.isTrue((boolean)Files.isDirectory(directory, new LinkOption[0]), (String)"The destination directory provided (%s) is not a directory", (Object[])new Object[]{directory});
        }
    }

    private void doDownloadDirectory(CompletableFuture<CompletedDirectoryDownload> returnFuture, DownloadDirectoryRequest downloadDirectoryRequest) {
        DownloadDirectoryHelper.validateDirectoryIfExists(downloadDirectoryRequest.destinationDirectory());
        String bucket = downloadDirectoryRequest.bucket();
        String delimiter = downloadDirectoryRequest.delimiter().orElse(null);
        String prefix = downloadDirectoryRequest.prefix().orElse("");
        ListObjectsV2Request request = (ListObjectsV2Request)((ListObjectsV2Request.Builder)ListObjectsV2Request.builder().bucket(bucket).prefix(prefix).delimiter(delimiter).applyMutation(downloadDirectoryRequest.listObjectsRequestTransformer())).build();
        ConcurrentLinkedQueue<FailedFileDownload> failedFileDownloads = new ConcurrentLinkedQueue<FailedFileDownload>();
        CompletableFuture<Void> allOfFutures = new CompletableFuture<Void>();
        AsyncBufferingSubscriber<DownloadFileContext> asyncBufferingSubscriber = new AsyncBufferingSubscriber<DownloadFileContext>(this.downloadSingleFile(returnFuture, downloadDirectoryRequest, failedFileDownloads), allOfFutures, 100);
        this.listObjectsHelper.listS3ObjectsRecursively(request).map(s3Object -> this.determineDestinationPath(downloadDirectoryRequest, (S3Object)s3Object)).filter((Predicate)downloadDirectoryRequest.filter()).subscribe(asyncBufferingSubscriber);
        allOfFutures.whenComplete((r, t) -> {
            if (t != null) {
                returnFuture.completeExceptionally((Throwable)SdkClientException.create((String)"Failed to send request", (Throwable)t));
            } else {
                returnFuture.complete(CompletedDirectoryDownload.builder().failedTransfers(failedFileDownloads).build());
            }
        });
    }

    private Function<DownloadFileContext, CompletableFuture<?>> downloadSingleFile(CompletableFuture<CompletedDirectoryDownload> returnFuture, DownloadDirectoryRequest downloadDirectoryRequest, Queue<FailedFileDownload> failedFileDownloads) {
        return downloadContext -> {
            CompletableFuture<CompletedFileDownload> future = this.doDownloadSingleFile(downloadDirectoryRequest, (Collection<FailedFileDownload>)failedFileDownloads, (DownloadFileContext)downloadContext);
            CompletableFutureUtils.forwardExceptionTo((CompletableFuture)returnFuture, future);
            return future;
        };
    }

    private DownloadFileContext determineDestinationPath(DownloadDirectoryRequest downloadDirectoryRequest, S3Object s3Object) {
        FileSystem fileSystem = downloadDirectoryRequest.destinationDirectory().getFileSystem();
        String delimiter = downloadDirectoryRequest.delimiter().orElse(null);
        String key = DownloadDirectoryHelper.normalizeKey(downloadDirectoryRequest, s3Object, delimiter);
        String relativePath = DownloadDirectoryHelper.getRelativePath(fileSystem, delimiter, key);
        Path destinationPath = downloadDirectoryRequest.destinationDirectory().resolve(relativePath);
        this.validatePath(downloadDirectoryRequest.destinationDirectory(), destinationPath, s3Object.key());
        return new DefaultDownloadFileContext(s3Object, destinationPath);
    }

    private void validatePath(Path destinationDirectory, Path targetPath, String key) {
        if (!targetPath.toAbsolutePath().normalize().startsWith(destinationDirectory.toAbsolutePath().normalize())) {
            throw SdkClientException.create((String)("Cannot download key " + key + ", its relative path resolves outside the parent directory."));
        }
    }

    private CompletableFuture<CompletedFileDownload> doDownloadSingleFile(DownloadDirectoryRequest downloadDirectoryRequest, Collection<FailedFileDownload> failedFileDownloads, DownloadFileContext downloadContext) {
        DownloadFileRequest downloadFileRequest = DownloadDirectoryHelper.downloadFileRequest(downloadDirectoryRequest, downloadContext);
        try {
            log.debug(() -> "Sending download request " + downloadFileRequest);
            DownloadDirectoryHelper.createParentDirectoriesIfNeeded(downloadContext.destination());
            CompletableFuture<CompletedFileDownload> executionFuture = this.downloadFileFunction.apply(downloadFileRequest).completionFuture();
            CompletionStage future = executionFuture.whenComplete((r, t) -> {
                if (t != null) {
                    failedFileDownloads.add((FailedFileDownload)FailedFileDownload.builder().exception(t instanceof CompletionException ? t.getCause() : t).request(downloadFileRequest).build());
                }
            });
            CompletableFutureUtils.forwardExceptionTo((CompletableFuture)future, executionFuture);
            return future;
        }
        catch (Throwable throwable) {
            failedFileDownloads.add((FailedFileDownload)FailedFileDownload.builder().exception(throwable).request(downloadFileRequest).build());
            return CompletableFutureUtils.failedFuture((Throwable)throwable);
        }
    }

    private static String normalizeKey(DownloadDirectoryRequest downloadDirectoryRequest, S3Object s3Object, String delimiter) {
        int delimiterLength = delimiter == null ? "/".length() : delimiter.length();
        return downloadDirectoryRequest.prefix().filter(prefix -> !prefix.isEmpty()).map(prefix -> s3Object.key().substring(prefix.length() + delimiterLength)).orElseGet(() -> ((S3Object)s3Object).key());
    }

    private static String getRelativePath(FileSystem fileSystem, String delimiter, String key) {
        if (delimiter == null) {
            return key;
        }
        if (fileSystem.getSeparator().equals(delimiter)) {
            return key;
        }
        return StringUtils.replace((String)key, (String)delimiter, (String)fileSystem.getSeparator());
    }

    private static DownloadFileRequest downloadFileRequest(DownloadDirectoryRequest downloadDirectoryRequest, DownloadFileContext downloadContext) {
        GetObjectRequest getObjectRequest = (GetObjectRequest)GetObjectRequest.builder().bucket(downloadDirectoryRequest.bucket()).key(downloadContext.source().key()).build();
        return (DownloadFileRequest)((DownloadFileRequest.Builder)DownloadFileRequest.builder().destination(downloadContext.destination()).getObjectRequest(getObjectRequest).applyMutation(downloadDirectoryRequest.downloadFileRequestTransformer())).build();
    }

    private static void createParentDirectoriesIfNeeded(Path destinationPath) {
        Path parentDirectory = destinationPath.getParent();
        try {
            if (parentDirectory != null) {
                Files.createDirectories(parentDirectory, new FileAttribute[0]);
            }
        }
        catch (IOException e) {
            throw SdkClientException.create((String)("Failed to create parent directories for " + destinationPath), (Throwable)e);
        }
    }
}

