/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rum.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This structure contains much of the configuration data for the app monitor.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AppMonitorConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<AppMonitorConfiguration.Builder, AppMonitorConfiguration> {
    private static final SdkField<Boolean> ALLOW_COOKIES_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AllowCookies").getter(getter(AppMonitorConfiguration::allowCookies))
            .setter(setter(Builder::allowCookies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowCookies").build()).build();

    private static final SdkField<Boolean> ENABLE_X_RAY_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableXRay").getter(getter(AppMonitorConfiguration::enableXRay)).setter(setter(Builder::enableXRay))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableXRay").build()).build();

    private static final SdkField<List<String>> EXCLUDED_PAGES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExcludedPages")
            .getter(getter(AppMonitorConfiguration::excludedPages))
            .setter(setter(Builder::excludedPages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludedPages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> FAVORITE_PAGES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("FavoritePages")
            .getter(getter(AppMonitorConfiguration::favoritePages))
            .setter(setter(Builder::favoritePages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FavoritePages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> GUEST_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GuestRoleArn").getter(getter(AppMonitorConfiguration::guestRoleArn))
            .setter(setter(Builder::guestRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GuestRoleArn").build()).build();

    private static final SdkField<String> IDENTITY_POOL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IdentityPoolId").getter(getter(AppMonitorConfiguration::identityPoolId))
            .setter(setter(Builder::identityPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentityPoolId").build()).build();

    private static final SdkField<List<String>> INCLUDED_PAGES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("IncludedPages")
            .getter(getter(AppMonitorConfiguration::includedPages))
            .setter(setter(Builder::includedPages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludedPages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> SESSION_SAMPLE_RATE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("SessionSampleRate").getter(getter(AppMonitorConfiguration::sessionSampleRate))
            .setter(setter(Builder::sessionSampleRate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SessionSampleRate").build()).build();

    private static final SdkField<List<String>> TELEMETRIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Telemetries")
            .getter(getter(AppMonitorConfiguration::telemetriesAsStrings))
            .setter(setter(Builder::telemetriesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Telemetries").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALLOW_COOKIES_FIELD,
            ENABLE_X_RAY_FIELD, EXCLUDED_PAGES_FIELD, FAVORITE_PAGES_FIELD, GUEST_ROLE_ARN_FIELD, IDENTITY_POOL_ID_FIELD,
            INCLUDED_PAGES_FIELD, SESSION_SAMPLE_RATE_FIELD, TELEMETRIES_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean allowCookies;

    private final Boolean enableXRay;

    private final List<String> excludedPages;

    private final List<String> favoritePages;

    private final String guestRoleArn;

    private final String identityPoolId;

    private final List<String> includedPages;

    private final Double sessionSampleRate;

    private final List<String> telemetries;

    private AppMonitorConfiguration(BuilderImpl builder) {
        this.allowCookies = builder.allowCookies;
        this.enableXRay = builder.enableXRay;
        this.excludedPages = builder.excludedPages;
        this.favoritePages = builder.favoritePages;
        this.guestRoleArn = builder.guestRoleArn;
        this.identityPoolId = builder.identityPoolId;
        this.includedPages = builder.includedPages;
        this.sessionSampleRate = builder.sessionSampleRate;
        this.telemetries = builder.telemetries;
    }

    /**
     * <p>
     * If you set this to <code>true</code>, the RUM web client sets two cookies, a session cookie and a user cookie.
     * The cookies allow the RUM web client to collect data relating to the number of users an application has and the
     * behavior of the application across a sequence of events. Cookies are stored in the top-level domain of the
     * current page.
     * </p>
     * 
     * @return If you set this to <code>true</code>, the RUM web client sets two cookies, a session cookie and a user
     *         cookie. The cookies allow the RUM web client to collect data relating to the number of users an
     *         application has and the behavior of the application across a sequence of events. Cookies are stored in
     *         the top-level domain of the current page.
     */
    public final Boolean allowCookies() {
        return allowCookies;
    }

    /**
     * <p>
     * If you set this to <code>true</code>, RUM enables X-Ray tracing for the user sessions that RUM samples. RUM adds
     * an X-Ray trace header to allowed HTTP requests. It also records an X-Ray segment for allowed HTTP requests. You
     * can see traces and segments from these user sessions in the X-Ray console and the CloudWatch ServiceLens console.
     * For more information, see <a href="https://docs.aws.amazon.com/xray/latest/devguide/aws-xray.html">What is
     * X-Ray?</a>
     * </p>
     * 
     * @return If you set this to <code>true</code>, RUM enables X-Ray tracing for the user sessions that RUM samples.
     *         RUM adds an X-Ray trace header to allowed HTTP requests. It also records an X-Ray segment for allowed
     *         HTTP requests. You can see traces and segments from these user sessions in the X-Ray console and the
     *         CloudWatch ServiceLens console. For more information, see <a
     *         href="https://docs.aws.amazon.com/xray/latest/devguide/aws-xray.html">What is X-Ray?</a>
     */
    public final Boolean enableXRay() {
        return enableXRay;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExcludedPages property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExcludedPages() {
        return excludedPages != null && !(excludedPages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of URLs in your website or application to exclude from RUM data collection.
     * </p>
     * <p>
     * You can't include both <code>ExcludedPages</code> and <code>IncludedPages</code> in the same operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExcludedPages} method.
     * </p>
     * 
     * @return A list of URLs in your website or application to exclude from RUM data collection.</p>
     *         <p>
     *         You can't include both <code>ExcludedPages</code> and <code>IncludedPages</code> in the same operation.
     */
    public final List<String> excludedPages() {
        return excludedPages;
    }

    /**
     * For responses, this returns true if the service returned a value for the FavoritePages property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFavoritePages() {
        return favoritePages != null && !(favoritePages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of pages in your application that are to be displayed with a "favorite" icon in the CloudWatch RUM
     * console.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFavoritePages} method.
     * </p>
     * 
     * @return A list of pages in your application that are to be displayed with a "favorite" icon in the CloudWatch RUM
     *         console.
     */
    public final List<String> favoritePages() {
        return favoritePages;
    }

    /**
     * <p>
     * The ARN of the guest IAM role that is attached to the Amazon Cognito identity pool that is used to authorize the
     * sending of data to RUM.
     * </p>
     * 
     * @return The ARN of the guest IAM role that is attached to the Amazon Cognito identity pool that is used to
     *         authorize the sending of data to RUM.
     */
    public final String guestRoleArn() {
        return guestRoleArn;
    }

    /**
     * <p>
     * The ID of the Amazon Cognito identity pool that is used to authorize the sending of data to RUM.
     * </p>
     * 
     * @return The ID of the Amazon Cognito identity pool that is used to authorize the sending of data to RUM.
     */
    public final String identityPoolId() {
        return identityPoolId;
    }

    /**
     * For responses, this returns true if the service returned a value for the IncludedPages property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIncludedPages() {
        return includedPages != null && !(includedPages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If this app monitor is to collect data from only certain pages in your application, this structure lists those
     * pages.
     * </p>
     * <p>
     * You can't include both <code>ExcludedPages</code> and <code>IncludedPages</code> in the same operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIncludedPages} method.
     * </p>
     * 
     * @return If this app monitor is to collect data from only certain pages in your application, this structure lists
     *         those pages. </p>
     *         <p>
     *         You can't include both <code>ExcludedPages</code> and <code>IncludedPages</code> in the same operation.
     */
    public final List<String> includedPages() {
        return includedPages;
    }

    /**
     * <p>
     * Specifies the portion of user sessions to use for RUM data collection. Choosing a higher portion gives you more
     * data but also incurs more costs.
     * </p>
     * <p>
     * The range for this value is 0 to 1 inclusive. Setting this to 1 means that 100% of user sessions are sampled, and
     * setting it to 0.1 means that 10% of user sessions are sampled.
     * </p>
     * <p>
     * If you omit this parameter, the default of 0.1 is used, and 10% of sessions will be sampled.
     * </p>
     * 
     * @return Specifies the portion of user sessions to use for RUM data collection. Choosing a higher portion gives
     *         you more data but also incurs more costs.</p>
     *         <p>
     *         The range for this value is 0 to 1 inclusive. Setting this to 1 means that 100% of user sessions are
     *         sampled, and setting it to 0.1 means that 10% of user sessions are sampled.
     *         </p>
     *         <p>
     *         If you omit this parameter, the default of 0.1 is used, and 10% of sessions will be sampled.
     */
    public final Double sessionSampleRate() {
        return sessionSampleRate;
    }

    /**
     * <p>
     * An array that lists the types of telemetry data that this app monitor is to collect.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>errors</code> indicates that RUM collects data about unhandled JavaScript errors raised by your
     * application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>performance</code> indicates that RUM collects performance data about how your application and its
     * resources are loaded and rendered. This includes Core Web Vitals.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>http</code> indicates that RUM collects data about HTTP errors thrown by your application.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTelemetries} method.
     * </p>
     * 
     * @return An array that lists the types of telemetry data that this app monitor is to collect.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>errors</code> indicates that RUM collects data about unhandled JavaScript errors raised by your
     *         application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>performance</code> indicates that RUM collects performance data about how your application and its
     *         resources are loaded and rendered. This includes Core Web Vitals.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>http</code> indicates that RUM collects data about HTTP errors thrown by your application.
     *         </p>
     *         </li>
     */
    public final List<Telemetry> telemetries() {
        return TelemetriesCopier.copyStringToEnum(telemetries);
    }

    /**
     * For responses, this returns true if the service returned a value for the Telemetries property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTelemetries() {
        return telemetries != null && !(telemetries instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array that lists the types of telemetry data that this app monitor is to collect.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>errors</code> indicates that RUM collects data about unhandled JavaScript errors raised by your
     * application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>performance</code> indicates that RUM collects performance data about how your application and its
     * resources are loaded and rendered. This includes Core Web Vitals.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>http</code> indicates that RUM collects data about HTTP errors thrown by your application.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTelemetries} method.
     * </p>
     * 
     * @return An array that lists the types of telemetry data that this app monitor is to collect.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>errors</code> indicates that RUM collects data about unhandled JavaScript errors raised by your
     *         application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>performance</code> indicates that RUM collects performance data about how your application and its
     *         resources are loaded and rendered. This includes Core Web Vitals.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>http</code> indicates that RUM collects data about HTTP errors thrown by your application.
     *         </p>
     *         </li>
     */
    public final List<String> telemetriesAsStrings() {
        return telemetries;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(allowCookies());
        hashCode = 31 * hashCode + Objects.hashCode(enableXRay());
        hashCode = 31 * hashCode + Objects.hashCode(hasExcludedPages() ? excludedPages() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasFavoritePages() ? favoritePages() : null);
        hashCode = 31 * hashCode + Objects.hashCode(guestRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(identityPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(hasIncludedPages() ? includedPages() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sessionSampleRate());
        hashCode = 31 * hashCode + Objects.hashCode(hasTelemetries() ? telemetriesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AppMonitorConfiguration)) {
            return false;
        }
        AppMonitorConfiguration other = (AppMonitorConfiguration) obj;
        return Objects.equals(allowCookies(), other.allowCookies()) && Objects.equals(enableXRay(), other.enableXRay())
                && hasExcludedPages() == other.hasExcludedPages() && Objects.equals(excludedPages(), other.excludedPages())
                && hasFavoritePages() == other.hasFavoritePages() && Objects.equals(favoritePages(), other.favoritePages())
                && Objects.equals(guestRoleArn(), other.guestRoleArn())
                && Objects.equals(identityPoolId(), other.identityPoolId()) && hasIncludedPages() == other.hasIncludedPages()
                && Objects.equals(includedPages(), other.includedPages())
                && Objects.equals(sessionSampleRate(), other.sessionSampleRate()) && hasTelemetries() == other.hasTelemetries()
                && Objects.equals(telemetriesAsStrings(), other.telemetriesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AppMonitorConfiguration").add("AllowCookies", allowCookies()).add("EnableXRay", enableXRay())
                .add("ExcludedPages", hasExcludedPages() ? excludedPages() : null)
                .add("FavoritePages", hasFavoritePages() ? favoritePages() : null).add("GuestRoleArn", guestRoleArn())
                .add("IdentityPoolId", identityPoolId()).add("IncludedPages", hasIncludedPages() ? includedPages() : null)
                .add("SessionSampleRate", sessionSampleRate())
                .add("Telemetries", hasTelemetries() ? telemetriesAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AllowCookies":
            return Optional.ofNullable(clazz.cast(allowCookies()));
        case "EnableXRay":
            return Optional.ofNullable(clazz.cast(enableXRay()));
        case "ExcludedPages":
            return Optional.ofNullable(clazz.cast(excludedPages()));
        case "FavoritePages":
            return Optional.ofNullable(clazz.cast(favoritePages()));
        case "GuestRoleArn":
            return Optional.ofNullable(clazz.cast(guestRoleArn()));
        case "IdentityPoolId":
            return Optional.ofNullable(clazz.cast(identityPoolId()));
        case "IncludedPages":
            return Optional.ofNullable(clazz.cast(includedPages()));
        case "SessionSampleRate":
            return Optional.ofNullable(clazz.cast(sessionSampleRate()));
        case "Telemetries":
            return Optional.ofNullable(clazz.cast(telemetriesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AppMonitorConfiguration, T> g) {
        return obj -> g.apply((AppMonitorConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AppMonitorConfiguration> {
        /**
         * <p>
         * If you set this to <code>true</code>, the RUM web client sets two cookies, a session cookie and a user
         * cookie. The cookies allow the RUM web client to collect data relating to the number of users an application
         * has and the behavior of the application across a sequence of events. Cookies are stored in the top-level
         * domain of the current page.
         * </p>
         * 
         * @param allowCookies
         *        If you set this to <code>true</code>, the RUM web client sets two cookies, a session cookie and a user
         *        cookie. The cookies allow the RUM web client to collect data relating to the number of users an
         *        application has and the behavior of the application across a sequence of events. Cookies are stored in
         *        the top-level domain of the current page.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowCookies(Boolean allowCookies);

        /**
         * <p>
         * If you set this to <code>true</code>, RUM enables X-Ray tracing for the user sessions that RUM samples. RUM
         * adds an X-Ray trace header to allowed HTTP requests. It also records an X-Ray segment for allowed HTTP
         * requests. You can see traces and segments from these user sessions in the X-Ray console and the CloudWatch
         * ServiceLens console. For more information, see <a
         * href="https://docs.aws.amazon.com/xray/latest/devguide/aws-xray.html">What is X-Ray?</a>
         * </p>
         * 
         * @param enableXRay
         *        If you set this to <code>true</code>, RUM enables X-Ray tracing for the user sessions that RUM
         *        samples. RUM adds an X-Ray trace header to allowed HTTP requests. It also records an X-Ray segment for
         *        allowed HTTP requests. You can see traces and segments from these user sessions in the X-Ray console
         *        and the CloudWatch ServiceLens console. For more information, see <a
         *        href="https://docs.aws.amazon.com/xray/latest/devguide/aws-xray.html">What is X-Ray?</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableXRay(Boolean enableXRay);

        /**
         * <p>
         * A list of URLs in your website or application to exclude from RUM data collection.
         * </p>
         * <p>
         * You can't include both <code>ExcludedPages</code> and <code>IncludedPages</code> in the same operation.
         * </p>
         * 
         * @param excludedPages
         *        A list of URLs in your website or application to exclude from RUM data collection.</p>
         *        <p>
         *        You can't include both <code>ExcludedPages</code> and <code>IncludedPages</code> in the same
         *        operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedPages(Collection<String> excludedPages);

        /**
         * <p>
         * A list of URLs in your website or application to exclude from RUM data collection.
         * </p>
         * <p>
         * You can't include both <code>ExcludedPages</code> and <code>IncludedPages</code> in the same operation.
         * </p>
         * 
         * @param excludedPages
         *        A list of URLs in your website or application to exclude from RUM data collection.</p>
         *        <p>
         *        You can't include both <code>ExcludedPages</code> and <code>IncludedPages</code> in the same
         *        operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedPages(String... excludedPages);

        /**
         * <p>
         * A list of pages in your application that are to be displayed with a "favorite" icon in the CloudWatch RUM
         * console.
         * </p>
         * 
         * @param favoritePages
         *        A list of pages in your application that are to be displayed with a "favorite" icon in the CloudWatch
         *        RUM console.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder favoritePages(Collection<String> favoritePages);

        /**
         * <p>
         * A list of pages in your application that are to be displayed with a "favorite" icon in the CloudWatch RUM
         * console.
         * </p>
         * 
         * @param favoritePages
         *        A list of pages in your application that are to be displayed with a "favorite" icon in the CloudWatch
         *        RUM console.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder favoritePages(String... favoritePages);

        /**
         * <p>
         * The ARN of the guest IAM role that is attached to the Amazon Cognito identity pool that is used to authorize
         * the sending of data to RUM.
         * </p>
         * 
         * @param guestRoleArn
         *        The ARN of the guest IAM role that is attached to the Amazon Cognito identity pool that is used to
         *        authorize the sending of data to RUM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder guestRoleArn(String guestRoleArn);

        /**
         * <p>
         * The ID of the Amazon Cognito identity pool that is used to authorize the sending of data to RUM.
         * </p>
         * 
         * @param identityPoolId
         *        The ID of the Amazon Cognito identity pool that is used to authorize the sending of data to RUM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identityPoolId(String identityPoolId);

        /**
         * <p>
         * If this app monitor is to collect data from only certain pages in your application, this structure lists
         * those pages.
         * </p>
         * <p>
         * You can't include both <code>ExcludedPages</code> and <code>IncludedPages</code> in the same operation.
         * </p>
         * 
         * @param includedPages
         *        If this app monitor is to collect data from only certain pages in your application, this structure
         *        lists those pages. </p>
         *        <p>
         *        You can't include both <code>ExcludedPages</code> and <code>IncludedPages</code> in the same
         *        operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includedPages(Collection<String> includedPages);

        /**
         * <p>
         * If this app monitor is to collect data from only certain pages in your application, this structure lists
         * those pages.
         * </p>
         * <p>
         * You can't include both <code>ExcludedPages</code> and <code>IncludedPages</code> in the same operation.
         * </p>
         * 
         * @param includedPages
         *        If this app monitor is to collect data from only certain pages in your application, this structure
         *        lists those pages. </p>
         *        <p>
         *        You can't include both <code>ExcludedPages</code> and <code>IncludedPages</code> in the same
         *        operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includedPages(String... includedPages);

        /**
         * <p>
         * Specifies the portion of user sessions to use for RUM data collection. Choosing a higher portion gives you
         * more data but also incurs more costs.
         * </p>
         * <p>
         * The range for this value is 0 to 1 inclusive. Setting this to 1 means that 100% of user sessions are sampled,
         * and setting it to 0.1 means that 10% of user sessions are sampled.
         * </p>
         * <p>
         * If you omit this parameter, the default of 0.1 is used, and 10% of sessions will be sampled.
         * </p>
         * 
         * @param sessionSampleRate
         *        Specifies the portion of user sessions to use for RUM data collection. Choosing a higher portion gives
         *        you more data but also incurs more costs.</p>
         *        <p>
         *        The range for this value is 0 to 1 inclusive. Setting this to 1 means that 100% of user sessions are
         *        sampled, and setting it to 0.1 means that 10% of user sessions are sampled.
         *        </p>
         *        <p>
         *        If you omit this parameter, the default of 0.1 is used, and 10% of sessions will be sampled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionSampleRate(Double sessionSampleRate);

        /**
         * <p>
         * An array that lists the types of telemetry data that this app monitor is to collect.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>errors</code> indicates that RUM collects data about unhandled JavaScript errors raised by your
         * application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>performance</code> indicates that RUM collects performance data about how your application and its
         * resources are loaded and rendered. This includes Core Web Vitals.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>http</code> indicates that RUM collects data about HTTP errors thrown by your application.
         * </p>
         * </li>
         * </ul>
         * 
         * @param telemetries
         *        An array that lists the types of telemetry data that this app monitor is to collect.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>errors</code> indicates that RUM collects data about unhandled JavaScript errors raised by your
         *        application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>performance</code> indicates that RUM collects performance data about how your application and
         *        its resources are loaded and rendered. This includes Core Web Vitals.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>http</code> indicates that RUM collects data about HTTP errors thrown by your application.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder telemetriesWithStrings(Collection<String> telemetries);

        /**
         * <p>
         * An array that lists the types of telemetry data that this app monitor is to collect.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>errors</code> indicates that RUM collects data about unhandled JavaScript errors raised by your
         * application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>performance</code> indicates that RUM collects performance data about how your application and its
         * resources are loaded and rendered. This includes Core Web Vitals.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>http</code> indicates that RUM collects data about HTTP errors thrown by your application.
         * </p>
         * </li>
         * </ul>
         * 
         * @param telemetries
         *        An array that lists the types of telemetry data that this app monitor is to collect.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>errors</code> indicates that RUM collects data about unhandled JavaScript errors raised by your
         *        application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>performance</code> indicates that RUM collects performance data about how your application and
         *        its resources are loaded and rendered. This includes Core Web Vitals.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>http</code> indicates that RUM collects data about HTTP errors thrown by your application.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder telemetriesWithStrings(String... telemetries);

        /**
         * <p>
         * An array that lists the types of telemetry data that this app monitor is to collect.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>errors</code> indicates that RUM collects data about unhandled JavaScript errors raised by your
         * application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>performance</code> indicates that RUM collects performance data about how your application and its
         * resources are loaded and rendered. This includes Core Web Vitals.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>http</code> indicates that RUM collects data about HTTP errors thrown by your application.
         * </p>
         * </li>
         * </ul>
         * 
         * @param telemetries
         *        An array that lists the types of telemetry data that this app monitor is to collect.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>errors</code> indicates that RUM collects data about unhandled JavaScript errors raised by your
         *        application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>performance</code> indicates that RUM collects performance data about how your application and
         *        its resources are loaded and rendered. This includes Core Web Vitals.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>http</code> indicates that RUM collects data about HTTP errors thrown by your application.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder telemetries(Collection<Telemetry> telemetries);

        /**
         * <p>
         * An array that lists the types of telemetry data that this app monitor is to collect.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>errors</code> indicates that RUM collects data about unhandled JavaScript errors raised by your
         * application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>performance</code> indicates that RUM collects performance data about how your application and its
         * resources are loaded and rendered. This includes Core Web Vitals.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>http</code> indicates that RUM collects data about HTTP errors thrown by your application.
         * </p>
         * </li>
         * </ul>
         * 
         * @param telemetries
         *        An array that lists the types of telemetry data that this app monitor is to collect.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>errors</code> indicates that RUM collects data about unhandled JavaScript errors raised by your
         *        application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>performance</code> indicates that RUM collects performance data about how your application and
         *        its resources are loaded and rendered. This includes Core Web Vitals.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>http</code> indicates that RUM collects data about HTTP errors thrown by your application.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder telemetries(Telemetry... telemetries);
    }

    static final class BuilderImpl implements Builder {
        private Boolean allowCookies;

        private Boolean enableXRay;

        private List<String> excludedPages = DefaultSdkAutoConstructList.getInstance();

        private List<String> favoritePages = DefaultSdkAutoConstructList.getInstance();

        private String guestRoleArn;

        private String identityPoolId;

        private List<String> includedPages = DefaultSdkAutoConstructList.getInstance();

        private Double sessionSampleRate;

        private List<String> telemetries = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AppMonitorConfiguration model) {
            allowCookies(model.allowCookies);
            enableXRay(model.enableXRay);
            excludedPages(model.excludedPages);
            favoritePages(model.favoritePages);
            guestRoleArn(model.guestRoleArn);
            identityPoolId(model.identityPoolId);
            includedPages(model.includedPages);
            sessionSampleRate(model.sessionSampleRate);
            telemetriesWithStrings(model.telemetries);
        }

        public final Boolean getAllowCookies() {
            return allowCookies;
        }

        public final void setAllowCookies(Boolean allowCookies) {
            this.allowCookies = allowCookies;
        }

        @Override
        public final Builder allowCookies(Boolean allowCookies) {
            this.allowCookies = allowCookies;
            return this;
        }

        public final Boolean getEnableXRay() {
            return enableXRay;
        }

        public final void setEnableXRay(Boolean enableXRay) {
            this.enableXRay = enableXRay;
        }

        @Override
        public final Builder enableXRay(Boolean enableXRay) {
            this.enableXRay = enableXRay;
            return this;
        }

        public final Collection<String> getExcludedPages() {
            if (excludedPages instanceof SdkAutoConstructList) {
                return null;
            }
            return excludedPages;
        }

        public final void setExcludedPages(Collection<String> excludedPages) {
            this.excludedPages = PagesCopier.copy(excludedPages);
        }

        @Override
        public final Builder excludedPages(Collection<String> excludedPages) {
            this.excludedPages = PagesCopier.copy(excludedPages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludedPages(String... excludedPages) {
            excludedPages(Arrays.asList(excludedPages));
            return this;
        }

        public final Collection<String> getFavoritePages() {
            if (favoritePages instanceof SdkAutoConstructList) {
                return null;
            }
            return favoritePages;
        }

        public final void setFavoritePages(Collection<String> favoritePages) {
            this.favoritePages = FavoritePagesCopier.copy(favoritePages);
        }

        @Override
        public final Builder favoritePages(Collection<String> favoritePages) {
            this.favoritePages = FavoritePagesCopier.copy(favoritePages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder favoritePages(String... favoritePages) {
            favoritePages(Arrays.asList(favoritePages));
            return this;
        }

        public final String getGuestRoleArn() {
            return guestRoleArn;
        }

        public final void setGuestRoleArn(String guestRoleArn) {
            this.guestRoleArn = guestRoleArn;
        }

        @Override
        public final Builder guestRoleArn(String guestRoleArn) {
            this.guestRoleArn = guestRoleArn;
            return this;
        }

        public final String getIdentityPoolId() {
            return identityPoolId;
        }

        public final void setIdentityPoolId(String identityPoolId) {
            this.identityPoolId = identityPoolId;
        }

        @Override
        public final Builder identityPoolId(String identityPoolId) {
            this.identityPoolId = identityPoolId;
            return this;
        }

        public final Collection<String> getIncludedPages() {
            if (includedPages instanceof SdkAutoConstructList) {
                return null;
            }
            return includedPages;
        }

        public final void setIncludedPages(Collection<String> includedPages) {
            this.includedPages = PagesCopier.copy(includedPages);
        }

        @Override
        public final Builder includedPages(Collection<String> includedPages) {
            this.includedPages = PagesCopier.copy(includedPages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder includedPages(String... includedPages) {
            includedPages(Arrays.asList(includedPages));
            return this;
        }

        public final Double getSessionSampleRate() {
            return sessionSampleRate;
        }

        public final void setSessionSampleRate(Double sessionSampleRate) {
            this.sessionSampleRate = sessionSampleRate;
        }

        @Override
        public final Builder sessionSampleRate(Double sessionSampleRate) {
            this.sessionSampleRate = sessionSampleRate;
            return this;
        }

        public final Collection<String> getTelemetries() {
            if (telemetries instanceof SdkAutoConstructList) {
                return null;
            }
            return telemetries;
        }

        public final void setTelemetries(Collection<String> telemetries) {
            this.telemetries = TelemetriesCopier.copy(telemetries);
        }

        @Override
        public final Builder telemetriesWithStrings(Collection<String> telemetries) {
            this.telemetries = TelemetriesCopier.copy(telemetries);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder telemetriesWithStrings(String... telemetries) {
            telemetriesWithStrings(Arrays.asList(telemetries));
            return this;
        }

        @Override
        public final Builder telemetries(Collection<Telemetry> telemetries) {
            this.telemetries = TelemetriesCopier.copyEnumToString(telemetries);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder telemetries(Telemetry... telemetries) {
            telemetries(Arrays.asList(telemetries));
            return this;
        }

        @Override
        public AppMonitorConfiguration build() {
            return new AppMonitorConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
