/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.robomaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a data source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataSourceConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<DataSourceConfig.Builder, DataSourceConfig> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DataSourceConfig::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> S3_BUCKET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DataSourceConfig::s3Bucket)).setter(setter(Builder::s3Bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("s3Bucket").build()).build();

    private static final SdkField<List<String>> S3_KEYS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DataSourceConfig::s3Keys))
            .setter(setter(Builder::s3Keys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("s3Keys").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, S3_BUCKET_FIELD,
            S3_KEYS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String s3Bucket;

    private final List<String> s3Keys;

    private DataSourceConfig(BuilderImpl builder) {
        this.name = builder.name;
        this.s3Bucket = builder.s3Bucket;
        this.s3Keys = builder.s3Keys;
    }

    /**
     * <p>
     * The name of the data source.
     * </p>
     * 
     * @return The name of the data source.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The S3 bucket where the data files are located.
     * </p>
     * 
     * @return The S3 bucket where the data files are located.
     */
    public String s3Bucket() {
        return s3Bucket;
    }

    /**
     * Returns true if the S3Keys property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasS3Keys() {
        return s3Keys != null && !(s3Keys instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of S3 keys identifying the data source files.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasS3Keys()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of S3 keys identifying the data source files.
     */
    public List<String> s3Keys() {
        return s3Keys;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(s3Bucket());
        hashCode = 31 * hashCode + Objects.hashCode(s3Keys());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataSourceConfig)) {
            return false;
        }
        DataSourceConfig other = (DataSourceConfig) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(s3Bucket(), other.s3Bucket())
                && Objects.equals(s3Keys(), other.s3Keys());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DataSourceConfig").add("Name", name()).add("S3Bucket", s3Bucket()).add("S3Keys", s3Keys())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "s3Bucket":
            return Optional.ofNullable(clazz.cast(s3Bucket()));
        case "s3Keys":
            return Optional.ofNullable(clazz.cast(s3Keys()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DataSourceConfig, T> g) {
        return obj -> g.apply((DataSourceConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataSourceConfig> {
        /**
         * <p>
         * The name of the data source.
         * </p>
         * 
         * @param name
         *        The name of the data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The S3 bucket where the data files are located.
         * </p>
         * 
         * @param s3Bucket
         *        The S3 bucket where the data files are located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Bucket(String s3Bucket);

        /**
         * <p>
         * The list of S3 keys identifying the data source files.
         * </p>
         * 
         * @param s3Keys
         *        The list of S3 keys identifying the data source files.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Keys(Collection<String> s3Keys);

        /**
         * <p>
         * The list of S3 keys identifying the data source files.
         * </p>
         * 
         * @param s3Keys
         *        The list of S3 keys identifying the data source files.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Keys(String... s3Keys);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String s3Bucket;

        private List<String> s3Keys = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DataSourceConfig model) {
            name(model.name);
            s3Bucket(model.s3Bucket);
            s3Keys(model.s3Keys);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getS3Bucket() {
            return s3Bucket;
        }

        @Override
        public final Builder s3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
            return this;
        }

        public final void setS3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
        }

        public final Collection<String> getS3Keys() {
            return s3Keys;
        }

        @Override
        public final Builder s3Keys(Collection<String> s3Keys) {
            this.s3Keys = S3KeysCopier.copy(s3Keys);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder s3Keys(String... s3Keys) {
            s3Keys(Arrays.asList(s3Keys));
            return this;
        }

        public final void setS3Keys(Collection<String> s3Keys) {
            this.s3Keys = S3KeysCopier.copy(s3Keys);
        }

        @Override
        public DataSourceConfig build() {
            return new DataSourceConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
