/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resourceexplorer2;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.resourceexplorer2.internal.ResourceExplorer2ServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.resourceexplorer2.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.resourceexplorer2.model.AccessDeniedException;
import software.amazon.awssdk.services.resourceexplorer2.model.AssociateDefaultViewRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.AssociateDefaultViewResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.BatchGetViewRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.BatchGetViewResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.ConflictException;
import software.amazon.awssdk.services.resourceexplorer2.model.CreateIndexRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.CreateIndexResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.CreateResourceExplorerSetupRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.CreateResourceExplorerSetupResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.CreateViewRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.CreateViewResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.DeleteIndexRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.DeleteIndexResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.DeleteResourceExplorerSetupRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.DeleteResourceExplorerSetupResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.DeleteViewRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.DeleteViewResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.DisassociateDefaultViewRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.DisassociateDefaultViewResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.GetAccountLevelServiceConfigurationRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.GetAccountLevelServiceConfigurationResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.GetDefaultViewRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.GetDefaultViewResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.GetIndexRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.GetIndexResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.GetManagedViewRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.GetManagedViewResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.GetResourceExplorerSetupRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.GetResourceExplorerSetupResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.GetServiceIndexRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.GetServiceIndexResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.GetServiceViewRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.GetServiceViewResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.GetViewRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.GetViewResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.InternalServerException;
import software.amazon.awssdk.services.resourceexplorer2.model.ListIndexesForMembersRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.ListIndexesForMembersResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.ListIndexesRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.ListIndexesResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.ListManagedViewsRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.ListManagedViewsResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.ListResourcesRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.ListResourcesResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.ListServiceIndexesRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.ListServiceIndexesResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.ListServiceViewsRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.ListServiceViewsResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.ListStreamingAccessForServicesRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.ListStreamingAccessForServicesResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.ListSupportedResourceTypesRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.ListSupportedResourceTypesResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.ListViewsRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.ListViewsResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.ResourceExplorer2Exception;
import software.amazon.awssdk.services.resourceexplorer2.model.ResourceNotFoundException;
import software.amazon.awssdk.services.resourceexplorer2.model.SearchRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.SearchResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.resourceexplorer2.model.TagResourceRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.TagResourceResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.ThrottlingException;
import software.amazon.awssdk.services.resourceexplorer2.model.UnauthorizedException;
import software.amazon.awssdk.services.resourceexplorer2.model.UntagResourceRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.UntagResourceResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.UpdateIndexTypeRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.UpdateIndexTypeResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.UpdateViewRequest;
import software.amazon.awssdk.services.resourceexplorer2.model.UpdateViewResponse;
import software.amazon.awssdk.services.resourceexplorer2.model.ValidationException;
import software.amazon.awssdk.services.resourceexplorer2.transform.AssociateDefaultViewRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.BatchGetViewRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.CreateIndexRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.CreateResourceExplorerSetupRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.CreateViewRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.DeleteIndexRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.DeleteResourceExplorerSetupRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.DeleteViewRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.DisassociateDefaultViewRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.GetAccountLevelServiceConfigurationRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.GetDefaultViewRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.GetIndexRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.GetManagedViewRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.GetResourceExplorerSetupRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.GetServiceIndexRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.GetServiceViewRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.GetViewRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.ListIndexesForMembersRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.ListIndexesRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.ListManagedViewsRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.ListResourcesRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.ListServiceIndexesRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.ListServiceViewsRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.ListStreamingAccessForServicesRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.ListSupportedResourceTypesRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.ListViewsRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.SearchRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.UpdateIndexTypeRequestMarshaller;
import software.amazon.awssdk.services.resourceexplorer2.transform.UpdateViewRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ResourceExplorer2AsyncClient}.
 *
 * @see ResourceExplorer2AsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultResourceExplorer2AsyncClient implements ResourceExplorer2AsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultResourceExplorer2AsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultResourceExplorer2AsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Resource_Explorer_2" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Sets the specified view as the default for the Amazon Web Services Region in which you call this operation. When
     * a user performs a <a>Search</a> that doesn't explicitly specify which view to use, then Amazon Web Services
     * Resource Explorer automatically chooses this default view for searches performed in this Amazon Web Services
     * Region.
     * </p>
     * <p>
     * If an Amazon Web Services Region doesn't have a default view configured, then users must explicitly specify a
     * view with every <code>Search</code> operation performed in that Region.
     * </p>
     *
     * @param associateDefaultViewRequest
     * @return A Java Future containing the result of the AssociateDefaultView operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.AssociateDefaultView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/AssociateDefaultView"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateDefaultViewResponse> associateDefaultView(
            AssociateDefaultViewRequest associateDefaultViewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateDefaultViewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateDefaultViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateDefaultView");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateDefaultViewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateDefaultViewResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<AssociateDefaultViewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateDefaultViewRequest, AssociateDefaultViewResponse>()
                            .withOperationName("AssociateDefaultView").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateDefaultViewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateDefaultViewRequest));
            CompletableFuture<AssociateDefaultViewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about a list of views.
     * </p>
     *
     * @param batchGetViewRequest
     * @return A Java Future containing the result of the BatchGetView operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>UnauthorizedException The principal making the request isn't permitted to perform the operation.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.BatchGetView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/BatchGetView"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetViewResponse> batchGetView(BatchGetViewRequest batchGetViewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetViewRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetView");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetViewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    BatchGetViewResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<BatchGetViewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetViewRequest, BatchGetViewResponse>()
                            .withOperationName("BatchGetView").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchGetViewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchGetViewRequest));
            CompletableFuture<BatchGetViewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Turns on Amazon Web Services Resource Explorer in the Amazon Web Services Region in which you called this
     * operation by creating an index. Resource Explorer begins discovering the resources in this Region and stores the
     * details about the resources in the index so that they can be queried by using the <a>Search</a> operation. You
     * can create only one index in a Region.
     * </p>
     * <note>
     * <p>
     * This operation creates only a <i>local</i> index. To promote the local index in one Amazon Web Services Region
     * into the aggregator index for the Amazon Web Services account, use the <a>UpdateIndexType</a> operation. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/manage-aggregator-region.html">Turning on
     * cross-Region search by creating an aggregator index</a> in the <i>Amazon Web Services Resource Explorer User
     * Guide</i>.
     * </p>
     * </note>
     * <p>
     * For more details about what happens when you turn on Resource Explorer in an Amazon Web Services Region, see <a
     * href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/manage-service-activate.html">Turn on
     * Resource Explorer to index your resources in an Amazon Web Services Region</a> in the <i>Amazon Web Services
     * Resource Explorer User Guide</i>.
     * </p>
     * <p>
     * If this is the first Amazon Web Services Region in which you've created an index for Resource Explorer, then this
     * operation also <a
     * href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/security_iam_service-linked-roles.html"
     * >creates a service-linked role</a> in your Amazon Web Services account that allows Resource Explorer to enumerate
     * your resources to populate the index.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Action</b>: <code>resource-explorer-2:CreateIndex</code>
     * </p>
     * <p>
     * <b>Resource</b>: The ARN of the index (as it will exist after the operation completes) in the Amazon Web Services
     * Region and account in which you're trying to create the index. Use the wildcard character (<code>*</code>) at the
     * end of the string to match the eventual UUID. For example, the following <code>Resource</code> element restricts
     * the role or user to creating an index in only the <code>us-east-2</code> Region of the specified account.
     * </p>
     * <p>
     * <code>"Resource": "arn:aws:resource-explorer-2:us-west-2:<i>&lt;account-id&gt;</i>:index/*"</code>
     * </p>
     * <p>
     * Alternatively, you can use <code>"Resource": "*"</code> to allow the role or user to create an index in any
     * Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Action</b>: <code>iam:CreateServiceLinkedRole</code>
     * </p>
     * <p>
     * <b>Resource</b>: No specific resource (*).
     * </p>
     * <p>
     * This permission is required only the first time you create an index to turn on Resource Explorer in the account.
     * Resource Explorer uses this to create the <a
     * href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/security_iam_service-linked-roles.html"
     * >service-linked role needed to index the resources in your account</a>. Resource Explorer uses the same
     * service-linked role for all additional indexes you create afterwards.
     * </p>
     * </li>
     * </ul>
     *
     * @param createIndexRequest
     * @return A Java Future containing the result of the CreateIndex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ConflictException If you attempted to create a view, then the request failed because either you
     *         specified parameters that didn’t match the original request, or you attempted to create a view with a
     *         name that already exists in this Amazon Web Services Region.</p>
     *         <p>
     *         If you attempted to create an index, then the request failed because either you specified parameters that
     *         didn't match the original request, or an index already exists in the current Amazon Web Services Region.
     *         </p>
     *         <p>
     *         If you attempted to update an index type to <code>AGGREGATOR</code>, then the request failed because you
     *         already have an <code>AGGREGATOR</code> index in a different Amazon Web Services Region.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.CreateIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/CreateIndex"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateIndexResponse> createIndex(CreateIndexRequest createIndexRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createIndexRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateIndex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateIndexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateIndexResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateIndexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateIndexRequest, CreateIndexResponse>()
                            .withOperationName("CreateIndex").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateIndexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createIndexRequest));
            CompletableFuture<CreateIndexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a Resource Explorer setup configuration across multiple Amazon Web Services Regions. This operation sets
     * up indexes and views in the specified Regions. This operation can also be used to set an aggregator Region for
     * cross-Region resource search.
     * </p>
     *
     * @param createResourceExplorerSetupRequest
     * @return A Java Future containing the result of the CreateResourceExplorerSetup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ConflictException If you attempted to create a view, then the request failed because either you
     *         specified parameters that didn’t match the original request, or you attempted to create a view with a
     *         name that already exists in this Amazon Web Services Region.</p>
     *         <p>
     *         If you attempted to create an index, then the request failed because either you specified parameters that
     *         didn't match the original request, or an index already exists in the current Amazon Web Services Region.
     *         </p>
     *         <p>
     *         If you attempted to update an index type to <code>AGGREGATOR</code>, then the request failed because you
     *         already have an <code>AGGREGATOR</code> index in a different Amazon Web Services Region.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.CreateResourceExplorerSetup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/CreateResourceExplorerSetup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateResourceExplorerSetupResponse> createResourceExplorerSetup(
            CreateResourceExplorerSetupRequest createResourceExplorerSetupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createResourceExplorerSetupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResourceExplorerSetupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResourceExplorerSetup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateResourceExplorerSetupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateResourceExplorerSetupResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateResourceExplorerSetupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateResourceExplorerSetupRequest, CreateResourceExplorerSetupResponse>()
                            .withOperationName("CreateResourceExplorerSetup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateResourceExplorerSetupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createResourceExplorerSetupRequest));
            CompletableFuture<CreateResourceExplorerSetupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a view that users can query by using the <a>Search</a> operation. Results from queries that you make
     * using this view include only resources that match the view's <code>Filters</code>. For more information about
     * Amazon Web Services Resource Explorer views, see <a
     * href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/manage-views.html">Managing views</a> in the
     * <i>Amazon Web Services Resource Explorer User Guide</i>.
     * </p>
     * <p>
     * Only the principals with an IAM identity-based policy that grants <code>Allow</code> to the <code>Search</code>
     * action on a <code>Resource</code> with the <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon resource name (ARN)</a>
     * of this view can <a>Search</a> using views you create with this operation.
     * </p>
     *
     * @param createViewRequest
     * @return A Java Future containing the result of the CreateView operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ConflictException If you attempted to create a view, then the request failed because either you
     *         specified parameters that didn’t match the original request, or you attempted to create a view with a
     *         name that already exists in this Amazon Web Services Region.</p>
     *         <p>
     *         If you attempted to create an index, then the request failed because either you specified parameters that
     *         didn't match the original request, or an index already exists in the current Amazon Web Services Region.
     *         </p>
     *         <p>
     *         If you attempted to update an index type to <code>AGGREGATOR</code>, then the request failed because you
     *         already have an <code>AGGREGATOR</code> index in a different Amazon Web Services Region.</li>
     *         <li>ServiceQuotaExceededException The request failed because it exceeds a service quota.</li>
     *         <li>UnauthorizedException The principal making the request isn't permitted to perform the operation.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.CreateView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/CreateView"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateViewResponse> createView(CreateViewRequest createViewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createViewRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateView");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateViewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateViewResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateViewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateViewRequest, CreateViewResponse>().withOperationName("CreateView")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateViewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createViewRequest));
            CompletableFuture<CreateViewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified index and turns off Amazon Web Services Resource Explorer in the specified Amazon Web
     * Services Region. When you delete an index, Resource Explorer stops discovering and indexing resources in that
     * Region. Resource Explorer also deletes all views in that Region. These actions occur as asynchronous background
     * tasks. You can check to see when the actions are complete by using the <a>GetIndex</a> operation and checking the
     * <code>Status</code> response value.
     * </p>
     * <note>
     * <p>
     * If the index you delete is the aggregator index for the Amazon Web Services account, you must wait 24 hours
     * before you can promote another local index to be the aggregator index for the account. Users can't perform
     * account-wide searches using Resource Explorer until another aggregator index is configured.
     * </p>
     * </note>
     *
     * @param deleteIndexRequest
     * @return A Java Future containing the result of the DeleteIndex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.DeleteIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/DeleteIndex"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteIndexResponse> deleteIndex(DeleteIndexRequest deleteIndexRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteIndexRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIndex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteIndexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteIndexResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteIndexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteIndexRequest, DeleteIndexResponse>()
                            .withOperationName("DeleteIndex").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteIndexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteIndexRequest));
            CompletableFuture<DeleteIndexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Resource Explorer setup configuration. This operation removes indexes and views from the specified
     * Regions or all Regions where Resource Explorer is configured.
     * </p>
     *
     * @param deleteResourceExplorerSetupRequest
     * @return A Java Future containing the result of the DeleteResourceExplorerSetup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ConflictException If you attempted to create a view, then the request failed because either you
     *         specified parameters that didn’t match the original request, or you attempted to create a view with a
     *         name that already exists in this Amazon Web Services Region.</p>
     *         <p>
     *         If you attempted to create an index, then the request failed because either you specified parameters that
     *         didn't match the original request, or an index already exists in the current Amazon Web Services Region.
     *         </p>
     *         <p>
     *         If you attempted to update an index type to <code>AGGREGATOR</code>, then the request failed because you
     *         already have an <code>AGGREGATOR</code> index in a different Amazon Web Services Region.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.DeleteResourceExplorerSetup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/DeleteResourceExplorerSetup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResourceExplorerSetupResponse> deleteResourceExplorerSetup(
            DeleteResourceExplorerSetupRequest deleteResourceExplorerSetupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourceExplorerSetupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourceExplorerSetupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourceExplorerSetup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResourceExplorerSetupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResourceExplorerSetupResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteResourceExplorerSetupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResourceExplorerSetupRequest, DeleteResourceExplorerSetupResponse>()
                            .withOperationName("DeleteResourceExplorerSetup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteResourceExplorerSetupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteResourceExplorerSetupRequest));
            CompletableFuture<DeleteResourceExplorerSetupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified view.
     * </p>
     * <p>
     * If the specified view is the default view for its Amazon Web Services Region, then all <a>Search</a> operations
     * in that Region must explicitly specify the view to use until you configure a new default by calling the
     * <a>AssociateDefaultView</a> operation.
     * </p>
     *
     * @param deleteViewRequest
     * @return A Java Future containing the result of the DeleteView operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>UnauthorizedException The principal making the request isn't permitted to perform the operation.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.DeleteView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/DeleteView"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteViewResponse> deleteView(DeleteViewRequest deleteViewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteViewRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteView");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteViewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteViewResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteViewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteViewRequest, DeleteViewResponse>().withOperationName("DeleteView")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteViewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteViewRequest));
            CompletableFuture<DeleteViewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * After you call this operation, the affected Amazon Web Services Region no longer has a default view. All
     * <a>Search</a> operations in that Region must explicitly specify a view or the operation fails. You can configure
     * a new default by calling the <a>AssociateDefaultView</a> operation.
     * </p>
     * <p>
     * If an Amazon Web Services Region doesn't have a default view configured, then users must explicitly specify a
     * view with every <code>Search</code> operation performed in that Region.
     * </p>
     *
     * @param disassociateDefaultViewRequest
     * @return A Java Future containing the result of the DisassociateDefaultView operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.DisassociateDefaultView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/DisassociateDefaultView"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateDefaultViewResponse> disassociateDefaultView(
            DisassociateDefaultViewRequest disassociateDefaultViewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateDefaultViewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateDefaultViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateDefaultView");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateDefaultViewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateDefaultViewResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DisassociateDefaultViewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateDefaultViewRequest, DisassociateDefaultViewResponse>()
                            .withOperationName("DisassociateDefaultView").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateDefaultViewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateDefaultViewRequest));
            CompletableFuture<DisassociateDefaultViewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the status of your account's Amazon Web Services service access, and validates the service linked role
     * required to access the multi-account search feature. Only the management account can invoke this API call.
     * </p>
     *
     * @param getAccountLevelServiceConfigurationRequest
     * @return A Java Future containing the result of the GetAccountLevelServiceConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.GetAccountLevelServiceConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/GetAccountLevelServiceConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccountLevelServiceConfigurationResponse> getAccountLevelServiceConfiguration(
            GetAccountLevelServiceConfigurationRequest getAccountLevelServiceConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAccountLevelServiceConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getAccountLevelServiceConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccountLevelServiceConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccountLevelServiceConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetAccountLevelServiceConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetAccountLevelServiceConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccountLevelServiceConfigurationRequest, GetAccountLevelServiceConfigurationResponse>()
                            .withOperationName("GetAccountLevelServiceConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAccountLevelServiceConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAccountLevelServiceConfigurationRequest));
            CompletableFuture<GetAccountLevelServiceConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the Amazon Resource Name (ARN) of the view that is the default for the Amazon Web Services Region in
     * which you call this operation. You can then call <a>GetView</a> to retrieve the details of that view.
     * </p>
     *
     * @param getDefaultViewRequest
     * @return A Java Future containing the result of the GetDefaultView operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.GetDefaultView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/GetDefaultView"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDefaultViewResponse> getDefaultView(GetDefaultViewRequest getDefaultViewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDefaultViewRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDefaultViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDefaultView");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDefaultViewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDefaultViewResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetDefaultViewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDefaultViewRequest, GetDefaultViewResponse>()
                            .withOperationName("GetDefaultView").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDefaultViewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDefaultViewRequest));
            CompletableFuture<GetDefaultViewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about the Amazon Web Services Resource Explorer index in the Amazon Web Services Region in
     * which you invoked the operation.
     * </p>
     *
     * @param getIndexRequest
     * @return A Java Future containing the result of the GetIndex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.GetIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/GetIndex" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetIndexResponse> getIndex(GetIndexRequest getIndexRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getIndexRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIndex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetIndexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetIndexResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetIndexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetIndexRequest, GetIndexResponse>().withOperationName("GetIndex")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetIndexRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getIndexRequest));
            CompletableFuture<GetIndexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details of the specified <a
     * href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/aws-managed-views.html">Amazon Web
     * Services-managed view</a>.
     * </p>
     *
     * @param getManagedViewRequest
     * @return A Java Future containing the result of the GetManagedView operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>UnauthorizedException The principal making the request isn't permitted to perform the operation.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.GetManagedView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/GetManagedView"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetManagedViewResponse> getManagedView(GetManagedViewRequest getManagedViewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getManagedViewRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getManagedViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetManagedView");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetManagedViewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetManagedViewResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetManagedViewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetManagedViewRequest, GetManagedViewResponse>()
                            .withOperationName("GetManagedView").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetManagedViewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getManagedViewRequest));
            CompletableFuture<GetManagedViewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the status and details of a Resource Explorer setup operation. This operation returns information about
     * the progress of creating or deleting Resource Explorer configurations across Regions.
     * </p>
     *
     * @param getResourceExplorerSetupRequest
     * @return A Java Future containing the result of the GetResourceExplorerSetup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.GetResourceExplorerSetup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/GetResourceExplorerSetup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceExplorerSetupResponse> getResourceExplorerSetup(
            GetResourceExplorerSetupRequest getResourceExplorerSetupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceExplorerSetupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceExplorerSetupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceExplorerSetup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceExplorerSetupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceExplorerSetupResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetResourceExplorerSetupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceExplorerSetupRequest, GetResourceExplorerSetupResponse>()
                            .withOperationName("GetResourceExplorerSetup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourceExplorerSetupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourceExplorerSetupRequest));
            CompletableFuture<GetResourceExplorerSetupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about the Resource Explorer index in the current Amazon Web Services Region. This operation
     * returns the ARN and type of the index if one exists.
     * </p>
     *
     * @param getServiceIndexRequest
     * @return A Java Future containing the result of the GetServiceIndex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.GetServiceIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/GetServiceIndex"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceIndexResponse> getServiceIndex(GetServiceIndexRequest getServiceIndexRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceIndexRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceIndex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceIndexResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServiceIndexResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetServiceIndexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceIndexRequest, GetServiceIndexResponse>()
                            .withOperationName("GetServiceIndex").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetServiceIndexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getServiceIndexRequest));
            CompletableFuture<GetServiceIndexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about a specific Resource Explorer service view. This operation returns the configuration and
     * properties of the specified view.
     * </p>
     *
     * @param getServiceViewRequest
     * @return A Java Future containing the result of the GetServiceView operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.GetServiceView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/GetServiceView"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceViewResponse> getServiceView(GetServiceViewRequest getServiceViewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceViewRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceView");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceViewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServiceViewResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetServiceViewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceViewRequest, GetServiceViewResponse>()
                            .withOperationName("GetServiceView").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetServiceViewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getServiceViewRequest));
            CompletableFuture<GetServiceViewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details of the specified view.
     * </p>
     *
     * @param getViewRequest
     * @return A Java Future containing the result of the GetView operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>UnauthorizedException The principal making the request isn't permitted to perform the operation.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.GetView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/GetView" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetViewResponse> getView(GetViewRequest getViewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getViewRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetView");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetViewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetViewResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetViewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetViewRequest, GetViewResponse>().withOperationName("GetView")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetViewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getViewRequest));
            CompletableFuture<GetViewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of all of the indexes in Amazon Web Services Regions that are currently collecting resource
     * information for Amazon Web Services Resource Explorer.
     * </p>
     *
     * @param listIndexesRequest
     * @return A Java Future containing the result of the ListIndexes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.ListIndexes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/ListIndexes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListIndexesResponse> listIndexes(ListIndexesRequest listIndexesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listIndexesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIndexesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIndexes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIndexesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListIndexesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListIndexesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIndexesRequest, ListIndexesResponse>()
                            .withOperationName("ListIndexes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListIndexesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listIndexesRequest));
            CompletableFuture<ListIndexesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of a member's indexes in all Amazon Web Services Regions that are currently collecting resource
     * information for Amazon Web Services Resource Explorer. Only the management account or a delegated administrator
     * with service access enabled can invoke this API call.
     * </p>
     *
     * @param listIndexesForMembersRequest
     * @return A Java Future containing the result of the ListIndexesForMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.ListIndexesForMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/ListIndexesForMembers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListIndexesForMembersResponse> listIndexesForMembers(
            ListIndexesForMembersRequest listIndexesForMembersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listIndexesForMembersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIndexesForMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIndexesForMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIndexesForMembersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListIndexesForMembersResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListIndexesForMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIndexesForMembersRequest, ListIndexesForMembersResponse>()
                            .withOperationName("ListIndexesForMembers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListIndexesForMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listIndexesForMembersRequest));
            CompletableFuture<ListIndexesForMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Amazon resource names (ARNs) of the <a
     * href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/aws-managed-views.html">Amazon Web
     * Services-managed views</a> available in the Amazon Web Services Region in which you call this operation.
     * </p>
     *
     * @param listManagedViewsRequest
     * @return A Java Future containing the result of the ListManagedViews operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>UnauthorizedException The principal making the request isn't permitted to perform the operation.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.ListManagedViews
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/ListManagedViews"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListManagedViewsResponse> listManagedViews(ListManagedViewsRequest listManagedViewsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listManagedViewsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listManagedViewsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListManagedViews");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListManagedViewsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListManagedViewsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListManagedViewsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListManagedViewsRequest, ListManagedViewsResponse>()
                            .withOperationName("ListManagedViews").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListManagedViewsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listManagedViewsRequest));
            CompletableFuture<ListManagedViewsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of resources and their details that match the specified criteria. This query must use a view. If
     * you don’t explicitly specify a view, then Resource Explorer uses the default view for the Amazon Web Services
     * Region in which you call this operation.
     * </p>
     *
     * @param listResourcesRequest
     * @return A Java Future containing the result of the ListResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>UnauthorizedException The principal making the request isn't permitted to perform the operation.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/ListResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourcesResponse> listResources(ListResourcesRequest listResourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourcesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListResourcesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourcesRequest, ListResourcesResponse>()
                            .withOperationName("ListResources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listResourcesRequest));
            CompletableFuture<ListResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all Resource Explorer indexes across the specified Amazon Web Services Regions. This operation returns
     * information about indexes including their ARNs, types, and Regions.
     * </p>
     *
     * @param listServiceIndexesRequest
     * @return A Java Future containing the result of the ListServiceIndexes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.ListServiceIndexes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/ListServiceIndexes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServiceIndexesResponse> listServiceIndexes(ListServiceIndexesRequest listServiceIndexesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServiceIndexesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServiceIndexesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceIndexes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServiceIndexesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListServiceIndexesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListServiceIndexesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServiceIndexesRequest, ListServiceIndexesResponse>()
                            .withOperationName("ListServiceIndexes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListServiceIndexesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listServiceIndexesRequest));
            CompletableFuture<ListServiceIndexesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all Resource Explorer service views available in the current Amazon Web Services account. This operation
     * returns the ARNs of available service views.
     * </p>
     *
     * @param listServiceViewsRequest
     * @return A Java Future containing the result of the ListServiceViews operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.ListServiceViews
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/ListServiceViews"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServiceViewsResponse> listServiceViews(ListServiceViewsRequest listServiceViewsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServiceViewsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServiceViewsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceViews");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServiceViewsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListServiceViewsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListServiceViewsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServiceViewsRequest, ListServiceViewsResponse>()
                            .withOperationName("ListServiceViews").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListServiceViewsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listServiceViewsRequest));
            CompletableFuture<ListServiceViewsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of Amazon Web Services services that have been granted streaming access to your Resource Explorer
     * data. Streaming access allows Amazon Web Services services to receive real-time updates about your resources as
     * they are indexed by Resource Explorer.
     * </p>
     *
     * @param listStreamingAccessForServicesRequest
     * @return A Java Future containing the result of the ListStreamingAccessForServices operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.ListStreamingAccessForServices
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/ListStreamingAccessForServices"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListStreamingAccessForServicesResponse> listStreamingAccessForServices(
            ListStreamingAccessForServicesRequest listStreamingAccessForServicesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStreamingAccessForServicesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listStreamingAccessForServicesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStreamingAccessForServices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStreamingAccessForServicesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListStreamingAccessForServicesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListStreamingAccessForServicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStreamingAccessForServicesRequest, ListStreamingAccessForServicesResponse>()
                            .withOperationName("ListStreamingAccessForServices").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListStreamingAccessForServicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listStreamingAccessForServicesRequest));
            CompletableFuture<ListStreamingAccessForServicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of all resource types currently supported by Amazon Web Services Resource Explorer.
     * </p>
     *
     * @param listSupportedResourceTypesRequest
     * @return A Java Future containing the result of the ListSupportedResourceTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.ListSupportedResourceTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/ListSupportedResourceTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSupportedResourceTypesResponse> listSupportedResourceTypes(
            ListSupportedResourceTypesRequest listSupportedResourceTypesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSupportedResourceTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSupportedResourceTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSupportedResourceTypes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSupportedResourceTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSupportedResourceTypesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListSupportedResourceTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSupportedResourceTypesRequest, ListSupportedResourceTypesResponse>()
                            .withOperationName("ListSupportedResourceTypes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSupportedResourceTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSupportedResourceTypesRequest));
            CompletableFuture<ListSupportedResourceTypesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags that are attached to the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>UnauthorizedException The principal making the request isn't permitted to perform the operation.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon resource
     * names (ARNs)</a> of the views available in the Amazon Web Services Region in which you call this operation.
     * </p>
     * <note>
     * <p>
     * Always check the <code>NextToken</code> response parameter for a <code>null</code> value when calling a paginated
     * operation. These operations can occasionally return an empty set of results even when there are more results
     * available. The <code>NextToken</code> response parameter value is <code>null</code> <i>only</i> when there are no
     * more results to display.
     * </p>
     * </note>
     *
     * @param listViewsRequest
     * @return A Java Future containing the result of the ListViews operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.ListViews
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/ListViews" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListViewsResponse> listViews(ListViewsRequest listViewsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listViewsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listViewsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListViews");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListViewsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListViewsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListViewsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListViewsRequest, ListViewsResponse>().withOperationName("ListViews")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListViewsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listViewsRequest));
            CompletableFuture<ListViewsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Searches for resources and displays details about all resources that match the specified criteria. You must
     * specify a query string.
     * </p>
     * <p>
     * All search queries must use a view. If you don't explicitly specify a view, then Amazon Web Services Resource
     * Explorer uses the default view for the Amazon Web Services Region in which you call this operation. The results
     * are the logical intersection of the results that match both the <code>QueryString</code> parameter supplied to
     * this operation and the <code>SearchFilter</code> parameter attached to the view.
     * </p>
     * <p>
     * For the complete syntax supported by the <code>QueryString</code> parameter, see <a
     * href="https://docs.aws.amazon.com/resource-explorer/latest/APIReference/about-query-syntax.html">Search query
     * syntax reference for Resource Explorer</a>.
     * </p>
     * <p>
     * If your search results are empty, or are missing results that you think should be there, see <a
     * href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/troubleshooting_search.html">Troubleshooting
     * Resource Explorer search</a>.
     * </p>
     *
     * @param searchRequest
     * @return A Java Future containing the result of the Search operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>UnauthorizedException The principal making the request isn't permitted to perform the operation.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.Search
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/Search" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchResponse> search(SearchRequest searchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Search");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SearchResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<SearchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchRequest, SearchResponse>().withOperationName("Search")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new SearchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(searchRequest));
            CompletableFuture<SearchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tag key and value pairs to an Amazon Web Services Resource Explorer view or index.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ConflictException If you attempted to create a view, then the request failed because either you
     *         specified parameters that didn’t match the original request, or you attempted to create a view with a
     *         name that already exists in this Amazon Web Services Region.</p>
     *         <p>
     *         If you attempted to create an index, then the request failed because either you specified parameters that
     *         didn't match the original request, or an index already exists in the current Amazon Web Services Region.
     *         </p>
     *         <p>
     *         If you attempted to update an index type to <code>AGGREGATOR</code>, then the request failed because you
     *         already have an <code>AGGREGATOR</code> index in a different Amazon Web Services Region.</li>
     *         <li>UnauthorizedException The principal making the request isn't permitted to perform the operation.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tag key and value pairs from an Amazon Web Services Resource Explorer view or index.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>UnauthorizedException The principal making the request isn't permitted to perform the operation.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Changes the type of the index from one of the following types to the other. For more information about indexes
     * and the role they perform in Amazon Web Services Resource Explorer, see <a
     * href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/manage-aggregator-region.html">Turning on
     * cross-Region search by creating an aggregator index</a> in the <i>Amazon Web Services Resource Explorer User
     * Guide</i>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>AGGREGATOR</code> index type</b>
     * </p>
     * <p>
     * The index contains information about resources from all Amazon Web Services Regions in the Amazon Web Services
     * account in which you've created a Resource Explorer index. Resource information from all other Regions is
     * replicated to this Region's index.
     * </p>
     * <p>
     * When you change the index type to <code>AGGREGATOR</code>, Resource Explorer turns on replication of all
     * discovered resource information from the other Amazon Web Services Regions in your account to this index. You can
     * then, from this Region only, perform resource search queries that span all Amazon Web Services Regions in the
     * Amazon Web Services account. Turning on replication from all other Regions is performed by asynchronous
     * background tasks. You can check the status of the asynchronous tasks by using the <a>GetIndex</a> operation. When
     * the asynchronous tasks complete, the <code>Status</code> response of that operation changes from
     * <code>UPDATING</code> to <code>ACTIVE</code>. After that, you can start to see results from other Amazon Web
     * Services Regions in query results. However, it can take several hours for replication from all other Regions to
     * complete.
     * </p>
     * <important>
     * <p>
     * You can have only one aggregator index per Amazon Web Services account. Before you can promote a different index
     * to be the aggregator index for the account, you must first demote the existing aggregator index to type
     * <code>LOCAL</code>.
     * </p>
     * </important></li>
     * <li>
     * <p>
     * <b> <code>LOCAL</code> index type</b>
     * </p>
     * <p>
     * The index contains information about resources in only the Amazon Web Services Region in which the index exists.
     * If an aggregator index in another Region exists, then information in this local index is replicated to the
     * aggregator index.
     * </p>
     * <p>
     * When you change the index type to <code>LOCAL</code>, Resource Explorer turns off the replication of resource
     * information from all other Amazon Web Services Regions in the Amazon Web Services account to this Region. The
     * aggregator index remains in the <code>UPDATING</code> state until all replication with other Regions successfully
     * stops. You can check the status of the asynchronous task by using the <a>GetIndex</a> operation. When Resource
     * Explorer successfully stops all replication with other Regions, the <code>Status</code> response of that
     * operation changes from <code>UPDATING</code> to <code>ACTIVE</code>. Separately, the resource information from
     * other Regions that was previously stored in the index is deleted within 30 days by another background task. Until
     * that asynchronous task completes, some results from other Regions can continue to appear in search results.
     * </p>
     * <important>
     * <p>
     * After you demote an aggregator index to a local index, you must wait 24 hours before you can promote another
     * index to be the new aggregator index for the account.
     * </p>
     * </important></li>
     * </ul>
     *
     * @param updateIndexTypeRequest
     * @return A Java Future containing the result of the UpdateIndexType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException You specified a resource that doesn't exist. Check the ID or ARN that you
     *         used to identity the resource, and try again.</li>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ConflictException If you attempted to create a view, then the request failed because either you
     *         specified parameters that didn’t match the original request, or you attempted to create a view with a
     *         name that already exists in this Amazon Web Services Region.</p>
     *         <p>
     *         If you attempted to create an index, then the request failed because either you specified parameters that
     *         didn't match the original request, or an index already exists in the current Amazon Web Services Region.
     *         </p>
     *         <p>
     *         If you attempted to update an index type to <code>AGGREGATOR</code>, then the request failed because you
     *         already have an <code>AGGREGATOR</code> index in a different Amazon Web Services Region.</li>
     *         <li>ServiceQuotaExceededException The request failed because it exceeds a service quota.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.UpdateIndexType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/UpdateIndexType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateIndexTypeResponse> updateIndexType(UpdateIndexTypeRequest updateIndexTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateIndexTypeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateIndexTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateIndexType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateIndexTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateIndexTypeResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateIndexTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateIndexTypeRequest, UpdateIndexTypeResponse>()
                            .withOperationName("UpdateIndexType").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateIndexTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateIndexTypeRequest));
            CompletableFuture<UpdateIndexTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies some of the details of a view. You can change the filter string and the list of included properties. You
     * can't change the name of the view.
     * </p>
     *
     * @param updateViewRequest
     * @return A Java Future containing the result of the UpdateView operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request failed because of internal service error. Try your request again
     *         later.</li>
     *         <li>ValidationException You provided an invalid value for one of the operation's parameters. Check the
     *         syntax for the operation, and try again.</li>
     *         <li>ServiceQuotaExceededException The request failed because it exceeds a service quota.</li>
     *         <li>UnauthorizedException The principal making the request isn't permitted to perform the operation.</li>
     *         <li>ThrottlingException The request failed because you exceeded a rate limit for this operation. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/quotas.html">Quotas for Resource
     *         Explorer</a>.</li>
     *         <li>AccessDeniedException The credentials that you used to call this operation don't have the minimum
     *         required permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceExplorer2Exception Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ResourceExplorer2AsyncClient.UpdateView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-explorer-2-2022-07-28/UpdateView"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateViewResponse> updateView(UpdateViewRequest updateViewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateViewRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Explorer 2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateView");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateViewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateViewResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateViewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateViewRequest, UpdateViewResponse>().withOperationName("UpdateView")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateViewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateViewRequest));
            CompletableFuture<UpdateViewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final ResourceExplorer2ServiceClientConfiguration serviceClientConfiguration() {
        return new ResourceExplorer2ServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ResourceExplorer2Exception::builder).protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        ResourceExplorer2ServiceClientConfigurationBuilder serviceConfigBuilder = new ResourceExplorer2ServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
