/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.repostspace.waiters;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Immutable;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.repostspace.RepostspaceClient;
import software.amazon.awssdk.services.repostspace.model.GetChannelRequest;
import software.amazon.awssdk.services.repostspace.model.GetChannelResponse;
import software.amazon.awssdk.services.repostspace.model.GetSpaceRequest;
import software.amazon.awssdk.services.repostspace.model.GetSpaceResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
@Immutable
public interface RepostspaceWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link RepostspaceClient#getChannel} API until the desired condition {@code ChannelCreated} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param getChannelRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetChannelResponse> waitUntilChannelCreated(GetChannelRequest getChannelRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RepostspaceClient#getChannel} API until the desired condition {@code ChannelCreated} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetChannelRequest#builder()}
     *
     * @param getChannelRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetChannelResponse> waitUntilChannelCreated(Consumer<GetChannelRequest.Builder> getChannelRequest) {
        return waitUntilChannelCreated(GetChannelRequest.builder().applyMutation(getChannelRequest).build());
    }

    /**
     * Polls {@link RepostspaceClient#getChannel} API until the desired condition {@code ChannelCreated} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param getChannelRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetChannelResponse> waitUntilChannelCreated(GetChannelRequest getChannelRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RepostspaceClient#getChannel} API until the desired condition {@code ChannelCreated} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getChannelRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetChannelResponse> waitUntilChannelCreated(Consumer<GetChannelRequest.Builder> getChannelRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilChannelCreated(GetChannelRequest.builder().applyMutation(getChannelRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link RepostspaceClient#getChannel} API until the desired condition {@code ChannelDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param getChannelRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetChannelResponse> waitUntilChannelDeleted(GetChannelRequest getChannelRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RepostspaceClient#getChannel} API until the desired condition {@code ChannelDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetChannelRequest#builder()}
     *
     * @param getChannelRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetChannelResponse> waitUntilChannelDeleted(Consumer<GetChannelRequest.Builder> getChannelRequest) {
        return waitUntilChannelDeleted(GetChannelRequest.builder().applyMutation(getChannelRequest).build());
    }

    /**
     * Polls {@link RepostspaceClient#getChannel} API until the desired condition {@code ChannelDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param getChannelRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetChannelResponse> waitUntilChannelDeleted(GetChannelRequest getChannelRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RepostspaceClient#getChannel} API until the desired condition {@code ChannelDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getChannelRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetChannelResponse> waitUntilChannelDeleted(Consumer<GetChannelRequest.Builder> getChannelRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilChannelDeleted(GetChannelRequest.builder().applyMutation(getChannelRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link RepostspaceClient#getSpace} API until the desired condition {@code SpaceCreated} is met, or until it
     * is determined that the resource will never enter into the desired state
     *
     * @param getSpaceRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetSpaceResponse> waitUntilSpaceCreated(GetSpaceRequest getSpaceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RepostspaceClient#getSpace} API until the desired condition {@code SpaceCreated} is met, or until it
     * is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetSpaceRequest#builder()}
     *
     * @param getSpaceRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetSpaceResponse> waitUntilSpaceCreated(Consumer<GetSpaceRequest.Builder> getSpaceRequest) {
        return waitUntilSpaceCreated(GetSpaceRequest.builder().applyMutation(getSpaceRequest).build());
    }

    /**
     * Polls {@link RepostspaceClient#getSpace} API until the desired condition {@code SpaceCreated} is met, or until it
     * is determined that the resource will never enter into the desired state
     *
     * @param getSpaceRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetSpaceResponse> waitUntilSpaceCreated(GetSpaceRequest getSpaceRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RepostspaceClient#getSpace} API until the desired condition {@code SpaceCreated} is met, or until it
     * is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getSpaceRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetSpaceResponse> waitUntilSpaceCreated(Consumer<GetSpaceRequest.Builder> getSpaceRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilSpaceCreated(GetSpaceRequest.builder().applyMutation(getSpaceRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link RepostspaceClient#getSpace} API until the desired condition {@code SpaceDeleted} is met, or until it
     * is determined that the resource will never enter into the desired state
     *
     * @param getSpaceRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetSpaceResponse> waitUntilSpaceDeleted(GetSpaceRequest getSpaceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RepostspaceClient#getSpace} API until the desired condition {@code SpaceDeleted} is met, or until it
     * is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetSpaceRequest#builder()}
     *
     * @param getSpaceRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetSpaceResponse> waitUntilSpaceDeleted(Consumer<GetSpaceRequest.Builder> getSpaceRequest) {
        return waitUntilSpaceDeleted(GetSpaceRequest.builder().applyMutation(getSpaceRequest).build());
    }

    /**
     * Polls {@link RepostspaceClient#getSpace} API until the desired condition {@code SpaceDeleted} is met, or until it
     * is determined that the resource will never enter into the desired state
     *
     * @param getSpaceRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetSpaceResponse> waitUntilSpaceDeleted(GetSpaceRequest getSpaceRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RepostspaceClient#getSpace} API until the desired condition {@code SpaceDeleted} is met, or until it
     * is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getSpaceRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetSpaceResponse> waitUntilSpaceDeleted(Consumer<GetSpaceRequest.Builder> getSpaceRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilSpaceDeleted(GetSpaceRequest.builder().applyMutation(getSpaceRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link RepostspaceWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultRepostspaceWaiter.builder();
    }

    /**
     * Create an instance of {@link RepostspaceWaiter} with the default configuration.
     * <p>
     * <b>A default {@link RepostspaceClient} will be created to poll resources. It is recommended to share a single
     * instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link RepostspaceWaiter}
     */
    static RepostspaceWaiter create() {
        return DefaultRepostspaceWaiter.builder().build();
    }

    interface Builder {
        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link RepostspaceClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(RepostspaceClient client);

        /**
         * Builds an instance of {@link RepostspaceWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link RepostspaceWaiter}
         */
        RepostspaceWaiter build();
    }
}
