/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpointemail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that contains information about the DKIM configuration for an email identity.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DkimAttributes implements SdkPojo, Serializable, ToCopyableBuilder<DkimAttributes.Builder, DkimAttributes> {
    private static final SdkField<Boolean> SIGNING_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(DkimAttributes::signingEnabled)).setter(setter(Builder::signingEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SigningEnabled").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DkimAttributes::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<String>> TOKENS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DkimAttributes::tokens))
            .setter(setter(Builder::tokens))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tokens").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SIGNING_ENABLED_FIELD,
            STATUS_FIELD, TOKENS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean signingEnabled;

    private final String status;

    private final List<String> tokens;

    private DkimAttributes(BuilderImpl builder) {
        this.signingEnabled = builder.signingEnabled;
        this.status = builder.status;
        this.tokens = builder.tokens;
    }

    /**
     * <p>
     * If the value is <code>true</code>, then the messages that Amazon Pinpoint sends from the identity are
     * DKIM-signed. If the value is <code>false</code>, then the messages that Amazon Pinpoint sends from the identity
     * aren't DKIM-signed.
     * </p>
     * 
     * @return If the value is <code>true</code>, then the messages that Amazon Pinpoint sends from the identity are
     *         DKIM-signed. If the value is <code>false</code>, then the messages that Amazon Pinpoint sends from the
     *         identity aren't DKIM-signed.
     */
    public Boolean signingEnabled() {
        return signingEnabled;
    }

    /**
     * <p>
     * Describes whether or not Amazon Pinpoint has successfully located the DKIM records in the DNS records for the
     * domain. The status can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PENDING</code> – Amazon Pinpoint hasn't yet located the DKIM records in the DNS configuration for the
     * domain, but will continue to attempt to locate them.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCESS</code> – Amazon Pinpoint located the DKIM records in the DNS configuration for the domain and
     * determined that they're correct. Amazon Pinpoint can now send DKIM-signed email from the identity.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> – Amazon Pinpoint was unable to locate the DKIM records in the DNS settings for the domain,
     * and won't continue to search for them.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TEMPORARY_FAILURE</code> – A temporary issue occurred, which prevented Amazon Pinpoint from determining the
     * DKIM status for the domain.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_STARTED</code> – Amazon Pinpoint hasn't yet started searching for the DKIM records in the DKIM records
     * for the domain.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DkimStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Describes whether or not Amazon Pinpoint has successfully located the DKIM records in the DNS records for
     *         the domain. The status can be one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PENDING</code> – Amazon Pinpoint hasn't yet located the DKIM records in the DNS configuration for
     *         the domain, but will continue to attempt to locate them.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCESS</code> – Amazon Pinpoint located the DKIM records in the DNS configuration for the domain
     *         and determined that they're correct. Amazon Pinpoint can now send DKIM-signed email from the identity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> – Amazon Pinpoint was unable to locate the DKIM records in the DNS settings for the
     *         domain, and won't continue to search for them.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TEMPORARY_FAILURE</code> – A temporary issue occurred, which prevented Amazon Pinpoint from
     *         determining the DKIM status for the domain.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_STARTED</code> – Amazon Pinpoint hasn't yet started searching for the DKIM records in the DKIM
     *         records for the domain.
     *         </p>
     *         </li>
     * @see DkimStatus
     */
    public DkimStatus status() {
        return DkimStatus.fromValue(status);
    }

    /**
     * <p>
     * Describes whether or not Amazon Pinpoint has successfully located the DKIM records in the DNS records for the
     * domain. The status can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PENDING</code> – Amazon Pinpoint hasn't yet located the DKIM records in the DNS configuration for the
     * domain, but will continue to attempt to locate them.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCESS</code> – Amazon Pinpoint located the DKIM records in the DNS configuration for the domain and
     * determined that they're correct. Amazon Pinpoint can now send DKIM-signed email from the identity.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> – Amazon Pinpoint was unable to locate the DKIM records in the DNS settings for the domain,
     * and won't continue to search for them.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TEMPORARY_FAILURE</code> – A temporary issue occurred, which prevented Amazon Pinpoint from determining the
     * DKIM status for the domain.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_STARTED</code> – Amazon Pinpoint hasn't yet started searching for the DKIM records in the DKIM records
     * for the domain.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DkimStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Describes whether or not Amazon Pinpoint has successfully located the DKIM records in the DNS records for
     *         the domain. The status can be one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PENDING</code> – Amazon Pinpoint hasn't yet located the DKIM records in the DNS configuration for
     *         the domain, but will continue to attempt to locate them.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCESS</code> – Amazon Pinpoint located the DKIM records in the DNS configuration for the domain
     *         and determined that they're correct. Amazon Pinpoint can now send DKIM-signed email from the identity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> – Amazon Pinpoint was unable to locate the DKIM records in the DNS settings for the
     *         domain, and won't continue to search for them.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TEMPORARY_FAILURE</code> – A temporary issue occurred, which prevented Amazon Pinpoint from
     *         determining the DKIM status for the domain.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_STARTED</code> – Amazon Pinpoint hasn't yet started searching for the DKIM records in the DKIM
     *         records for the domain.
     *         </p>
     *         </li>
     * @see DkimStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * Returns true if the Tokens property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTokens() {
        return tokens != null && !(tokens instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of unique strings that you use to create a set of CNAME records that you add to the DNS configuration for
     * your domain. When Amazon Pinpoint detects these records in the DNS configuration for your domain, the DKIM
     * authentication process is complete. Amazon Pinpoint usually detects these records within about 72 hours of adding
     * them to the DNS configuration for your domain.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTokens()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A set of unique strings that you use to create a set of CNAME records that you add to the DNS
     *         configuration for your domain. When Amazon Pinpoint detects these records in the DNS configuration for
     *         your domain, the DKIM authentication process is complete. Amazon Pinpoint usually detects these records
     *         within about 72 hours of adding them to the DNS configuration for your domain.
     */
    public List<String> tokens() {
        return tokens;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(signingEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tokens());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DkimAttributes)) {
            return false;
        }
        DkimAttributes other = (DkimAttributes) obj;
        return Objects.equals(signingEnabled(), other.signingEnabled())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(tokens(), other.tokens());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DkimAttributes").add("SigningEnabled", signingEnabled()).add("Status", statusAsString())
                .add("Tokens", tokens()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SigningEnabled":
            return Optional.ofNullable(clazz.cast(signingEnabled()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Tokens":
            return Optional.ofNullable(clazz.cast(tokens()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DkimAttributes, T> g) {
        return obj -> g.apply((DkimAttributes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DkimAttributes> {
        /**
         * <p>
         * If the value is <code>true</code>, then the messages that Amazon Pinpoint sends from the identity are
         * DKIM-signed. If the value is <code>false</code>, then the messages that Amazon Pinpoint sends from the
         * identity aren't DKIM-signed.
         * </p>
         * 
         * @param signingEnabled
         *        If the value is <code>true</code>, then the messages that Amazon Pinpoint sends from the identity are
         *        DKIM-signed. If the value is <code>false</code>, then the messages that Amazon Pinpoint sends from the
         *        identity aren't DKIM-signed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signingEnabled(Boolean signingEnabled);

        /**
         * <p>
         * Describes whether or not Amazon Pinpoint has successfully located the DKIM records in the DNS records for the
         * domain. The status can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PENDING</code> – Amazon Pinpoint hasn't yet located the DKIM records in the DNS configuration for the
         * domain, but will continue to attempt to locate them.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCESS</code> – Amazon Pinpoint located the DKIM records in the DNS configuration for the domain and
         * determined that they're correct. Amazon Pinpoint can now send DKIM-signed email from the identity.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> – Amazon Pinpoint was unable to locate the DKIM records in the DNS settings for the
         * domain, and won't continue to search for them.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TEMPORARY_FAILURE</code> – A temporary issue occurred, which prevented Amazon Pinpoint from determining
         * the DKIM status for the domain.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_STARTED</code> – Amazon Pinpoint hasn't yet started searching for the DKIM records in the DKIM
         * records for the domain.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Describes whether or not Amazon Pinpoint has successfully located the DKIM records in the DNS records
         *        for the domain. The status can be one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PENDING</code> – Amazon Pinpoint hasn't yet located the DKIM records in the DNS configuration
         *        for the domain, but will continue to attempt to locate them.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCESS</code> – Amazon Pinpoint located the DKIM records in the DNS configuration for the
         *        domain and determined that they're correct. Amazon Pinpoint can now send DKIM-signed email from the
         *        identity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> – Amazon Pinpoint was unable to locate the DKIM records in the DNS settings for
         *        the domain, and won't continue to search for them.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TEMPORARY_FAILURE</code> – A temporary issue occurred, which prevented Amazon Pinpoint from
         *        determining the DKIM status for the domain.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_STARTED</code> – Amazon Pinpoint hasn't yet started searching for the DKIM records in the
         *        DKIM records for the domain.
         *        </p>
         *        </li>
         * @see DkimStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DkimStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Describes whether or not Amazon Pinpoint has successfully located the DKIM records in the DNS records for the
         * domain. The status can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PENDING</code> – Amazon Pinpoint hasn't yet located the DKIM records in the DNS configuration for the
         * domain, but will continue to attempt to locate them.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCESS</code> – Amazon Pinpoint located the DKIM records in the DNS configuration for the domain and
         * determined that they're correct. Amazon Pinpoint can now send DKIM-signed email from the identity.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> – Amazon Pinpoint was unable to locate the DKIM records in the DNS settings for the
         * domain, and won't continue to search for them.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TEMPORARY_FAILURE</code> – A temporary issue occurred, which prevented Amazon Pinpoint from determining
         * the DKIM status for the domain.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_STARTED</code> – Amazon Pinpoint hasn't yet started searching for the DKIM records in the DKIM
         * records for the domain.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Describes whether or not Amazon Pinpoint has successfully located the DKIM records in the DNS records
         *        for the domain. The status can be one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PENDING</code> – Amazon Pinpoint hasn't yet located the DKIM records in the DNS configuration
         *        for the domain, but will continue to attempt to locate them.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCESS</code> – Amazon Pinpoint located the DKIM records in the DNS configuration for the
         *        domain and determined that they're correct. Amazon Pinpoint can now send DKIM-signed email from the
         *        identity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> – Amazon Pinpoint was unable to locate the DKIM records in the DNS settings for
         *        the domain, and won't continue to search for them.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TEMPORARY_FAILURE</code> – A temporary issue occurred, which prevented Amazon Pinpoint from
         *        determining the DKIM status for the domain.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_STARTED</code> – Amazon Pinpoint hasn't yet started searching for the DKIM records in the
         *        DKIM records for the domain.
         *        </p>
         *        </li>
         * @see DkimStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DkimStatus
         */
        Builder status(DkimStatus status);

        /**
         * <p>
         * A set of unique strings that you use to create a set of CNAME records that you add to the DNS configuration
         * for your domain. When Amazon Pinpoint detects these records in the DNS configuration for your domain, the
         * DKIM authentication process is complete. Amazon Pinpoint usually detects these records within about 72 hours
         * of adding them to the DNS configuration for your domain.
         * </p>
         * 
         * @param tokens
         *        A set of unique strings that you use to create a set of CNAME records that you add to the DNS
         *        configuration for your domain. When Amazon Pinpoint detects these records in the DNS configuration for
         *        your domain, the DKIM authentication process is complete. Amazon Pinpoint usually detects these
         *        records within about 72 hours of adding them to the DNS configuration for your domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokens(Collection<String> tokens);

        /**
         * <p>
         * A set of unique strings that you use to create a set of CNAME records that you add to the DNS configuration
         * for your domain. When Amazon Pinpoint detects these records in the DNS configuration for your domain, the
         * DKIM authentication process is complete. Amazon Pinpoint usually detects these records within about 72 hours
         * of adding them to the DNS configuration for your domain.
         * </p>
         * 
         * @param tokens
         *        A set of unique strings that you use to create a set of CNAME records that you add to the DNS
         *        configuration for your domain. When Amazon Pinpoint detects these records in the DNS configuration for
         *        your domain, the DKIM authentication process is complete. Amazon Pinpoint usually detects these
         *        records within about 72 hours of adding them to the DNS configuration for your domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokens(String... tokens);
    }

    static final class BuilderImpl implements Builder {
        private Boolean signingEnabled;

        private String status;

        private List<String> tokens = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DkimAttributes model) {
            signingEnabled(model.signingEnabled);
            status(model.status);
            tokens(model.tokens);
        }

        public final Boolean getSigningEnabled() {
            return signingEnabled;
        }

        @Override
        public final Builder signingEnabled(Boolean signingEnabled) {
            this.signingEnabled = signingEnabled;
            return this;
        }

        public final void setSigningEnabled(Boolean signingEnabled) {
            this.signingEnabled = signingEnabled;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(DkimStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Collection<String> getTokens() {
            return tokens;
        }

        @Override
        public final Builder tokens(Collection<String> tokens) {
            this.tokens = DnsTokenListCopier.copy(tokens);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tokens(String... tokens) {
            tokens(Arrays.asList(tokens));
            return this;
        }

        public final void setTokens(Collection<String> tokens) {
            this.tokens = DnsTokenListCopier.copy(tokens);
        }

        @Override
        public DkimAttributes build() {
            return new DkimAttributes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
