/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpoint.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies a batch of endpoints and events to process.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EventsBatch implements SdkPojo, Serializable, ToCopyableBuilder<EventsBatch.Builder, EventsBatch> {
    private static final SdkField<PublicEndpoint> ENDPOINT_FIELD = SdkField.<PublicEndpoint> builder(MarshallingType.SDK_POJO)
            .getter(getter(EventsBatch::endpoint)).setter(setter(Builder::endpoint)).constructor(PublicEndpoint::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoint").build()).build();

    private static final SdkField<Map<String, Event>> EVENTS_FIELD = SdkField
            .<Map<String, Event>> builder(MarshallingType.MAP)
            .getter(getter(EventsBatch::events))
            .setter(setter(Builder::events))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Events").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Event> builder(MarshallingType.SDK_POJO)
                                            .constructor(Event::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENDPOINT_FIELD, EVENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final PublicEndpoint endpoint;

    private final Map<String, Event> events;

    private EventsBatch(BuilderImpl builder) {
        this.endpoint = builder.endpoint;
        this.events = builder.events;
    }

    /**
     * <p>
     * A set of properties and attributes that are associated with the endpoint.
     * </p>
     * 
     * @return A set of properties and attributes that are associated with the endpoint.
     */
    public PublicEndpoint endpoint() {
        return endpoint;
    }

    /**
     * Returns true if the Events property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasEvents() {
        return events != null && !(events instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A set of properties that are associated with the event.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEvents()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A set of properties that are associated with the event.
     */
    public Map<String, Event> events() {
        return events;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(endpoint());
        hashCode = 31 * hashCode + Objects.hashCode(events());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EventsBatch)) {
            return false;
        }
        EventsBatch other = (EventsBatch) obj;
        return Objects.equals(endpoint(), other.endpoint()) && Objects.equals(events(), other.events());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("EventsBatch").add("Endpoint", endpoint()).add("Events", events()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Endpoint":
            return Optional.ofNullable(clazz.cast(endpoint()));
        case "Events":
            return Optional.ofNullable(clazz.cast(events()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EventsBatch, T> g) {
        return obj -> g.apply((EventsBatch) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EventsBatch> {
        /**
         * <p>
         * A set of properties and attributes that are associated with the endpoint.
         * </p>
         * 
         * @param endpoint
         *        A set of properties and attributes that are associated with the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoint(PublicEndpoint endpoint);

        /**
         * <p>
         * A set of properties and attributes that are associated with the endpoint.
         * </p>
         * This is a convenience that creates an instance of the {@link PublicEndpoint.Builder} avoiding the need to
         * create one manually via {@link PublicEndpoint#builder()}.
         *
         * When the {@link Consumer} completes, {@link PublicEndpoint.Builder#build()} is called immediately and its
         * result is passed to {@link #endpoint(PublicEndpoint)}.
         * 
         * @param endpoint
         *        a consumer that will call methods on {@link PublicEndpoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #endpoint(PublicEndpoint)
         */
        default Builder endpoint(Consumer<PublicEndpoint.Builder> endpoint) {
            return endpoint(PublicEndpoint.builder().applyMutation(endpoint).build());
        }

        /**
         * <p>
         * A set of properties that are associated with the event.
         * </p>
         * 
         * @param events
         *        A set of properties that are associated with the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder events(Map<String, Event> events);
    }

    static final class BuilderImpl implements Builder {
        private PublicEndpoint endpoint;

        private Map<String, Event> events = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EventsBatch model) {
            endpoint(model.endpoint);
            events(model.events);
        }

        public final PublicEndpoint.Builder getEndpoint() {
            return endpoint != null ? endpoint.toBuilder() : null;
        }

        @Override
        public final Builder endpoint(PublicEndpoint endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        public final void setEndpoint(PublicEndpoint.BuilderImpl endpoint) {
            this.endpoint = endpoint != null ? endpoint.build() : null;
        }

        public final Map<String, Event.Builder> getEvents() {
            return events != null ? CollectionUtils.mapValues(events, Event::toBuilder) : null;
        }

        @Override
        public final Builder events(Map<String, Event> events) {
            this.events = MapOfEventCopier.copy(events);
            return this;
        }

        public final void setEvents(Map<String, Event.BuilderImpl> events) {
            this.events = MapOfEventCopier.copyFromBuilder(events);
        }

        @Override
        public EventsBatch build() {
            return new EventsBatch(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
