/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpoint.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about the results of a request to send a message to an endpoint address.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MessageResponse implements SdkPojo, Serializable, ToCopyableBuilder<MessageResponse.Builder, MessageResponse> {
    private static final SdkField<String> APPLICATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MessageResponse::applicationId)).setter(setter(Builder::applicationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationId").build()).build();

    private static final SdkField<Map<String, EndpointMessageResult>> ENDPOINT_RESULT_FIELD = SdkField
            .<Map<String, EndpointMessageResult>> builder(MarshallingType.MAP)
            .getter(getter(MessageResponse::endpointResult))
            .setter(setter(Builder::endpointResult))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointResult").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<EndpointMessageResult> builder(MarshallingType.SDK_POJO)
                                            .constructor(EndpointMessageResult::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> REQUEST_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MessageResponse::requestId)).setter(setter(Builder::requestId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestId").build()).build();

    private static final SdkField<Map<String, MessageResult>> RESULT_FIELD = SdkField
            .<Map<String, MessageResult>> builder(MarshallingType.MAP)
            .getter(getter(MessageResponse::result))
            .setter(setter(Builder::result))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Result").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<MessageResult> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageResult::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APPLICATION_ID_FIELD,
            ENDPOINT_RESULT_FIELD, REQUEST_ID_FIELD, RESULT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String applicationId;

    private final Map<String, EndpointMessageResult> endpointResult;

    private final String requestId;

    private final Map<String, MessageResult> result;

    private MessageResponse(BuilderImpl builder) {
        this.applicationId = builder.applicationId;
        this.endpointResult = builder.endpointResult;
        this.requestId = builder.requestId;
        this.result = builder.result;
    }

    /**
     * <p>
     * The unique identifier for the application that was used to send the message.
     * </p>
     * 
     * @return The unique identifier for the application that was used to send the message.
     */
    public String applicationId() {
        return applicationId;
    }

    /**
     * Returns true if the EndpointResult property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasEndpointResult() {
        return endpointResult != null && !(endpointResult instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map that contains a multipart response for each address that the message was sent to. In the map, the endpoint
     * ID is the key and the result is the value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEndpointResult()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A map that contains a multipart response for each address that the message was sent to. In the map, the
     *         endpoint ID is the key and the result is the value.
     */
    public Map<String, EndpointMessageResult> endpointResult() {
        return endpointResult;
    }

    /**
     * <p>
     * The identifier for the original request that the message was delivered for.
     * </p>
     * 
     * @return The identifier for the original request that the message was delivered for.
     */
    public String requestId() {
        return requestId;
    }

    /**
     * Returns true if the Result property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasResult() {
        return result != null && !(result instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map that contains a multipart response for each address (email address, phone number, or push notification
     * token) that the message was sent to. In the map, the address is the key and the result is the value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResult()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A map that contains a multipart response for each address (email address, phone number, or push
     *         notification token) that the message was sent to. In the map, the address is the key and the result is
     *         the value.
     */
    public Map<String, MessageResult> result() {
        return result;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(applicationId());
        hashCode = 31 * hashCode + Objects.hashCode(endpointResult());
        hashCode = 31 * hashCode + Objects.hashCode(requestId());
        hashCode = 31 * hashCode + Objects.hashCode(result());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MessageResponse)) {
            return false;
        }
        MessageResponse other = (MessageResponse) obj;
        return Objects.equals(applicationId(), other.applicationId()) && Objects.equals(endpointResult(), other.endpointResult())
                && Objects.equals(requestId(), other.requestId()) && Objects.equals(result(), other.result());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("MessageResponse").add("ApplicationId", applicationId()).add("EndpointResult", endpointResult())
                .add("RequestId", requestId()).add("Result", result()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ApplicationId":
            return Optional.ofNullable(clazz.cast(applicationId()));
        case "EndpointResult":
            return Optional.ofNullable(clazz.cast(endpointResult()));
        case "RequestId":
            return Optional.ofNullable(clazz.cast(requestId()));
        case "Result":
            return Optional.ofNullable(clazz.cast(result()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MessageResponse, T> g) {
        return obj -> g.apply((MessageResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MessageResponse> {
        /**
         * <p>
         * The unique identifier for the application that was used to send the message.
         * </p>
         * 
         * @param applicationId
         *        The unique identifier for the application that was used to send the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationId(String applicationId);

        /**
         * <p>
         * A map that contains a multipart response for each address that the message was sent to. In the map, the
         * endpoint ID is the key and the result is the value.
         * </p>
         * 
         * @param endpointResult
         *        A map that contains a multipart response for each address that the message was sent to. In the map,
         *        the endpoint ID is the key and the result is the value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointResult(Map<String, EndpointMessageResult> endpointResult);

        /**
         * <p>
         * The identifier for the original request that the message was delivered for.
         * </p>
         * 
         * @param requestId
         *        The identifier for the original request that the message was delivered for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestId(String requestId);

        /**
         * <p>
         * A map that contains a multipart response for each address (email address, phone number, or push notification
         * token) that the message was sent to. In the map, the address is the key and the result is the value.
         * </p>
         * 
         * @param result
         *        A map that contains a multipart response for each address (email address, phone number, or push
         *        notification token) that the message was sent to. In the map, the address is the key and the result is
         *        the value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder result(Map<String, MessageResult> result);
    }

    static final class BuilderImpl implements Builder {
        private String applicationId;

        private Map<String, EndpointMessageResult> endpointResult = DefaultSdkAutoConstructMap.getInstance();

        private String requestId;

        private Map<String, MessageResult> result = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MessageResponse model) {
            applicationId(model.applicationId);
            endpointResult(model.endpointResult);
            requestId(model.requestId);
            result(model.result);
        }

        public final String getApplicationId() {
            return applicationId;
        }

        @Override
        public final Builder applicationId(String applicationId) {
            this.applicationId = applicationId;
            return this;
        }

        public final void setApplicationId(String applicationId) {
            this.applicationId = applicationId;
        }

        public final Map<String, EndpointMessageResult.Builder> getEndpointResult() {
            return endpointResult != null ? CollectionUtils.mapValues(endpointResult, EndpointMessageResult::toBuilder) : null;
        }

        @Override
        public final Builder endpointResult(Map<String, EndpointMessageResult> endpointResult) {
            this.endpointResult = MapOfEndpointMessageResultCopier.copy(endpointResult);
            return this;
        }

        public final void setEndpointResult(Map<String, EndpointMessageResult.BuilderImpl> endpointResult) {
            this.endpointResult = MapOfEndpointMessageResultCopier.copyFromBuilder(endpointResult);
        }

        public final String getRequestId() {
            return requestId;
        }

        @Override
        public final Builder requestId(String requestId) {
            this.requestId = requestId;
            return this;
        }

        public final void setRequestId(String requestId) {
            this.requestId = requestId;
        }

        public final Map<String, MessageResult.Builder> getResult() {
            return result != null ? CollectionUtils.mapValues(result, MessageResult::toBuilder) : null;
        }

        @Override
        public final Builder result(Map<String, MessageResult> result) {
            this.result = MapOfMessageResultCopier.copy(result);
            return this;
        }

        public final void setResult(Map<String, MessageResult.BuilderImpl> result) {
            this.result = MapOfMessageResultCopier.copyFromBuilder(result);
        }

        @Override
        public MessageResponse build() {
            return new MessageResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
