/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pcaconnectorscep;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.pcaconnectorscep.internal.PcaConnectorScepServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.pcaconnectorscep.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.pcaconnectorscep.model.AccessDeniedException;
import software.amazon.awssdk.services.pcaconnectorscep.model.BadRequestException;
import software.amazon.awssdk.services.pcaconnectorscep.model.ConflictException;
import software.amazon.awssdk.services.pcaconnectorscep.model.CreateChallengeRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.CreateChallengeResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.CreateConnectorRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.CreateConnectorResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.DeleteChallengeRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.DeleteChallengeResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.DeleteConnectorRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.DeleteConnectorResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetChallengeMetadataRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetChallengeMetadataResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetChallengePasswordRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetChallengePasswordResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetConnectorRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetConnectorResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.InternalServerException;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListChallengeMetadataRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListChallengeMetadataResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListConnectorsRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListConnectorsResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.PcaConnectorScepException;
import software.amazon.awssdk.services.pcaconnectorscep.model.ResourceNotFoundException;
import software.amazon.awssdk.services.pcaconnectorscep.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.pcaconnectorscep.model.TagResourceRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.TagResourceResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.ThrottlingException;
import software.amazon.awssdk.services.pcaconnectorscep.model.UntagResourceRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.UntagResourceResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.ValidationException;
import software.amazon.awssdk.services.pcaconnectorscep.transform.CreateChallengeRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.CreateConnectorRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.DeleteChallengeRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.DeleteConnectorRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.GetChallengeMetadataRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.GetChallengePasswordRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.GetConnectorRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.ListChallengeMetadataRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.ListConnectorsRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorscep.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link PcaConnectorScepAsyncClient}.
 *
 * @see PcaConnectorScepAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPcaConnectorScepAsyncClient implements PcaConnectorScepAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPcaConnectorScepAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPcaConnectorScepAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Pca_Connector_Scep" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * For general-purpose connectors. Creates a <i>challenge password</i> for the specified connector. The SCEP
     * protocol uses a challenge password to authenticate a request before issuing a certificate from a certificate
     * authority (CA). Your SCEP clients include the challenge password as part of their certificate request to
     * Connector for SCEP. To retrieve the connector Amazon Resource Names (ARNs) for the connectors in your account,
     * call <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_ListConnectors.html"
     * >ListConnectors</a>.
     * </p>
     * <p>
     * To create additional challenge passwords for the connector, call <code>CreateChallenge</code> again. We recommend
     * frequently rotating your challenge passwords.
     * </p>
     *
     * @param createChallengeRequest
     * @return A Java Future containing the result of the CreateChallenge operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might be
     *         incorrectly specified, or it might have a status other than <code>ACTIVE</code>.</li>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a name
     *         tag, or an invalid pagination token.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You can receive this error if you attempt to perform an operation and you don't
     *         have the required permissions. This can be caused by insufficient permissions in policies attached to
     *         your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ConflictException This request can't be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorScepException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorScepAsyncClient.CreateChallenge
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/CreateChallenge"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateChallengeResponse> createChallenge(CreateChallengeRequest createChallengeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createChallengeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChallengeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChallenge");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateChallengeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateChallengeResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateChallengeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateChallengeRequest, CreateChallengeResponse>()
                            .withOperationName("CreateChallenge").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateChallengeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createChallengeRequest));
            CompletableFuture<CreateChallengeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a SCEP connector. A SCEP connector links Amazon Web Services Private Certificate Authority to your
     * SCEP-compatible devices and mobile device management (MDM) systems. Before you create a connector, you must
     * complete a set of prerequisites, including creation of a private certificate authority (CA) to use with this
     * connector. For more information, see <a href=
     * "https://docs.aws.amazon.com/privateca/latest/userguide/scep-connector.htmlconnector-for-scep-prerequisites.html"
     * >Connector for SCEP prerequisites</a>.
     * </p>
     *
     * @param createConnectorRequest
     * @return A Java Future containing the result of the CreateConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might be
     *         incorrectly specified, or it might have a status other than <code>ACTIVE</code>.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a name
     *         tag, or an invalid pagination token.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You can receive this error if you attempt to perform an operation and you don't
     *         have the required permissions. This can be caused by insufficient permissions in policies attached to
     *         your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ConflictException This request can't be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorScepException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorScepAsyncClient.CreateConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/CreateConnector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConnectorResponse> createConnector(CreateConnectorRequest createConnectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConnectorResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConnectorRequest, CreateConnectorResponse>()
                            .withOperationName("CreateConnector").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createConnectorRequest));
            CompletableFuture<CreateConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Challenge.html"
     * >Challenge</a>.
     * </p>
     *
     * @param deleteChallengeRequest
     * @return A Java Future containing the result of the DeleteChallenge operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might be
     *         incorrectly specified, or it might have a status other than <code>ACTIVE</code>.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a name
     *         tag, or an invalid pagination token.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You can receive this error if you attempt to perform an operation and you don't
     *         have the required permissions. This can be caused by insufficient permissions in policies attached to
     *         your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ConflictException This request can't be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorScepException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorScepAsyncClient.DeleteChallenge
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/DeleteChallenge"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChallengeResponse> deleteChallenge(DeleteChallengeRequest deleteChallengeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteChallengeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChallengeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChallenge");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChallengeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteChallengeResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteChallengeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChallengeRequest, DeleteChallengeResponse>()
                            .withOperationName("DeleteChallenge").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteChallengeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteChallengeRequest));
            CompletableFuture<DeleteChallengeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Connector.html"
     * >Connector</a>. This operation also deletes any challenges associated with the connector.
     * </p>
     *
     * @param deleteConnectorRequest
     * @return A Java Future containing the result of the DeleteConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might be
     *         incorrectly specified, or it might have a status other than <code>ACTIVE</code>.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a name
     *         tag, or an invalid pagination token.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You can receive this error if you attempt to perform an operation and you don't
     *         have the required permissions. This can be caused by insufficient permissions in policies attached to
     *         your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ConflictException This request can't be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorScepException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorScepAsyncClient.DeleteConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/DeleteConnector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConnectorResponse> deleteConnector(DeleteConnectorRequest deleteConnectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteConnectorResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConnectorRequest, DeleteConnectorResponse>()
                            .withOperationName("DeleteConnector").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteConnectorRequest));
            CompletableFuture<DeleteConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the metadata for the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Challenge.html"
     * >Challenge</a>.
     * </p>
     *
     * @param getChallengeMetadataRequest
     * @return A Java Future containing the result of the GetChallengeMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might be
     *         incorrectly specified, or it might have a status other than <code>ACTIVE</code>.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a name
     *         tag, or an invalid pagination token.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You can receive this error if you attempt to perform an operation and you don't
     *         have the required permissions. This can be caused by insufficient permissions in policies attached to
     *         your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorScepException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorScepAsyncClient.GetChallengeMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/GetChallengeMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetChallengeMetadataResponse> getChallengeMetadata(
            GetChallengeMetadataRequest getChallengeMetadataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getChallengeMetadataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChallengeMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChallengeMetadata");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetChallengeMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetChallengeMetadataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetChallengeMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetChallengeMetadataRequest, GetChallengeMetadataResponse>()
                            .withOperationName("GetChallengeMetadata").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetChallengeMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getChallengeMetadataRequest));
            CompletableFuture<GetChallengeMetadataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the challenge password for the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Challenge.html"
     * >Challenge</a>.
     * </p>
     *
     * @param getChallengePasswordRequest
     * @return A Java Future containing the result of the GetChallengePassword operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might be
     *         incorrectly specified, or it might have a status other than <code>ACTIVE</code>.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a name
     *         tag, or an invalid pagination token.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You can receive this error if you attempt to perform an operation and you don't
     *         have the required permissions. This can be caused by insufficient permissions in policies attached to
     *         your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorScepException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorScepAsyncClient.GetChallengePassword
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/GetChallengePassword"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetChallengePasswordResponse> getChallengePassword(
            GetChallengePasswordRequest getChallengePasswordRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getChallengePasswordRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChallengePasswordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChallengePassword");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetChallengePasswordResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetChallengePasswordResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetChallengePasswordResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetChallengePasswordRequest, GetChallengePasswordResponse>()
                            .withOperationName("GetChallengePassword").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetChallengePasswordRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getChallengePasswordRequest));
            CompletableFuture<GetChallengePasswordResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Connector.html"
     * >Connector</a>. Calling this action returns important details about the connector, such as the public SCEP URL
     * where your clients can request certificates.
     * </p>
     *
     * @param getConnectorRequest
     * @return A Java Future containing the result of the GetConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might be
     *         incorrectly specified, or it might have a status other than <code>ACTIVE</code>.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a name
     *         tag, or an invalid pagination token.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You can receive this error if you attempt to perform an operation and you don't
     *         have the required permissions. This can be caused by insufficient permissions in policies attached to
     *         your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorScepException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorScepAsyncClient.GetConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/GetConnector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetConnectorResponse> getConnector(GetConnectorRequest getConnectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConnectorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConnectorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetConnectorResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConnectorRequest, GetConnectorResponse>()
                            .withOperationName("GetConnector").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getConnectorRequest));
            CompletableFuture<GetConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the challenge metadata for the specified ARN.
     * </p>
     *
     * @param listChallengeMetadataRequest
     * @return A Java Future containing the result of the ListChallengeMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might be
     *         incorrectly specified, or it might have a status other than <code>ACTIVE</code>.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a name
     *         tag, or an invalid pagination token.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You can receive this error if you attempt to perform an operation and you don't
     *         have the required permissions. This can be caused by insufficient permissions in policies attached to
     *         your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorScepException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorScepAsyncClient.ListChallengeMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListChallengeMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListChallengeMetadataResponse> listChallengeMetadata(
            ListChallengeMetadataRequest listChallengeMetadataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChallengeMetadataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChallengeMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChallengeMetadata");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChallengeMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListChallengeMetadataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListChallengeMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChallengeMetadataRequest, ListChallengeMetadataResponse>()
                            .withOperationName("ListChallengeMetadata").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListChallengeMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listChallengeMetadataRequest));
            CompletableFuture<ListChallengeMetadataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the connectors belonging to your Amazon Web Services account.
     * </p>
     *
     * @param listConnectorsRequest
     * @return A Java Future containing the result of the ListConnectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a name
     *         tag, or an invalid pagination token.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You can receive this error if you attempt to perform an operation and you don't
     *         have the required permissions. This can be caused by insufficient permissions in policies attached to
     *         your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorScepException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorScepAsyncClient.ListConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListConnectors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListConnectorsResponse> listConnectors(ListConnectorsRequest listConnectorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConnectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConnectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConnectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConnectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConnectorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListConnectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConnectorsRequest, ListConnectorsResponse>()
                            .withOperationName("ListConnectors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListConnectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listConnectorsRequest));
            CompletableFuture<ListConnectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the tags associated with the specified resource. Tags are key-value pairs that you can use to
     * categorize and manage your resources, for purposes like billing. For example, you might set the tag key to
     * "customer" and the value to the customer name or ID. You can specify one or more tags to add to each Amazon Web
     * Services resource, up to 50 tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might be
     *         incorrectly specified, or it might have a status other than <code>ACTIVE</code>.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a name
     *         tag, or an invalid pagination token.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You can receive this error if you attempt to perform an operation and you don't
     *         have the required permissions. This can be caused by insufficient permissions in policies attached to
     *         your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorScepException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorScepAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to your resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might be
     *         incorrectly specified, or it might have a status other than <code>ACTIVE</code>.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a name
     *         tag, or an invalid pagination token.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You can receive this error if you attempt to perform an operation and you don't
     *         have the required permissions. This can be caused by insufficient permissions in policies attached to
     *         your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorScepException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorScepAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from your resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might be
     *         incorrectly specified, or it might have a status other than <code>ACTIVE</code>.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a name
     *         tag, or an invalid pagination token.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You can receive this error if you attempt to perform an operation and you don't
     *         have the required permissions. This can be caused by insufficient permissions in policies attached to
     *         your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorScepException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorScepAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Scep");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final PcaConnectorScepServiceClientConfiguration serviceClientConfiguration() {
        return new PcaConnectorScepServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PcaConnectorScepException::builder).protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        PcaConnectorScepServiceClientConfigurationBuilder serviceConfigBuilder = new PcaConnectorScepServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
