/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.odb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration for Amazon Web Services Key Management Service (KMS) access from the ODB network.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KmsAccess implements SdkPojo, Serializable, ToCopyableBuilder<KmsAccess.Builder, KmsAccess> {
    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(KmsAccess::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<List<String>> IPV4_ADDRESSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ipv4Addresses")
            .getter(getter(KmsAccess::ipv4Addresses))
            .setter(setter(Builder::ipv4Addresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ipv4Addresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("domainName").getter(getter(KmsAccess::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("domainName").build()).build();

    private static final SdkField<String> KMS_POLICY_DOCUMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("kmsPolicyDocument").getter(getter(KmsAccess::kmsPolicyDocument))
            .setter(setter(Builder::kmsPolicyDocument))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kmsPolicyDocument").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD,
            IPV4_ADDRESSES_FIELD, DOMAIN_NAME_FIELD, KMS_POLICY_DOCUMENT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String status;

    private final List<String> ipv4Addresses;

    private final String domainName;

    private final String kmsPolicyDocument;

    private KmsAccess(BuilderImpl builder) {
        this.status = builder.status;
        this.ipv4Addresses = builder.ipv4Addresses;
        this.domainName = builder.domainName;
        this.kmsPolicyDocument = builder.kmsPolicyDocument;
    }

    /**
     * <p>
     * The current status of the KMS access configuration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ManagedResourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the KMS access configuration.
     * @see ManagedResourceStatus
     */
    public final ManagedResourceStatus status() {
        return ManagedResourceStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the KMS access configuration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ManagedResourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the KMS access configuration.
     * @see ManagedResourceStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the Ipv4Addresses property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIpv4Addresses() {
        return ipv4Addresses != null && !(ipv4Addresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IPv4 addresses allowed for KMS access.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIpv4Addresses} method.
     * </p>
     * 
     * @return The IPv4 addresses allowed for KMS access.
     */
    public final List<String> ipv4Addresses() {
        return ipv4Addresses;
    }

    /**
     * <p>
     * The domain name for KMS access configuration.
     * </p>
     * 
     * @return The domain name for KMS access configuration.
     */
    public final String domainName() {
        return domainName;
    }

    /**
     * <p>
     * The KMS policy document that defines permissions for key usage.
     * </p>
     * 
     * @return The KMS policy document that defines permissions for key usage.
     */
    public final String kmsPolicyDocument() {
        return kmsPolicyDocument;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasIpv4Addresses() ? ipv4Addresses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(kmsPolicyDocument());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KmsAccess)) {
            return false;
        }
        KmsAccess other = (KmsAccess) obj;
        return Objects.equals(statusAsString(), other.statusAsString()) && hasIpv4Addresses() == other.hasIpv4Addresses()
                && Objects.equals(ipv4Addresses(), other.ipv4Addresses()) && Objects.equals(domainName(), other.domainName())
                && Objects.equals(kmsPolicyDocument(), other.kmsPolicyDocument());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("KmsAccess").add("Status", statusAsString())
                .add("Ipv4Addresses", hasIpv4Addresses() ? ipv4Addresses() : null).add("DomainName", domainName())
                .add("KmsPolicyDocument", kmsPolicyDocument()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "ipv4Addresses":
            return Optional.ofNullable(clazz.cast(ipv4Addresses()));
        case "domainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "kmsPolicyDocument":
            return Optional.ofNullable(clazz.cast(kmsPolicyDocument()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("status", STATUS_FIELD);
        map.put("ipv4Addresses", IPV4_ADDRESSES_FIELD);
        map.put("domainName", DOMAIN_NAME_FIELD);
        map.put("kmsPolicyDocument", KMS_POLICY_DOCUMENT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<KmsAccess, T> g) {
        return obj -> g.apply((KmsAccess) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KmsAccess> {
        /**
         * <p>
         * The current status of the KMS access configuration.
         * </p>
         * 
         * @param status
         *        The current status of the KMS access configuration.
         * @see ManagedResourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ManagedResourceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the KMS access configuration.
         * </p>
         * 
         * @param status
         *        The current status of the KMS access configuration.
         * @see ManagedResourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ManagedResourceStatus
         */
        Builder status(ManagedResourceStatus status);

        /**
         * <p>
         * The IPv4 addresses allowed for KMS access.
         * </p>
         * 
         * @param ipv4Addresses
         *        The IPv4 addresses allowed for KMS access.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv4Addresses(Collection<String> ipv4Addresses);

        /**
         * <p>
         * The IPv4 addresses allowed for KMS access.
         * </p>
         * 
         * @param ipv4Addresses
         *        The IPv4 addresses allowed for KMS access.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv4Addresses(String... ipv4Addresses);

        /**
         * <p>
         * The domain name for KMS access configuration.
         * </p>
         * 
         * @param domainName
         *        The domain name for KMS access configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * The KMS policy document that defines permissions for key usage.
         * </p>
         * 
         * @param kmsPolicyDocument
         *        The KMS policy document that defines permissions for key usage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsPolicyDocument(String kmsPolicyDocument);
    }

    static final class BuilderImpl implements Builder {
        private String status;

        private List<String> ipv4Addresses = DefaultSdkAutoConstructList.getInstance();

        private String domainName;

        private String kmsPolicyDocument;

        private BuilderImpl() {
        }

        private BuilderImpl(KmsAccess model) {
            status(model.status);
            ipv4Addresses(model.ipv4Addresses);
            domainName(model.domainName);
            kmsPolicyDocument(model.kmsPolicyDocument);
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ManagedResourceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Collection<String> getIpv4Addresses() {
            if (ipv4Addresses instanceof SdkAutoConstructList) {
                return null;
            }
            return ipv4Addresses;
        }

        public final void setIpv4Addresses(Collection<String> ipv4Addresses) {
            this.ipv4Addresses = StringListCopier.copy(ipv4Addresses);
        }

        @Override
        public final Builder ipv4Addresses(Collection<String> ipv4Addresses) {
            this.ipv4Addresses = StringListCopier.copy(ipv4Addresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipv4Addresses(String... ipv4Addresses) {
            ipv4Addresses(Arrays.asList(ipv4Addresses));
            return this;
        }

        public final String getDomainName() {
            return domainName;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final String getKmsPolicyDocument() {
            return kmsPolicyDocument;
        }

        public final void setKmsPolicyDocument(String kmsPolicyDocument) {
            this.kmsPolicyDocument = kmsPolicyDocument;
        }

        @Override
        public final Builder kmsPolicyDocument(String kmsPolicyDocument) {
            this.kmsPolicyDocument = kmsPolicyDocument;
            return this;
        }

        @Override
        public KmsAccess build() {
            return new KmsAccess(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
