/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkflowmonitor;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.networkflowmonitor.internal.NetworkFlowMonitorServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.networkflowmonitor.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.networkflowmonitor.model.AccessDeniedException;
import software.amazon.awssdk.services.networkflowmonitor.model.ConflictException;
import software.amazon.awssdk.services.networkflowmonitor.model.CreateMonitorRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.CreateMonitorResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.CreateScopeRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.CreateScopeResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.DeleteMonitorRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.DeleteMonitorResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.DeleteScopeRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.DeleteScopeResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetMonitorRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetMonitorResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryResultsMonitorTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryResultsMonitorTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryResultsWorkloadInsightsTopContributorsDataRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryResultsWorkloadInsightsTopContributorsDataResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryResultsWorkloadInsightsTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryResultsWorkloadInsightsTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryStatusMonitorTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryStatusMonitorTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryStatusWorkloadInsightsTopContributorsDataRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryStatusWorkloadInsightsTopContributorsDataResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryStatusWorkloadInsightsTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryStatusWorkloadInsightsTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetScopeRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetScopeResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.InternalServerException;
import software.amazon.awssdk.services.networkflowmonitor.model.ListMonitorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.ListMonitorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.ListScopesRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.ListScopesResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.NetworkFlowMonitorException;
import software.amazon.awssdk.services.networkflowmonitor.model.ResourceNotFoundException;
import software.amazon.awssdk.services.networkflowmonitor.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.networkflowmonitor.model.StartQueryMonitorTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.StartQueryMonitorTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.StartQueryWorkloadInsightsTopContributorsDataRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.StartQueryWorkloadInsightsTopContributorsDataResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.StartQueryWorkloadInsightsTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.StartQueryWorkloadInsightsTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.StopQueryMonitorTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.StopQueryMonitorTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.StopQueryWorkloadInsightsTopContributorsDataRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.StopQueryWorkloadInsightsTopContributorsDataResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.StopQueryWorkloadInsightsTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.StopQueryWorkloadInsightsTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.TagResourceRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.TagResourceResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.ThrottlingException;
import software.amazon.awssdk.services.networkflowmonitor.model.UntagResourceRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.UntagResourceResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.UpdateMonitorRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.UpdateMonitorResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.UpdateScopeRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.UpdateScopeResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.ValidationException;
import software.amazon.awssdk.services.networkflowmonitor.transform.CreateMonitorRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.CreateScopeRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.DeleteMonitorRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.DeleteScopeRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetMonitorRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetQueryResultsMonitorTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetQueryResultsWorkloadInsightsTopContributorsDataRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetQueryResultsWorkloadInsightsTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetQueryStatusMonitorTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetQueryStatusWorkloadInsightsTopContributorsDataRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetQueryStatusWorkloadInsightsTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetScopeRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.ListMonitorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.ListScopesRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.StartQueryMonitorTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.StartQueryWorkloadInsightsTopContributorsDataRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.StartQueryWorkloadInsightsTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.StopQueryMonitorTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.StopQueryWorkloadInsightsTopContributorsDataRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.StopQueryWorkloadInsightsTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.UpdateMonitorRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.UpdateScopeRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link NetworkFlowMonitorAsyncClient}.
 *
 * @see NetworkFlowMonitorAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultNetworkFlowMonitorAsyncClient implements NetworkFlowMonitorAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultNetworkFlowMonitorAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultNetworkFlowMonitorAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "NetworkFlowMonitor" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Create a monitor for specific network flows between local and remote resources, so that you can monitor network
     * performance for one or several of your workloads. For each monitor, Network Flow Monitor publishes detailed
     * end-to-end performance metrics and a network health indicator (NHI) that informs you whether there were Amazon
     * Web Services network issues for one or more of the network flows tracked by a monitor, during a time period that
     * you choose.
     * </p>
     *
     * @param createMonitorRequest
     * @return A Java Future containing the result of the CreateMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ConflictException The requested resource is in use.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.CreateMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/CreateMonitor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMonitorResponse> createMonitor(CreateMonitorRequest createMonitorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMonitorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMonitor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateMonitorResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateMonitorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMonitorRequest, CreateMonitorResponse>()
                            .withOperationName("CreateMonitor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateMonitorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createMonitorRequest));
            CompletableFuture<CreateMonitorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * In Network Flow Monitor, you specify a scope for the service to generate metrics for. By using the scope, Network
     * Flow Monitor can generate a topology of all the resources to measure performance metrics for. When you create a
     * scope, you enable permissions for Network Flow Monitor.
     * </p>
     * <p>
     * A scope is a Region-account pair or multiple Region-account pairs. Network Flow Monitor uses your scope to
     * determine all the resources (the topology) where Network Flow Monitor will gather network flow performance
     * metrics for you. To provide performance metrics, Network Flow Monitor uses the data that is sent by the Network
     * Flow Monitor agents you install on the resources.
     * </p>
     * <p>
     * To define the Region-account pairs for your scope, the Network Flow Monitor API uses the following constucts,
     * which allow for future flexibility in defining scopes:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>Targets</i>, which are arrays of targetResources.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>Target resources</i>, which are Region-targetIdentifier pairs.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>Target identifiers</i>, made up of a targetID (currently always an account ID) and a targetType (currently
     * always an account).
     * </p>
     * </li>
     * </ul>
     *
     * @param createScopeRequest
     * @return A Java Future containing the result of the CreateScope operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ConflictException The requested resource is in use.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.CreateScope
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/CreateScope"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateScopeResponse> createScope(CreateScopeRequest createScopeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createScopeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createScopeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateScope");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateScopeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateScopeResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateScopeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateScopeRequest, CreateScopeResponse>()
                            .withOperationName("CreateScope").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateScopeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createScopeRequest));
            CompletableFuture<CreateScopeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a monitor in Network Flow Monitor.
     * </p>
     *
     * @param deleteMonitorRequest
     * @return A Java Future containing the result of the DeleteMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request specifies a resource that doesn't exist.</li>
     *         <li>ConflictException The requested resource is in use.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.DeleteMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/DeleteMonitor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMonitorResponse> deleteMonitor(DeleteMonitorRequest deleteMonitorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMonitorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMonitor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteMonitorResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteMonitorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMonitorRequest, DeleteMonitorResponse>()
                            .withOperationName("DeleteMonitor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMonitorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMonitorRequest));
            CompletableFuture<DeleteMonitorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a scope that has been defined.
     * </p>
     *
     * @param deleteScopeRequest
     * @return A Java Future containing the result of the DeleteScope operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ResourceNotFoundException The request specifies a resource that doesn't exist.</li>
     *         <li>ConflictException The requested resource is in use.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.DeleteScope
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/DeleteScope"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteScopeResponse> deleteScope(DeleteScopeRequest deleteScopeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteScopeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteScopeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteScope");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteScopeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteScopeResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteScopeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteScopeRequest, DeleteScopeResponse>()
                            .withOperationName("DeleteScope").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteScopeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteScopeRequest));
            CompletableFuture<DeleteScopeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a monitor in Network Flow Monitor based on a monitor name. The information returned
     * includes the Amazon Resource Name (ARN), create time, modified time, resources included in the monitor, and
     * status information.
     * </p>
     *
     * @param getMonitorRequest
     * @return A Java Future containing the result of the GetMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request specifies a resource that doesn't exist.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.GetMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMonitorResponse> getMonitor(GetMonitorRequest getMonitorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMonitorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMonitor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMonitorResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetMonitorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMonitorRequest, GetMonitorResponse>().withOperationName("GetMonitor")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetMonitorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMonitorRequest));
            CompletableFuture<GetMonitorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Return the data for a query with the Network Flow Monitor query interface. You specify the query that you want to
     * return results for by providing a query ID and a monitor name. This query returns the top contributors for a
     * specific monitor.
     * </p>
     * <p>
     * Create a query ID for this call by calling the corresponding API call to start the query,
     * <code>StartQueryMonitorTopContributors</code>. Use the scope ID that was returned for your account by
     * <code>CreateScope</code>.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     *
     * @param getQueryResultsMonitorTopContributorsRequest
     * @return A Java Future containing the result of the GetQueryResultsMonitorTopContributors operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ResourceNotFoundException The request specifies a resource that doesn't exist.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.GetQueryResultsMonitorTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetQueryResultsMonitorTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetQueryResultsMonitorTopContributorsResponse> getQueryResultsMonitorTopContributors(
            GetQueryResultsMonitorTopContributorsRequest getQueryResultsMonitorTopContributorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQueryResultsMonitorTopContributorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getQueryResultsMonitorTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryResultsMonitorTopContributors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueryResultsMonitorTopContributorsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetQueryResultsMonitorTopContributorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetQueryResultsMonitorTopContributorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueryResultsMonitorTopContributorsRequest, GetQueryResultsMonitorTopContributorsResponse>()
                            .withOperationName("GetQueryResultsMonitorTopContributors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetQueryResultsMonitorTopContributorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getQueryResultsMonitorTopContributorsRequest));
            CompletableFuture<GetQueryResultsMonitorTopContributorsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Return the data for a query with the Network Flow Monitor query interface. You specify the query that you want to
     * return results for by providing a query ID and a monitor name.
     * </p>
     * <p>
     * This query returns the top contributors for a scope for workload insights. Workload insights provide a high level
     * view of network flow performance data collected by agents. To return the data for the top contributors, see
     * <code>GetQueryResultsWorkloadInsightsTopContributorsData</code>.
     * </p>
     * <p>
     * Create a query ID for this call by calling the corresponding API call to start the query,
     * <code>StartQueryWorkloadInsightsTopContributors</code>. Use the scope ID that was returned for your account by
     * <code>CreateScope</code>.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     *
     * @param getQueryResultsWorkloadInsightsTopContributorsRequest
     * @return A Java Future containing the result of the GetQueryResultsWorkloadInsightsTopContributors operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ResourceNotFoundException The request specifies a resource that doesn't exist.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.GetQueryResultsWorkloadInsightsTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetQueryResultsWorkloadInsightsTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetQueryResultsWorkloadInsightsTopContributorsResponse> getQueryResultsWorkloadInsightsTopContributors(
            GetQueryResultsWorkloadInsightsTopContributorsRequest getQueryResultsWorkloadInsightsTopContributorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                getQueryResultsWorkloadInsightsTopContributorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getQueryResultsWorkloadInsightsTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryResultsWorkloadInsightsTopContributors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueryResultsWorkloadInsightsTopContributorsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetQueryResultsWorkloadInsightsTopContributorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetQueryResultsWorkloadInsightsTopContributorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueryResultsWorkloadInsightsTopContributorsRequest, GetQueryResultsWorkloadInsightsTopContributorsResponse>()
                            .withOperationName("GetQueryResultsWorkloadInsightsTopContributors")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetQueryResultsWorkloadInsightsTopContributorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getQueryResultsWorkloadInsightsTopContributorsRequest));
            CompletableFuture<GetQueryResultsWorkloadInsightsTopContributorsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Return the data for a query with the Network Flow Monitor query interface. Specify the query that you want to
     * return results for by providing a query ID and a scope ID.
     * </p>
     * <p>
     * This query returns the data for top contributors for workload insights for a specific scope. Workload insights
     * provide a high level view of network flow performance data collected by agents for a scope. To return just the
     * top contributors, see <code>GetQueryResultsWorkloadInsightsTopContributors</code>.
     * </p>
     * <p>
     * Create a query ID for this call by calling the corresponding API call to start the query,
     * <code>StartQueryWorkloadInsightsTopContributorsData</code>. Use the scope ID that was returned for your account
     * by <code>CreateScope</code>.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     * <p>
     * The top contributor network flows overall are for a specific metric type, for example, the number of
     * retransmissions.
     * </p>
     *
     * @param getQueryResultsWorkloadInsightsTopContributorsDataRequest
     * @return A Java Future containing the result of the GetQueryResultsWorkloadInsightsTopContributorsData operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ResourceNotFoundException The request specifies a resource that doesn't exist.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.GetQueryResultsWorkloadInsightsTopContributorsData
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetQueryResultsWorkloadInsightsTopContributorsData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetQueryResultsWorkloadInsightsTopContributorsDataResponse> getQueryResultsWorkloadInsightsTopContributorsData(
            GetQueryResultsWorkloadInsightsTopContributorsDataRequest getQueryResultsWorkloadInsightsTopContributorsDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                getQueryResultsWorkloadInsightsTopContributorsDataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getQueryResultsWorkloadInsightsTopContributorsDataRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryResultsWorkloadInsightsTopContributorsData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueryResultsWorkloadInsightsTopContributorsDataResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetQueryResultsWorkloadInsightsTopContributorsDataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetQueryResultsWorkloadInsightsTopContributorsDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueryResultsWorkloadInsightsTopContributorsDataRequest, GetQueryResultsWorkloadInsightsTopContributorsDataResponse>()
                            .withOperationName("GetQueryResultsWorkloadInsightsTopContributorsData")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(
                                    new GetQueryResultsWorkloadInsightsTopContributorsDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getQueryResultsWorkloadInsightsTopContributorsDataRequest));
            CompletableFuture<GetQueryResultsWorkloadInsightsTopContributorsDataResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the current status of a query for the Network Flow Monitor query interface, for a specified query ID and
     * monitor. This call returns the query status for the top contributors for a monitor.
     * </p>
     * <p>
     * When you create a query, use this call to check the status of the query to make sure that it has has
     * <code>SUCCEEDED</code> before you review the results. Use the same query ID that you used for the corresponding
     * API call to start (create) the query, <code>StartQueryMonitorTopContributors</code>.
     * </p>
     * <p>
     * When you run a query, use this call to check the status of the query to make sure that the query has
     * <code>SUCCEEDED</code> before you review the results.
     * </p>
     *
     * @param getQueryStatusMonitorTopContributorsRequest
     * @return A Java Future containing the result of the GetQueryStatusMonitorTopContributors operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.GetQueryStatusMonitorTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetQueryStatusMonitorTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetQueryStatusMonitorTopContributorsResponse> getQueryStatusMonitorTopContributors(
            GetQueryStatusMonitorTopContributorsRequest getQueryStatusMonitorTopContributorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQueryStatusMonitorTopContributorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getQueryStatusMonitorTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryStatusMonitorTopContributors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueryStatusMonitorTopContributorsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetQueryStatusMonitorTopContributorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetQueryStatusMonitorTopContributorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueryStatusMonitorTopContributorsRequest, GetQueryStatusMonitorTopContributorsResponse>()
                            .withOperationName("GetQueryStatusMonitorTopContributors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetQueryStatusMonitorTopContributorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getQueryStatusMonitorTopContributorsRequest));
            CompletableFuture<GetQueryStatusMonitorTopContributorsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Return the data for a query with the Network Flow Monitor query interface. Specify the query that you want to
     * return results for by providing a query ID and a monitor name. This query returns the top contributors for
     * workload insights.
     * </p>
     * <p>
     * When you start a query, use this call to check the status of the query to make sure that it has has
     * <code>SUCCEEDED</code> before you review the results. Use the same query ID that you used for the corresponding
     * API call to start the query, <code>StartQueryWorkloadInsightsTopContributors</code>.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     *
     * @param getQueryStatusWorkloadInsightsTopContributorsRequest
     * @return A Java Future containing the result of the GetQueryStatusWorkloadInsightsTopContributors operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.GetQueryStatusWorkloadInsightsTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetQueryStatusWorkloadInsightsTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetQueryStatusWorkloadInsightsTopContributorsResponse> getQueryStatusWorkloadInsightsTopContributors(
            GetQueryStatusWorkloadInsightsTopContributorsRequest getQueryStatusWorkloadInsightsTopContributorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                getQueryStatusWorkloadInsightsTopContributorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getQueryStatusWorkloadInsightsTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryStatusWorkloadInsightsTopContributors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueryStatusWorkloadInsightsTopContributorsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetQueryStatusWorkloadInsightsTopContributorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetQueryStatusWorkloadInsightsTopContributorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueryStatusWorkloadInsightsTopContributorsRequest, GetQueryStatusWorkloadInsightsTopContributorsResponse>()
                            .withOperationName("GetQueryStatusWorkloadInsightsTopContributors")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetQueryStatusWorkloadInsightsTopContributorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getQueryStatusWorkloadInsightsTopContributorsRequest));
            CompletableFuture<GetQueryStatusWorkloadInsightsTopContributorsResponse> whenCompleted = executeFuture.whenComplete((
                    r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the current status of a query for the Network Flow Monitor query interface, for a specified query ID and
     * monitor. This call returns the query status for the top contributors data for workload insights.
     * </p>
     * <p>
     * When you start a query, use this call to check the status of the query to make sure that it has has
     * <code>SUCCEEDED</code> before you review the results. Use the same query ID that you used for the corresponding
     * API call to start the query, <code>StartQueryWorkloadInsightsTopContributorsData</code>.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     * <p>
     * The top contributor network flows overall are for a specific metric type, for example, the number of
     * retransmissions.
     * </p>
     *
     * @param getQueryStatusWorkloadInsightsTopContributorsDataRequest
     * @return A Java Future containing the result of the GetQueryStatusWorkloadInsightsTopContributorsData operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.GetQueryStatusWorkloadInsightsTopContributorsData
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetQueryStatusWorkloadInsightsTopContributorsData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetQueryStatusWorkloadInsightsTopContributorsDataResponse> getQueryStatusWorkloadInsightsTopContributorsData(
            GetQueryStatusWorkloadInsightsTopContributorsDataRequest getQueryStatusWorkloadInsightsTopContributorsDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                getQueryStatusWorkloadInsightsTopContributorsDataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getQueryStatusWorkloadInsightsTopContributorsDataRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryStatusWorkloadInsightsTopContributorsData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueryStatusWorkloadInsightsTopContributorsDataResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetQueryStatusWorkloadInsightsTopContributorsDataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetQueryStatusWorkloadInsightsTopContributorsDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueryStatusWorkloadInsightsTopContributorsDataRequest, GetQueryStatusWorkloadInsightsTopContributorsDataResponse>()
                            .withOperationName("GetQueryStatusWorkloadInsightsTopContributorsData")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(
                                    new GetQueryStatusWorkloadInsightsTopContributorsDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getQueryStatusWorkloadInsightsTopContributorsDataRequest));
            CompletableFuture<GetQueryStatusWorkloadInsightsTopContributorsDataResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a scope, including the name, status, tags, and target details. The scope in Network Flow
     * Monitor is an account.
     * </p>
     *
     * @param getScopeRequest
     * @return A Java Future containing the result of the GetScope operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ResourceNotFoundException The request specifies a resource that doesn't exist.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.GetScope
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetScope" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetScopeResponse> getScope(GetScopeRequest getScopeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getScopeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getScopeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetScope");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetScopeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetScopeResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetScopeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetScopeRequest, GetScopeResponse>().withOperationName("GetScope")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetScopeRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getScopeRequest));
            CompletableFuture<GetScopeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all monitors in an account. Optionally, you can list only monitors that have a specific status, by using the
     * <code>STATUS</code> parameter.
     * </p>
     *
     * @param listMonitorsRequest
     * @return A Java Future containing the result of the ListMonitors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.ListMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/ListMonitors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMonitorsResponse> listMonitors(ListMonitorsRequest listMonitorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMonitorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMonitorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMonitors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMonitorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListMonitorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListMonitorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMonitorsRequest, ListMonitorsResponse>()
                            .withOperationName("ListMonitors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMonitorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMonitorsRequest));
            CompletableFuture<ListMonitorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all the scopes for an account.
     * </p>
     *
     * @param listScopesRequest
     * @return A Java Future containing the result of the ListScopes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.ListScopes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/ListScopes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListScopesResponse> listScopes(ListScopesRequest listScopesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listScopesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listScopesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListScopes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListScopesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListScopesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListScopesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListScopesRequest, ListScopesResponse>().withOperationName("ListScopes")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListScopesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listScopesRequest));
            CompletableFuture<ListScopesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns all the tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request specifies a resource that doesn't exist.</li>
     *         <li>ConflictException The requested resource is in use.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a query that you can use with the Network Flow Monitor query interface to return the top contributors for
     * a monitor. Specify the monitor that you want to create the query for.
     * </p>
     * <p>
     * The call returns a query ID that you can use with <a href=
     * "https://docs.aws.amazon.com/networkflowmonitor/2.0/APIReference/API_GetQueryResultsMonitorTopContributors.html">
     * GetQueryResultsMonitorTopContributors</a> to run the query and return the top contributors for a specific
     * monitor.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable APIs for the top contributors that you want to be returned.
     * </p>
     *
     * @param startQueryMonitorTopContributorsRequest
     * @return A Java Future containing the result of the StartQueryMonitorTopContributors operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.StartQueryMonitorTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/StartQueryMonitorTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartQueryMonitorTopContributorsResponse> startQueryMonitorTopContributors(
            StartQueryMonitorTopContributorsRequest startQueryMonitorTopContributorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startQueryMonitorTopContributorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startQueryMonitorTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartQueryMonitorTopContributors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartQueryMonitorTopContributorsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, StartQueryMonitorTopContributorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartQueryMonitorTopContributorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartQueryMonitorTopContributorsRequest, StartQueryMonitorTopContributorsResponse>()
                            .withOperationName("StartQueryMonitorTopContributors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartQueryMonitorTopContributorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startQueryMonitorTopContributorsRequest));
            CompletableFuture<StartQueryMonitorTopContributorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a query with the Network Flow Monitor query interface that you can run to return workload insights top
     * contributors. Specify the scope that you want to create a query for.
     * </p>
     * <p>
     * The call returns a query ID that you can use with <a href=
     * "https://docs.aws.amazon.com/networkflowmonitor/2.0/APIReference/API_GetQueryResultsWorkloadInsightsTopContributors.html"
     * > GetQueryResultsWorkloadInsightsTopContributors</a> to run the query and return the top contributors for the
     * workload insights for a scope.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable APIs for the top contributors that you want to be returned.
     * </p>
     *
     * @param startQueryWorkloadInsightsTopContributorsRequest
     * @return A Java Future containing the result of the StartQueryWorkloadInsightsTopContributors operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.StartQueryWorkloadInsightsTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/StartQueryWorkloadInsightsTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartQueryWorkloadInsightsTopContributorsResponse> startQueryWorkloadInsightsTopContributors(
            StartQueryWorkloadInsightsTopContributorsRequest startQueryWorkloadInsightsTopContributorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                startQueryWorkloadInsightsTopContributorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startQueryWorkloadInsightsTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartQueryWorkloadInsightsTopContributors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartQueryWorkloadInsightsTopContributorsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, StartQueryWorkloadInsightsTopContributorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartQueryWorkloadInsightsTopContributorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartQueryWorkloadInsightsTopContributorsRequest, StartQueryWorkloadInsightsTopContributorsResponse>()
                            .withOperationName("StartQueryWorkloadInsightsTopContributors")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartQueryWorkloadInsightsTopContributorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startQueryWorkloadInsightsTopContributorsRequest));
            CompletableFuture<StartQueryWorkloadInsightsTopContributorsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a query with the Network Flow Monitor query interface that you can run to return data for workload
     * insights top contributors. Specify the scope that you want to create a query for.
     * </p>
     * <p>
     * The call returns a query ID that you can use with <a href=
     * "https://docs.aws.amazon.com/networkflowmonitor/2.0/APIReference/API_GetQueryResultsWorkloadInsightsTopContributorsData.html"
     * > GetQueryResultsWorkloadInsightsTopContributorsData</a> to run the query and return the data for the top
     * contributors for the workload insights for a scope.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     *
     * @param startQueryWorkloadInsightsTopContributorsDataRequest
     * @return A Java Future containing the result of the StartQueryWorkloadInsightsTopContributorsData operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.StartQueryWorkloadInsightsTopContributorsData
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/StartQueryWorkloadInsightsTopContributorsData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartQueryWorkloadInsightsTopContributorsDataResponse> startQueryWorkloadInsightsTopContributorsData(
            StartQueryWorkloadInsightsTopContributorsDataRequest startQueryWorkloadInsightsTopContributorsDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                startQueryWorkloadInsightsTopContributorsDataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startQueryWorkloadInsightsTopContributorsDataRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartQueryWorkloadInsightsTopContributorsData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartQueryWorkloadInsightsTopContributorsDataResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, StartQueryWorkloadInsightsTopContributorsDataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartQueryWorkloadInsightsTopContributorsDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartQueryWorkloadInsightsTopContributorsDataRequest, StartQueryWorkloadInsightsTopContributorsDataResponse>()
                            .withOperationName("StartQueryWorkloadInsightsTopContributorsData")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartQueryWorkloadInsightsTopContributorsDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startQueryWorkloadInsightsTopContributorsDataRequest));
            CompletableFuture<StartQueryWorkloadInsightsTopContributorsDataResponse> whenCompleted = executeFuture.whenComplete((
                    r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stop a top contributors query for a monitor. Specify the query that you want to stop by providing a query ID and
     * a monitor name.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     *
     * @param stopQueryMonitorTopContributorsRequest
     * @return A Java Future containing the result of the StopQueryMonitorTopContributors operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.StopQueryMonitorTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/StopQueryMonitorTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopQueryMonitorTopContributorsResponse> stopQueryMonitorTopContributors(
            StopQueryMonitorTopContributorsRequest stopQueryMonitorTopContributorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopQueryMonitorTopContributorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                stopQueryMonitorTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopQueryMonitorTopContributors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopQueryMonitorTopContributorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopQueryMonitorTopContributorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StopQueryMonitorTopContributorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopQueryMonitorTopContributorsRequest, StopQueryMonitorTopContributorsResponse>()
                            .withOperationName("StopQueryMonitorTopContributors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopQueryMonitorTopContributorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopQueryMonitorTopContributorsRequest));
            CompletableFuture<StopQueryMonitorTopContributorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stop a top contributors query for workload insights. Specify the query that you want to stop by providing a query
     * ID and a scope ID.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     *
     * @param stopQueryWorkloadInsightsTopContributorsRequest
     * @return A Java Future containing the result of the StopQueryWorkloadInsightsTopContributors operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.StopQueryWorkloadInsightsTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/StopQueryWorkloadInsightsTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopQueryWorkloadInsightsTopContributorsResponse> stopQueryWorkloadInsightsTopContributors(
            StopQueryWorkloadInsightsTopContributorsRequest stopQueryWorkloadInsightsTopContributorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                stopQueryWorkloadInsightsTopContributorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                stopQueryWorkloadInsightsTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopQueryWorkloadInsightsTopContributors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopQueryWorkloadInsightsTopContributorsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, StopQueryWorkloadInsightsTopContributorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StopQueryWorkloadInsightsTopContributorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopQueryWorkloadInsightsTopContributorsRequest, StopQueryWorkloadInsightsTopContributorsResponse>()
                            .withOperationName("StopQueryWorkloadInsightsTopContributors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopQueryWorkloadInsightsTopContributorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopQueryWorkloadInsightsTopContributorsRequest));
            CompletableFuture<StopQueryWorkloadInsightsTopContributorsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stop a top contributors data query for workload insights. Specify the query that you want to stop by providing a
     * query ID and a scope ID.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     *
     * @param stopQueryWorkloadInsightsTopContributorsDataRequest
     * @return A Java Future containing the result of the StopQueryWorkloadInsightsTopContributorsData operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.StopQueryWorkloadInsightsTopContributorsData
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/StopQueryWorkloadInsightsTopContributorsData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopQueryWorkloadInsightsTopContributorsDataResponse> stopQueryWorkloadInsightsTopContributorsData(
            StopQueryWorkloadInsightsTopContributorsDataRequest stopQueryWorkloadInsightsTopContributorsDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                stopQueryWorkloadInsightsTopContributorsDataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                stopQueryWorkloadInsightsTopContributorsDataRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopQueryWorkloadInsightsTopContributorsData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopQueryWorkloadInsightsTopContributorsDataResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, StopQueryWorkloadInsightsTopContributorsDataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StopQueryWorkloadInsightsTopContributorsDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopQueryWorkloadInsightsTopContributorsDataRequest, StopQueryWorkloadInsightsTopContributorsDataResponse>()
                            .withOperationName("StopQueryWorkloadInsightsTopContributorsData")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopQueryWorkloadInsightsTopContributorsDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopQueryWorkloadInsightsTopContributorsDataRequest));
            CompletableFuture<StopQueryWorkloadInsightsTopContributorsDataResponse> whenCompleted = executeFuture.whenComplete((
                    r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a tag to a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request specifies a resource that doesn't exist.</li>
     *         <li>ConflictException The requested resource is in use.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request specifies a resource that doesn't exist.</li>
     *         <li>ConflictException The requested resource is in use.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a monitor to add or remove local or remote resources.
     * </p>
     *
     * @param updateMonitorRequest
     * @return A Java Future containing the result of the UpdateMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request specifies a resource that doesn't exist.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.UpdateMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/UpdateMonitor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMonitorResponse> updateMonitor(UpdateMonitorRequest updateMonitorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateMonitorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMonitor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateMonitorResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateMonitorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMonitorRequest, UpdateMonitorResponse>()
                            .withOperationName("UpdateMonitor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateMonitorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateMonitorRequest));
            CompletableFuture<UpdateMonitorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a scope to add or remove resources that you want to be available for Network Flow Monitor to generate
     * metrics for, when you have active agents on those resources sending metrics reports to the Network Flow Monitor
     * backend.
     * </p>
     *
     * @param updateScopeRequest
     * @return A Java Future containing the result of the UpdateScope operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded a service quota.</li>
     *         <li>ResourceNotFoundException The request specifies a resource that doesn't exist.</li>
     *         <li>ConflictException The requested resource is in use.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NetworkFlowMonitorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample NetworkFlowMonitorAsyncClient.UpdateScope
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/UpdateScope"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateScopeResponse> updateScope(UpdateScopeRequest updateScopeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateScopeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateScopeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateScope");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateScopeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateScopeResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateScopeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateScopeRequest, UpdateScopeResponse>()
                            .withOperationName("UpdateScope").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateScopeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateScopeRequest));
            CompletableFuture<UpdateScopeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final NetworkFlowMonitorServiceClientConfiguration serviceClientConfiguration() {
        return new NetworkFlowMonitorServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(NetworkFlowMonitorException::builder).protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        NetworkFlowMonitorServiceClientConfigurationBuilder serviceConfigBuilder = new NetworkFlowMonitorServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
