/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkfirewall.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A single 5-tuple stateful rule, for use in a stateful rule group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StatefulRule implements SdkPojo, Serializable, ToCopyableBuilder<StatefulRule.Builder, StatefulRule> {
    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Action")
            .getter(getter(StatefulRule::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<Header> HEADER_FIELD = SdkField.<Header> builder(MarshallingType.SDK_POJO).memberName("Header")
            .getter(getter(StatefulRule::header)).setter(setter(Builder::header)).constructor(Header::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Header").build()).build();

    private static final SdkField<List<RuleOption>> RULE_OPTIONS_FIELD = SdkField
            .<List<RuleOption>> builder(MarshallingType.LIST)
            .memberName("RuleOptions")
            .getter(getter(StatefulRule::ruleOptions))
            .setter(setter(Builder::ruleOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RuleOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(RuleOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTION_FIELD, HEADER_FIELD,
            RULE_OPTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String action;

    private final Header header;

    private final List<RuleOption> ruleOptions;

    private StatefulRule(BuilderImpl builder) {
        this.action = builder.action;
        this.header = builder.header;
        this.ruleOptions = builder.ruleOptions;
    }

    /**
     * <p>
     * Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the stateful
     * rule criteria. For all actions, Network Firewall performs the specified action and discontinues stateful
     * inspection of the traffic flow.
     * </p>
     * <p>
     * The actions for a stateful rule are defined as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>PASS</b> - Permits the packets to go to the intended destination.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log message, if alert
     * logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ALERT</b> - Permits the packets to go to the intended destination and sends an alert log message, if alert
     * logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
     * </p>
     * <p>
     * You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule with
     * <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change the action to
     * <code>DROP</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link StatefulAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the
     *         stateful rule criteria. For all actions, Network Firewall performs the specified action and discontinues
     *         stateful inspection of the traffic flow. </p>
     *         <p>
     *         The actions for a stateful rule are defined as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>PASS</b> - Permits the packets to go to the intended destination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log message,
     *         if alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ALERT</b> - Permits the packets to go to the intended destination and sends an alert log message, if
     *         alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
     *         </p>
     *         <p>
     *         You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule
     *         with <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change
     *         the action to <code>DROP</code>.
     *         </p>
     *         </li>
     * @see StatefulAction
     */
    public final StatefulAction action() {
        return StatefulAction.fromValue(action);
    }

    /**
     * <p>
     * Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the stateful
     * rule criteria. For all actions, Network Firewall performs the specified action and discontinues stateful
     * inspection of the traffic flow.
     * </p>
     * <p>
     * The actions for a stateful rule are defined as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>PASS</b> - Permits the packets to go to the intended destination.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log message, if alert
     * logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ALERT</b> - Permits the packets to go to the intended destination and sends an alert log message, if alert
     * logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
     * </p>
     * <p>
     * You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule with
     * <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change the action to
     * <code>DROP</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link StatefulAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the
     *         stateful rule criteria. For all actions, Network Firewall performs the specified action and discontinues
     *         stateful inspection of the traffic flow. </p>
     *         <p>
     *         The actions for a stateful rule are defined as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>PASS</b> - Permits the packets to go to the intended destination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log message,
     *         if alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ALERT</b> - Permits the packets to go to the intended destination and sends an alert log message, if
     *         alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
     *         </p>
     *         <p>
     *         You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule
     *         with <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change
     *         the action to <code>DROP</code>.
     *         </p>
     *         </li>
     * @see StatefulAction
     */
    public final String actionAsString() {
        return action;
    }

    /**
     * <p>
     * The stateful 5-tuple inspection criteria for this rule, used to inspect traffic flows.
     * </p>
     * 
     * @return The stateful 5-tuple inspection criteria for this rule, used to inspect traffic flows.
     */
    public final Header header() {
        return header;
    }

    /**
     * Returns true if the RuleOptions property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasRuleOptions() {
        return ruleOptions != null && !(ruleOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p/>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRuleOptions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return
     */
    public final List<RuleOption> ruleOptions() {
        return ruleOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(header());
        hashCode = 31 * hashCode + Objects.hashCode(hasRuleOptions() ? ruleOptions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StatefulRule)) {
            return false;
        }
        StatefulRule other = (StatefulRule) obj;
        return Objects.equals(actionAsString(), other.actionAsString()) && Objects.equals(header(), other.header())
                && hasRuleOptions() == other.hasRuleOptions() && Objects.equals(ruleOptions(), other.ruleOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StatefulRule").add("Action", actionAsString()).add("Header", header())
                .add("RuleOptions", hasRuleOptions() ? ruleOptions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        case "Header":
            return Optional.ofNullable(clazz.cast(header()));
        case "RuleOptions":
            return Optional.ofNullable(clazz.cast(ruleOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StatefulRule, T> g) {
        return obj -> g.apply((StatefulRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StatefulRule> {
        /**
         * <p>
         * Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the stateful
         * rule criteria. For all actions, Network Firewall performs the specified action and discontinues stateful
         * inspection of the traffic flow.
         * </p>
         * <p>
         * The actions for a stateful rule are defined as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>PASS</b> - Permits the packets to go to the intended destination.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log message, if
         * alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ALERT</b> - Permits the packets to go to the intended destination and sends an alert log message, if alert
         * logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
         * </p>
         * <p>
         * You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule with
         * <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change the action
         * to <code>DROP</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the
         *        stateful rule criteria. For all actions, Network Firewall performs the specified action and
         *        discontinues stateful inspection of the traffic flow. </p>
         *        <p>
         *        The actions for a stateful rule are defined as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>PASS</b> - Permits the packets to go to the intended destination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log
         *        message, if alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ALERT</b> - Permits the packets to go to the intended destination and sends an alert log message,
         *        if alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
         *        </p>
         *        <p>
         *        You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule
         *        with <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change
         *        the action to <code>DROP</code>.
         *        </p>
         *        </li>
         * @see StatefulAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatefulAction
         */
        Builder action(String action);

        /**
         * <p>
         * Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the stateful
         * rule criteria. For all actions, Network Firewall performs the specified action and discontinues stateful
         * inspection of the traffic flow.
         * </p>
         * <p>
         * The actions for a stateful rule are defined as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>PASS</b> - Permits the packets to go to the intended destination.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log message, if
         * alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ALERT</b> - Permits the packets to go to the intended destination and sends an alert log message, if alert
         * logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
         * </p>
         * <p>
         * You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule with
         * <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change the action
         * to <code>DROP</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the
         *        stateful rule criteria. For all actions, Network Firewall performs the specified action and
         *        discontinues stateful inspection of the traffic flow. </p>
         *        <p>
         *        The actions for a stateful rule are defined as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>PASS</b> - Permits the packets to go to the intended destination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log
         *        message, if alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ALERT</b> - Permits the packets to go to the intended destination and sends an alert log message,
         *        if alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
         *        </p>
         *        <p>
         *        You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule
         *        with <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change
         *        the action to <code>DROP</code>.
         *        </p>
         *        </li>
         * @see StatefulAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatefulAction
         */
        Builder action(StatefulAction action);

        /**
         * <p>
         * The stateful 5-tuple inspection criteria for this rule, used to inspect traffic flows.
         * </p>
         * 
         * @param header
         *        The stateful 5-tuple inspection criteria for this rule, used to inspect traffic flows.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder header(Header header);

        /**
         * <p>
         * The stateful 5-tuple inspection criteria for this rule, used to inspect traffic flows.
         * </p>
         * This is a convenience that creates an instance of the {@link Header.Builder} avoiding the need to create one
         * manually via {@link Header#builder()}.
         *
         * When the {@link Consumer} completes, {@link Header.Builder#build()} is called immediately and its result is
         * passed to {@link #header(Header)}.
         * 
         * @param header
         *        a consumer that will call methods on {@link Header.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #header(Header)
         */
        default Builder header(Consumer<Header.Builder> header) {
            return header(Header.builder().applyMutation(header).build());
        }

        /**
         * <p/>
         * 
         * @param ruleOptions
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleOptions(Collection<RuleOption> ruleOptions);

        /**
         * <p/>
         * 
         * @param ruleOptions
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleOptions(RuleOption... ruleOptions);

        /**
         * <p/>
         * This is a convenience that creates an instance of the {@link List<RuleOption>.Builder} avoiding the need to
         * create one manually via {@link List<RuleOption>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<RuleOption>.Builder#build()} is called immediately and its
         * result is passed to {@link #ruleOptions(List<RuleOption>)}.
         * 
         * @param ruleOptions
         *        a consumer that will call methods on {@link List<RuleOption>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ruleOptions(List<RuleOption>)
         */
        Builder ruleOptions(Consumer<RuleOption.Builder>... ruleOptions);
    }

    static final class BuilderImpl implements Builder {
        private String action;

        private Header header;

        private List<RuleOption> ruleOptions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(StatefulRule model) {
            action(model.action);
            header(model.header);
            ruleOptions(model.ruleOptions);
        }

        public final String getAction() {
            return action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(StatefulAction action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        public final Header.Builder getHeader() {
            return header != null ? header.toBuilder() : null;
        }

        @Override
        public final Builder header(Header header) {
            this.header = header;
            return this;
        }

        public final void setHeader(Header.BuilderImpl header) {
            this.header = header != null ? header.build() : null;
        }

        public final Collection<RuleOption.Builder> getRuleOptions() {
            if (ruleOptions instanceof SdkAutoConstructList) {
                return null;
            }
            return ruleOptions != null ? ruleOptions.stream().map(RuleOption::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder ruleOptions(Collection<RuleOption> ruleOptions) {
            this.ruleOptions = RuleOptionsCopier.copy(ruleOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ruleOptions(RuleOption... ruleOptions) {
            ruleOptions(Arrays.asList(ruleOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ruleOptions(Consumer<RuleOption.Builder>... ruleOptions) {
            ruleOptions(Stream.of(ruleOptions).map(c -> RuleOption.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRuleOptions(Collection<RuleOption.BuilderImpl> ruleOptions) {
            this.ruleOptions = RuleOptionsCopier.copyFromBuilder(ruleOptions);
        }

        @Override
        public StatefulRule build() {
            return new StatefulRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
