/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkfirewall.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The inspection criteria and action for a single stateless rule. AWS Network Firewall inspects each packet for the
 * specified matching criteria. When a packet matches the criteria, Network Firewall performs the rule's actions on the
 * packet.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RuleDefinition implements SdkPojo, Serializable, ToCopyableBuilder<RuleDefinition.Builder, RuleDefinition> {
    private static final SdkField<MatchAttributes> MATCH_ATTRIBUTES_FIELD = SdkField
            .<MatchAttributes> builder(MarshallingType.SDK_POJO).memberName("MatchAttributes")
            .getter(getter(RuleDefinition::matchAttributes)).setter(setter(Builder::matchAttributes))
            .constructor(MatchAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MatchAttributes").build()).build();

    private static final SdkField<List<String>> ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Actions")
            .getter(getter(RuleDefinition::actions))
            .setter(setter(Builder::actions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Actions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MATCH_ATTRIBUTES_FIELD,
            ACTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final MatchAttributes matchAttributes;

    private final List<String> actions;

    private RuleDefinition(BuilderImpl builder) {
        this.matchAttributes = builder.matchAttributes;
        this.actions = builder.actions;
    }

    /**
     * <p>
     * Criteria for Network Firewall to use to inspect an individual packet in stateless rule inspection. Each match
     * attributes set can include one or more items such as IP address, CIDR range, port number, protocol, and TCP
     * flags.
     * </p>
     * 
     * @return Criteria for Network Firewall to use to inspect an individual packet in stateless rule inspection. Each
     *         match attributes set can include one or more items such as IP address, CIDR range, port number, protocol,
     *         and TCP flags.
     */
    public final MatchAttributes matchAttributes() {
        return matchAttributes;
    }

    /**
     * Returns true if the Actions property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasActions() {
        return actions != null && !(actions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The actions to take on a packet that matches one of the stateless rule definition's match attributes. You must
     * specify a standard action and you can add custom actions.
     * </p>
     * <note>
     * <p>
     * Network Firewall only forwards a packet for stateful rule inspection if you specify
     * <code>aws:forward_to_sfe</code> for a rule that the packet matches, or if the packet doesn't match any stateless
     * rule and you specify <code>aws:forward_to_sfe</code> for the <code>StatelessDefaultActions</code> setting for the
     * <a>FirewallPolicy</a>.
     * </p>
     * </note>
     * <p>
     * For every rule, you must specify exactly one of the following standard actions.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>aws:pass</b> - Discontinues all inspection of the packet and permits it to go to its intended destination.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>aws:drop</b> - Discontinues all inspection of the packet and blocks it from going to its intended destination.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>aws:forward_to_sfe</b> - Discontinues stateless inspection of the packet and forwards it to the stateful rule
     * engine for inspection.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Additionally, you can specify a custom action. To do this, you define a custom action by name and type, then
     * provide the name you've assigned to the action in this <code>Actions</code> setting. For information about the
     * options, see <a>CustomAction</a>.
     * </p>
     * <p>
     * To provide more than one action in this setting, separate the settings with a comma. For example, if you have a
     * custom <code>PublishMetrics</code> action that you've named <code>MyMetricsAction</code>, then you could specify
     * the standard action <code>aws:pass</code> and the custom action with <code>[“aws:pass”, “MyMetricsAction”]</code>
     * .
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasActions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The actions to take on a packet that matches one of the stateless rule definition's match attributes. You
     *         must specify a standard action and you can add custom actions. </p> <note>
     *         <p>
     *         Network Firewall only forwards a packet for stateful rule inspection if you specify
     *         <code>aws:forward_to_sfe</code> for a rule that the packet matches, or if the packet doesn't match any
     *         stateless rule and you specify <code>aws:forward_to_sfe</code> for the
     *         <code>StatelessDefaultActions</code> setting for the <a>FirewallPolicy</a>.
     *         </p>
     *         </note>
     *         <p>
     *         For every rule, you must specify exactly one of the following standard actions.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>aws:pass</b> - Discontinues all inspection of the packet and permits it to go to its intended
     *         destination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>aws:drop</b> - Discontinues all inspection of the packet and blocks it from going to its intended
     *         destination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>aws:forward_to_sfe</b> - Discontinues stateless inspection of the packet and forwards it to the
     *         stateful rule engine for inspection.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Additionally, you can specify a custom action. To do this, you define a custom action by name and type,
     *         then provide the name you've assigned to the action in this <code>Actions</code> setting. For information
     *         about the options, see <a>CustomAction</a>.
     *         </p>
     *         <p>
     *         To provide more than one action in this setting, separate the settings with a comma. For example, if you
     *         have a custom <code>PublishMetrics</code> action that you've named <code>MyMetricsAction</code>, then you
     *         could specify the standard action <code>aws:pass</code> and the custom action with
     *         <code>[“aws:pass”, “MyMetricsAction”]</code>.
     */
    public final List<String> actions() {
        return actions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(matchAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(hasActions() ? actions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RuleDefinition)) {
            return false;
        }
        RuleDefinition other = (RuleDefinition) obj;
        return Objects.equals(matchAttributes(), other.matchAttributes()) && hasActions() == other.hasActions()
                && Objects.equals(actions(), other.actions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RuleDefinition").add("MatchAttributes", matchAttributes())
                .add("Actions", hasActions() ? actions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MatchAttributes":
            return Optional.ofNullable(clazz.cast(matchAttributes()));
        case "Actions":
            return Optional.ofNullable(clazz.cast(actions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RuleDefinition, T> g) {
        return obj -> g.apply((RuleDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RuleDefinition> {
        /**
         * <p>
         * Criteria for Network Firewall to use to inspect an individual packet in stateless rule inspection. Each match
         * attributes set can include one or more items such as IP address, CIDR range, port number, protocol, and TCP
         * flags.
         * </p>
         * 
         * @param matchAttributes
         *        Criteria for Network Firewall to use to inspect an individual packet in stateless rule inspection.
         *        Each match attributes set can include one or more items such as IP address, CIDR range, port number,
         *        protocol, and TCP flags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchAttributes(MatchAttributes matchAttributes);

        /**
         * <p>
         * Criteria for Network Firewall to use to inspect an individual packet in stateless rule inspection. Each match
         * attributes set can include one or more items such as IP address, CIDR range, port number, protocol, and TCP
         * flags.
         * </p>
         * This is a convenience that creates an instance of the {@link MatchAttributes.Builder} avoiding the need to
         * create one manually via {@link MatchAttributes#builder()}.
         *
         * When the {@link Consumer} completes, {@link MatchAttributes.Builder#build()} is called immediately and its
         * result is passed to {@link #matchAttributes(MatchAttributes)}.
         * 
         * @param matchAttributes
         *        a consumer that will call methods on {@link MatchAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #matchAttributes(MatchAttributes)
         */
        default Builder matchAttributes(Consumer<MatchAttributes.Builder> matchAttributes) {
            return matchAttributes(MatchAttributes.builder().applyMutation(matchAttributes).build());
        }

        /**
         * <p>
         * The actions to take on a packet that matches one of the stateless rule definition's match attributes. You
         * must specify a standard action and you can add custom actions.
         * </p>
         * <note>
         * <p>
         * Network Firewall only forwards a packet for stateful rule inspection if you specify
         * <code>aws:forward_to_sfe</code> for a rule that the packet matches, or if the packet doesn't match any
         * stateless rule and you specify <code>aws:forward_to_sfe</code> for the <code>StatelessDefaultActions</code>
         * setting for the <a>FirewallPolicy</a>.
         * </p>
         * </note>
         * <p>
         * For every rule, you must specify exactly one of the following standard actions.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>aws:pass</b> - Discontinues all inspection of the packet and permits it to go to its intended destination.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>aws:drop</b> - Discontinues all inspection of the packet and blocks it from going to its intended
         * destination.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>aws:forward_to_sfe</b> - Discontinues stateless inspection of the packet and forwards it to the stateful
         * rule engine for inspection.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Additionally, you can specify a custom action. To do this, you define a custom action by name and type, then
         * provide the name you've assigned to the action in this <code>Actions</code> setting. For information about
         * the options, see <a>CustomAction</a>.
         * </p>
         * <p>
         * To provide more than one action in this setting, separate the settings with a comma. For example, if you have
         * a custom <code>PublishMetrics</code> action that you've named <code>MyMetricsAction</code>, then you could
         * specify the standard action <code>aws:pass</code> and the custom action with
         * <code>[“aws:pass”, “MyMetricsAction”]</code>.
         * </p>
         * 
         * @param actions
         *        The actions to take on a packet that matches one of the stateless rule definition's match attributes.
         *        You must specify a standard action and you can add custom actions. </p> <note>
         *        <p>
         *        Network Firewall only forwards a packet for stateful rule inspection if you specify
         *        <code>aws:forward_to_sfe</code> for a rule that the packet matches, or if the packet doesn't match any
         *        stateless rule and you specify <code>aws:forward_to_sfe</code> for the
         *        <code>StatelessDefaultActions</code> setting for the <a>FirewallPolicy</a>.
         *        </p>
         *        </note>
         *        <p>
         *        For every rule, you must specify exactly one of the following standard actions.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>aws:pass</b> - Discontinues all inspection of the packet and permits it to go to its intended
         *        destination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>aws:drop</b> - Discontinues all inspection of the packet and blocks it from going to its intended
         *        destination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>aws:forward_to_sfe</b> - Discontinues stateless inspection of the packet and forwards it to the
         *        stateful rule engine for inspection.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Additionally, you can specify a custom action. To do this, you define a custom action by name and
         *        type, then provide the name you've assigned to the action in this <code>Actions</code> setting. For
         *        information about the options, see <a>CustomAction</a>.
         *        </p>
         *        <p>
         *        To provide more than one action in this setting, separate the settings with a comma. For example, if
         *        you have a custom <code>PublishMetrics</code> action that you've named <code>MyMetricsAction</code>,
         *        then you could specify the standard action <code>aws:pass</code> and the custom action with
         *        <code>[“aws:pass”, “MyMetricsAction”]</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(Collection<String> actions);

        /**
         * <p>
         * The actions to take on a packet that matches one of the stateless rule definition's match attributes. You
         * must specify a standard action and you can add custom actions.
         * </p>
         * <note>
         * <p>
         * Network Firewall only forwards a packet for stateful rule inspection if you specify
         * <code>aws:forward_to_sfe</code> for a rule that the packet matches, or if the packet doesn't match any
         * stateless rule and you specify <code>aws:forward_to_sfe</code> for the <code>StatelessDefaultActions</code>
         * setting for the <a>FirewallPolicy</a>.
         * </p>
         * </note>
         * <p>
         * For every rule, you must specify exactly one of the following standard actions.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>aws:pass</b> - Discontinues all inspection of the packet and permits it to go to its intended destination.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>aws:drop</b> - Discontinues all inspection of the packet and blocks it from going to its intended
         * destination.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>aws:forward_to_sfe</b> - Discontinues stateless inspection of the packet and forwards it to the stateful
         * rule engine for inspection.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Additionally, you can specify a custom action. To do this, you define a custom action by name and type, then
         * provide the name you've assigned to the action in this <code>Actions</code> setting. For information about
         * the options, see <a>CustomAction</a>.
         * </p>
         * <p>
         * To provide more than one action in this setting, separate the settings with a comma. For example, if you have
         * a custom <code>PublishMetrics</code> action that you've named <code>MyMetricsAction</code>, then you could
         * specify the standard action <code>aws:pass</code> and the custom action with
         * <code>[“aws:pass”, “MyMetricsAction”]</code>.
         * </p>
         * 
         * @param actions
         *        The actions to take on a packet that matches one of the stateless rule definition's match attributes.
         *        You must specify a standard action and you can add custom actions. </p> <note>
         *        <p>
         *        Network Firewall only forwards a packet for stateful rule inspection if you specify
         *        <code>aws:forward_to_sfe</code> for a rule that the packet matches, or if the packet doesn't match any
         *        stateless rule and you specify <code>aws:forward_to_sfe</code> for the
         *        <code>StatelessDefaultActions</code> setting for the <a>FirewallPolicy</a>.
         *        </p>
         *        </note>
         *        <p>
         *        For every rule, you must specify exactly one of the following standard actions.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>aws:pass</b> - Discontinues all inspection of the packet and permits it to go to its intended
         *        destination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>aws:drop</b> - Discontinues all inspection of the packet and blocks it from going to its intended
         *        destination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>aws:forward_to_sfe</b> - Discontinues stateless inspection of the packet and forwards it to the
         *        stateful rule engine for inspection.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Additionally, you can specify a custom action. To do this, you define a custom action by name and
         *        type, then provide the name you've assigned to the action in this <code>Actions</code> setting. For
         *        information about the options, see <a>CustomAction</a>.
         *        </p>
         *        <p>
         *        To provide more than one action in this setting, separate the settings with a comma. For example, if
         *        you have a custom <code>PublishMetrics</code> action that you've named <code>MyMetricsAction</code>,
         *        then you could specify the standard action <code>aws:pass</code> and the custom action with
         *        <code>[“aws:pass”, “MyMetricsAction”]</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(String... actions);
    }

    static final class BuilderImpl implements Builder {
        private MatchAttributes matchAttributes;

        private List<String> actions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RuleDefinition model) {
            matchAttributes(model.matchAttributes);
            actions(model.actions);
        }

        public final MatchAttributes.Builder getMatchAttributes() {
            return matchAttributes != null ? matchAttributes.toBuilder() : null;
        }

        @Override
        public final Builder matchAttributes(MatchAttributes matchAttributes) {
            this.matchAttributes = matchAttributes;
            return this;
        }

        public final void setMatchAttributes(MatchAttributes.BuilderImpl matchAttributes) {
            this.matchAttributes = matchAttributes != null ? matchAttributes.build() : null;
        }

        public final Collection<String> getActions() {
            if (actions instanceof SdkAutoConstructList) {
                return null;
            }
            return actions;
        }

        @Override
        public final Builder actions(Collection<String> actions) {
            this.actions = StatelessActionsCopier.copy(actions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(String... actions) {
            actions(Arrays.asList(actions));
            return this;
        }

        public final void setActions(Collection<String> actions) {
            this.actions = StatelessActionsCopier.copy(actions);
        }

        @Override
        public RuleDefinition build() {
            return new RuleDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
