/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mq;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mq.model.BadRequestException;
import software.amazon.awssdk.services.mq.model.ConflictException;
import software.amazon.awssdk.services.mq.model.CreateBrokerRequest;
import software.amazon.awssdk.services.mq.model.CreateBrokerResponse;
import software.amazon.awssdk.services.mq.model.CreateConfigurationRequest;
import software.amazon.awssdk.services.mq.model.CreateConfigurationResponse;
import software.amazon.awssdk.services.mq.model.CreateTagsRequest;
import software.amazon.awssdk.services.mq.model.CreateTagsResponse;
import software.amazon.awssdk.services.mq.model.CreateUserRequest;
import software.amazon.awssdk.services.mq.model.CreateUserResponse;
import software.amazon.awssdk.services.mq.model.DeleteBrokerRequest;
import software.amazon.awssdk.services.mq.model.DeleteBrokerResponse;
import software.amazon.awssdk.services.mq.model.DeleteTagsRequest;
import software.amazon.awssdk.services.mq.model.DeleteTagsResponse;
import software.amazon.awssdk.services.mq.model.DeleteUserRequest;
import software.amazon.awssdk.services.mq.model.DeleteUserResponse;
import software.amazon.awssdk.services.mq.model.DescribeBrokerEngineTypesRequest;
import software.amazon.awssdk.services.mq.model.DescribeBrokerEngineTypesResponse;
import software.amazon.awssdk.services.mq.model.DescribeBrokerInstanceOptionsRequest;
import software.amazon.awssdk.services.mq.model.DescribeBrokerInstanceOptionsResponse;
import software.amazon.awssdk.services.mq.model.DescribeBrokerRequest;
import software.amazon.awssdk.services.mq.model.DescribeBrokerResponse;
import software.amazon.awssdk.services.mq.model.DescribeConfigurationRequest;
import software.amazon.awssdk.services.mq.model.DescribeConfigurationResponse;
import software.amazon.awssdk.services.mq.model.DescribeConfigurationRevisionRequest;
import software.amazon.awssdk.services.mq.model.DescribeConfigurationRevisionResponse;
import software.amazon.awssdk.services.mq.model.DescribeUserRequest;
import software.amazon.awssdk.services.mq.model.DescribeUserResponse;
import software.amazon.awssdk.services.mq.model.ForbiddenException;
import software.amazon.awssdk.services.mq.model.InternalServerErrorException;
import software.amazon.awssdk.services.mq.model.ListBrokersRequest;
import software.amazon.awssdk.services.mq.model.ListBrokersResponse;
import software.amazon.awssdk.services.mq.model.ListConfigurationRevisionsRequest;
import software.amazon.awssdk.services.mq.model.ListConfigurationRevisionsResponse;
import software.amazon.awssdk.services.mq.model.ListConfigurationsRequest;
import software.amazon.awssdk.services.mq.model.ListConfigurationsResponse;
import software.amazon.awssdk.services.mq.model.ListTagsRequest;
import software.amazon.awssdk.services.mq.model.ListTagsResponse;
import software.amazon.awssdk.services.mq.model.ListUsersRequest;
import software.amazon.awssdk.services.mq.model.ListUsersResponse;
import software.amazon.awssdk.services.mq.model.MqException;
import software.amazon.awssdk.services.mq.model.MqRequest;
import software.amazon.awssdk.services.mq.model.NotFoundException;
import software.amazon.awssdk.services.mq.model.RebootBrokerRequest;
import software.amazon.awssdk.services.mq.model.RebootBrokerResponse;
import software.amazon.awssdk.services.mq.model.UnauthorizedException;
import software.amazon.awssdk.services.mq.model.UpdateBrokerRequest;
import software.amazon.awssdk.services.mq.model.UpdateBrokerResponse;
import software.amazon.awssdk.services.mq.model.UpdateConfigurationRequest;
import software.amazon.awssdk.services.mq.model.UpdateConfigurationResponse;
import software.amazon.awssdk.services.mq.model.UpdateUserRequest;
import software.amazon.awssdk.services.mq.model.UpdateUserResponse;
import software.amazon.awssdk.services.mq.paginators.ListBrokersIterable;
import software.amazon.awssdk.services.mq.transform.CreateBrokerRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.CreateConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.CreateTagsRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.CreateUserRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DeleteBrokerRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DeleteTagsRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DeleteUserRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DescribeBrokerEngineTypesRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DescribeBrokerInstanceOptionsRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DescribeBrokerRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DescribeConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DescribeConfigurationRevisionRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DescribeUserRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.ListBrokersRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.ListConfigurationRevisionsRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.ListConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.ListTagsRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.ListUsersRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.RebootBrokerRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.UpdateBrokerRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.UpdateConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.UpdateUserRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MqClient}.
 *
 * @see MqClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMqClient implements MqClient {
    private static final Logger log = Logger.loggerFor(DefaultMqClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMqClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Creates a broker. Note: This API is asynchronous.
     *
     * @param createBrokerRequest
     *        Creates a broker using the specified properties.
     * @return Result of the CreateBroker operation returned by the service.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ConflictException
     *         HTTP Status Code 409: Conflict. This broker name already exists. Retry your request with another name.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.CreateBroker
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/CreateBroker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateBrokerResponse createBroker(CreateBrokerRequest createBrokerRequest) throws BadRequestException,
            UnauthorizedException, InternalServerErrorException, ConflictException, ForbiddenException, AwsServiceException,
            SdkClientException, MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBrokerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateBrokerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBrokerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBroker");

            return clientHandler.execute(new ClientExecutionParams<CreateBrokerRequest, CreateBrokerResponse>()
                    .withOperationName("CreateBroker").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createBrokerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBrokerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Creates a new configuration for the specified configuration name. Amazon MQ uses the default configuration (the
     * engine type and version).
     *
     * @param createConfigurationRequest
     *        Creates a new configuration for the specified configuration name. Amazon MQ uses the default configuration
     *        (the engine type and version).
     * @return Result of the CreateConfiguration operation returned by the service.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ConflictException
     *         HTTP Status Code 409: Conflict. This configuration name already exists. Retry your request with another
     *         configuration name.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your input and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.CreateConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/CreateConfiguration" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateConfigurationResponse createConfiguration(CreateConfigurationRequest createConfigurationRequest)
            throws BadRequestException, InternalServerErrorException, ConflictException, ForbiddenException, AwsServiceException,
            SdkClientException, MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfiguration");

            return clientHandler.execute(new ClientExecutionParams<CreateConfigurationRequest, CreateConfigurationResponse>()
                    .withOperationName("CreateConfiguration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createConfigurationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Add a tag to a resource.
     *
     * @param createTagsRequest
     *        A map of the key-value pairs for the resource tag.
     * @return Result of the CreateTags operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.CreateTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/CreateTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateTagsResponse createTags(CreateTagsRequest createTagsRequest) throws NotFoundException, BadRequestException,
            InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException, MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTags");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateTagsRequest, CreateTagsResponse>().withOperationName("CreateTags")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createTagsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Creates an ActiveMQ user.
     *
     * @param createUserRequest
     *        Creates a new ActiveMQ user.
     * @return Result of the CreateUser operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ConflictException
     *         HTTP Status Code 409: Conflict. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.CreateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateUserResponse createUser(CreateUserRequest createUserRequest) throws NotFoundException, BadRequestException,
            InternalServerErrorException, ConflictException, ForbiddenException, AwsServiceException, SdkClientException,
            MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateUser");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateUserRequest, CreateUserResponse>().withOperationName("CreateUser")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createUserRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Deletes a broker. Note: This API is asynchronous.
     *
     * @param deleteBrokerRequest
     * @return Result of the DeleteBroker operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.DeleteBroker
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DeleteBroker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteBrokerResponse deleteBroker(DeleteBrokerRequest deleteBrokerRequest) throws NotFoundException,
            BadRequestException, InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException,
            MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBrokerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBrokerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBrokerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBroker");

            return clientHandler.execute(new ClientExecutionParams<DeleteBrokerRequest, DeleteBrokerResponse>()
                    .withOperationName("DeleteBroker").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteBrokerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBrokerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Removes a tag from a resource.
     *
     * @param deleteTagsRequest
     * @return Result of the DeleteTags operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.DeleteTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DeleteTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteTagsResponse deleteTags(DeleteTagsRequest deleteTagsRequest) throws NotFoundException, BadRequestException,
            InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException, MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTags");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteTagsRequest, DeleteTagsResponse>().withOperationName("DeleteTags")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteTagsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Deletes an ActiveMQ user.
     *
     * @param deleteUserRequest
     * @return Result of the DeleteUser operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.DeleteUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteUserResponse deleteUser(DeleteUserRequest deleteUserRequest) throws NotFoundException, BadRequestException,
            InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException, MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteUser");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteUserRequest, DeleteUserResponse>().withOperationName("DeleteUser")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteUserRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns information about the specified broker.
     *
     * @param describeBrokerRequest
     * @return Result of the DescribeBroker operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.DescribeBroker
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DescribeBroker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeBrokerResponse describeBroker(DescribeBrokerRequest describeBrokerRequest) throws NotFoundException,
            BadRequestException, InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException,
            MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBrokerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeBrokerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBrokerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBroker");

            return clientHandler.execute(new ClientExecutionParams<DescribeBrokerRequest, DescribeBrokerResponse>()
                    .withOperationName("DescribeBroker").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeBrokerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeBrokerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Describe available engine types and versions.
     *
     * @param describeBrokerEngineTypesRequest
     * @return Result of the DescribeBrokerEngineTypes operation returned by the service.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.DescribeBrokerEngineTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DescribeBrokerEngineTypes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeBrokerEngineTypesResponse describeBrokerEngineTypes(
            DescribeBrokerEngineTypesRequest describeBrokerEngineTypesRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException, MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBrokerEngineTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeBrokerEngineTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBrokerEngineTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBrokerEngineTypes");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeBrokerEngineTypesRequest, DescribeBrokerEngineTypesResponse>()
                            .withOperationName("DescribeBrokerEngineTypes").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeBrokerEngineTypesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeBrokerEngineTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Describe available broker instance options.
     *
     * @param describeBrokerInstanceOptionsRequest
     * @return Result of the DescribeBrokerInstanceOptions operation returned by the service.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.DescribeBrokerInstanceOptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DescribeBrokerInstanceOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeBrokerInstanceOptionsResponse describeBrokerInstanceOptions(
            DescribeBrokerInstanceOptionsRequest describeBrokerInstanceOptionsRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException, MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBrokerInstanceOptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeBrokerInstanceOptionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeBrokerInstanceOptionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBrokerInstanceOptions");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeBrokerInstanceOptionsRequest, DescribeBrokerInstanceOptionsResponse>()
                            .withOperationName("DescribeBrokerInstanceOptions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeBrokerInstanceOptionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeBrokerInstanceOptionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns information about the specified configuration.
     *
     * @param describeConfigurationRequest
     * @return Result of the DescribeConfiguration operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.DescribeConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DescribeConfiguration" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeConfigurationResponse describeConfiguration(DescribeConfigurationRequest describeConfigurationRequest)
            throws NotFoundException, BadRequestException, InternalServerErrorException, ForbiddenException, AwsServiceException,
            SdkClientException, MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConfiguration");

            return clientHandler.execute(new ClientExecutionParams<DescribeConfigurationRequest, DescribeConfigurationResponse>()
                    .withOperationName("DescribeConfiguration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeConfigurationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns the specified configuration revision for the specified configuration.
     *
     * @param describeConfigurationRevisionRequest
     * @return Result of the DescribeConfigurationRevision operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.DescribeConfigurationRevision
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DescribeConfigurationRevision"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeConfigurationRevisionResponse describeConfigurationRevision(
            DescribeConfigurationRevisionRequest describeConfigurationRevisionRequest) throws NotFoundException,
            BadRequestException, InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException,
            MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeConfigurationRevisionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeConfigurationRevisionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeConfigurationRevisionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConfigurationRevision");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeConfigurationRevisionRequest, DescribeConfigurationRevisionResponse>()
                            .withOperationName("DescribeConfigurationRevision").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeConfigurationRevisionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeConfigurationRevisionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns information about an ActiveMQ user.
     *
     * @param describeUserRequest
     * @return Result of the DescribeUser operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.DescribeUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DescribeUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeUserResponse describeUser(DescribeUserRequest describeUserRequest) throws NotFoundException,
            BadRequestException, InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException,
            MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUser");

            return clientHandler.execute(new ClientExecutionParams<DescribeUserRequest, DescribeUserResponse>()
                    .withOperationName("DescribeUser").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a list of all brokers.
     *
     * @param listBrokersRequest
     * @return Result of the ListBrokers operation returned by the service.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.ListBrokers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/ListBrokers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBrokersResponse listBrokers(ListBrokersRequest listBrokersRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException, MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBrokersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListBrokersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBrokersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBrokers");

            return clientHandler.execute(new ClientExecutionParams<ListBrokersRequest, ListBrokersResponse>()
                    .withOperationName("ListBrokers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listBrokersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListBrokersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a list of all brokers.<br/>
     * <p>
     * This is a variant of {@link #listBrokers(software.amazon.awssdk.services.mq.model.ListBrokersRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mq.paginators.ListBrokersIterable responses = client.listBrokersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mq.paginators.ListBrokersIterable responses = client.listBrokersPaginator(request);
     *     for (software.amazon.awssdk.services.mq.model.ListBrokersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mq.paginators.ListBrokersIterable responses = client.listBrokersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listBrokers(software.amazon.awssdk.services.mq.model.ListBrokersRequest)} operation.</b>
     * </p>
     *
     * @param listBrokersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.ListBrokers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/ListBrokers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBrokersIterable listBrokersPaginator(ListBrokersRequest listBrokersRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException, MqException {
        return new ListBrokersIterable(this, applyPaginatorUserAgent(listBrokersRequest));
    }

    /**
     * Returns a list of all revisions for the specified configuration.
     *
     * @param listConfigurationRevisionsRequest
     * @return Result of the ListConfigurationRevisions operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.ListConfigurationRevisions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/ListConfigurationRevisions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListConfigurationRevisionsResponse listConfigurationRevisions(
            ListConfigurationRevisionsRequest listConfigurationRevisionsRequest) throws NotFoundException, BadRequestException,
            InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException, MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConfigurationRevisionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListConfigurationRevisionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConfigurationRevisionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfigurationRevisions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListConfigurationRevisionsRequest, ListConfigurationRevisionsResponse>()
                            .withOperationName("ListConfigurationRevisions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listConfigurationRevisionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListConfigurationRevisionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a list of all configurations.
     *
     * @param listConfigurationsRequest
     * @return Result of the ListConfigurations operation returned by the service.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.ListConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/ListConfigurations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListConfigurationsResponse listConfigurations(ListConfigurationsRequest listConfigurationsRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, AwsServiceException,
            SdkClientException, MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfigurations");

            return clientHandler.execute(new ClientExecutionParams<ListConfigurationsRequest, ListConfigurationsResponse>()
                    .withOperationName("ListConfigurations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listConfigurationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Lists tags for a resource.
     *
     * @param listTagsRequest
     * @return Result of the ListTags operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.ListTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsResponse listTags(ListTagsRequest listTagsRequest) throws NotFoundException, BadRequestException,
            InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException, MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTags");

            return clientHandler.execute(new ClientExecutionParams<ListTagsRequest, ListTagsResponse>()
                    .withOperationName("ListTags").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a list of all ActiveMQ users.
     *
     * @param listUsersRequest
     * @return Result of the ListUsers operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.ListUsers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListUsersResponse listUsers(ListUsersRequest listUsersRequest) throws NotFoundException, BadRequestException,
            InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException, MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListUsersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUsers");

            return clientHandler.execute(new ClientExecutionParams<ListUsersRequest, ListUsersResponse>()
                    .withOperationName("ListUsers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listUsersRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListUsersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Reboots a broker. Note: This API is asynchronous.
     *
     * @param rebootBrokerRequest
     * @return Result of the RebootBroker operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.RebootBroker
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/RebootBroker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RebootBrokerResponse rebootBroker(RebootBrokerRequest rebootBrokerRequest) throws NotFoundException,
            BadRequestException, InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException,
            MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RebootBrokerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RebootBrokerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rebootBrokerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RebootBroker");

            return clientHandler.execute(new ClientExecutionParams<RebootBrokerRequest, RebootBrokerResponse>()
                    .withOperationName("RebootBroker").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(rebootBrokerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RebootBrokerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Adds a pending configuration change to a broker.
     *
     * @param updateBrokerRequest
     *        Updates the broker using the specified properties.
     * @return Result of the UpdateBroker operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ConflictException
     *         HTTP Status Code 409: Conflict. Concurrent broker update detected. Retrying your request might resolve
     *         the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.UpdateBroker
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/UpdateBroker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateBrokerResponse updateBroker(UpdateBrokerRequest updateBrokerRequest) throws NotFoundException,
            BadRequestException, InternalServerErrorException, ConflictException, ForbiddenException, AwsServiceException,
            SdkClientException, MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateBrokerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateBrokerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBrokerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBroker");

            return clientHandler.execute(new ClientExecutionParams<UpdateBrokerRequest, UpdateBrokerResponse>()
                    .withOperationName("UpdateBroker").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateBrokerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateBrokerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Updates the specified configuration.
     *
     * @param updateConfigurationRequest
     *        Updates the specified configuration.
     * @return Result of the UpdateConfiguration operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ConflictException
     *         HTTP Status Code 409: Conflict. Concurrent update to configuration. Retry to create a new revision.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your input and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.UpdateConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/UpdateConfiguration" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateConfigurationResponse updateConfiguration(UpdateConfigurationRequest updateConfigurationRequest)
            throws NotFoundException, BadRequestException, InternalServerErrorException, ConflictException, ForbiddenException,
            AwsServiceException, SdkClientException, MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfiguration");

            return clientHandler.execute(new ClientExecutionParams<UpdateConfigurationRequest, UpdateConfigurationResponse>()
                    .withOperationName("UpdateConfiguration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateConfigurationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Updates the information for an ActiveMQ user.
     *
     * @param updateUserRequest
     *        Updates the information for an ActiveMQ user.
     * @return Result of the UpdateUser operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws ConflictException
     *         HTTP Status Code 409: Conflict. Retrying your request might resolve the issue.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MqException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MqClient.UpdateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateUserResponse updateUser(UpdateUserRequest updateUserRequest) throws NotFoundException, BadRequestException,
            InternalServerErrorException, ConflictException, ForbiddenException, AwsServiceException, SdkClientException,
            MqException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUser");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateUserRequest, UpdateUserResponse>().withOperationName("UpdateUser")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateUserRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MqException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends MqRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
