/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mq.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Creates a broker using the specified properties.
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateBrokerRequest extends MqRequest implements
        ToCopyableBuilder<CreateBrokerRequest.Builder, CreateBrokerRequest> {
    private static final SdkField<Boolean> AUTO_MINOR_VERSION_UPGRADE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(CreateBrokerRequest::autoMinorVersionUpgrade)).setter(setter(Builder::autoMinorVersionUpgrade))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoMinorVersionUpgrade").build())
            .build();

    private static final SdkField<String> BROKER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateBrokerRequest::brokerName)).setter(setter(Builder::brokerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("brokerName").build()).build();

    private static final SdkField<ConfigurationId> CONFIGURATION_FIELD = SdkField
            .<ConfigurationId> builder(MarshallingType.SDK_POJO).getter(getter(CreateBrokerRequest::configuration))
            .setter(setter(Builder::configuration)).constructor(ConfigurationId::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configuration").build()).build();

    private static final SdkField<String> CREATOR_REQUEST_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CreateBrokerRequest::creatorRequestId))
            .setter(setter(Builder::creatorRequestId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creatorRequestId").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<String> DEPLOYMENT_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateBrokerRequest::deploymentModeAsString)).setter(setter(Builder::deploymentMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deploymentMode").build()).build();

    private static final SdkField<EncryptionOptions> ENCRYPTION_OPTIONS_FIELD = SdkField
            .<EncryptionOptions> builder(MarshallingType.SDK_POJO).getter(getter(CreateBrokerRequest::encryptionOptions))
            .setter(setter(Builder::encryptionOptions)).constructor(EncryptionOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionOptions").build()).build();

    private static final SdkField<String> ENGINE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateBrokerRequest::engineTypeAsString)).setter(setter(Builder::engineType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("engineType").build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateBrokerRequest::engineVersion)).setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("engineVersion").build()).build();

    private static final SdkField<String> HOST_INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateBrokerRequest::hostInstanceType)).setter(setter(Builder::hostInstanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hostInstanceType").build()).build();

    private static final SdkField<Logs> LOGS_FIELD = SdkField.<Logs> builder(MarshallingType.SDK_POJO)
            .getter(getter(CreateBrokerRequest::logs)).setter(setter(Builder::logs)).constructor(Logs::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logs").build()).build();

    private static final SdkField<WeeklyStartTime> MAINTENANCE_WINDOW_START_TIME_FIELD = SdkField
            .<WeeklyStartTime> builder(MarshallingType.SDK_POJO)
            .getter(getter(CreateBrokerRequest::maintenanceWindowStartTime))
            .setter(setter(Builder::maintenanceWindowStartTime))
            .constructor(WeeklyStartTime::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maintenanceWindowStartTime").build())
            .build();

    private static final SdkField<Boolean> PUBLICLY_ACCESSIBLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(CreateBrokerRequest::publiclyAccessible)).setter(setter(Builder::publiclyAccessible))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publiclyAccessible").build())
            .build();

    private static final SdkField<List<String>> SECURITY_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(CreateBrokerRequest::securityGroups))
            .setter(setter(Builder::securityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("securityGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STORAGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateBrokerRequest::storageTypeAsString)).setter(setter(Builder::storageType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("storageType").build()).build();

    private static final SdkField<List<String>> SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(CreateBrokerRequest::subnetIds))
            .setter(setter(Builder::subnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(CreateBrokerRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<User>> USERS_FIELD = SdkField
            .<List<User>> builder(MarshallingType.LIST)
            .getter(getter(CreateBrokerRequest::users))
            .setter(setter(Builder::users))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("users").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<User> builder(MarshallingType.SDK_POJO)
                                            .constructor(User::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            AUTO_MINOR_VERSION_UPGRADE_FIELD, BROKER_NAME_FIELD, CONFIGURATION_FIELD, CREATOR_REQUEST_ID_FIELD,
            DEPLOYMENT_MODE_FIELD, ENCRYPTION_OPTIONS_FIELD, ENGINE_TYPE_FIELD, ENGINE_VERSION_FIELD, HOST_INSTANCE_TYPE_FIELD,
            LOGS_FIELD, MAINTENANCE_WINDOW_START_TIME_FIELD, PUBLICLY_ACCESSIBLE_FIELD, SECURITY_GROUPS_FIELD,
            STORAGE_TYPE_FIELD, SUBNET_IDS_FIELD, TAGS_FIELD, USERS_FIELD));

    private final Boolean autoMinorVersionUpgrade;

    private final String brokerName;

    private final ConfigurationId configuration;

    private final String creatorRequestId;

    private final String deploymentMode;

    private final EncryptionOptions encryptionOptions;

    private final String engineType;

    private final String engineVersion;

    private final String hostInstanceType;

    private final Logs logs;

    private final WeeklyStartTime maintenanceWindowStartTime;

    private final Boolean publiclyAccessible;

    private final List<String> securityGroups;

    private final String storageType;

    private final List<String> subnetIds;

    private final Map<String, String> tags;

    private final List<User> users;

    private CreateBrokerRequest(BuilderImpl builder) {
        super(builder);
        this.autoMinorVersionUpgrade = builder.autoMinorVersionUpgrade;
        this.brokerName = builder.brokerName;
        this.configuration = builder.configuration;
        this.creatorRequestId = builder.creatorRequestId;
        this.deploymentMode = builder.deploymentMode;
        this.encryptionOptions = builder.encryptionOptions;
        this.engineType = builder.engineType;
        this.engineVersion = builder.engineVersion;
        this.hostInstanceType = builder.hostInstanceType;
        this.logs = builder.logs;
        this.maintenanceWindowStartTime = builder.maintenanceWindowStartTime;
        this.publiclyAccessible = builder.publiclyAccessible;
        this.securityGroups = builder.securityGroups;
        this.storageType = builder.storageType;
        this.subnetIds = builder.subnetIds;
        this.tags = builder.tags;
        this.users = builder.users;
    }

    /**
     * Required. Enables automatic upgrades to new minor versions for brokers, as Apache releases the versions. The
     * automatic upgrades occur during the maintenance window of the broker or after a manual broker reboot.
     * 
     * @return Required. Enables automatic upgrades to new minor versions for brokers, as Apache releases the versions.
     *         The automatic upgrades occur during the maintenance window of the broker or after a manual broker reboot.
     */
    public Boolean autoMinorVersionUpgrade() {
        return autoMinorVersionUpgrade;
    }

    /**
     * Required. The name of the broker. This value must be unique in your AWS account, 1-50 characters long, must
     * contain only letters, numbers, dashes, and underscores, and must not contain whitespaces, brackets, wildcard
     * characters, or special characters.
     * 
     * @return Required. The name of the broker. This value must be unique in your AWS account, 1-50 characters long,
     *         must contain only letters, numbers, dashes, and underscores, and must not contain whitespaces, brackets,
     *         wildcard characters, or special characters.
     */
    public String brokerName() {
        return brokerName;
    }

    /**
     * A list of information about the configuration.
     * 
     * @return A list of information about the configuration.
     */
    public ConfigurationId configuration() {
        return configuration;
    }

    /**
     * The unique ID that the requester receives for the created broker. Amazon MQ passes your ID with the API action.
     * Note: We recommend using a Universally Unique Identifier (UUID) for the creatorRequestId. You may omit the
     * creatorRequestId if your application doesn't require idempotency.
     * 
     * @return The unique ID that the requester receives for the created broker. Amazon MQ passes your ID with the API
     *         action. Note: We recommend using a Universally Unique Identifier (UUID) for the creatorRequestId. You may
     *         omit the creatorRequestId if your application doesn't require idempotency.
     */
    public String creatorRequestId() {
        return creatorRequestId;
    }

    /**
     * Required. The deployment mode of the broker.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deploymentMode}
     * will return {@link DeploymentMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #deploymentModeAsString}.
     * </p>
     * 
     * @return Required. The deployment mode of the broker.
     * @see DeploymentMode
     */
    public DeploymentMode deploymentMode() {
        return DeploymentMode.fromValue(deploymentMode);
    }

    /**
     * Required. The deployment mode of the broker.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deploymentMode}
     * will return {@link DeploymentMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #deploymentModeAsString}.
     * </p>
     * 
     * @return Required. The deployment mode of the broker.
     * @see DeploymentMode
     */
    public String deploymentModeAsString() {
        return deploymentMode;
    }

    /**
     * Encryption options for the broker.
     * 
     * @return Encryption options for the broker.
     */
    public EncryptionOptions encryptionOptions() {
        return encryptionOptions;
    }

    /**
     * Required. The type of broker engine. Note: Currently, Amazon MQ supports only ACTIVEMQ.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #engineType} will
     * return {@link EngineType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #engineTypeAsString}.
     * </p>
     * 
     * @return Required. The type of broker engine. Note: Currently, Amazon MQ supports only ACTIVEMQ.
     * @see EngineType
     */
    public EngineType engineType() {
        return EngineType.fromValue(engineType);
    }

    /**
     * Required. The type of broker engine. Note: Currently, Amazon MQ supports only ACTIVEMQ.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #engineType} will
     * return {@link EngineType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #engineTypeAsString}.
     * </p>
     * 
     * @return Required. The type of broker engine. Note: Currently, Amazon MQ supports only ACTIVEMQ.
     * @see EngineType
     */
    public String engineTypeAsString() {
        return engineType;
    }

    /**
     * Required. The version of the broker engine. For a list of supported engine versions, see
     * https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/broker-engine.html
     * 
     * @return Required. The version of the broker engine. For a list of supported engine versions, see
     *         https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/broker-engine.html
     */
    public String engineVersion() {
        return engineVersion;
    }

    /**
     * Required. The broker's instance type.
     * 
     * @return Required. The broker's instance type.
     */
    public String hostInstanceType() {
        return hostInstanceType;
    }

    /**
     * Enables Amazon CloudWatch logging for brokers.
     * 
     * @return Enables Amazon CloudWatch logging for brokers.
     */
    public Logs logs() {
        return logs;
    }

    /**
     * The parameters that determine the WeeklyStartTime.
     * 
     * @return The parameters that determine the WeeklyStartTime.
     */
    public WeeklyStartTime maintenanceWindowStartTime() {
        return maintenanceWindowStartTime;
    }

    /**
     * Required. Enables connections from applications outside of the VPC that hosts the broker's subnets.
     * 
     * @return Required. Enables connections from applications outside of the VPC that hosts the broker's subnets.
     */
    public Boolean publiclyAccessible() {
        return publiclyAccessible;
    }

    /**
     * Returns true if the SecurityGroups property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasSecurityGroups() {
        return securityGroups != null && !(securityGroups instanceof SdkAutoConstructList);
    }

    /**
     * The list of security groups (1 minimum, 5 maximum) that authorizes connections to brokers.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSecurityGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of security groups (1 minimum, 5 maximum) that authorizes connections to brokers.
     */
    public List<String> securityGroups() {
        return securityGroups;
    }

    /**
     * The broker's storage type.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageType} will
     * return {@link BrokerStorageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageTypeAsString}.
     * </p>
     * 
     * @return The broker's storage type.
     * @see BrokerStorageType
     */
    public BrokerStorageType storageType() {
        return BrokerStorageType.fromValue(storageType);
    }

    /**
     * The broker's storage type.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageType} will
     * return {@link BrokerStorageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageTypeAsString}.
     * </p>
     * 
     * @return The broker's storage type.
     * @see BrokerStorageType
     */
    public String storageTypeAsString() {
        return storageType;
    }

    /**
     * Returns true if the SubnetIds property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasSubnetIds() {
        return subnetIds != null && !(subnetIds instanceof SdkAutoConstructList);
    }

    /**
     * The list of groups (2 maximum) that define which subnets and IP ranges the broker can use from different
     * Availability Zones. A SINGLE_INSTANCE deployment requires one subnet (for example, the default subnet). An
     * ACTIVE_STANDBY_MULTI_AZ deployment requires two subnets.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSubnetIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of groups (2 maximum) that define which subnets and IP ranges the broker can use from different
     *         Availability Zones. A SINGLE_INSTANCE deployment requires one subnet (for example, the default subnet).
     *         An ACTIVE_STANDBY_MULTI_AZ deployment requires two subnets.
     */
    public List<String> subnetIds() {
        return subnetIds;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * Create tags when creating the broker.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Create tags when creating the broker.
     */
    public Map<String, String> tags() {
        return tags;
    }

    /**
     * Returns true if the Users property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasUsers() {
        return users != null && !(users instanceof SdkAutoConstructList);
    }

    /**
     * Required. The list of ActiveMQ users (persons or applications) who can access queues and topics. This value can
     * contain only alphanumeric characters, dashes, periods, underscores, and tildes (- . _ ~). This value must be
     * 2-100 characters long.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasUsers()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Required. The list of ActiveMQ users (persons or applications) who can access queues and topics. This
     *         value can contain only alphanumeric characters, dashes, periods, underscores, and tildes (- . _ ~). This
     *         value must be 2-100 characters long.
     */
    public List<User> users() {
        return users;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(autoMinorVersionUpgrade());
        hashCode = 31 * hashCode + Objects.hashCode(brokerName());
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        hashCode = 31 * hashCode + Objects.hashCode(creatorRequestId());
        hashCode = 31 * hashCode + Objects.hashCode(deploymentModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionOptions());
        hashCode = 31 * hashCode + Objects.hashCode(engineTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hostInstanceType());
        hashCode = 31 * hashCode + Objects.hashCode(logs());
        hashCode = 31 * hashCode + Objects.hashCode(maintenanceWindowStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(publiclyAccessible());
        hashCode = 31 * hashCode + Objects.hashCode(securityGroups());
        hashCode = 31 * hashCode + Objects.hashCode(storageTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(subnetIds());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(users());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateBrokerRequest)) {
            return false;
        }
        CreateBrokerRequest other = (CreateBrokerRequest) obj;
        return Objects.equals(autoMinorVersionUpgrade(), other.autoMinorVersionUpgrade())
                && Objects.equals(brokerName(), other.brokerName()) && Objects.equals(configuration(), other.configuration())
                && Objects.equals(creatorRequestId(), other.creatorRequestId())
                && Objects.equals(deploymentModeAsString(), other.deploymentModeAsString())
                && Objects.equals(encryptionOptions(), other.encryptionOptions())
                && Objects.equals(engineTypeAsString(), other.engineTypeAsString())
                && Objects.equals(engineVersion(), other.engineVersion())
                && Objects.equals(hostInstanceType(), other.hostInstanceType()) && Objects.equals(logs(), other.logs())
                && Objects.equals(maintenanceWindowStartTime(), other.maintenanceWindowStartTime())
                && Objects.equals(publiclyAccessible(), other.publiclyAccessible())
                && Objects.equals(securityGroups(), other.securityGroups())
                && Objects.equals(storageTypeAsString(), other.storageTypeAsString())
                && Objects.equals(subnetIds(), other.subnetIds()) && Objects.equals(tags(), other.tags())
                && Objects.equals(users(), other.users());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateBrokerRequest").add("AutoMinorVersionUpgrade", autoMinorVersionUpgrade())
                .add("BrokerName", brokerName()).add("Configuration", configuration())
                .add("CreatorRequestId", creatorRequestId()).add("DeploymentMode", deploymentModeAsString())
                .add("EncryptionOptions", encryptionOptions()).add("EngineType", engineTypeAsString())
                .add("EngineVersion", engineVersion()).add("HostInstanceType", hostInstanceType()).add("Logs", logs())
                .add("MaintenanceWindowStartTime", maintenanceWindowStartTime()).add("PubliclyAccessible", publiclyAccessible())
                .add("SecurityGroups", securityGroups()).add("StorageType", storageTypeAsString()).add("SubnetIds", subnetIds())
                .add("Tags", tags()).add("Users", users()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AutoMinorVersionUpgrade":
            return Optional.ofNullable(clazz.cast(autoMinorVersionUpgrade()));
        case "BrokerName":
            return Optional.ofNullable(clazz.cast(brokerName()));
        case "Configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "CreatorRequestId":
            return Optional.ofNullable(clazz.cast(creatorRequestId()));
        case "DeploymentMode":
            return Optional.ofNullable(clazz.cast(deploymentModeAsString()));
        case "EncryptionOptions":
            return Optional.ofNullable(clazz.cast(encryptionOptions()));
        case "EngineType":
            return Optional.ofNullable(clazz.cast(engineTypeAsString()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "HostInstanceType":
            return Optional.ofNullable(clazz.cast(hostInstanceType()));
        case "Logs":
            return Optional.ofNullable(clazz.cast(logs()));
        case "MaintenanceWindowStartTime":
            return Optional.ofNullable(clazz.cast(maintenanceWindowStartTime()));
        case "PubliclyAccessible":
            return Optional.ofNullable(clazz.cast(publiclyAccessible()));
        case "SecurityGroups":
            return Optional.ofNullable(clazz.cast(securityGroups()));
        case "StorageType":
            return Optional.ofNullable(clazz.cast(storageTypeAsString()));
        case "SubnetIds":
            return Optional.ofNullable(clazz.cast(subnetIds()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "Users":
            return Optional.ofNullable(clazz.cast(users()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateBrokerRequest, T> g) {
        return obj -> g.apply((CreateBrokerRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends MqRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateBrokerRequest> {
        /**
         * Required. Enables automatic upgrades to new minor versions for brokers, as Apache releases the versions. The
         * automatic upgrades occur during the maintenance window of the broker or after a manual broker reboot.
         * 
         * @param autoMinorVersionUpgrade
         *        Required. Enables automatic upgrades to new minor versions for brokers, as Apache releases the
         *        versions. The automatic upgrades occur during the maintenance window of the broker or after a manual
         *        broker reboot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoMinorVersionUpgrade(Boolean autoMinorVersionUpgrade);

        /**
         * Required. The name of the broker. This value must be unique in your AWS account, 1-50 characters long, must
         * contain only letters, numbers, dashes, and underscores, and must not contain whitespaces, brackets, wildcard
         * characters, or special characters.
         * 
         * @param brokerName
         *        Required. The name of the broker. This value must be unique in your AWS account, 1-50 characters long,
         *        must contain only letters, numbers, dashes, and underscores, and must not contain whitespaces,
         *        brackets, wildcard characters, or special characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder brokerName(String brokerName);

        /**
         * A list of information about the configuration.
         * 
         * @param configuration
         *        A list of information about the configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(ConfigurationId configuration);

        /**
         * A list of information about the configuration. This is a convenience that creates an instance of the
         * {@link ConfigurationId.Builder} avoiding the need to create one manually via
         * {@link ConfigurationId#builder()}.
         *
         * When the {@link Consumer} completes, {@link ConfigurationId.Builder#build()} is called immediately and its
         * result is passed to {@link #configuration(ConfigurationId)}.
         * 
         * @param configuration
         *        a consumer that will call methods on {@link ConfigurationId.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configuration(ConfigurationId)
         */
        default Builder configuration(Consumer<ConfigurationId.Builder> configuration) {
            return configuration(ConfigurationId.builder().applyMutation(configuration).build());
        }

        /**
         * The unique ID that the requester receives for the created broker. Amazon MQ passes your ID with the API
         * action. Note: We recommend using a Universally Unique Identifier (UUID) for the creatorRequestId. You may
         * omit the creatorRequestId if your application doesn't require idempotency.
         * 
         * @param creatorRequestId
         *        The unique ID that the requester receives for the created broker. Amazon MQ passes your ID with the
         *        API action. Note: We recommend using a Universally Unique Identifier (UUID) for the creatorRequestId.
         *        You may omit the creatorRequestId if your application doesn't require idempotency.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creatorRequestId(String creatorRequestId);

        /**
         * Required. The deployment mode of the broker.
         * 
         * @param deploymentMode
         *        Required. The deployment mode of the broker.
         * @see DeploymentMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeploymentMode
         */
        Builder deploymentMode(String deploymentMode);

        /**
         * Required. The deployment mode of the broker.
         * 
         * @param deploymentMode
         *        Required. The deployment mode of the broker.
         * @see DeploymentMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeploymentMode
         */
        Builder deploymentMode(DeploymentMode deploymentMode);

        /**
         * Encryption options for the broker.
         * 
         * @param encryptionOptions
         *        Encryption options for the broker.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionOptions(EncryptionOptions encryptionOptions);

        /**
         * Encryption options for the broker. This is a convenience that creates an instance of the
         * {@link EncryptionOptions.Builder} avoiding the need to create one manually via
         * {@link EncryptionOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link EncryptionOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #encryptionOptions(EncryptionOptions)}.
         * 
         * @param encryptionOptions
         *        a consumer that will call methods on {@link EncryptionOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionOptions(EncryptionOptions)
         */
        default Builder encryptionOptions(Consumer<EncryptionOptions.Builder> encryptionOptions) {
            return encryptionOptions(EncryptionOptions.builder().applyMutation(encryptionOptions).build());
        }

        /**
         * Required. The type of broker engine. Note: Currently, Amazon MQ supports only ACTIVEMQ.
         * 
         * @param engineType
         *        Required. The type of broker engine. Note: Currently, Amazon MQ supports only ACTIVEMQ.
         * @see EngineType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EngineType
         */
        Builder engineType(String engineType);

        /**
         * Required. The type of broker engine. Note: Currently, Amazon MQ supports only ACTIVEMQ.
         * 
         * @param engineType
         *        Required. The type of broker engine. Note: Currently, Amazon MQ supports only ACTIVEMQ.
         * @see EngineType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EngineType
         */
        Builder engineType(EngineType engineType);

        /**
         * Required. The version of the broker engine. For a list of supported engine versions, see
         * https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/broker-engine.html
         * 
         * @param engineVersion
         *        Required. The version of the broker engine. For a list of supported engine versions, see
         *        https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/broker-engine.html
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * Required. The broker's instance type.
         * 
         * @param hostInstanceType
         *        Required. The broker's instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostInstanceType(String hostInstanceType);

        /**
         * Enables Amazon CloudWatch logging for brokers.
         * 
         * @param logs
         *        Enables Amazon CloudWatch logging for brokers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logs(Logs logs);

        /**
         * Enables Amazon CloudWatch logging for brokers. This is a convenience that creates an instance of the
         * {@link Logs.Builder} avoiding the need to create one manually via {@link Logs#builder()}.
         *
         * When the {@link Consumer} completes, {@link Logs.Builder#build()} is called immediately and its result is
         * passed to {@link #logs(Logs)}.
         * 
         * @param logs
         *        a consumer that will call methods on {@link Logs.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logs(Logs)
         */
        default Builder logs(Consumer<Logs.Builder> logs) {
            return logs(Logs.builder().applyMutation(logs).build());
        }

        /**
         * The parameters that determine the WeeklyStartTime.
         * 
         * @param maintenanceWindowStartTime
         *        The parameters that determine the WeeklyStartTime.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maintenanceWindowStartTime(WeeklyStartTime maintenanceWindowStartTime);

        /**
         * The parameters that determine the WeeklyStartTime. This is a convenience that creates an instance of the
         * {@link WeeklyStartTime.Builder} avoiding the need to create one manually via
         * {@link WeeklyStartTime#builder()}.
         *
         * When the {@link Consumer} completes, {@link WeeklyStartTime.Builder#build()} is called immediately and its
         * result is passed to {@link #maintenanceWindowStartTime(WeeklyStartTime)}.
         * 
         * @param maintenanceWindowStartTime
         *        a consumer that will call methods on {@link WeeklyStartTime.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #maintenanceWindowStartTime(WeeklyStartTime)
         */
        default Builder maintenanceWindowStartTime(Consumer<WeeklyStartTime.Builder> maintenanceWindowStartTime) {
            return maintenanceWindowStartTime(WeeklyStartTime.builder().applyMutation(maintenanceWindowStartTime).build());
        }

        /**
         * Required. Enables connections from applications outside of the VPC that hosts the broker's subnets.
         * 
         * @param publiclyAccessible
         *        Required. Enables connections from applications outside of the VPC that hosts the broker's subnets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publiclyAccessible(Boolean publiclyAccessible);

        /**
         * The list of security groups (1 minimum, 5 maximum) that authorizes connections to brokers.
         * 
         * @param securityGroups
         *        The list of security groups (1 minimum, 5 maximum) that authorizes connections to brokers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(Collection<String> securityGroups);

        /**
         * The list of security groups (1 minimum, 5 maximum) that authorizes connections to brokers.
         * 
         * @param securityGroups
         *        The list of security groups (1 minimum, 5 maximum) that authorizes connections to brokers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(String... securityGroups);

        /**
         * The broker's storage type.
         * 
         * @param storageType
         *        The broker's storage type.
         * @see BrokerStorageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BrokerStorageType
         */
        Builder storageType(String storageType);

        /**
         * The broker's storage type.
         * 
         * @param storageType
         *        The broker's storage type.
         * @see BrokerStorageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BrokerStorageType
         */
        Builder storageType(BrokerStorageType storageType);

        /**
         * The list of groups (2 maximum) that define which subnets and IP ranges the broker can use from different
         * Availability Zones. A SINGLE_INSTANCE deployment requires one subnet (for example, the default subnet). An
         * ACTIVE_STANDBY_MULTI_AZ deployment requires two subnets.
         * 
         * @param subnetIds
         *        The list of groups (2 maximum) that define which subnets and IP ranges the broker can use from
         *        different Availability Zones. A SINGLE_INSTANCE deployment requires one subnet (for example, the
         *        default subnet). An ACTIVE_STANDBY_MULTI_AZ deployment requires two subnets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(Collection<String> subnetIds);

        /**
         * The list of groups (2 maximum) that define which subnets and IP ranges the broker can use from different
         * Availability Zones. A SINGLE_INSTANCE deployment requires one subnet (for example, the default subnet). An
         * ACTIVE_STANDBY_MULTI_AZ deployment requires two subnets.
         * 
         * @param subnetIds
         *        The list of groups (2 maximum) that define which subnets and IP ranges the broker can use from
         *        different Availability Zones. A SINGLE_INSTANCE deployment requires one subnet (for example, the
         *        default subnet). An ACTIVE_STANDBY_MULTI_AZ deployment requires two subnets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(String... subnetIds);

        /**
         * Create tags when creating the broker.
         * 
         * @param tags
         *        Create tags when creating the broker.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * Required. The list of ActiveMQ users (persons or applications) who can access queues and topics. This value
         * can contain only alphanumeric characters, dashes, periods, underscores, and tildes (- . _ ~). This value must
         * be 2-100 characters long.
         * 
         * @param users
         *        Required. The list of ActiveMQ users (persons or applications) who can access queues and topics. This
         *        value can contain only alphanumeric characters, dashes, periods, underscores, and tildes (- . _ ~).
         *        This value must be 2-100 characters long.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder users(Collection<User> users);

        /**
         * Required. The list of ActiveMQ users (persons or applications) who can access queues and topics. This value
         * can contain only alphanumeric characters, dashes, periods, underscores, and tildes (- . _ ~). This value must
         * be 2-100 characters long.
         * 
         * @param users
         *        Required. The list of ActiveMQ users (persons or applications) who can access queues and topics. This
         *        value can contain only alphanumeric characters, dashes, periods, underscores, and tildes (- . _ ~).
         *        This value must be 2-100 characters long.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder users(User... users);

        /**
         * Required. The list of ActiveMQ users (persons or applications) who can access queues and topics. This value
         * can contain only alphanumeric characters, dashes, periods, underscores, and tildes (- . _ ~). This value must
         * be 2-100 characters long. This is a convenience that creates an instance of the {@link List<User>.Builder}
         * avoiding the need to create one manually via {@link List<User>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<User>.Builder#build()} is called immediately and its result
         * is passed to {@link #users(List<User>)}.
         * 
         * @param users
         *        a consumer that will call methods on {@link List<User>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #users(List<User>)
         */
        Builder users(Consumer<User.Builder>... users);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends MqRequest.BuilderImpl implements Builder {
        private Boolean autoMinorVersionUpgrade;

        private String brokerName;

        private ConfigurationId configuration;

        private String creatorRequestId;

        private String deploymentMode;

        private EncryptionOptions encryptionOptions;

        private String engineType;

        private String engineVersion;

        private String hostInstanceType;

        private Logs logs;

        private WeeklyStartTime maintenanceWindowStartTime;

        private Boolean publiclyAccessible;

        private List<String> securityGroups = DefaultSdkAutoConstructList.getInstance();

        private String storageType;

        private List<String> subnetIds = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private List<User> users = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateBrokerRequest model) {
            super(model);
            autoMinorVersionUpgrade(model.autoMinorVersionUpgrade);
            brokerName(model.brokerName);
            configuration(model.configuration);
            creatorRequestId(model.creatorRequestId);
            deploymentMode(model.deploymentMode);
            encryptionOptions(model.encryptionOptions);
            engineType(model.engineType);
            engineVersion(model.engineVersion);
            hostInstanceType(model.hostInstanceType);
            logs(model.logs);
            maintenanceWindowStartTime(model.maintenanceWindowStartTime);
            publiclyAccessible(model.publiclyAccessible);
            securityGroups(model.securityGroups);
            storageType(model.storageType);
            subnetIds(model.subnetIds);
            tags(model.tags);
            users(model.users);
        }

        public final Boolean getAutoMinorVersionUpgrade() {
            return autoMinorVersionUpgrade;
        }

        @Override
        public final Builder autoMinorVersionUpgrade(Boolean autoMinorVersionUpgrade) {
            this.autoMinorVersionUpgrade = autoMinorVersionUpgrade;
            return this;
        }

        public final void setAutoMinorVersionUpgrade(Boolean autoMinorVersionUpgrade) {
            this.autoMinorVersionUpgrade = autoMinorVersionUpgrade;
        }

        public final String getBrokerName() {
            return brokerName;
        }

        @Override
        public final Builder brokerName(String brokerName) {
            this.brokerName = brokerName;
            return this;
        }

        public final void setBrokerName(String brokerName) {
            this.brokerName = brokerName;
        }

        public final ConfigurationId.Builder getConfiguration() {
            return configuration != null ? configuration.toBuilder() : null;
        }

        @Override
        public final Builder configuration(ConfigurationId configuration) {
            this.configuration = configuration;
            return this;
        }

        public final void setConfiguration(ConfigurationId.BuilderImpl configuration) {
            this.configuration = configuration != null ? configuration.build() : null;
        }

        public final String getCreatorRequestId() {
            return creatorRequestId;
        }

        @Override
        public final Builder creatorRequestId(String creatorRequestId) {
            this.creatorRequestId = creatorRequestId;
            return this;
        }

        public final void setCreatorRequestId(String creatorRequestId) {
            this.creatorRequestId = creatorRequestId;
        }

        public final String getDeploymentModeAsString() {
            return deploymentMode;
        }

        @Override
        public final Builder deploymentMode(String deploymentMode) {
            this.deploymentMode = deploymentMode;
            return this;
        }

        @Override
        public final Builder deploymentMode(DeploymentMode deploymentMode) {
            this.deploymentMode(deploymentMode == null ? null : deploymentMode.toString());
            return this;
        }

        public final void setDeploymentMode(String deploymentMode) {
            this.deploymentMode = deploymentMode;
        }

        public final EncryptionOptions.Builder getEncryptionOptions() {
            return encryptionOptions != null ? encryptionOptions.toBuilder() : null;
        }

        @Override
        public final Builder encryptionOptions(EncryptionOptions encryptionOptions) {
            this.encryptionOptions = encryptionOptions;
            return this;
        }

        public final void setEncryptionOptions(EncryptionOptions.BuilderImpl encryptionOptions) {
            this.encryptionOptions = encryptionOptions != null ? encryptionOptions.build() : null;
        }

        public final String getEngineTypeAsString() {
            return engineType;
        }

        @Override
        public final Builder engineType(String engineType) {
            this.engineType = engineType;
            return this;
        }

        @Override
        public final Builder engineType(EngineType engineType) {
            this.engineType(engineType == null ? null : engineType.toString());
            return this;
        }

        public final void setEngineType(String engineType) {
            this.engineType = engineType;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        public final String getHostInstanceType() {
            return hostInstanceType;
        }

        @Override
        public final Builder hostInstanceType(String hostInstanceType) {
            this.hostInstanceType = hostInstanceType;
            return this;
        }

        public final void setHostInstanceType(String hostInstanceType) {
            this.hostInstanceType = hostInstanceType;
        }

        public final Logs.Builder getLogs() {
            return logs != null ? logs.toBuilder() : null;
        }

        @Override
        public final Builder logs(Logs logs) {
            this.logs = logs;
            return this;
        }

        public final void setLogs(Logs.BuilderImpl logs) {
            this.logs = logs != null ? logs.build() : null;
        }

        public final WeeklyStartTime.Builder getMaintenanceWindowStartTime() {
            return maintenanceWindowStartTime != null ? maintenanceWindowStartTime.toBuilder() : null;
        }

        @Override
        public final Builder maintenanceWindowStartTime(WeeklyStartTime maintenanceWindowStartTime) {
            this.maintenanceWindowStartTime = maintenanceWindowStartTime;
            return this;
        }

        public final void setMaintenanceWindowStartTime(WeeklyStartTime.BuilderImpl maintenanceWindowStartTime) {
            this.maintenanceWindowStartTime = maintenanceWindowStartTime != null ? maintenanceWindowStartTime.build() : null;
        }

        public final Boolean getPubliclyAccessible() {
            return publiclyAccessible;
        }

        @Override
        public final Builder publiclyAccessible(Boolean publiclyAccessible) {
            this.publiclyAccessible = publiclyAccessible;
            return this;
        }

        public final void setPubliclyAccessible(Boolean publiclyAccessible) {
            this.publiclyAccessible = publiclyAccessible;
        }

        public final Collection<String> getSecurityGroups() {
            return securityGroups;
        }

        @Override
        public final Builder securityGroups(Collection<String> securityGroups) {
            this.securityGroups = ___listOf__stringCopier.copy(securityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroups(String... securityGroups) {
            securityGroups(Arrays.asList(securityGroups));
            return this;
        }

        public final void setSecurityGroups(Collection<String> securityGroups) {
            this.securityGroups = ___listOf__stringCopier.copy(securityGroups);
        }

        public final String getStorageTypeAsString() {
            return storageType;
        }

        @Override
        public final Builder storageType(String storageType) {
            this.storageType = storageType;
            return this;
        }

        @Override
        public final Builder storageType(BrokerStorageType storageType) {
            this.storageType(storageType == null ? null : storageType.toString());
            return this;
        }

        public final void setStorageType(String storageType) {
            this.storageType = storageType;
        }

        public final Collection<String> getSubnetIds() {
            return subnetIds;
        }

        @Override
        public final Builder subnetIds(Collection<String> subnetIds) {
            this.subnetIds = ___listOf__stringCopier.copy(subnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetIds(String... subnetIds) {
            subnetIds(Arrays.asList(subnetIds));
            return this;
        }

        public final void setSubnetIds(Collection<String> subnetIds) {
            this.subnetIds = ___listOf__stringCopier.copy(subnetIds);
        }

        public final Map<String, String> getTags() {
            return tags;
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = ___mapOf__stringCopier.copy(tags);
            return this;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = ___mapOf__stringCopier.copy(tags);
        }

        public final Collection<User.Builder> getUsers() {
            return users != null ? users.stream().map(User::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder users(Collection<User> users) {
            this.users = ___listOfUserCopier.copy(users);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder users(User... users) {
            users(Arrays.asList(users));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder users(Consumer<User.Builder>... users) {
            users(Stream.of(users).map(c -> User.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setUsers(Collection<User.BuilderImpl> users) {
            this.users = ___listOfUserCopier.copyFromBuilder(users);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateBrokerRequest build() {
            return new CreateBrokerRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
