/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.migrationhubconfig;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.migrationhubconfig.model.AccessDeniedException;
import software.amazon.awssdk.services.migrationhubconfig.model.CreateHomeRegionControlRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.CreateHomeRegionControlResponse;
import software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsResponse;
import software.amazon.awssdk.services.migrationhubconfig.model.DryRunOperationException;
import software.amazon.awssdk.services.migrationhubconfig.model.GetHomeRegionRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.GetHomeRegionResponse;
import software.amazon.awssdk.services.migrationhubconfig.model.InternalServerErrorException;
import software.amazon.awssdk.services.migrationhubconfig.model.InvalidInputException;
import software.amazon.awssdk.services.migrationhubconfig.model.MigrationHubConfigException;
import software.amazon.awssdk.services.migrationhubconfig.model.MigrationHubConfigRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.ServiceUnavailableException;
import software.amazon.awssdk.services.migrationhubconfig.model.ThrottlingException;
import software.amazon.awssdk.services.migrationhubconfig.paginators.DescribeHomeRegionControlsPublisher;
import software.amazon.awssdk.services.migrationhubconfig.transform.CreateHomeRegionControlRequestMarshaller;
import software.amazon.awssdk.services.migrationhubconfig.transform.DescribeHomeRegionControlsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubconfig.transform.GetHomeRegionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MigrationHubConfigAsyncClient}.
 *
 * @see MigrationHubConfigAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMigrationHubConfigAsyncClient implements MigrationHubConfigAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMigrationHubConfigAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMigrationHubConfigAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * This API sets up the home region for the calling account only.
     * </p>
     *
     * @param createHomeRegionControlRequest
     * @return A Java Future containing the result of the CreateHomeRegionControl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when a request fails due to temporary unavailability of
     *         the service.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>DryRunOperationException Exception raised to indicate that authorization of an action was successful,
     *         when the <code>DryRun</code> flag is set to true.</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubConfigException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubConfigAsyncClient.CreateHomeRegionControl
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/CreateHomeRegionControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateHomeRegionControlResponse> createHomeRegionControl(
            CreateHomeRegionControlRequest createHomeRegionControlRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHomeRegionControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHub Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHomeRegionControl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateHomeRegionControlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateHomeRegionControlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateHomeRegionControlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateHomeRegionControlRequest, CreateHomeRegionControlResponse>()
                            .withOperationName("CreateHomeRegionControl")
                            .withMarshaller(new CreateHomeRegionControlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createHomeRegionControlRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createHomeRegionControlRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateHomeRegionControlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This API permits filtering on the <code>ControlId</code> and <code>HomeRegion</code> fields.
     * </p>
     *
     * @param describeHomeRegionControlsRequest
     * @return A Java Future containing the result of the DescribeHomeRegionControls operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when a request fails due to temporary unavailability of
     *         the service.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubConfigException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubConfigAsyncClient.DescribeHomeRegionControls
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/DescribeHomeRegionControls"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeHomeRegionControlsResponse> describeHomeRegionControls(
            DescribeHomeRegionControlsRequest describeHomeRegionControlsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeHomeRegionControlsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHub Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeHomeRegionControls");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeHomeRegionControlsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeHomeRegionControlsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeHomeRegionControlsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeHomeRegionControlsRequest, DescribeHomeRegionControlsResponse>()
                            .withOperationName("DescribeHomeRegionControls")
                            .withMarshaller(new DescribeHomeRegionControlsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeHomeRegionControlsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeHomeRegionControlsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeHomeRegionControlsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This API permits filtering on the <code>ControlId</code> and <code>HomeRegion</code> fields.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeHomeRegionControls(software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubconfig.paginators.DescribeHomeRegionControlsPublisher publisher = client.describeHomeRegionControlsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubconfig.paginators.DescribeHomeRegionControlsPublisher publisher = client.describeHomeRegionControlsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeHomeRegionControls(software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeHomeRegionControlsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when a request fails due to temporary unavailability of
     *         the service.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubConfigException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubConfigAsyncClient.DescribeHomeRegionControls
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/DescribeHomeRegionControls"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeHomeRegionControlsPublisher describeHomeRegionControlsPaginator(
            DescribeHomeRegionControlsRequest describeHomeRegionControlsRequest) {
        return new DescribeHomeRegionControlsPublisher(this, applyPaginatorUserAgent(describeHomeRegionControlsRequest));
    }

    /**
     * <p>
     * Returns the calling account’s home region, if configured. This API is used by other AWS services to determine the
     * regional endpoint for calling AWS Application Discovery Service and Migration Hub. You must call
     * <code>GetHomeRegion</code> at least once before you call any other AWS Application Discovery Service and AWS
     * Migration Hub APIs, to obtain the account's Migration Hub home region.
     * </p>
     *
     * @param getHomeRegionRequest
     * @return A Java Future containing the result of the GetHomeRegion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when a request fails due to temporary unavailability of
     *         the service.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubConfigException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubConfigAsyncClient.GetHomeRegion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/GetHomeRegion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetHomeRegionResponse> getHomeRegion(GetHomeRegionRequest getHomeRegionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getHomeRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHub Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHomeRegion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetHomeRegionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetHomeRegionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetHomeRegionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetHomeRegionRequest, GetHomeRegionResponse>()
                            .withOperationName("GetHomeRegion")
                            .withMarshaller(new GetHomeRegionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getHomeRegionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getHomeRegionRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetHomeRegionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MigrationHubConfigException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DryRunOperation")
                                .exceptionBuilderSupplier(DryRunOperationException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends MigrationHubConfigRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
