/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.medialive.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * The settings for Automatic Input Failover.
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutomaticInputFailoverSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<AutomaticInputFailoverSettings.Builder, AutomaticInputFailoverSettings> {
    private static final SdkField<Integer> ERROR_CLEAR_TIME_MSEC_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ErrorClearTimeMsec").getter(getter(AutomaticInputFailoverSettings::errorClearTimeMsec))
            .setter(setter(Builder::errorClearTimeMsec))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("errorClearTimeMsec").build())
            .build();

    private static final SdkField<List<FailoverCondition>> FAILOVER_CONDITIONS_FIELD = SdkField
            .<List<FailoverCondition>> builder(MarshallingType.LIST)
            .memberName("FailoverConditions")
            .getter(getter(AutomaticInputFailoverSettings::failoverConditions))
            .setter(setter(Builder::failoverConditions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("failoverConditions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<FailoverCondition> builder(MarshallingType.SDK_POJO)
                                            .constructor(FailoverCondition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> INPUT_PREFERENCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InputPreference").getter(getter(AutomaticInputFailoverSettings::inputPreferenceAsString))
            .setter(setter(Builder::inputPreference))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inputPreference").build()).build();

    private static final SdkField<String> SECONDARY_INPUT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecondaryInputId").getter(getter(AutomaticInputFailoverSettings::secondaryInputId))
            .setter(setter(Builder::secondaryInputId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secondaryInputId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ERROR_CLEAR_TIME_MSEC_FIELD,
            FAILOVER_CONDITIONS_FIELD, INPUT_PREFERENCE_FIELD, SECONDARY_INPUT_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer errorClearTimeMsec;

    private final List<FailoverCondition> failoverConditions;

    private final String inputPreference;

    private final String secondaryInputId;

    private AutomaticInputFailoverSettings(BuilderImpl builder) {
        this.errorClearTimeMsec = builder.errorClearTimeMsec;
        this.failoverConditions = builder.failoverConditions;
        this.inputPreference = builder.inputPreference;
        this.secondaryInputId = builder.secondaryInputId;
    }

    /**
     * This clear time defines the requirement a recovered input must meet to be considered healthy. The input must have
     * no failover conditions for this length of time. Enter a time in milliseconds. This value is particularly
     * important if the input_preference for the failover pair is set to PRIMARY_INPUT_PREFERRED, because after this
     * time, MediaLive will switch back to the primary input.
     * 
     * @return This clear time defines the requirement a recovered input must meet to be considered healthy. The input
     *         must have no failover conditions for this length of time. Enter a time in milliseconds. This value is
     *         particularly important if the input_preference for the failover pair is set to PRIMARY_INPUT_PREFERRED,
     *         because after this time, MediaLive will switch back to the primary input.
     */
    public final Integer errorClearTimeMsec() {
        return errorClearTimeMsec;
    }

    /**
     * Returns true if the FailoverConditions property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasFailoverConditions() {
        return failoverConditions != null && !(failoverConditions instanceof SdkAutoConstructList);
    }

    /**
     * A list of failover conditions. If any of these conditions occur, MediaLive will perform a failover to the other
     * input.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasFailoverConditions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of failover conditions. If any of these conditions occur, MediaLive will perform a failover to the
     *         other input.
     */
    public final List<FailoverCondition> failoverConditions() {
        return failoverConditions;
    }

    /**
     * Input preference when deciding which input to make active when a previously failed input has recovered.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #inputPreference}
     * will return {@link InputPreference#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #inputPreferenceAsString}.
     * </p>
     * 
     * @return Input preference when deciding which input to make active when a previously failed input has recovered.
     * @see InputPreference
     */
    public final InputPreference inputPreference() {
        return InputPreference.fromValue(inputPreference);
    }

    /**
     * Input preference when deciding which input to make active when a previously failed input has recovered.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #inputPreference}
     * will return {@link InputPreference#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #inputPreferenceAsString}.
     * </p>
     * 
     * @return Input preference when deciding which input to make active when a previously failed input has recovered.
     * @see InputPreference
     */
    public final String inputPreferenceAsString() {
        return inputPreference;
    }

    /**
     * The input ID of the secondary input in the automatic input failover pair.
     * 
     * @return The input ID of the secondary input in the automatic input failover pair.
     */
    public final String secondaryInputId() {
        return secondaryInputId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(errorClearTimeMsec());
        hashCode = 31 * hashCode + Objects.hashCode(hasFailoverConditions() ? failoverConditions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(inputPreferenceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(secondaryInputId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutomaticInputFailoverSettings)) {
            return false;
        }
        AutomaticInputFailoverSettings other = (AutomaticInputFailoverSettings) obj;
        return Objects.equals(errorClearTimeMsec(), other.errorClearTimeMsec())
                && hasFailoverConditions() == other.hasFailoverConditions()
                && Objects.equals(failoverConditions(), other.failoverConditions())
                && Objects.equals(inputPreferenceAsString(), other.inputPreferenceAsString())
                && Objects.equals(secondaryInputId(), other.secondaryInputId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AutomaticInputFailoverSettings").add("ErrorClearTimeMsec", errorClearTimeMsec())
                .add("FailoverConditions", hasFailoverConditions() ? failoverConditions() : null)
                .add("InputPreference", inputPreferenceAsString()).add("SecondaryInputId", secondaryInputId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ErrorClearTimeMsec":
            return Optional.ofNullable(clazz.cast(errorClearTimeMsec()));
        case "FailoverConditions":
            return Optional.ofNullable(clazz.cast(failoverConditions()));
        case "InputPreference":
            return Optional.ofNullable(clazz.cast(inputPreferenceAsString()));
        case "SecondaryInputId":
            return Optional.ofNullable(clazz.cast(secondaryInputId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AutomaticInputFailoverSettings, T> g) {
        return obj -> g.apply((AutomaticInputFailoverSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutomaticInputFailoverSettings> {
        /**
         * This clear time defines the requirement a recovered input must meet to be considered healthy. The input must
         * have no failover conditions for this length of time. Enter a time in milliseconds. This value is particularly
         * important if the input_preference for the failover pair is set to PRIMARY_INPUT_PREFERRED, because after this
         * time, MediaLive will switch back to the primary input.
         * 
         * @param errorClearTimeMsec
         *        This clear time defines the requirement a recovered input must meet to be considered healthy. The
         *        input must have no failover conditions for this length of time. Enter a time in milliseconds. This
         *        value is particularly important if the input_preference for the failover pair is set to
         *        PRIMARY_INPUT_PREFERRED, because after this time, MediaLive will switch back to the primary input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorClearTimeMsec(Integer errorClearTimeMsec);

        /**
         * A list of failover conditions. If any of these conditions occur, MediaLive will perform a failover to the
         * other input.
         * 
         * @param failoverConditions
         *        A list of failover conditions. If any of these conditions occur, MediaLive will perform a failover to
         *        the other input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failoverConditions(Collection<FailoverCondition> failoverConditions);

        /**
         * A list of failover conditions. If any of these conditions occur, MediaLive will perform a failover to the
         * other input.
         * 
         * @param failoverConditions
         *        A list of failover conditions. If any of these conditions occur, MediaLive will perform a failover to
         *        the other input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failoverConditions(FailoverCondition... failoverConditions);

        /**
         * A list of failover conditions. If any of these conditions occur, MediaLive will perform a failover to the
         * other input. This is a convenience that creates an instance of the {@link List<FailoverCondition>.Builder}
         * avoiding the need to create one manually via {@link List<FailoverCondition>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<FailoverCondition>.Builder#build()} is called immediately
         * and its result is passed to {@link #failoverConditions(List<FailoverCondition>)}.
         * 
         * @param failoverConditions
         *        a consumer that will call methods on {@link List<FailoverCondition>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #failoverConditions(List<FailoverCondition>)
         */
        Builder failoverConditions(Consumer<FailoverCondition.Builder>... failoverConditions);

        /**
         * Input preference when deciding which input to make active when a previously failed input has recovered.
         * 
         * @param inputPreference
         *        Input preference when deciding which input to make active when a previously failed input has
         *        recovered.
         * @see InputPreference
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputPreference
         */
        Builder inputPreference(String inputPreference);

        /**
         * Input preference when deciding which input to make active when a previously failed input has recovered.
         * 
         * @param inputPreference
         *        Input preference when deciding which input to make active when a previously failed input has
         *        recovered.
         * @see InputPreference
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputPreference
         */
        Builder inputPreference(InputPreference inputPreference);

        /**
         * The input ID of the secondary input in the automatic input failover pair.
         * 
         * @param secondaryInputId
         *        The input ID of the secondary input in the automatic input failover pair.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondaryInputId(String secondaryInputId);
    }

    static final class BuilderImpl implements Builder {
        private Integer errorClearTimeMsec;

        private List<FailoverCondition> failoverConditions = DefaultSdkAutoConstructList.getInstance();

        private String inputPreference;

        private String secondaryInputId;

        private BuilderImpl() {
        }

        private BuilderImpl(AutomaticInputFailoverSettings model) {
            errorClearTimeMsec(model.errorClearTimeMsec);
            failoverConditions(model.failoverConditions);
            inputPreference(model.inputPreference);
            secondaryInputId(model.secondaryInputId);
        }

        public final Integer getErrorClearTimeMsec() {
            return errorClearTimeMsec;
        }

        @Override
        public final Builder errorClearTimeMsec(Integer errorClearTimeMsec) {
            this.errorClearTimeMsec = errorClearTimeMsec;
            return this;
        }

        public final void setErrorClearTimeMsec(Integer errorClearTimeMsec) {
            this.errorClearTimeMsec = errorClearTimeMsec;
        }

        public final Collection<FailoverCondition.Builder> getFailoverConditions() {
            if (failoverConditions instanceof SdkAutoConstructList) {
                return null;
            }
            return failoverConditions != null ? failoverConditions.stream().map(FailoverCondition::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder failoverConditions(Collection<FailoverCondition> failoverConditions) {
            this.failoverConditions = ___listOfFailoverConditionCopier.copy(failoverConditions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder failoverConditions(FailoverCondition... failoverConditions) {
            failoverConditions(Arrays.asList(failoverConditions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder failoverConditions(Consumer<FailoverCondition.Builder>... failoverConditions) {
            failoverConditions(Stream.of(failoverConditions).map(c -> FailoverCondition.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setFailoverConditions(Collection<FailoverCondition.BuilderImpl> failoverConditions) {
            this.failoverConditions = ___listOfFailoverConditionCopier.copyFromBuilder(failoverConditions);
        }

        public final String getInputPreference() {
            return inputPreference;
        }

        @Override
        public final Builder inputPreference(String inputPreference) {
            this.inputPreference = inputPreference;
            return this;
        }

        @Override
        public final Builder inputPreference(InputPreference inputPreference) {
            this.inputPreference(inputPreference == null ? null : inputPreference.toString());
            return this;
        }

        public final void setInputPreference(String inputPreference) {
            this.inputPreference = inputPreference;
        }

        public final String getSecondaryInputId() {
            return secondaryInputId;
        }

        @Override
        public final Builder secondaryInputId(String secondaryInputId) {
            this.secondaryInputId = secondaryInputId;
            return this;
        }

        public final void setSecondaryInputId(String secondaryInputId) {
            this.secondaryInputId = secondaryInputId;
        }

        @Override
        public AutomaticInputFailoverSettings build() {
            return new AutomaticInputFailoverSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
