/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconvert.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * MPEG-2 TS container settings. These apply to outputs in a File output group when the output's container
 * (ContainerType) is MPEG-2 Transport Stream (M2TS). In these assets, data is organized by the program map table (PMT).
 * Each transport stream program contains subsets of data, including audio, video, and metadata. Each of these subsets
 * of data has a numerical label called a packet identifier (PID). Each transport stream program corresponds to one
 * MediaConvert output. The PMT lists the types of data in a program along with their PID. Downstream systems and
 * players use the program map table to look up the PID for each type of data it accesses and then uses the PIDs to
 * locate specific data within the asset.
 */
@Generated("software.amazon.awssdk:codegen")
public final class M2tsSettings implements SdkPojo, Serializable, ToCopyableBuilder<M2tsSettings.Builder, M2tsSettings> {
    private static final SdkField<String> AUDIO_BUFFER_MODEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(M2tsSettings::audioBufferModelAsString)).setter(setter(Builder::audioBufferModel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("audioBufferModel").build()).build();

    private static final SdkField<Integer> AUDIO_FRAMES_PER_PES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(M2tsSettings::audioFramesPerPes)).setter(setter(Builder::audioFramesPerPes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("audioFramesPerPes").build()).build();

    private static final SdkField<List<Integer>> AUDIO_PIDS_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .getter(getter(M2tsSettings::audioPids))
            .setter(setter(Builder::audioPids))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("audioPids").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> BITRATE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(M2tsSettings::bitrate)).setter(setter(Builder::bitrate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bitrate").build()).build();

    private static final SdkField<String> BUFFER_MODEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(M2tsSettings::bufferModelAsString)).setter(setter(Builder::bufferModel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bufferModel").build()).build();

    private static final SdkField<DvbNitSettings> DVB_NIT_SETTINGS_FIELD = SdkField
            .<DvbNitSettings> builder(MarshallingType.SDK_POJO).getter(getter(M2tsSettings::dvbNitSettings))
            .setter(setter(Builder::dvbNitSettings)).constructor(DvbNitSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dvbNitSettings").build()).build();

    private static final SdkField<DvbSdtSettings> DVB_SDT_SETTINGS_FIELD = SdkField
            .<DvbSdtSettings> builder(MarshallingType.SDK_POJO).getter(getter(M2tsSettings::dvbSdtSettings))
            .setter(setter(Builder::dvbSdtSettings)).constructor(DvbSdtSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dvbSdtSettings").build()).build();

    private static final SdkField<List<Integer>> DVB_SUB_PIDS_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .getter(getter(M2tsSettings::dvbSubPids))
            .setter(setter(Builder::dvbSubPids))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dvbSubPids").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DvbTdtSettings> DVB_TDT_SETTINGS_FIELD = SdkField
            .<DvbTdtSettings> builder(MarshallingType.SDK_POJO).getter(getter(M2tsSettings::dvbTdtSettings))
            .setter(setter(Builder::dvbTdtSettings)).constructor(DvbTdtSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dvbTdtSettings").build()).build();

    private static final SdkField<Integer> DVB_TELETEXT_PID_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(M2tsSettings::dvbTeletextPid)).setter(setter(Builder::dvbTeletextPid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dvbTeletextPid").build()).build();

    private static final SdkField<String> EBP_AUDIO_INTERVAL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(M2tsSettings::ebpAudioIntervalAsString)).setter(setter(Builder::ebpAudioInterval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ebpAudioInterval").build()).build();

    private static final SdkField<String> EBP_PLACEMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(M2tsSettings::ebpPlacementAsString)).setter(setter(Builder::ebpPlacement))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ebpPlacement").build()).build();

    private static final SdkField<String> ES_RATE_IN_PES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(M2tsSettings::esRateInPesAsString)).setter(setter(Builder::esRateInPes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("esRateInPes").build()).build();

    private static final SdkField<String> FORCE_TS_VIDEO_EBP_ORDER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(M2tsSettings::forceTsVideoEbpOrderAsString)).setter(setter(Builder::forceTsVideoEbpOrder))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("forceTsVideoEbpOrder").build())
            .build();

    private static final SdkField<Double> FRAGMENT_TIME_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(M2tsSettings::fragmentTime)).setter(setter(Builder::fragmentTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fragmentTime").build()).build();

    private static final SdkField<Integer> MAX_PCR_INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(M2tsSettings::maxPcrInterval)).setter(setter(Builder::maxPcrInterval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxPcrInterval").build()).build();

    private static final SdkField<Integer> MIN_EBP_INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(M2tsSettings::minEbpInterval)).setter(setter(Builder::minEbpInterval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("minEbpInterval").build()).build();

    private static final SdkField<String> NIELSEN_ID3_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(M2tsSettings::nielsenId3AsString)).setter(setter(Builder::nielsenId3))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nielsenId3").build()).build();

    private static final SdkField<Double> NULL_PACKET_BITRATE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(M2tsSettings::nullPacketBitrate)).setter(setter(Builder::nullPacketBitrate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nullPacketBitrate").build()).build();

    private static final SdkField<Integer> PAT_INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(M2tsSettings::patInterval)).setter(setter(Builder::patInterval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("patInterval").build()).build();

    private static final SdkField<String> PCR_CONTROL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(M2tsSettings::pcrControlAsString)).setter(setter(Builder::pcrControl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pcrControl").build()).build();

    private static final SdkField<Integer> PCR_PID_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(M2tsSettings::pcrPid)).setter(setter(Builder::pcrPid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pcrPid").build()).build();

    private static final SdkField<Integer> PMT_INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(M2tsSettings::pmtInterval)).setter(setter(Builder::pmtInterval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pmtInterval").build()).build();

    private static final SdkField<Integer> PMT_PID_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(M2tsSettings::pmtPid)).setter(setter(Builder::pmtPid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pmtPid").build()).build();

    private static final SdkField<Integer> PRIVATE_METADATA_PID_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(M2tsSettings::privateMetadataPid)).setter(setter(Builder::privateMetadataPid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("privateMetadataPid").build())
            .build();

    private static final SdkField<Integer> PROGRAM_NUMBER_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(M2tsSettings::programNumber)).setter(setter(Builder::programNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("programNumber").build()).build();

    private static final SdkField<String> RATE_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(M2tsSettings::rateModeAsString)).setter(setter(Builder::rateMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rateMode").build()).build();

    private static final SdkField<M2tsScte35Esam> SCTE35_ESAM_FIELD = SdkField.<M2tsScte35Esam> builder(MarshallingType.SDK_POJO)
            .getter(getter(M2tsSettings::scte35Esam)).setter(setter(Builder::scte35Esam)).constructor(M2tsScte35Esam::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scte35Esam").build()).build();

    private static final SdkField<Integer> SCTE35_PID_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(M2tsSettings::scte35Pid)).setter(setter(Builder::scte35Pid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scte35Pid").build()).build();

    private static final SdkField<String> SCTE35_SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(M2tsSettings::scte35SourceAsString)).setter(setter(Builder::scte35Source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scte35Source").build()).build();

    private static final SdkField<String> SEGMENTATION_MARKERS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(M2tsSettings::segmentationMarkersAsString)).setter(setter(Builder::segmentationMarkers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("segmentationMarkers").build())
            .build();

    private static final SdkField<String> SEGMENTATION_STYLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(M2tsSettings::segmentationStyleAsString)).setter(setter(Builder::segmentationStyle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("segmentationStyle").build()).build();

    private static final SdkField<Double> SEGMENTATION_TIME_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(M2tsSettings::segmentationTime)).setter(setter(Builder::segmentationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("segmentationTime").build()).build();

    private static final SdkField<Integer> TIMED_METADATA_PID_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(M2tsSettings::timedMetadataPid)).setter(setter(Builder::timedMetadataPid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timedMetadataPid").build()).build();

    private static final SdkField<Integer> TRANSPORT_STREAM_ID_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(M2tsSettings::transportStreamId)).setter(setter(Builder::transportStreamId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("transportStreamId").build()).build();

    private static final SdkField<Integer> VIDEO_PID_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(M2tsSettings::videoPid)).setter(setter(Builder::videoPid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("videoPid").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AUDIO_BUFFER_MODEL_FIELD,
            AUDIO_FRAMES_PER_PES_FIELD, AUDIO_PIDS_FIELD, BITRATE_FIELD, BUFFER_MODEL_FIELD, DVB_NIT_SETTINGS_FIELD,
            DVB_SDT_SETTINGS_FIELD, DVB_SUB_PIDS_FIELD, DVB_TDT_SETTINGS_FIELD, DVB_TELETEXT_PID_FIELD, EBP_AUDIO_INTERVAL_FIELD,
            EBP_PLACEMENT_FIELD, ES_RATE_IN_PES_FIELD, FORCE_TS_VIDEO_EBP_ORDER_FIELD, FRAGMENT_TIME_FIELD,
            MAX_PCR_INTERVAL_FIELD, MIN_EBP_INTERVAL_FIELD, NIELSEN_ID3_FIELD, NULL_PACKET_BITRATE_FIELD, PAT_INTERVAL_FIELD,
            PCR_CONTROL_FIELD, PCR_PID_FIELD, PMT_INTERVAL_FIELD, PMT_PID_FIELD, PRIVATE_METADATA_PID_FIELD,
            PROGRAM_NUMBER_FIELD, RATE_MODE_FIELD, SCTE35_ESAM_FIELD, SCTE35_PID_FIELD, SCTE35_SOURCE_FIELD,
            SEGMENTATION_MARKERS_FIELD, SEGMENTATION_STYLE_FIELD, SEGMENTATION_TIME_FIELD, TIMED_METADATA_PID_FIELD,
            TRANSPORT_STREAM_ID_FIELD, VIDEO_PID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String audioBufferModel;

    private final Integer audioFramesPerPes;

    private final List<Integer> audioPids;

    private final Integer bitrate;

    private final String bufferModel;

    private final DvbNitSettings dvbNitSettings;

    private final DvbSdtSettings dvbSdtSettings;

    private final List<Integer> dvbSubPids;

    private final DvbTdtSettings dvbTdtSettings;

    private final Integer dvbTeletextPid;

    private final String ebpAudioInterval;

    private final String ebpPlacement;

    private final String esRateInPes;

    private final String forceTsVideoEbpOrder;

    private final Double fragmentTime;

    private final Integer maxPcrInterval;

    private final Integer minEbpInterval;

    private final String nielsenId3;

    private final Double nullPacketBitrate;

    private final Integer patInterval;

    private final String pcrControl;

    private final Integer pcrPid;

    private final Integer pmtInterval;

    private final Integer pmtPid;

    private final Integer privateMetadataPid;

    private final Integer programNumber;

    private final String rateMode;

    private final M2tsScte35Esam scte35Esam;

    private final Integer scte35Pid;

    private final String scte35Source;

    private final String segmentationMarkers;

    private final String segmentationStyle;

    private final Double segmentationTime;

    private final Integer timedMetadataPid;

    private final Integer transportStreamId;

    private final Integer videoPid;

    private M2tsSettings(BuilderImpl builder) {
        this.audioBufferModel = builder.audioBufferModel;
        this.audioFramesPerPes = builder.audioFramesPerPes;
        this.audioPids = builder.audioPids;
        this.bitrate = builder.bitrate;
        this.bufferModel = builder.bufferModel;
        this.dvbNitSettings = builder.dvbNitSettings;
        this.dvbSdtSettings = builder.dvbSdtSettings;
        this.dvbSubPids = builder.dvbSubPids;
        this.dvbTdtSettings = builder.dvbTdtSettings;
        this.dvbTeletextPid = builder.dvbTeletextPid;
        this.ebpAudioInterval = builder.ebpAudioInterval;
        this.ebpPlacement = builder.ebpPlacement;
        this.esRateInPes = builder.esRateInPes;
        this.forceTsVideoEbpOrder = builder.forceTsVideoEbpOrder;
        this.fragmentTime = builder.fragmentTime;
        this.maxPcrInterval = builder.maxPcrInterval;
        this.minEbpInterval = builder.minEbpInterval;
        this.nielsenId3 = builder.nielsenId3;
        this.nullPacketBitrate = builder.nullPacketBitrate;
        this.patInterval = builder.patInterval;
        this.pcrControl = builder.pcrControl;
        this.pcrPid = builder.pcrPid;
        this.pmtInterval = builder.pmtInterval;
        this.pmtPid = builder.pmtPid;
        this.privateMetadataPid = builder.privateMetadataPid;
        this.programNumber = builder.programNumber;
        this.rateMode = builder.rateMode;
        this.scte35Esam = builder.scte35Esam;
        this.scte35Pid = builder.scte35Pid;
        this.scte35Source = builder.scte35Source;
        this.segmentationMarkers = builder.segmentationMarkers;
        this.segmentationStyle = builder.segmentationStyle;
        this.segmentationTime = builder.segmentationTime;
        this.timedMetadataPid = builder.timedMetadataPid;
        this.transportStreamId = builder.transportStreamId;
        this.videoPid = builder.videoPid;
    }

    /**
     * Selects between the DVB and ATSC buffer models for Dolby Digital audio.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #audioBufferModel}
     * will return {@link M2tsAudioBufferModel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #audioBufferModelAsString}.
     * </p>
     * 
     * @return Selects between the DVB and ATSC buffer models for Dolby Digital audio.
     * @see M2tsAudioBufferModel
     */
    public M2tsAudioBufferModel audioBufferModel() {
        return M2tsAudioBufferModel.fromValue(audioBufferModel);
    }

    /**
     * Selects between the DVB and ATSC buffer models for Dolby Digital audio.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #audioBufferModel}
     * will return {@link M2tsAudioBufferModel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #audioBufferModelAsString}.
     * </p>
     * 
     * @return Selects between the DVB and ATSC buffer models for Dolby Digital audio.
     * @see M2tsAudioBufferModel
     */
    public String audioBufferModelAsString() {
        return audioBufferModel;
    }

    /**
     * The number of audio frames to insert for each PES packet.
     * 
     * @return The number of audio frames to insert for each PES packet.
     */
    public Integer audioFramesPerPes() {
        return audioFramesPerPes;
    }

    /**
     * Returns true if the AudioPids property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasAudioPids() {
        return audioPids != null && !(audioPids instanceof SdkAutoConstructList);
    }

    /**
     * Specify the packet identifiers (PIDs) for any elementary audio streams you include in this output. Specify
     * multiple PIDs as a JSON array. Default is the range 482-492.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAudioPids()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specify the packet identifiers (PIDs) for any elementary audio streams you include in this output.
     *         Specify multiple PIDs as a JSON array. Default is the range 482-492.
     */
    public List<Integer> audioPids() {
        return audioPids;
    }

    /**
     * Specify the output bitrate of the transport stream in bits per second. Setting to 0 lets the muxer automatically
     * determine the appropriate bitrate. Other common values are 3750000, 7500000, and 15000000.
     * 
     * @return Specify the output bitrate of the transport stream in bits per second. Setting to 0 lets the muxer
     *         automatically determine the appropriate bitrate. Other common values are 3750000, 7500000, and 15000000.
     */
    public Integer bitrate() {
        return bitrate;
    }

    /**
     * Controls what buffer model to use for accurate interleaving. If set to MULTIPLEX, use multiplex buffer model. If
     * set to NONE, this can lead to lower latency, but low-memory devices may not be able to play back the stream
     * without interruptions.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #bufferModel} will
     * return {@link M2tsBufferModel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #bufferModelAsString}.
     * </p>
     * 
     * @return Controls what buffer model to use for accurate interleaving. If set to MULTIPLEX, use multiplex buffer
     *         model. If set to NONE, this can lead to lower latency, but low-memory devices may not be able to play
     *         back the stream without interruptions.
     * @see M2tsBufferModel
     */
    public M2tsBufferModel bufferModel() {
        return M2tsBufferModel.fromValue(bufferModel);
    }

    /**
     * Controls what buffer model to use for accurate interleaving. If set to MULTIPLEX, use multiplex buffer model. If
     * set to NONE, this can lead to lower latency, but low-memory devices may not be able to play back the stream
     * without interruptions.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #bufferModel} will
     * return {@link M2tsBufferModel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #bufferModelAsString}.
     * </p>
     * 
     * @return Controls what buffer model to use for accurate interleaving. If set to MULTIPLEX, use multiplex buffer
     *         model. If set to NONE, this can lead to lower latency, but low-memory devices may not be able to play
     *         back the stream without interruptions.
     * @see M2tsBufferModel
     */
    public String bufferModelAsString() {
        return bufferModel;
    }

    /**
     * Inserts DVB Network Information Table (NIT) at the specified table repetition interval.
     * 
     * @return Inserts DVB Network Information Table (NIT) at the specified table repetition interval.
     */
    public DvbNitSettings dvbNitSettings() {
        return dvbNitSettings;
    }

    /**
     * Inserts DVB Service Description Table (NIT) at the specified table repetition interval.
     * 
     * @return Inserts DVB Service Description Table (NIT) at the specified table repetition interval.
     */
    public DvbSdtSettings dvbSdtSettings() {
        return dvbSdtSettings;
    }

    /**
     * Returns true if the DvbSubPids property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDvbSubPids() {
        return dvbSubPids != null && !(dvbSubPids instanceof SdkAutoConstructList);
    }

    /**
     * Specify the packet identifiers (PIDs) for DVB subtitle data included in this output. Specify multiple PIDs as a
     * JSON array. Default is the range 460-479.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDvbSubPids()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specify the packet identifiers (PIDs) for DVB subtitle data included in this output. Specify multiple
     *         PIDs as a JSON array. Default is the range 460-479.
     */
    public List<Integer> dvbSubPids() {
        return dvbSubPids;
    }

    /**
     * Inserts DVB Time and Date Table (TDT) at the specified table repetition interval.
     * 
     * @return Inserts DVB Time and Date Table (TDT) at the specified table repetition interval.
     */
    public DvbTdtSettings dvbTdtSettings() {
        return dvbTdtSettings;
    }

    /**
     * Specify the packet identifier (PID) for DVB teletext data you include in this output. Default is 499.
     * 
     * @return Specify the packet identifier (PID) for DVB teletext data you include in this output. Default is 499.
     */
    public Integer dvbTeletextPid() {
        return dvbTeletextPid;
    }

    /**
     * When set to VIDEO_AND_FIXED_INTERVALS, audio EBP markers will be added to partitions 3 and 4. The interval
     * between these additional markers will be fixed, and will be slightly shorter than the video EBP marker interval.
     * When set to VIDEO_INTERVAL, these additional markers will not be inserted. Only applicable when EBP segmentation
     * markers are is selected (segmentationMarkers is EBP or EBP_LEGACY).
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ebpAudioInterval}
     * will return {@link M2tsEbpAudioInterval#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #ebpAudioIntervalAsString}.
     * </p>
     * 
     * @return When set to VIDEO_AND_FIXED_INTERVALS, audio EBP markers will be added to partitions 3 and 4. The
     *         interval between these additional markers will be fixed, and will be slightly shorter than the video EBP
     *         marker interval. When set to VIDEO_INTERVAL, these additional markers will not be inserted. Only
     *         applicable when EBP segmentation markers are is selected (segmentationMarkers is EBP or EBP_LEGACY).
     * @see M2tsEbpAudioInterval
     */
    public M2tsEbpAudioInterval ebpAudioInterval() {
        return M2tsEbpAudioInterval.fromValue(ebpAudioInterval);
    }

    /**
     * When set to VIDEO_AND_FIXED_INTERVALS, audio EBP markers will be added to partitions 3 and 4. The interval
     * between these additional markers will be fixed, and will be slightly shorter than the video EBP marker interval.
     * When set to VIDEO_INTERVAL, these additional markers will not be inserted. Only applicable when EBP segmentation
     * markers are is selected (segmentationMarkers is EBP or EBP_LEGACY).
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ebpAudioInterval}
     * will return {@link M2tsEbpAudioInterval#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #ebpAudioIntervalAsString}.
     * </p>
     * 
     * @return When set to VIDEO_AND_FIXED_INTERVALS, audio EBP markers will be added to partitions 3 and 4. The
     *         interval between these additional markers will be fixed, and will be slightly shorter than the video EBP
     *         marker interval. When set to VIDEO_INTERVAL, these additional markers will not be inserted. Only
     *         applicable when EBP segmentation markers are is selected (segmentationMarkers is EBP or EBP_LEGACY).
     * @see M2tsEbpAudioInterval
     */
    public String ebpAudioIntervalAsString() {
        return ebpAudioInterval;
    }

    /**
     * Selects which PIDs to place EBP markers on. They can either be placed only on the video PID, or on both the video
     * PID and all audio PIDs. Only applicable when EBP segmentation markers are is selected (segmentationMarkers is EBP
     * or EBP_LEGACY).
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ebpPlacement} will
     * return {@link M2tsEbpPlacement#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ebpPlacementAsString}.
     * </p>
     * 
     * @return Selects which PIDs to place EBP markers on. They can either be placed only on the video PID, or on both
     *         the video PID and all audio PIDs. Only applicable when EBP segmentation markers are is selected
     *         (segmentationMarkers is EBP or EBP_LEGACY).
     * @see M2tsEbpPlacement
     */
    public M2tsEbpPlacement ebpPlacement() {
        return M2tsEbpPlacement.fromValue(ebpPlacement);
    }

    /**
     * Selects which PIDs to place EBP markers on. They can either be placed only on the video PID, or on both the video
     * PID and all audio PIDs. Only applicable when EBP segmentation markers are is selected (segmentationMarkers is EBP
     * or EBP_LEGACY).
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ebpPlacement} will
     * return {@link M2tsEbpPlacement#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ebpPlacementAsString}.
     * </p>
     * 
     * @return Selects which PIDs to place EBP markers on. They can either be placed only on the video PID, or on both
     *         the video PID and all audio PIDs. Only applicable when EBP segmentation markers are is selected
     *         (segmentationMarkers is EBP or EBP_LEGACY).
     * @see M2tsEbpPlacement
     */
    public String ebpPlacementAsString() {
        return ebpPlacement;
    }

    /**
     * Controls whether to include the ES Rate field in the PES header.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #esRateInPes} will
     * return {@link M2tsEsRateInPes#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #esRateInPesAsString}.
     * </p>
     * 
     * @return Controls whether to include the ES Rate field in the PES header.
     * @see M2tsEsRateInPes
     */
    public M2tsEsRateInPes esRateInPes() {
        return M2tsEsRateInPes.fromValue(esRateInPes);
    }

    /**
     * Controls whether to include the ES Rate field in the PES header.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #esRateInPes} will
     * return {@link M2tsEsRateInPes#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #esRateInPesAsString}.
     * </p>
     * 
     * @return Controls whether to include the ES Rate field in the PES header.
     * @see M2tsEsRateInPes
     */
    public String esRateInPesAsString() {
        return esRateInPes;
    }

    /**
     * Keep the default value (DEFAULT) unless you know that your audio EBP markers are incorrectly appearing before
     * your video EBP markers. To correct this problem, set this value to Force (FORCE).
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #forceTsVideoEbpOrder} will return {@link M2tsForceTsVideoEbpOrder#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #forceTsVideoEbpOrderAsString}.
     * </p>
     * 
     * @return Keep the default value (DEFAULT) unless you know that your audio EBP markers are incorrectly appearing
     *         before your video EBP markers. To correct this problem, set this value to Force (FORCE).
     * @see M2tsForceTsVideoEbpOrder
     */
    public M2tsForceTsVideoEbpOrder forceTsVideoEbpOrder() {
        return M2tsForceTsVideoEbpOrder.fromValue(forceTsVideoEbpOrder);
    }

    /**
     * Keep the default value (DEFAULT) unless you know that your audio EBP markers are incorrectly appearing before
     * your video EBP markers. To correct this problem, set this value to Force (FORCE).
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #forceTsVideoEbpOrder} will return {@link M2tsForceTsVideoEbpOrder#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #forceTsVideoEbpOrderAsString}.
     * </p>
     * 
     * @return Keep the default value (DEFAULT) unless you know that your audio EBP markers are incorrectly appearing
     *         before your video EBP markers. To correct this problem, set this value to Force (FORCE).
     * @see M2tsForceTsVideoEbpOrder
     */
    public String forceTsVideoEbpOrderAsString() {
        return forceTsVideoEbpOrder;
    }

    /**
     * The length, in seconds, of each fragment. Only used with EBP markers.
     * 
     * @return The length, in seconds, of each fragment. Only used with EBP markers.
     */
    public Double fragmentTime() {
        return fragmentTime;
    }

    /**
     * Specify the maximum time, in milliseconds, between Program Clock References (PCRs) inserted into the transport
     * stream.
     * 
     * @return Specify the maximum time, in milliseconds, between Program Clock References (PCRs) inserted into the
     *         transport stream.
     */
    public Integer maxPcrInterval() {
        return maxPcrInterval;
    }

    /**
     * When set, enforces that Encoder Boundary Points do not come within the specified time interval of each other by
     * looking ahead at input video. If another EBP is going to come in within the specified time interval, the current
     * EBP is not emitted, and the segment is "stretched" to the next marker. The lookahead value does not add latency
     * to the system. The Live Event must be configured elsewhere to create sufficient latency to make the lookahead
     * accurate.
     * 
     * @return When set, enforces that Encoder Boundary Points do not come within the specified time interval of each
     *         other by looking ahead at input video. If another EBP is going to come in within the specified time
     *         interval, the current EBP is not emitted, and the segment is "stretched" to the next marker. The
     *         lookahead value does not add latency to the system. The Live Event must be configured elsewhere to create
     *         sufficient latency to make the lookahead accurate.
     */
    public Integer minEbpInterval() {
        return minEbpInterval;
    }

    /**
     * If INSERT, Nielsen inaudible tones for media tracking will be detected in the input audio and an equivalent ID3
     * tag will be inserted in the output.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #nielsenId3} will
     * return {@link M2tsNielsenId3#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nielsenId3AsString}.
     * </p>
     * 
     * @return If INSERT, Nielsen inaudible tones for media tracking will be detected in the input audio and an
     *         equivalent ID3 tag will be inserted in the output.
     * @see M2tsNielsenId3
     */
    public M2tsNielsenId3 nielsenId3() {
        return M2tsNielsenId3.fromValue(nielsenId3);
    }

    /**
     * If INSERT, Nielsen inaudible tones for media tracking will be detected in the input audio and an equivalent ID3
     * tag will be inserted in the output.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #nielsenId3} will
     * return {@link M2tsNielsenId3#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nielsenId3AsString}.
     * </p>
     * 
     * @return If INSERT, Nielsen inaudible tones for media tracking will be detected in the input audio and an
     *         equivalent ID3 tag will be inserted in the output.
     * @see M2tsNielsenId3
     */
    public String nielsenId3AsString() {
        return nielsenId3;
    }

    /**
     * Value in bits per second of extra null packets to insert into the transport stream. This can be used if a
     * downstream encryption system requires periodic null packets.
     * 
     * @return Value in bits per second of extra null packets to insert into the transport stream. This can be used if a
     *         downstream encryption system requires periodic null packets.
     */
    public Double nullPacketBitrate() {
        return nullPacketBitrate;
    }

    /**
     * The number of milliseconds between instances of this table in the output transport stream.
     * 
     * @return The number of milliseconds between instances of this table in the output transport stream.
     */
    public Integer patInterval() {
        return patInterval;
    }

    /**
     * When set to PCR_EVERY_PES_PACKET, a Program Clock Reference value is inserted for every Packetized Elementary
     * Stream (PES) header. This is effective only when the PCR PID is the same as the video or audio elementary stream.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #pcrControl} will
     * return {@link M2tsPcrControl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #pcrControlAsString}.
     * </p>
     * 
     * @return When set to PCR_EVERY_PES_PACKET, a Program Clock Reference value is inserted for every Packetized
     *         Elementary Stream (PES) header. This is effective only when the PCR PID is the same as the video or audio
     *         elementary stream.
     * @see M2tsPcrControl
     */
    public M2tsPcrControl pcrControl() {
        return M2tsPcrControl.fromValue(pcrControl);
    }

    /**
     * When set to PCR_EVERY_PES_PACKET, a Program Clock Reference value is inserted for every Packetized Elementary
     * Stream (PES) header. This is effective only when the PCR PID is the same as the video or audio elementary stream.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #pcrControl} will
     * return {@link M2tsPcrControl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #pcrControlAsString}.
     * </p>
     * 
     * @return When set to PCR_EVERY_PES_PACKET, a Program Clock Reference value is inserted for every Packetized
     *         Elementary Stream (PES) header. This is effective only when the PCR PID is the same as the video or audio
     *         elementary stream.
     * @see M2tsPcrControl
     */
    public String pcrControlAsString() {
        return pcrControl;
    }

    /**
     * Specify the packet identifier (PID) for the program clock reference (PCR) in this output. If you do not specify a
     * value, the service will use the value for Video PID (VideoPid).
     * 
     * @return Specify the packet identifier (PID) for the program clock reference (PCR) in this output. If you do not
     *         specify a value, the service will use the value for Video PID (VideoPid).
     */
    public Integer pcrPid() {
        return pcrPid;
    }

    /**
     * Specify the number of milliseconds between instances of the program map table (PMT) in the output transport
     * stream.
     * 
     * @return Specify the number of milliseconds between instances of the program map table (PMT) in the output
     *         transport stream.
     */
    public Integer pmtInterval() {
        return pmtInterval;
    }

    /**
     * Specify the packet identifier (PID) for the program map table (PMT) itself. Default is 480.
     * 
     * @return Specify the packet identifier (PID) for the program map table (PMT) itself. Default is 480.
     */
    public Integer pmtPid() {
        return pmtPid;
    }

    /**
     * Specify the packet identifier (PID) of the private metadata stream. Default is 503.
     * 
     * @return Specify the packet identifier (PID) of the private metadata stream. Default is 503.
     */
    public Integer privateMetadataPid() {
        return privateMetadataPid;
    }

    /**
     * Use Program number (programNumber) to specify the program number used in the program map table (PMT) for this
     * output. Default is 1. Program numbers and program map tables are parts of MPEG-2 transport stream containers,
     * used for organizing data.
     * 
     * @return Use Program number (programNumber) to specify the program number used in the program map table (PMT) for
     *         this output. Default is 1. Program numbers and program map tables are parts of MPEG-2 transport stream
     *         containers, used for organizing data.
     */
    public Integer programNumber() {
        return programNumber;
    }

    /**
     * When set to CBR, inserts null packets into transport stream to fill specified bitrate. When set to VBR, the
     * bitrate setting acts as the maximum bitrate, but the output will not be padded up to that bitrate.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #rateMode} will
     * return {@link M2tsRateMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #rateModeAsString}.
     * </p>
     * 
     * @return When set to CBR, inserts null packets into transport stream to fill specified bitrate. When set to VBR,
     *         the bitrate setting acts as the maximum bitrate, but the output will not be padded up to that bitrate.
     * @see M2tsRateMode
     */
    public M2tsRateMode rateMode() {
        return M2tsRateMode.fromValue(rateMode);
    }

    /**
     * When set to CBR, inserts null packets into transport stream to fill specified bitrate. When set to VBR, the
     * bitrate setting acts as the maximum bitrate, but the output will not be padded up to that bitrate.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #rateMode} will
     * return {@link M2tsRateMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #rateModeAsString}.
     * </p>
     * 
     * @return When set to CBR, inserts null packets into transport stream to fill specified bitrate. When set to VBR,
     *         the bitrate setting acts as the maximum bitrate, but the output will not be padded up to that bitrate.
     * @see M2tsRateMode
     */
    public String rateModeAsString() {
        return rateMode;
    }

    /**
     * Include this in your job settings to put SCTE-35 markers in your HLS and transport stream outputs at the
     * insertion points that you specify in an ESAM XML document. Provide the document in the setting SCC XML (sccXml).
     * 
     * @return Include this in your job settings to put SCTE-35 markers in your HLS and transport stream outputs at the
     *         insertion points that you specify in an ESAM XML document. Provide the document in the setting SCC XML
     *         (sccXml).
     */
    public M2tsScte35Esam scte35Esam() {
        return scte35Esam;
    }

    /**
     * Specify the packet identifier (PID) of the SCTE-35 stream in the transport stream.
     * 
     * @return Specify the packet identifier (PID) of the SCTE-35 stream in the transport stream.
     */
    public Integer scte35Pid() {
        return scte35Pid;
    }

    /**
     * For SCTE-35 markers from your input-- Choose Passthrough (PASSTHROUGH) if you want SCTE-35 markers that appear in
     * your input to also appear in this output. Choose None (NONE) if you don't want SCTE-35 markers in this output.
     * For SCTE-35 markers from an ESAM XML document-- Choose None (NONE). Also provide the ESAM XML as a string in the
     * setting Signal processing notification XML (sccXml). Also enable ESAM SCTE-35 (include the property scte35Esam).
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scte35Source} will
     * return {@link M2tsScte35Source#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scte35SourceAsString}.
     * </p>
     * 
     * @return For SCTE-35 markers from your input-- Choose Passthrough (PASSTHROUGH) if you want SCTE-35 markers that
     *         appear in your input to also appear in this output. Choose None (NONE) if you don't want SCTE-35 markers
     *         in this output. For SCTE-35 markers from an ESAM XML document-- Choose None (NONE). Also provide the ESAM
     *         XML as a string in the setting Signal processing notification XML (sccXml). Also enable ESAM SCTE-35
     *         (include the property scte35Esam).
     * @see M2tsScte35Source
     */
    public M2tsScte35Source scte35Source() {
        return M2tsScte35Source.fromValue(scte35Source);
    }

    /**
     * For SCTE-35 markers from your input-- Choose Passthrough (PASSTHROUGH) if you want SCTE-35 markers that appear in
     * your input to also appear in this output. Choose None (NONE) if you don't want SCTE-35 markers in this output.
     * For SCTE-35 markers from an ESAM XML document-- Choose None (NONE). Also provide the ESAM XML as a string in the
     * setting Signal processing notification XML (sccXml). Also enable ESAM SCTE-35 (include the property scte35Esam).
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scte35Source} will
     * return {@link M2tsScte35Source#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scte35SourceAsString}.
     * </p>
     * 
     * @return For SCTE-35 markers from your input-- Choose Passthrough (PASSTHROUGH) if you want SCTE-35 markers that
     *         appear in your input to also appear in this output. Choose None (NONE) if you don't want SCTE-35 markers
     *         in this output. For SCTE-35 markers from an ESAM XML document-- Choose None (NONE). Also provide the ESAM
     *         XML as a string in the setting Signal processing notification XML (sccXml). Also enable ESAM SCTE-35
     *         (include the property scte35Esam).
     * @see M2tsScte35Source
     */
    public String scte35SourceAsString() {
        return scte35Source;
    }

    /**
     * Inserts segmentation markers at each segmentation_time period. rai_segstart sets the Random Access Indicator bit
     * in the adaptation field. rai_adapt sets the RAI bit and adds the current timecode in the private data bytes.
     * psi_segstart inserts PAT and PMT tables at the start of segments. ebp adds Encoder Boundary Point information to
     * the adaptation field as per OpenCable specification OC-SP-EBP-I01-130118. ebp_legacy adds Encoder Boundary Point
     * information to the adaptation field using a legacy proprietary format.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #segmentationMarkers} will return {@link M2tsSegmentationMarkers#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #segmentationMarkersAsString}.
     * </p>
     * 
     * @return Inserts segmentation markers at each segmentation_time period. rai_segstart sets the Random Access
     *         Indicator bit in the adaptation field. rai_adapt sets the RAI bit and adds the current timecode in the
     *         private data bytes. psi_segstart inserts PAT and PMT tables at the start of segments. ebp adds Encoder
     *         Boundary Point information to the adaptation field as per OpenCable specification OC-SP-EBP-I01-130118.
     *         ebp_legacy adds Encoder Boundary Point information to the adaptation field using a legacy proprietary
     *         format.
     * @see M2tsSegmentationMarkers
     */
    public M2tsSegmentationMarkers segmentationMarkers() {
        return M2tsSegmentationMarkers.fromValue(segmentationMarkers);
    }

    /**
     * Inserts segmentation markers at each segmentation_time period. rai_segstart sets the Random Access Indicator bit
     * in the adaptation field. rai_adapt sets the RAI bit and adds the current timecode in the private data bytes.
     * psi_segstart inserts PAT and PMT tables at the start of segments. ebp adds Encoder Boundary Point information to
     * the adaptation field as per OpenCable specification OC-SP-EBP-I01-130118. ebp_legacy adds Encoder Boundary Point
     * information to the adaptation field using a legacy proprietary format.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #segmentationMarkers} will return {@link M2tsSegmentationMarkers#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #segmentationMarkersAsString}.
     * </p>
     * 
     * @return Inserts segmentation markers at each segmentation_time period. rai_segstart sets the Random Access
     *         Indicator bit in the adaptation field. rai_adapt sets the RAI bit and adds the current timecode in the
     *         private data bytes. psi_segstart inserts PAT and PMT tables at the start of segments. ebp adds Encoder
     *         Boundary Point information to the adaptation field as per OpenCable specification OC-SP-EBP-I01-130118.
     *         ebp_legacy adds Encoder Boundary Point information to the adaptation field using a legacy proprietary
     *         format.
     * @see M2tsSegmentationMarkers
     */
    public String segmentationMarkersAsString() {
        return segmentationMarkers;
    }

    /**
     * The segmentation style parameter controls how segmentation markers are inserted into the transport stream. With
     * avails, it is possible that segments may be truncated, which can influence where future segmentation markers are
     * inserted. When a segmentation style of "reset_cadence" is selected and a segment is truncated due to an avail, we
     * will reset the segmentation cadence. This means the subsequent segment will have a duration of of
     * $segmentation_time seconds. When a segmentation style of "maintain_cadence" is selected and a segment is
     * truncated due to an avail, we will not reset the segmentation cadence. This means the subsequent segment will
     * likely be truncated as well. However, all segments after that will have a duration of $segmentation_time seconds.
     * Note that EBP lookahead is a slight exception to this rule.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #segmentationStyle}
     * will return {@link M2tsSegmentationStyle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #segmentationStyleAsString}.
     * </p>
     * 
     * @return The segmentation style parameter controls how segmentation markers are inserted into the transport
     *         stream. With avails, it is possible that segments may be truncated, which can influence where future
     *         segmentation markers are inserted. When a segmentation style of "reset_cadence" is selected and a segment
     *         is truncated due to an avail, we will reset the segmentation cadence. This means the subsequent segment
     *         will have a duration of of $segmentation_time seconds. When a segmentation style of "maintain_cadence" is
     *         selected and a segment is truncated due to an avail, we will not reset the segmentation cadence. This
     *         means the subsequent segment will likely be truncated as well. However, all segments after that will have
     *         a duration of $segmentation_time seconds. Note that EBP lookahead is a slight exception to this rule.
     * @see M2tsSegmentationStyle
     */
    public M2tsSegmentationStyle segmentationStyle() {
        return M2tsSegmentationStyle.fromValue(segmentationStyle);
    }

    /**
     * The segmentation style parameter controls how segmentation markers are inserted into the transport stream. With
     * avails, it is possible that segments may be truncated, which can influence where future segmentation markers are
     * inserted. When a segmentation style of "reset_cadence" is selected and a segment is truncated due to an avail, we
     * will reset the segmentation cadence. This means the subsequent segment will have a duration of of
     * $segmentation_time seconds. When a segmentation style of "maintain_cadence" is selected and a segment is
     * truncated due to an avail, we will not reset the segmentation cadence. This means the subsequent segment will
     * likely be truncated as well. However, all segments after that will have a duration of $segmentation_time seconds.
     * Note that EBP lookahead is a slight exception to this rule.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #segmentationStyle}
     * will return {@link M2tsSegmentationStyle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #segmentationStyleAsString}.
     * </p>
     * 
     * @return The segmentation style parameter controls how segmentation markers are inserted into the transport
     *         stream. With avails, it is possible that segments may be truncated, which can influence where future
     *         segmentation markers are inserted. When a segmentation style of "reset_cadence" is selected and a segment
     *         is truncated due to an avail, we will reset the segmentation cadence. This means the subsequent segment
     *         will have a duration of of $segmentation_time seconds. When a segmentation style of "maintain_cadence" is
     *         selected and a segment is truncated due to an avail, we will not reset the segmentation cadence. This
     *         means the subsequent segment will likely be truncated as well. However, all segments after that will have
     *         a duration of $segmentation_time seconds. Note that EBP lookahead is a slight exception to this rule.
     * @see M2tsSegmentationStyle
     */
    public String segmentationStyleAsString() {
        return segmentationStyle;
    }

    /**
     * Specify the length, in seconds, of each segment. Required unless markers is set to _none_.
     * 
     * @return Specify the length, in seconds, of each segment. Required unless markers is set to _none_.
     */
    public Double segmentationTime() {
        return segmentationTime;
    }

    /**
     * Specify the packet identifier (PID) for timed metadata in this output. Default is 502.
     * 
     * @return Specify the packet identifier (PID) for timed metadata in this output. Default is 502.
     */
    public Integer timedMetadataPid() {
        return timedMetadataPid;
    }

    /**
     * Specify the ID for the transport stream itself in the program map table for this output. Transport stream IDs and
     * program map tables are parts of MPEG-2 transport stream containers, used for organizing data.
     * 
     * @return Specify the ID for the transport stream itself in the program map table for this output. Transport stream
     *         IDs and program map tables are parts of MPEG-2 transport stream containers, used for organizing data.
     */
    public Integer transportStreamId() {
        return transportStreamId;
    }

    /**
     * Specify the packet identifier (PID) of the elementary video stream in the transport stream.
     * 
     * @return Specify the packet identifier (PID) of the elementary video stream in the transport stream.
     */
    public Integer videoPid() {
        return videoPid;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(audioBufferModelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(audioFramesPerPes());
        hashCode = 31 * hashCode + Objects.hashCode(audioPids());
        hashCode = 31 * hashCode + Objects.hashCode(bitrate());
        hashCode = 31 * hashCode + Objects.hashCode(bufferModelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dvbNitSettings());
        hashCode = 31 * hashCode + Objects.hashCode(dvbSdtSettings());
        hashCode = 31 * hashCode + Objects.hashCode(dvbSubPids());
        hashCode = 31 * hashCode + Objects.hashCode(dvbTdtSettings());
        hashCode = 31 * hashCode + Objects.hashCode(dvbTeletextPid());
        hashCode = 31 * hashCode + Objects.hashCode(ebpAudioIntervalAsString());
        hashCode = 31 * hashCode + Objects.hashCode(ebpPlacementAsString());
        hashCode = 31 * hashCode + Objects.hashCode(esRateInPesAsString());
        hashCode = 31 * hashCode + Objects.hashCode(forceTsVideoEbpOrderAsString());
        hashCode = 31 * hashCode + Objects.hashCode(fragmentTime());
        hashCode = 31 * hashCode + Objects.hashCode(maxPcrInterval());
        hashCode = 31 * hashCode + Objects.hashCode(minEbpInterval());
        hashCode = 31 * hashCode + Objects.hashCode(nielsenId3AsString());
        hashCode = 31 * hashCode + Objects.hashCode(nullPacketBitrate());
        hashCode = 31 * hashCode + Objects.hashCode(patInterval());
        hashCode = 31 * hashCode + Objects.hashCode(pcrControlAsString());
        hashCode = 31 * hashCode + Objects.hashCode(pcrPid());
        hashCode = 31 * hashCode + Objects.hashCode(pmtInterval());
        hashCode = 31 * hashCode + Objects.hashCode(pmtPid());
        hashCode = 31 * hashCode + Objects.hashCode(privateMetadataPid());
        hashCode = 31 * hashCode + Objects.hashCode(programNumber());
        hashCode = 31 * hashCode + Objects.hashCode(rateModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(scte35Esam());
        hashCode = 31 * hashCode + Objects.hashCode(scte35Pid());
        hashCode = 31 * hashCode + Objects.hashCode(scte35SourceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(segmentationMarkersAsString());
        hashCode = 31 * hashCode + Objects.hashCode(segmentationStyleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(segmentationTime());
        hashCode = 31 * hashCode + Objects.hashCode(timedMetadataPid());
        hashCode = 31 * hashCode + Objects.hashCode(transportStreamId());
        hashCode = 31 * hashCode + Objects.hashCode(videoPid());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof M2tsSettings)) {
            return false;
        }
        M2tsSettings other = (M2tsSettings) obj;
        return Objects.equals(audioBufferModelAsString(), other.audioBufferModelAsString())
                && Objects.equals(audioFramesPerPes(), other.audioFramesPerPes())
                && Objects.equals(audioPids(), other.audioPids()) && Objects.equals(bitrate(), other.bitrate())
                && Objects.equals(bufferModelAsString(), other.bufferModelAsString())
                && Objects.equals(dvbNitSettings(), other.dvbNitSettings())
                && Objects.equals(dvbSdtSettings(), other.dvbSdtSettings()) && Objects.equals(dvbSubPids(), other.dvbSubPids())
                && Objects.equals(dvbTdtSettings(), other.dvbTdtSettings())
                && Objects.equals(dvbTeletextPid(), other.dvbTeletextPid())
                && Objects.equals(ebpAudioIntervalAsString(), other.ebpAudioIntervalAsString())
                && Objects.equals(ebpPlacementAsString(), other.ebpPlacementAsString())
                && Objects.equals(esRateInPesAsString(), other.esRateInPesAsString())
                && Objects.equals(forceTsVideoEbpOrderAsString(), other.forceTsVideoEbpOrderAsString())
                && Objects.equals(fragmentTime(), other.fragmentTime())
                && Objects.equals(maxPcrInterval(), other.maxPcrInterval())
                && Objects.equals(minEbpInterval(), other.minEbpInterval())
                && Objects.equals(nielsenId3AsString(), other.nielsenId3AsString())
                && Objects.equals(nullPacketBitrate(), other.nullPacketBitrate())
                && Objects.equals(patInterval(), other.patInterval())
                && Objects.equals(pcrControlAsString(), other.pcrControlAsString()) && Objects.equals(pcrPid(), other.pcrPid())
                && Objects.equals(pmtInterval(), other.pmtInterval()) && Objects.equals(pmtPid(), other.pmtPid())
                && Objects.equals(privateMetadataPid(), other.privateMetadataPid())
                && Objects.equals(programNumber(), other.programNumber())
                && Objects.equals(rateModeAsString(), other.rateModeAsString())
                && Objects.equals(scte35Esam(), other.scte35Esam()) && Objects.equals(scte35Pid(), other.scte35Pid())
                && Objects.equals(scte35SourceAsString(), other.scte35SourceAsString())
                && Objects.equals(segmentationMarkersAsString(), other.segmentationMarkersAsString())
                && Objects.equals(segmentationStyleAsString(), other.segmentationStyleAsString())
                && Objects.equals(segmentationTime(), other.segmentationTime())
                && Objects.equals(timedMetadataPid(), other.timedMetadataPid())
                && Objects.equals(transportStreamId(), other.transportStreamId()) && Objects.equals(videoPid(), other.videoPid());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("M2tsSettings").add("AudioBufferModel", audioBufferModelAsString())
                .add("AudioFramesPerPes", audioFramesPerPes()).add("AudioPids", audioPids()).add("Bitrate", bitrate())
                .add("BufferModel", bufferModelAsString()).add("DvbNitSettings", dvbNitSettings())
                .add("DvbSdtSettings", dvbSdtSettings()).add("DvbSubPids", dvbSubPids()).add("DvbTdtSettings", dvbTdtSettings())
                .add("DvbTeletextPid", dvbTeletextPid()).add("EbpAudioInterval", ebpAudioIntervalAsString())
                .add("EbpPlacement", ebpPlacementAsString()).add("EsRateInPes", esRateInPesAsString())
                .add("ForceTsVideoEbpOrder", forceTsVideoEbpOrderAsString()).add("FragmentTime", fragmentTime())
                .add("MaxPcrInterval", maxPcrInterval()).add("MinEbpInterval", minEbpInterval())
                .add("NielsenId3", nielsenId3AsString()).add("NullPacketBitrate", nullPacketBitrate())
                .add("PatInterval", patInterval()).add("PcrControl", pcrControlAsString()).add("PcrPid", pcrPid())
                .add("PmtInterval", pmtInterval()).add("PmtPid", pmtPid()).add("PrivateMetadataPid", privateMetadataPid())
                .add("ProgramNumber", programNumber()).add("RateMode", rateModeAsString()).add("Scte35Esam", scte35Esam())
                .add("Scte35Pid", scte35Pid()).add("Scte35Source", scte35SourceAsString())
                .add("SegmentationMarkers", segmentationMarkersAsString()).add("SegmentationStyle", segmentationStyleAsString())
                .add("SegmentationTime", segmentationTime()).add("TimedMetadataPid", timedMetadataPid())
                .add("TransportStreamId", transportStreamId()).add("VideoPid", videoPid()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AudioBufferModel":
            return Optional.ofNullable(clazz.cast(audioBufferModelAsString()));
        case "AudioFramesPerPes":
            return Optional.ofNullable(clazz.cast(audioFramesPerPes()));
        case "AudioPids":
            return Optional.ofNullable(clazz.cast(audioPids()));
        case "Bitrate":
            return Optional.ofNullable(clazz.cast(bitrate()));
        case "BufferModel":
            return Optional.ofNullable(clazz.cast(bufferModelAsString()));
        case "DvbNitSettings":
            return Optional.ofNullable(clazz.cast(dvbNitSettings()));
        case "DvbSdtSettings":
            return Optional.ofNullable(clazz.cast(dvbSdtSettings()));
        case "DvbSubPids":
            return Optional.ofNullable(clazz.cast(dvbSubPids()));
        case "DvbTdtSettings":
            return Optional.ofNullable(clazz.cast(dvbTdtSettings()));
        case "DvbTeletextPid":
            return Optional.ofNullable(clazz.cast(dvbTeletextPid()));
        case "EbpAudioInterval":
            return Optional.ofNullable(clazz.cast(ebpAudioIntervalAsString()));
        case "EbpPlacement":
            return Optional.ofNullable(clazz.cast(ebpPlacementAsString()));
        case "EsRateInPes":
            return Optional.ofNullable(clazz.cast(esRateInPesAsString()));
        case "ForceTsVideoEbpOrder":
            return Optional.ofNullable(clazz.cast(forceTsVideoEbpOrderAsString()));
        case "FragmentTime":
            return Optional.ofNullable(clazz.cast(fragmentTime()));
        case "MaxPcrInterval":
            return Optional.ofNullable(clazz.cast(maxPcrInterval()));
        case "MinEbpInterval":
            return Optional.ofNullable(clazz.cast(minEbpInterval()));
        case "NielsenId3":
            return Optional.ofNullable(clazz.cast(nielsenId3AsString()));
        case "NullPacketBitrate":
            return Optional.ofNullable(clazz.cast(nullPacketBitrate()));
        case "PatInterval":
            return Optional.ofNullable(clazz.cast(patInterval()));
        case "PcrControl":
            return Optional.ofNullable(clazz.cast(pcrControlAsString()));
        case "PcrPid":
            return Optional.ofNullable(clazz.cast(pcrPid()));
        case "PmtInterval":
            return Optional.ofNullable(clazz.cast(pmtInterval()));
        case "PmtPid":
            return Optional.ofNullable(clazz.cast(pmtPid()));
        case "PrivateMetadataPid":
            return Optional.ofNullable(clazz.cast(privateMetadataPid()));
        case "ProgramNumber":
            return Optional.ofNullable(clazz.cast(programNumber()));
        case "RateMode":
            return Optional.ofNullable(clazz.cast(rateModeAsString()));
        case "Scte35Esam":
            return Optional.ofNullable(clazz.cast(scte35Esam()));
        case "Scte35Pid":
            return Optional.ofNullable(clazz.cast(scte35Pid()));
        case "Scte35Source":
            return Optional.ofNullable(clazz.cast(scte35SourceAsString()));
        case "SegmentationMarkers":
            return Optional.ofNullable(clazz.cast(segmentationMarkersAsString()));
        case "SegmentationStyle":
            return Optional.ofNullable(clazz.cast(segmentationStyleAsString()));
        case "SegmentationTime":
            return Optional.ofNullable(clazz.cast(segmentationTime()));
        case "TimedMetadataPid":
            return Optional.ofNullable(clazz.cast(timedMetadataPid()));
        case "TransportStreamId":
            return Optional.ofNullable(clazz.cast(transportStreamId()));
        case "VideoPid":
            return Optional.ofNullable(clazz.cast(videoPid()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<M2tsSettings, T> g) {
        return obj -> g.apply((M2tsSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, M2tsSettings> {
        /**
         * Selects between the DVB and ATSC buffer models for Dolby Digital audio.
         * 
         * @param audioBufferModel
         *        Selects between the DVB and ATSC buffer models for Dolby Digital audio.
         * @see M2tsAudioBufferModel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsAudioBufferModel
         */
        Builder audioBufferModel(String audioBufferModel);

        /**
         * Selects between the DVB and ATSC buffer models for Dolby Digital audio.
         * 
         * @param audioBufferModel
         *        Selects between the DVB and ATSC buffer models for Dolby Digital audio.
         * @see M2tsAudioBufferModel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsAudioBufferModel
         */
        Builder audioBufferModel(M2tsAudioBufferModel audioBufferModel);

        /**
         * The number of audio frames to insert for each PES packet.
         * 
         * @param audioFramesPerPes
         *        The number of audio frames to insert for each PES packet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audioFramesPerPes(Integer audioFramesPerPes);

        /**
         * Specify the packet identifiers (PIDs) for any elementary audio streams you include in this output. Specify
         * multiple PIDs as a JSON array. Default is the range 482-492.
         * 
         * @param audioPids
         *        Specify the packet identifiers (PIDs) for any elementary audio streams you include in this output.
         *        Specify multiple PIDs as a JSON array. Default is the range 482-492.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audioPids(Collection<Integer> audioPids);

        /**
         * Specify the packet identifiers (PIDs) for any elementary audio streams you include in this output. Specify
         * multiple PIDs as a JSON array. Default is the range 482-492.
         * 
         * @param audioPids
         *        Specify the packet identifiers (PIDs) for any elementary audio streams you include in this output.
         *        Specify multiple PIDs as a JSON array. Default is the range 482-492.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audioPids(Integer... audioPids);

        /**
         * Specify the output bitrate of the transport stream in bits per second. Setting to 0 lets the muxer
         * automatically determine the appropriate bitrate. Other common values are 3750000, 7500000, and 15000000.
         * 
         * @param bitrate
         *        Specify the output bitrate of the transport stream in bits per second. Setting to 0 lets the muxer
         *        automatically determine the appropriate bitrate. Other common values are 3750000, 7500000, and
         *        15000000.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bitrate(Integer bitrate);

        /**
         * Controls what buffer model to use for accurate interleaving. If set to MULTIPLEX, use multiplex buffer model.
         * If set to NONE, this can lead to lower latency, but low-memory devices may not be able to play back the
         * stream without interruptions.
         * 
         * @param bufferModel
         *        Controls what buffer model to use for accurate interleaving. If set to MULTIPLEX, use multiplex buffer
         *        model. If set to NONE, this can lead to lower latency, but low-memory devices may not be able to play
         *        back the stream without interruptions.
         * @see M2tsBufferModel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsBufferModel
         */
        Builder bufferModel(String bufferModel);

        /**
         * Controls what buffer model to use for accurate interleaving. If set to MULTIPLEX, use multiplex buffer model.
         * If set to NONE, this can lead to lower latency, but low-memory devices may not be able to play back the
         * stream without interruptions.
         * 
         * @param bufferModel
         *        Controls what buffer model to use for accurate interleaving. If set to MULTIPLEX, use multiplex buffer
         *        model. If set to NONE, this can lead to lower latency, but low-memory devices may not be able to play
         *        back the stream without interruptions.
         * @see M2tsBufferModel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsBufferModel
         */
        Builder bufferModel(M2tsBufferModel bufferModel);

        /**
         * Inserts DVB Network Information Table (NIT) at the specified table repetition interval.
         * 
         * @param dvbNitSettings
         *        Inserts DVB Network Information Table (NIT) at the specified table repetition interval.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dvbNitSettings(DvbNitSettings dvbNitSettings);

        /**
         * Inserts DVB Network Information Table (NIT) at the specified table repetition interval. This is a convenience
         * that creates an instance of the {@link DvbNitSettings.Builder} avoiding the need to create one manually via
         * {@link DvbNitSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link DvbNitSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #dvbNitSettings(DvbNitSettings)}.
         * 
         * @param dvbNitSettings
         *        a consumer that will call methods on {@link DvbNitSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dvbNitSettings(DvbNitSettings)
         */
        default Builder dvbNitSettings(Consumer<DvbNitSettings.Builder> dvbNitSettings) {
            return dvbNitSettings(DvbNitSettings.builder().applyMutation(dvbNitSettings).build());
        }

        /**
         * Inserts DVB Service Description Table (NIT) at the specified table repetition interval.
         * 
         * @param dvbSdtSettings
         *        Inserts DVB Service Description Table (NIT) at the specified table repetition interval.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dvbSdtSettings(DvbSdtSettings dvbSdtSettings);

        /**
         * Inserts DVB Service Description Table (NIT) at the specified table repetition interval. This is a convenience
         * that creates an instance of the {@link DvbSdtSettings.Builder} avoiding the need to create one manually via
         * {@link DvbSdtSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link DvbSdtSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #dvbSdtSettings(DvbSdtSettings)}.
         * 
         * @param dvbSdtSettings
         *        a consumer that will call methods on {@link DvbSdtSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dvbSdtSettings(DvbSdtSettings)
         */
        default Builder dvbSdtSettings(Consumer<DvbSdtSettings.Builder> dvbSdtSettings) {
            return dvbSdtSettings(DvbSdtSettings.builder().applyMutation(dvbSdtSettings).build());
        }

        /**
         * Specify the packet identifiers (PIDs) for DVB subtitle data included in this output. Specify multiple PIDs as
         * a JSON array. Default is the range 460-479.
         * 
         * @param dvbSubPids
         *        Specify the packet identifiers (PIDs) for DVB subtitle data included in this output. Specify multiple
         *        PIDs as a JSON array. Default is the range 460-479.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dvbSubPids(Collection<Integer> dvbSubPids);

        /**
         * Specify the packet identifiers (PIDs) for DVB subtitle data included in this output. Specify multiple PIDs as
         * a JSON array. Default is the range 460-479.
         * 
         * @param dvbSubPids
         *        Specify the packet identifiers (PIDs) for DVB subtitle data included in this output. Specify multiple
         *        PIDs as a JSON array. Default is the range 460-479.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dvbSubPids(Integer... dvbSubPids);

        /**
         * Inserts DVB Time and Date Table (TDT) at the specified table repetition interval.
         * 
         * @param dvbTdtSettings
         *        Inserts DVB Time and Date Table (TDT) at the specified table repetition interval.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dvbTdtSettings(DvbTdtSettings dvbTdtSettings);

        /**
         * Inserts DVB Time and Date Table (TDT) at the specified table repetition interval. This is a convenience that
         * creates an instance of the {@link DvbTdtSettings.Builder} avoiding the need to create one manually via
         * {@link DvbTdtSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link DvbTdtSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #dvbTdtSettings(DvbTdtSettings)}.
         * 
         * @param dvbTdtSettings
         *        a consumer that will call methods on {@link DvbTdtSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dvbTdtSettings(DvbTdtSettings)
         */
        default Builder dvbTdtSettings(Consumer<DvbTdtSettings.Builder> dvbTdtSettings) {
            return dvbTdtSettings(DvbTdtSettings.builder().applyMutation(dvbTdtSettings).build());
        }

        /**
         * Specify the packet identifier (PID) for DVB teletext data you include in this output. Default is 499.
         * 
         * @param dvbTeletextPid
         *        Specify the packet identifier (PID) for DVB teletext data you include in this output. Default is 499.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dvbTeletextPid(Integer dvbTeletextPid);

        /**
         * When set to VIDEO_AND_FIXED_INTERVALS, audio EBP markers will be added to partitions 3 and 4. The interval
         * between these additional markers will be fixed, and will be slightly shorter than the video EBP marker
         * interval. When set to VIDEO_INTERVAL, these additional markers will not be inserted. Only applicable when EBP
         * segmentation markers are is selected (segmentationMarkers is EBP or EBP_LEGACY).
         * 
         * @param ebpAudioInterval
         *        When set to VIDEO_AND_FIXED_INTERVALS, audio EBP markers will be added to partitions 3 and 4. The
         *        interval between these additional markers will be fixed, and will be slightly shorter than the video
         *        EBP marker interval. When set to VIDEO_INTERVAL, these additional markers will not be inserted. Only
         *        applicable when EBP segmentation markers are is selected (segmentationMarkers is EBP or EBP_LEGACY).
         * @see M2tsEbpAudioInterval
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsEbpAudioInterval
         */
        Builder ebpAudioInterval(String ebpAudioInterval);

        /**
         * When set to VIDEO_AND_FIXED_INTERVALS, audio EBP markers will be added to partitions 3 and 4. The interval
         * between these additional markers will be fixed, and will be slightly shorter than the video EBP marker
         * interval. When set to VIDEO_INTERVAL, these additional markers will not be inserted. Only applicable when EBP
         * segmentation markers are is selected (segmentationMarkers is EBP or EBP_LEGACY).
         * 
         * @param ebpAudioInterval
         *        When set to VIDEO_AND_FIXED_INTERVALS, audio EBP markers will be added to partitions 3 and 4. The
         *        interval between these additional markers will be fixed, and will be slightly shorter than the video
         *        EBP marker interval. When set to VIDEO_INTERVAL, these additional markers will not be inserted. Only
         *        applicable when EBP segmentation markers are is selected (segmentationMarkers is EBP or EBP_LEGACY).
         * @see M2tsEbpAudioInterval
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsEbpAudioInterval
         */
        Builder ebpAudioInterval(M2tsEbpAudioInterval ebpAudioInterval);

        /**
         * Selects which PIDs to place EBP markers on. They can either be placed only on the video PID, or on both the
         * video PID and all audio PIDs. Only applicable when EBP segmentation markers are is selected
         * (segmentationMarkers is EBP or EBP_LEGACY).
         * 
         * @param ebpPlacement
         *        Selects which PIDs to place EBP markers on. They can either be placed only on the video PID, or on
         *        both the video PID and all audio PIDs. Only applicable when EBP segmentation markers are is selected
         *        (segmentationMarkers is EBP or EBP_LEGACY).
         * @see M2tsEbpPlacement
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsEbpPlacement
         */
        Builder ebpPlacement(String ebpPlacement);

        /**
         * Selects which PIDs to place EBP markers on. They can either be placed only on the video PID, or on both the
         * video PID and all audio PIDs. Only applicable when EBP segmentation markers are is selected
         * (segmentationMarkers is EBP or EBP_LEGACY).
         * 
         * @param ebpPlacement
         *        Selects which PIDs to place EBP markers on. They can either be placed only on the video PID, or on
         *        both the video PID and all audio PIDs. Only applicable when EBP segmentation markers are is selected
         *        (segmentationMarkers is EBP or EBP_LEGACY).
         * @see M2tsEbpPlacement
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsEbpPlacement
         */
        Builder ebpPlacement(M2tsEbpPlacement ebpPlacement);

        /**
         * Controls whether to include the ES Rate field in the PES header.
         * 
         * @param esRateInPes
         *        Controls whether to include the ES Rate field in the PES header.
         * @see M2tsEsRateInPes
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsEsRateInPes
         */
        Builder esRateInPes(String esRateInPes);

        /**
         * Controls whether to include the ES Rate field in the PES header.
         * 
         * @param esRateInPes
         *        Controls whether to include the ES Rate field in the PES header.
         * @see M2tsEsRateInPes
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsEsRateInPes
         */
        Builder esRateInPes(M2tsEsRateInPes esRateInPes);

        /**
         * Keep the default value (DEFAULT) unless you know that your audio EBP markers are incorrectly appearing before
         * your video EBP markers. To correct this problem, set this value to Force (FORCE).
         * 
         * @param forceTsVideoEbpOrder
         *        Keep the default value (DEFAULT) unless you know that your audio EBP markers are incorrectly appearing
         *        before your video EBP markers. To correct this problem, set this value to Force (FORCE).
         * @see M2tsForceTsVideoEbpOrder
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsForceTsVideoEbpOrder
         */
        Builder forceTsVideoEbpOrder(String forceTsVideoEbpOrder);

        /**
         * Keep the default value (DEFAULT) unless you know that your audio EBP markers are incorrectly appearing before
         * your video EBP markers. To correct this problem, set this value to Force (FORCE).
         * 
         * @param forceTsVideoEbpOrder
         *        Keep the default value (DEFAULT) unless you know that your audio EBP markers are incorrectly appearing
         *        before your video EBP markers. To correct this problem, set this value to Force (FORCE).
         * @see M2tsForceTsVideoEbpOrder
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsForceTsVideoEbpOrder
         */
        Builder forceTsVideoEbpOrder(M2tsForceTsVideoEbpOrder forceTsVideoEbpOrder);

        /**
         * The length, in seconds, of each fragment. Only used with EBP markers.
         * 
         * @param fragmentTime
         *        The length, in seconds, of each fragment. Only used with EBP markers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fragmentTime(Double fragmentTime);

        /**
         * Specify the maximum time, in milliseconds, between Program Clock References (PCRs) inserted into the
         * transport stream.
         * 
         * @param maxPcrInterval
         *        Specify the maximum time, in milliseconds, between Program Clock References (PCRs) inserted into the
         *        transport stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxPcrInterval(Integer maxPcrInterval);

        /**
         * When set, enforces that Encoder Boundary Points do not come within the specified time interval of each other
         * by looking ahead at input video. If another EBP is going to come in within the specified time interval, the
         * current EBP is not emitted, and the segment is "stretched" to the next marker. The lookahead value does not
         * add latency to the system. The Live Event must be configured elsewhere to create sufficient latency to make
         * the lookahead accurate.
         * 
         * @param minEbpInterval
         *        When set, enforces that Encoder Boundary Points do not come within the specified time interval of each
         *        other by looking ahead at input video. If another EBP is going to come in within the specified time
         *        interval, the current EBP is not emitted, and the segment is "stretched" to the next marker. The
         *        lookahead value does not add latency to the system. The Live Event must be configured elsewhere to
         *        create sufficient latency to make the lookahead accurate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minEbpInterval(Integer minEbpInterval);

        /**
         * If INSERT, Nielsen inaudible tones for media tracking will be detected in the input audio and an equivalent
         * ID3 tag will be inserted in the output.
         * 
         * @param nielsenId3
         *        If INSERT, Nielsen inaudible tones for media tracking will be detected in the input audio and an
         *        equivalent ID3 tag will be inserted in the output.
         * @see M2tsNielsenId3
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsNielsenId3
         */
        Builder nielsenId3(String nielsenId3);

        /**
         * If INSERT, Nielsen inaudible tones for media tracking will be detected in the input audio and an equivalent
         * ID3 tag will be inserted in the output.
         * 
         * @param nielsenId3
         *        If INSERT, Nielsen inaudible tones for media tracking will be detected in the input audio and an
         *        equivalent ID3 tag will be inserted in the output.
         * @see M2tsNielsenId3
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsNielsenId3
         */
        Builder nielsenId3(M2tsNielsenId3 nielsenId3);

        /**
         * Value in bits per second of extra null packets to insert into the transport stream. This can be used if a
         * downstream encryption system requires periodic null packets.
         * 
         * @param nullPacketBitrate
         *        Value in bits per second of extra null packets to insert into the transport stream. This can be used
         *        if a downstream encryption system requires periodic null packets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nullPacketBitrate(Double nullPacketBitrate);

        /**
         * The number of milliseconds between instances of this table in the output transport stream.
         * 
         * @param patInterval
         *        The number of milliseconds between instances of this table in the output transport stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder patInterval(Integer patInterval);

        /**
         * When set to PCR_EVERY_PES_PACKET, a Program Clock Reference value is inserted for every Packetized Elementary
         * Stream (PES) header. This is effective only when the PCR PID is the same as the video or audio elementary
         * stream.
         * 
         * @param pcrControl
         *        When set to PCR_EVERY_PES_PACKET, a Program Clock Reference value is inserted for every Packetized
         *        Elementary Stream (PES) header. This is effective only when the PCR PID is the same as the video or
         *        audio elementary stream.
         * @see M2tsPcrControl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsPcrControl
         */
        Builder pcrControl(String pcrControl);

        /**
         * When set to PCR_EVERY_PES_PACKET, a Program Clock Reference value is inserted for every Packetized Elementary
         * Stream (PES) header. This is effective only when the PCR PID is the same as the video or audio elementary
         * stream.
         * 
         * @param pcrControl
         *        When set to PCR_EVERY_PES_PACKET, a Program Clock Reference value is inserted for every Packetized
         *        Elementary Stream (PES) header. This is effective only when the PCR PID is the same as the video or
         *        audio elementary stream.
         * @see M2tsPcrControl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsPcrControl
         */
        Builder pcrControl(M2tsPcrControl pcrControl);

        /**
         * Specify the packet identifier (PID) for the program clock reference (PCR) in this output. If you do not
         * specify a value, the service will use the value for Video PID (VideoPid).
         * 
         * @param pcrPid
         *        Specify the packet identifier (PID) for the program clock reference (PCR) in this output. If you do
         *        not specify a value, the service will use the value for Video PID (VideoPid).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pcrPid(Integer pcrPid);

        /**
         * Specify the number of milliseconds between instances of the program map table (PMT) in the output transport
         * stream.
         * 
         * @param pmtInterval
         *        Specify the number of milliseconds between instances of the program map table (PMT) in the output
         *        transport stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pmtInterval(Integer pmtInterval);

        /**
         * Specify the packet identifier (PID) for the program map table (PMT) itself. Default is 480.
         * 
         * @param pmtPid
         *        Specify the packet identifier (PID) for the program map table (PMT) itself. Default is 480.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pmtPid(Integer pmtPid);

        /**
         * Specify the packet identifier (PID) of the private metadata stream. Default is 503.
         * 
         * @param privateMetadataPid
         *        Specify the packet identifier (PID) of the private metadata stream. Default is 503.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateMetadataPid(Integer privateMetadataPid);

        /**
         * Use Program number (programNumber) to specify the program number used in the program map table (PMT) for this
         * output. Default is 1. Program numbers and program map tables are parts of MPEG-2 transport stream containers,
         * used for organizing data.
         * 
         * @param programNumber
         *        Use Program number (programNumber) to specify the program number used in the program map table (PMT)
         *        for this output. Default is 1. Program numbers and program map tables are parts of MPEG-2 transport
         *        stream containers, used for organizing data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder programNumber(Integer programNumber);

        /**
         * When set to CBR, inserts null packets into transport stream to fill specified bitrate. When set to VBR, the
         * bitrate setting acts as the maximum bitrate, but the output will not be padded up to that bitrate.
         * 
         * @param rateMode
         *        When set to CBR, inserts null packets into transport stream to fill specified bitrate. When set to
         *        VBR, the bitrate setting acts as the maximum bitrate, but the output will not be padded up to that
         *        bitrate.
         * @see M2tsRateMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsRateMode
         */
        Builder rateMode(String rateMode);

        /**
         * When set to CBR, inserts null packets into transport stream to fill specified bitrate. When set to VBR, the
         * bitrate setting acts as the maximum bitrate, but the output will not be padded up to that bitrate.
         * 
         * @param rateMode
         *        When set to CBR, inserts null packets into transport stream to fill specified bitrate. When set to
         *        VBR, the bitrate setting acts as the maximum bitrate, but the output will not be padded up to that
         *        bitrate.
         * @see M2tsRateMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsRateMode
         */
        Builder rateMode(M2tsRateMode rateMode);

        /**
         * Include this in your job settings to put SCTE-35 markers in your HLS and transport stream outputs at the
         * insertion points that you specify in an ESAM XML document. Provide the document in the setting SCC XML
         * (sccXml).
         * 
         * @param scte35Esam
         *        Include this in your job settings to put SCTE-35 markers in your HLS and transport stream outputs at
         *        the insertion points that you specify in an ESAM XML document. Provide the document in the setting SCC
         *        XML (sccXml).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scte35Esam(M2tsScte35Esam scte35Esam);

        /**
         * Include this in your job settings to put SCTE-35 markers in your HLS and transport stream outputs at the
         * insertion points that you specify in an ESAM XML document. Provide the document in the setting SCC XML
         * (sccXml). This is a convenience that creates an instance of the {@link M2tsScte35Esam.Builder} avoiding the
         * need to create one manually via {@link M2tsScte35Esam#builder()}.
         *
         * When the {@link Consumer} completes, {@link M2tsScte35Esam.Builder#build()} is called immediately and its
         * result is passed to {@link #scte35Esam(M2tsScte35Esam)}.
         * 
         * @param scte35Esam
         *        a consumer that will call methods on {@link M2tsScte35Esam.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scte35Esam(M2tsScte35Esam)
         */
        default Builder scte35Esam(Consumer<M2tsScte35Esam.Builder> scte35Esam) {
            return scte35Esam(M2tsScte35Esam.builder().applyMutation(scte35Esam).build());
        }

        /**
         * Specify the packet identifier (PID) of the SCTE-35 stream in the transport stream.
         * 
         * @param scte35Pid
         *        Specify the packet identifier (PID) of the SCTE-35 stream in the transport stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scte35Pid(Integer scte35Pid);

        /**
         * For SCTE-35 markers from your input-- Choose Passthrough (PASSTHROUGH) if you want SCTE-35 markers that
         * appear in your input to also appear in this output. Choose None (NONE) if you don't want SCTE-35 markers in
         * this output. For SCTE-35 markers from an ESAM XML document-- Choose None (NONE). Also provide the ESAM XML as
         * a string in the setting Signal processing notification XML (sccXml). Also enable ESAM SCTE-35 (include the
         * property scte35Esam).
         * 
         * @param scte35Source
         *        For SCTE-35 markers from your input-- Choose Passthrough (PASSTHROUGH) if you want SCTE-35 markers
         *        that appear in your input to also appear in this output. Choose None (NONE) if you don't want SCTE-35
         *        markers in this output. For SCTE-35 markers from an ESAM XML document-- Choose None (NONE). Also
         *        provide the ESAM XML as a string in the setting Signal processing notification XML (sccXml). Also
         *        enable ESAM SCTE-35 (include the property scte35Esam).
         * @see M2tsScte35Source
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsScte35Source
         */
        Builder scte35Source(String scte35Source);

        /**
         * For SCTE-35 markers from your input-- Choose Passthrough (PASSTHROUGH) if you want SCTE-35 markers that
         * appear in your input to also appear in this output. Choose None (NONE) if you don't want SCTE-35 markers in
         * this output. For SCTE-35 markers from an ESAM XML document-- Choose None (NONE). Also provide the ESAM XML as
         * a string in the setting Signal processing notification XML (sccXml). Also enable ESAM SCTE-35 (include the
         * property scte35Esam).
         * 
         * @param scte35Source
         *        For SCTE-35 markers from your input-- Choose Passthrough (PASSTHROUGH) if you want SCTE-35 markers
         *        that appear in your input to also appear in this output. Choose None (NONE) if you don't want SCTE-35
         *        markers in this output. For SCTE-35 markers from an ESAM XML document-- Choose None (NONE). Also
         *        provide the ESAM XML as a string in the setting Signal processing notification XML (sccXml). Also
         *        enable ESAM SCTE-35 (include the property scte35Esam).
         * @see M2tsScte35Source
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsScte35Source
         */
        Builder scte35Source(M2tsScte35Source scte35Source);

        /**
         * Inserts segmentation markers at each segmentation_time period. rai_segstart sets the Random Access Indicator
         * bit in the adaptation field. rai_adapt sets the RAI bit and adds the current timecode in the private data
         * bytes. psi_segstart inserts PAT and PMT tables at the start of segments. ebp adds Encoder Boundary Point
         * information to the adaptation field as per OpenCable specification OC-SP-EBP-I01-130118. ebp_legacy adds
         * Encoder Boundary Point information to the adaptation field using a legacy proprietary format.
         * 
         * @param segmentationMarkers
         *        Inserts segmentation markers at each segmentation_time period. rai_segstart sets the Random Access
         *        Indicator bit in the adaptation field. rai_adapt sets the RAI bit and adds the current timecode in the
         *        private data bytes. psi_segstart inserts PAT and PMT tables at the start of segments. ebp adds Encoder
         *        Boundary Point information to the adaptation field as per OpenCable specification
         *        OC-SP-EBP-I01-130118. ebp_legacy adds Encoder Boundary Point information to the adaptation field using
         *        a legacy proprietary format.
         * @see M2tsSegmentationMarkers
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsSegmentationMarkers
         */
        Builder segmentationMarkers(String segmentationMarkers);

        /**
         * Inserts segmentation markers at each segmentation_time period. rai_segstart sets the Random Access Indicator
         * bit in the adaptation field. rai_adapt sets the RAI bit and adds the current timecode in the private data
         * bytes. psi_segstart inserts PAT and PMT tables at the start of segments. ebp adds Encoder Boundary Point
         * information to the adaptation field as per OpenCable specification OC-SP-EBP-I01-130118. ebp_legacy adds
         * Encoder Boundary Point information to the adaptation field using a legacy proprietary format.
         * 
         * @param segmentationMarkers
         *        Inserts segmentation markers at each segmentation_time period. rai_segstart sets the Random Access
         *        Indicator bit in the adaptation field. rai_adapt sets the RAI bit and adds the current timecode in the
         *        private data bytes. psi_segstart inserts PAT and PMT tables at the start of segments. ebp adds Encoder
         *        Boundary Point information to the adaptation field as per OpenCable specification
         *        OC-SP-EBP-I01-130118. ebp_legacy adds Encoder Boundary Point information to the adaptation field using
         *        a legacy proprietary format.
         * @see M2tsSegmentationMarkers
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsSegmentationMarkers
         */
        Builder segmentationMarkers(M2tsSegmentationMarkers segmentationMarkers);

        /**
         * The segmentation style parameter controls how segmentation markers are inserted into the transport stream.
         * With avails, it is possible that segments may be truncated, which can influence where future segmentation
         * markers are inserted. When a segmentation style of "reset_cadence" is selected and a segment is truncated due
         * to an avail, we will reset the segmentation cadence. This means the subsequent segment will have a duration
         * of of $segmentation_time seconds. When a segmentation style of "maintain_cadence" is selected and a segment
         * is truncated due to an avail, we will not reset the segmentation cadence. This means the subsequent segment
         * will likely be truncated as well. However, all segments after that will have a duration of $segmentation_time
         * seconds. Note that EBP lookahead is a slight exception to this rule.
         * 
         * @param segmentationStyle
         *        The segmentation style parameter controls how segmentation markers are inserted into the transport
         *        stream. With avails, it is possible that segments may be truncated, which can influence where future
         *        segmentation markers are inserted. When a segmentation style of "reset_cadence" is selected and a
         *        segment is truncated due to an avail, we will reset the segmentation cadence. This means the
         *        subsequent segment will have a duration of of $segmentation_time seconds. When a segmentation style of
         *        "maintain_cadence" is selected and a segment is truncated due to an avail, we will not reset the
         *        segmentation cadence. This means the subsequent segment will likely be truncated as well. However, all
         *        segments after that will have a duration of $segmentation_time seconds. Note that EBP lookahead is a
         *        slight exception to this rule.
         * @see M2tsSegmentationStyle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsSegmentationStyle
         */
        Builder segmentationStyle(String segmentationStyle);

        /**
         * The segmentation style parameter controls how segmentation markers are inserted into the transport stream.
         * With avails, it is possible that segments may be truncated, which can influence where future segmentation
         * markers are inserted. When a segmentation style of "reset_cadence" is selected and a segment is truncated due
         * to an avail, we will reset the segmentation cadence. This means the subsequent segment will have a duration
         * of of $segmentation_time seconds. When a segmentation style of "maintain_cadence" is selected and a segment
         * is truncated due to an avail, we will not reset the segmentation cadence. This means the subsequent segment
         * will likely be truncated as well. However, all segments after that will have a duration of $segmentation_time
         * seconds. Note that EBP lookahead is a slight exception to this rule.
         * 
         * @param segmentationStyle
         *        The segmentation style parameter controls how segmentation markers are inserted into the transport
         *        stream. With avails, it is possible that segments may be truncated, which can influence where future
         *        segmentation markers are inserted. When a segmentation style of "reset_cadence" is selected and a
         *        segment is truncated due to an avail, we will reset the segmentation cadence. This means the
         *        subsequent segment will have a duration of of $segmentation_time seconds. When a segmentation style of
         *        "maintain_cadence" is selected and a segment is truncated due to an avail, we will not reset the
         *        segmentation cadence. This means the subsequent segment will likely be truncated as well. However, all
         *        segments after that will have a duration of $segmentation_time seconds. Note that EBP lookahead is a
         *        slight exception to this rule.
         * @see M2tsSegmentationStyle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see M2tsSegmentationStyle
         */
        Builder segmentationStyle(M2tsSegmentationStyle segmentationStyle);

        /**
         * Specify the length, in seconds, of each segment. Required unless markers is set to _none_.
         * 
         * @param segmentationTime
         *        Specify the length, in seconds, of each segment. Required unless markers is set to _none_.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder segmentationTime(Double segmentationTime);

        /**
         * Specify the packet identifier (PID) for timed metadata in this output. Default is 502.
         * 
         * @param timedMetadataPid
         *        Specify the packet identifier (PID) for timed metadata in this output. Default is 502.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timedMetadataPid(Integer timedMetadataPid);

        /**
         * Specify the ID for the transport stream itself in the program map table for this output. Transport stream IDs
         * and program map tables are parts of MPEG-2 transport stream containers, used for organizing data.
         * 
         * @param transportStreamId
         *        Specify the ID for the transport stream itself in the program map table for this output. Transport
         *        stream IDs and program map tables are parts of MPEG-2 transport stream containers, used for organizing
         *        data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transportStreamId(Integer transportStreamId);

        /**
         * Specify the packet identifier (PID) of the elementary video stream in the transport stream.
         * 
         * @param videoPid
         *        Specify the packet identifier (PID) of the elementary video stream in the transport stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder videoPid(Integer videoPid);
    }

    static final class BuilderImpl implements Builder {
        private String audioBufferModel;

        private Integer audioFramesPerPes;

        private List<Integer> audioPids = DefaultSdkAutoConstructList.getInstance();

        private Integer bitrate;

        private String bufferModel;

        private DvbNitSettings dvbNitSettings;

        private DvbSdtSettings dvbSdtSettings;

        private List<Integer> dvbSubPids = DefaultSdkAutoConstructList.getInstance();

        private DvbTdtSettings dvbTdtSettings;

        private Integer dvbTeletextPid;

        private String ebpAudioInterval;

        private String ebpPlacement;

        private String esRateInPes;

        private String forceTsVideoEbpOrder;

        private Double fragmentTime;

        private Integer maxPcrInterval;

        private Integer minEbpInterval;

        private String nielsenId3;

        private Double nullPacketBitrate;

        private Integer patInterval;

        private String pcrControl;

        private Integer pcrPid;

        private Integer pmtInterval;

        private Integer pmtPid;

        private Integer privateMetadataPid;

        private Integer programNumber;

        private String rateMode;

        private M2tsScte35Esam scte35Esam;

        private Integer scte35Pid;

        private String scte35Source;

        private String segmentationMarkers;

        private String segmentationStyle;

        private Double segmentationTime;

        private Integer timedMetadataPid;

        private Integer transportStreamId;

        private Integer videoPid;

        private BuilderImpl() {
        }

        private BuilderImpl(M2tsSettings model) {
            audioBufferModel(model.audioBufferModel);
            audioFramesPerPes(model.audioFramesPerPes);
            audioPids(model.audioPids);
            bitrate(model.bitrate);
            bufferModel(model.bufferModel);
            dvbNitSettings(model.dvbNitSettings);
            dvbSdtSettings(model.dvbSdtSettings);
            dvbSubPids(model.dvbSubPids);
            dvbTdtSettings(model.dvbTdtSettings);
            dvbTeletextPid(model.dvbTeletextPid);
            ebpAudioInterval(model.ebpAudioInterval);
            ebpPlacement(model.ebpPlacement);
            esRateInPes(model.esRateInPes);
            forceTsVideoEbpOrder(model.forceTsVideoEbpOrder);
            fragmentTime(model.fragmentTime);
            maxPcrInterval(model.maxPcrInterval);
            minEbpInterval(model.minEbpInterval);
            nielsenId3(model.nielsenId3);
            nullPacketBitrate(model.nullPacketBitrate);
            patInterval(model.patInterval);
            pcrControl(model.pcrControl);
            pcrPid(model.pcrPid);
            pmtInterval(model.pmtInterval);
            pmtPid(model.pmtPid);
            privateMetadataPid(model.privateMetadataPid);
            programNumber(model.programNumber);
            rateMode(model.rateMode);
            scte35Esam(model.scte35Esam);
            scte35Pid(model.scte35Pid);
            scte35Source(model.scte35Source);
            segmentationMarkers(model.segmentationMarkers);
            segmentationStyle(model.segmentationStyle);
            segmentationTime(model.segmentationTime);
            timedMetadataPid(model.timedMetadataPid);
            transportStreamId(model.transportStreamId);
            videoPid(model.videoPid);
        }

        public final String getAudioBufferModel() {
            return audioBufferModel;
        }

        @Override
        public final Builder audioBufferModel(String audioBufferModel) {
            this.audioBufferModel = audioBufferModel;
            return this;
        }

        @Override
        public final Builder audioBufferModel(M2tsAudioBufferModel audioBufferModel) {
            this.audioBufferModel(audioBufferModel == null ? null : audioBufferModel.toString());
            return this;
        }

        public final void setAudioBufferModel(String audioBufferModel) {
            this.audioBufferModel = audioBufferModel;
        }

        public final Integer getAudioFramesPerPes() {
            return audioFramesPerPes;
        }

        @Override
        public final Builder audioFramesPerPes(Integer audioFramesPerPes) {
            this.audioFramesPerPes = audioFramesPerPes;
            return this;
        }

        public final void setAudioFramesPerPes(Integer audioFramesPerPes) {
            this.audioFramesPerPes = audioFramesPerPes;
        }

        public final Collection<Integer> getAudioPids() {
            return audioPids;
        }

        @Override
        public final Builder audioPids(Collection<Integer> audioPids) {
            this.audioPids = ___listOf__integerMin32Max8182Copier.copy(audioPids);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder audioPids(Integer... audioPids) {
            audioPids(Arrays.asList(audioPids));
            return this;
        }

        public final void setAudioPids(Collection<Integer> audioPids) {
            this.audioPids = ___listOf__integerMin32Max8182Copier.copy(audioPids);
        }

        public final Integer getBitrate() {
            return bitrate;
        }

        @Override
        public final Builder bitrate(Integer bitrate) {
            this.bitrate = bitrate;
            return this;
        }

        public final void setBitrate(Integer bitrate) {
            this.bitrate = bitrate;
        }

        public final String getBufferModel() {
            return bufferModel;
        }

        @Override
        public final Builder bufferModel(String bufferModel) {
            this.bufferModel = bufferModel;
            return this;
        }

        @Override
        public final Builder bufferModel(M2tsBufferModel bufferModel) {
            this.bufferModel(bufferModel == null ? null : bufferModel.toString());
            return this;
        }

        public final void setBufferModel(String bufferModel) {
            this.bufferModel = bufferModel;
        }

        public final DvbNitSettings.Builder getDvbNitSettings() {
            return dvbNitSettings != null ? dvbNitSettings.toBuilder() : null;
        }

        @Override
        public final Builder dvbNitSettings(DvbNitSettings dvbNitSettings) {
            this.dvbNitSettings = dvbNitSettings;
            return this;
        }

        public final void setDvbNitSettings(DvbNitSettings.BuilderImpl dvbNitSettings) {
            this.dvbNitSettings = dvbNitSettings != null ? dvbNitSettings.build() : null;
        }

        public final DvbSdtSettings.Builder getDvbSdtSettings() {
            return dvbSdtSettings != null ? dvbSdtSettings.toBuilder() : null;
        }

        @Override
        public final Builder dvbSdtSettings(DvbSdtSettings dvbSdtSettings) {
            this.dvbSdtSettings = dvbSdtSettings;
            return this;
        }

        public final void setDvbSdtSettings(DvbSdtSettings.BuilderImpl dvbSdtSettings) {
            this.dvbSdtSettings = dvbSdtSettings != null ? dvbSdtSettings.build() : null;
        }

        public final Collection<Integer> getDvbSubPids() {
            return dvbSubPids;
        }

        @Override
        public final Builder dvbSubPids(Collection<Integer> dvbSubPids) {
            this.dvbSubPids = ___listOf__integerMin32Max8182Copier.copy(dvbSubPids);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dvbSubPids(Integer... dvbSubPids) {
            dvbSubPids(Arrays.asList(dvbSubPids));
            return this;
        }

        public final void setDvbSubPids(Collection<Integer> dvbSubPids) {
            this.dvbSubPids = ___listOf__integerMin32Max8182Copier.copy(dvbSubPids);
        }

        public final DvbTdtSettings.Builder getDvbTdtSettings() {
            return dvbTdtSettings != null ? dvbTdtSettings.toBuilder() : null;
        }

        @Override
        public final Builder dvbTdtSettings(DvbTdtSettings dvbTdtSettings) {
            this.dvbTdtSettings = dvbTdtSettings;
            return this;
        }

        public final void setDvbTdtSettings(DvbTdtSettings.BuilderImpl dvbTdtSettings) {
            this.dvbTdtSettings = dvbTdtSettings != null ? dvbTdtSettings.build() : null;
        }

        public final Integer getDvbTeletextPid() {
            return dvbTeletextPid;
        }

        @Override
        public final Builder dvbTeletextPid(Integer dvbTeletextPid) {
            this.dvbTeletextPid = dvbTeletextPid;
            return this;
        }

        public final void setDvbTeletextPid(Integer dvbTeletextPid) {
            this.dvbTeletextPid = dvbTeletextPid;
        }

        public final String getEbpAudioInterval() {
            return ebpAudioInterval;
        }

        @Override
        public final Builder ebpAudioInterval(String ebpAudioInterval) {
            this.ebpAudioInterval = ebpAudioInterval;
            return this;
        }

        @Override
        public final Builder ebpAudioInterval(M2tsEbpAudioInterval ebpAudioInterval) {
            this.ebpAudioInterval(ebpAudioInterval == null ? null : ebpAudioInterval.toString());
            return this;
        }

        public final void setEbpAudioInterval(String ebpAudioInterval) {
            this.ebpAudioInterval = ebpAudioInterval;
        }

        public final String getEbpPlacement() {
            return ebpPlacement;
        }

        @Override
        public final Builder ebpPlacement(String ebpPlacement) {
            this.ebpPlacement = ebpPlacement;
            return this;
        }

        @Override
        public final Builder ebpPlacement(M2tsEbpPlacement ebpPlacement) {
            this.ebpPlacement(ebpPlacement == null ? null : ebpPlacement.toString());
            return this;
        }

        public final void setEbpPlacement(String ebpPlacement) {
            this.ebpPlacement = ebpPlacement;
        }

        public final String getEsRateInPes() {
            return esRateInPes;
        }

        @Override
        public final Builder esRateInPes(String esRateInPes) {
            this.esRateInPes = esRateInPes;
            return this;
        }

        @Override
        public final Builder esRateInPes(M2tsEsRateInPes esRateInPes) {
            this.esRateInPes(esRateInPes == null ? null : esRateInPes.toString());
            return this;
        }

        public final void setEsRateInPes(String esRateInPes) {
            this.esRateInPes = esRateInPes;
        }

        public final String getForceTsVideoEbpOrder() {
            return forceTsVideoEbpOrder;
        }

        @Override
        public final Builder forceTsVideoEbpOrder(String forceTsVideoEbpOrder) {
            this.forceTsVideoEbpOrder = forceTsVideoEbpOrder;
            return this;
        }

        @Override
        public final Builder forceTsVideoEbpOrder(M2tsForceTsVideoEbpOrder forceTsVideoEbpOrder) {
            this.forceTsVideoEbpOrder(forceTsVideoEbpOrder == null ? null : forceTsVideoEbpOrder.toString());
            return this;
        }

        public final void setForceTsVideoEbpOrder(String forceTsVideoEbpOrder) {
            this.forceTsVideoEbpOrder = forceTsVideoEbpOrder;
        }

        public final Double getFragmentTime() {
            return fragmentTime;
        }

        @Override
        public final Builder fragmentTime(Double fragmentTime) {
            this.fragmentTime = fragmentTime;
            return this;
        }

        public final void setFragmentTime(Double fragmentTime) {
            this.fragmentTime = fragmentTime;
        }

        public final Integer getMaxPcrInterval() {
            return maxPcrInterval;
        }

        @Override
        public final Builder maxPcrInterval(Integer maxPcrInterval) {
            this.maxPcrInterval = maxPcrInterval;
            return this;
        }

        public final void setMaxPcrInterval(Integer maxPcrInterval) {
            this.maxPcrInterval = maxPcrInterval;
        }

        public final Integer getMinEbpInterval() {
            return minEbpInterval;
        }

        @Override
        public final Builder minEbpInterval(Integer minEbpInterval) {
            this.minEbpInterval = minEbpInterval;
            return this;
        }

        public final void setMinEbpInterval(Integer minEbpInterval) {
            this.minEbpInterval = minEbpInterval;
        }

        public final String getNielsenId3() {
            return nielsenId3;
        }

        @Override
        public final Builder nielsenId3(String nielsenId3) {
            this.nielsenId3 = nielsenId3;
            return this;
        }

        @Override
        public final Builder nielsenId3(M2tsNielsenId3 nielsenId3) {
            this.nielsenId3(nielsenId3 == null ? null : nielsenId3.toString());
            return this;
        }

        public final void setNielsenId3(String nielsenId3) {
            this.nielsenId3 = nielsenId3;
        }

        public final Double getNullPacketBitrate() {
            return nullPacketBitrate;
        }

        @Override
        public final Builder nullPacketBitrate(Double nullPacketBitrate) {
            this.nullPacketBitrate = nullPacketBitrate;
            return this;
        }

        public final void setNullPacketBitrate(Double nullPacketBitrate) {
            this.nullPacketBitrate = nullPacketBitrate;
        }

        public final Integer getPatInterval() {
            return patInterval;
        }

        @Override
        public final Builder patInterval(Integer patInterval) {
            this.patInterval = patInterval;
            return this;
        }

        public final void setPatInterval(Integer patInterval) {
            this.patInterval = patInterval;
        }

        public final String getPcrControl() {
            return pcrControl;
        }

        @Override
        public final Builder pcrControl(String pcrControl) {
            this.pcrControl = pcrControl;
            return this;
        }

        @Override
        public final Builder pcrControl(M2tsPcrControl pcrControl) {
            this.pcrControl(pcrControl == null ? null : pcrControl.toString());
            return this;
        }

        public final void setPcrControl(String pcrControl) {
            this.pcrControl = pcrControl;
        }

        public final Integer getPcrPid() {
            return pcrPid;
        }

        @Override
        public final Builder pcrPid(Integer pcrPid) {
            this.pcrPid = pcrPid;
            return this;
        }

        public final void setPcrPid(Integer pcrPid) {
            this.pcrPid = pcrPid;
        }

        public final Integer getPmtInterval() {
            return pmtInterval;
        }

        @Override
        public final Builder pmtInterval(Integer pmtInterval) {
            this.pmtInterval = pmtInterval;
            return this;
        }

        public final void setPmtInterval(Integer pmtInterval) {
            this.pmtInterval = pmtInterval;
        }

        public final Integer getPmtPid() {
            return pmtPid;
        }

        @Override
        public final Builder pmtPid(Integer pmtPid) {
            this.pmtPid = pmtPid;
            return this;
        }

        public final void setPmtPid(Integer pmtPid) {
            this.pmtPid = pmtPid;
        }

        public final Integer getPrivateMetadataPid() {
            return privateMetadataPid;
        }

        @Override
        public final Builder privateMetadataPid(Integer privateMetadataPid) {
            this.privateMetadataPid = privateMetadataPid;
            return this;
        }

        public final void setPrivateMetadataPid(Integer privateMetadataPid) {
            this.privateMetadataPid = privateMetadataPid;
        }

        public final Integer getProgramNumber() {
            return programNumber;
        }

        @Override
        public final Builder programNumber(Integer programNumber) {
            this.programNumber = programNumber;
            return this;
        }

        public final void setProgramNumber(Integer programNumber) {
            this.programNumber = programNumber;
        }

        public final String getRateMode() {
            return rateMode;
        }

        @Override
        public final Builder rateMode(String rateMode) {
            this.rateMode = rateMode;
            return this;
        }

        @Override
        public final Builder rateMode(M2tsRateMode rateMode) {
            this.rateMode(rateMode == null ? null : rateMode.toString());
            return this;
        }

        public final void setRateMode(String rateMode) {
            this.rateMode = rateMode;
        }

        public final M2tsScte35Esam.Builder getScte35Esam() {
            return scte35Esam != null ? scte35Esam.toBuilder() : null;
        }

        @Override
        public final Builder scte35Esam(M2tsScte35Esam scte35Esam) {
            this.scte35Esam = scte35Esam;
            return this;
        }

        public final void setScte35Esam(M2tsScte35Esam.BuilderImpl scte35Esam) {
            this.scte35Esam = scte35Esam != null ? scte35Esam.build() : null;
        }

        public final Integer getScte35Pid() {
            return scte35Pid;
        }

        @Override
        public final Builder scte35Pid(Integer scte35Pid) {
            this.scte35Pid = scte35Pid;
            return this;
        }

        public final void setScte35Pid(Integer scte35Pid) {
            this.scte35Pid = scte35Pid;
        }

        public final String getScte35Source() {
            return scte35Source;
        }

        @Override
        public final Builder scte35Source(String scte35Source) {
            this.scte35Source = scte35Source;
            return this;
        }

        @Override
        public final Builder scte35Source(M2tsScte35Source scte35Source) {
            this.scte35Source(scte35Source == null ? null : scte35Source.toString());
            return this;
        }

        public final void setScte35Source(String scte35Source) {
            this.scte35Source = scte35Source;
        }

        public final String getSegmentationMarkers() {
            return segmentationMarkers;
        }

        @Override
        public final Builder segmentationMarkers(String segmentationMarkers) {
            this.segmentationMarkers = segmentationMarkers;
            return this;
        }

        @Override
        public final Builder segmentationMarkers(M2tsSegmentationMarkers segmentationMarkers) {
            this.segmentationMarkers(segmentationMarkers == null ? null : segmentationMarkers.toString());
            return this;
        }

        public final void setSegmentationMarkers(String segmentationMarkers) {
            this.segmentationMarkers = segmentationMarkers;
        }

        public final String getSegmentationStyle() {
            return segmentationStyle;
        }

        @Override
        public final Builder segmentationStyle(String segmentationStyle) {
            this.segmentationStyle = segmentationStyle;
            return this;
        }

        @Override
        public final Builder segmentationStyle(M2tsSegmentationStyle segmentationStyle) {
            this.segmentationStyle(segmentationStyle == null ? null : segmentationStyle.toString());
            return this;
        }

        public final void setSegmentationStyle(String segmentationStyle) {
            this.segmentationStyle = segmentationStyle;
        }

        public final Double getSegmentationTime() {
            return segmentationTime;
        }

        @Override
        public final Builder segmentationTime(Double segmentationTime) {
            this.segmentationTime = segmentationTime;
            return this;
        }

        public final void setSegmentationTime(Double segmentationTime) {
            this.segmentationTime = segmentationTime;
        }

        public final Integer getTimedMetadataPid() {
            return timedMetadataPid;
        }

        @Override
        public final Builder timedMetadataPid(Integer timedMetadataPid) {
            this.timedMetadataPid = timedMetadataPid;
            return this;
        }

        public final void setTimedMetadataPid(Integer timedMetadataPid) {
            this.timedMetadataPid = timedMetadataPid;
        }

        public final Integer getTransportStreamId() {
            return transportStreamId;
        }

        @Override
        public final Builder transportStreamId(Integer transportStreamId) {
            this.transportStreamId = transportStreamId;
            return this;
        }

        public final void setTransportStreamId(Integer transportStreamId) {
            this.transportStreamId = transportStreamId;
        }

        public final Integer getVideoPid() {
            return videoPid;
        }

        @Override
        public final Builder videoPid(Integer videoPid) {
            this.videoPid = videoPid;
            return this;
        }

        public final void setVideoPid(Integer videoPid) {
            this.videoPid = videoPid;
        }

        @Override
        public M2tsSettings build() {
            return new M2tsSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
