/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconnect.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration settings for a router output using the SRT (Secure Reliable Transport) protocol in listener mode,
 * including the port, minimum latency, and encryption key configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SrtListenerRouterOutputConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<SrtListenerRouterOutputConfiguration.Builder, SrtListenerRouterOutputConfiguration> {
    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Port")
            .getter(getter(SrtListenerRouterOutputConfiguration::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("port").build()).build();

    private static final SdkField<Long> MINIMUM_LATENCY_MILLISECONDS_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("MinimumLatencyMilliseconds")
            .getter(getter(SrtListenerRouterOutputConfiguration::minimumLatencyMilliseconds))
            .setter(setter(Builder::minimumLatencyMilliseconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("minimumLatencyMilliseconds").build())
            .build();

    private static final SdkField<SrtEncryptionConfiguration> ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<SrtEncryptionConfiguration> builder(MarshallingType.SDK_POJO).memberName("EncryptionConfiguration")
            .getter(getter(SrtListenerRouterOutputConfiguration::encryptionConfiguration))
            .setter(setter(Builder::encryptionConfiguration)).constructor(SrtEncryptionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PORT_FIELD,
            MINIMUM_LATENCY_MILLISECONDS_FIELD, ENCRYPTION_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer port;

    private final Long minimumLatencyMilliseconds;

    private final SrtEncryptionConfiguration encryptionConfiguration;

    private SrtListenerRouterOutputConfiguration(BuilderImpl builder) {
        this.port = builder.port;
        this.minimumLatencyMilliseconds = builder.minimumLatencyMilliseconds;
        this.encryptionConfiguration = builder.encryptionConfiguration;
    }

    /**
     * <p>
     * The port number for the SRT protocol in listener mode.
     * </p>
     * 
     * @return The port number for the SRT protocol in listener mode.
     */
    public final Integer port() {
        return port;
    }

    /**
     * <p>
     * The minimum latency in milliseconds for the SRT protocol in listener mode.
     * </p>
     * 
     * @return The minimum latency in milliseconds for the SRT protocol in listener mode.
     */
    public final Long minimumLatencyMilliseconds() {
        return minimumLatencyMilliseconds;
    }

    /**
     * <p>
     * Defines the encryption settings for an SRT listener output, including the encryption key configuration and
     * associated parameters.
     * </p>
     * 
     * @return Defines the encryption settings for an SRT listener output, including the encryption key configuration
     *         and associated parameters.
     */
    public final SrtEncryptionConfiguration encryptionConfiguration() {
        return encryptionConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(minimumLatencyMilliseconds());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SrtListenerRouterOutputConfiguration)) {
            return false;
        }
        SrtListenerRouterOutputConfiguration other = (SrtListenerRouterOutputConfiguration) obj;
        return Objects.equals(port(), other.port())
                && Objects.equals(minimumLatencyMilliseconds(), other.minimumLatencyMilliseconds())
                && Objects.equals(encryptionConfiguration(), other.encryptionConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SrtListenerRouterOutputConfiguration").add("Port", port())
                .add("MinimumLatencyMilliseconds", minimumLatencyMilliseconds())
                .add("EncryptionConfiguration", encryptionConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "MinimumLatencyMilliseconds":
            return Optional.ofNullable(clazz.cast(minimumLatencyMilliseconds()));
        case "EncryptionConfiguration":
            return Optional.ofNullable(clazz.cast(encryptionConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("port", PORT_FIELD);
        map.put("minimumLatencyMilliseconds", MINIMUM_LATENCY_MILLISECONDS_FIELD);
        map.put("encryptionConfiguration", ENCRYPTION_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SrtListenerRouterOutputConfiguration, T> g) {
        return obj -> g.apply((SrtListenerRouterOutputConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SrtListenerRouterOutputConfiguration> {
        /**
         * <p>
         * The port number for the SRT protocol in listener mode.
         * </p>
         * 
         * @param port
         *        The port number for the SRT protocol in listener mode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * The minimum latency in milliseconds for the SRT protocol in listener mode.
         * </p>
         * 
         * @param minimumLatencyMilliseconds
         *        The minimum latency in milliseconds for the SRT protocol in listener mode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minimumLatencyMilliseconds(Long minimumLatencyMilliseconds);

        /**
         * <p>
         * Defines the encryption settings for an SRT listener output, including the encryption key configuration and
         * associated parameters.
         * </p>
         * 
         * @param encryptionConfiguration
         *        Defines the encryption settings for an SRT listener output, including the encryption key configuration
         *        and associated parameters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfiguration(SrtEncryptionConfiguration encryptionConfiguration);

        /**
         * <p>
         * Defines the encryption settings for an SRT listener output, including the encryption key configuration and
         * associated parameters.
         * </p>
         * This is a convenience method that creates an instance of the {@link SrtEncryptionConfiguration.Builder}
         * avoiding the need to create one manually via {@link SrtEncryptionConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SrtEncryptionConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #encryptionConfiguration(SrtEncryptionConfiguration)}.
         * 
         * @param encryptionConfiguration
         *        a consumer that will call methods on {@link SrtEncryptionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionConfiguration(SrtEncryptionConfiguration)
         */
        default Builder encryptionConfiguration(Consumer<SrtEncryptionConfiguration.Builder> encryptionConfiguration) {
            return encryptionConfiguration(SrtEncryptionConfiguration.builder().applyMutation(encryptionConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Integer port;

        private Long minimumLatencyMilliseconds;

        private SrtEncryptionConfiguration encryptionConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(SrtListenerRouterOutputConfiguration model) {
            port(model.port);
            minimumLatencyMilliseconds(model.minimumLatencyMilliseconds);
            encryptionConfiguration(model.encryptionConfiguration);
        }

        public final Integer getPort() {
            return port;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final Long getMinimumLatencyMilliseconds() {
            return minimumLatencyMilliseconds;
        }

        public final void setMinimumLatencyMilliseconds(Long minimumLatencyMilliseconds) {
            this.minimumLatencyMilliseconds = minimumLatencyMilliseconds;
        }

        @Override
        public final Builder minimumLatencyMilliseconds(Long minimumLatencyMilliseconds) {
            this.minimumLatencyMilliseconds = minimumLatencyMilliseconds;
            return this;
        }

        public final SrtEncryptionConfiguration.Builder getEncryptionConfiguration() {
            return encryptionConfiguration != null ? encryptionConfiguration.toBuilder() : null;
        }

        public final void setEncryptionConfiguration(SrtEncryptionConfiguration.BuilderImpl encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration != null ? encryptionConfiguration.build() : null;
        }

        @Override
        public final Builder encryptionConfiguration(SrtEncryptionConfiguration encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration;
            return this;
        }

        @Override
        public SrtListenerRouterOutputConfiguration build() {
            return new SrtListenerRouterOutputConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
