/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconnect.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A router output in AWS Elemental MediaConnect. A router output is a destination for media content that can receive
 * input from one or more router inputs.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RouterOutput implements SdkPojo, Serializable, ToCopyableBuilder<RouterOutput.Builder, RouterOutput> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(RouterOutput::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(RouterOutput::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(RouterOutput::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(RouterOutput::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> OUTPUT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutputType").getter(getter(RouterOutput::outputTypeAsString)).setter(setter(Builder::outputType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("outputType").build()).build();

    private static final SdkField<RouterOutputConfiguration> CONFIGURATION_FIELD = SdkField
            .<RouterOutputConfiguration> builder(MarshallingType.SDK_POJO).memberName("Configuration")
            .getter(getter(RouterOutput::configuration)).setter(setter(Builder::configuration))
            .constructor(RouterOutputConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configuration").build()).build();

    private static final SdkField<String> ROUTED_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoutedState").getter(getter(RouterOutput::routedStateAsString)).setter(setter(Builder::routedState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("routedState").build()).build();

    private static final SdkField<String> REGION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RegionName").getter(getter(RouterOutput::regionName)).setter(setter(Builder::regionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("regionName").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZone").getter(getter(RouterOutput::availabilityZone))
            .setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("availabilityZone").build()).build();

    private static final SdkField<Long> MAXIMUM_BITRATE_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("MaximumBitrate").getter(getter(RouterOutput::maximumBitrate)).setter(setter(Builder::maximumBitrate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maximumBitrate").build()).build();

    private static final SdkField<String> ROUTING_SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoutingScope").getter(getter(RouterOutput::routingScopeAsString)).setter(setter(Builder::routingScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("routingScope").build()).build();

    private static final SdkField<String> TIER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Tier")
            .getter(getter(RouterOutput::tierAsString)).setter(setter(Builder::tier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tier").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt")
            .getter(getter(RouterOutput::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdatedAt")
            .getter(getter(RouterOutput::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<List<RouterOutputMessage>> MESSAGES_FIELD = SdkField
            .<List<RouterOutputMessage>> builder(MarshallingType.LIST)
            .memberName("Messages")
            .getter(getter(RouterOutput::messages))
            .setter(setter(Builder::messages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("messages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RouterOutputMessage> builder(MarshallingType.SDK_POJO)
                                            .constructor(RouterOutputMessage::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(RouterOutput::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<RouterOutputStreamDetails> STREAM_DETAILS_FIELD = SdkField
            .<RouterOutputStreamDetails> builder(MarshallingType.SDK_POJO).memberName("StreamDetails")
            .getter(getter(RouterOutput::streamDetails)).setter(setter(Builder::streamDetails))
            .constructor(RouterOutputStreamDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("streamDetails").build()).build();

    private static final SdkField<String> IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IpAddress").getter(getter(RouterOutput::ipAddress)).setter(setter(Builder::ipAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ipAddress").build()).build();

    private static final SdkField<String> ROUTED_INPUT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoutedInputArn").getter(getter(RouterOutput::routedInputArn)).setter(setter(Builder::routedInputArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("routedInputArn").build()).build();

    private static final SdkField<String> MAINTENANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaintenanceType").getter(getter(RouterOutput::maintenanceTypeAsString))
            .setter(setter(Builder::maintenanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maintenanceType").build()).build();

    private static final SdkField<MaintenanceConfiguration> MAINTENANCE_CONFIGURATION_FIELD = SdkField
            .<MaintenanceConfiguration> builder(MarshallingType.SDK_POJO).memberName("MaintenanceConfiguration")
            .getter(getter(RouterOutput::maintenanceConfiguration)).setter(setter(Builder::maintenanceConfiguration))
            .constructor(MaintenanceConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maintenanceConfiguration").build())
            .build();

    private static final SdkField<String> MAINTENANCE_SCHEDULE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaintenanceScheduleType").getter(getter(RouterOutput::maintenanceScheduleTypeAsString))
            .setter(setter(Builder::maintenanceScheduleType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maintenanceScheduleType").build())
            .build();

    private static final SdkField<MaintenanceSchedule> MAINTENANCE_SCHEDULE_FIELD = SdkField
            .<MaintenanceSchedule> builder(MarshallingType.SDK_POJO).memberName("MaintenanceSchedule")
            .getter(getter(RouterOutput::maintenanceSchedule)).setter(setter(Builder::maintenanceSchedule))
            .constructor(MaintenanceSchedule::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maintenanceSchedule").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            ID_FIELD, STATE_FIELD, OUTPUT_TYPE_FIELD, CONFIGURATION_FIELD, ROUTED_STATE_FIELD, REGION_NAME_FIELD,
            AVAILABILITY_ZONE_FIELD, MAXIMUM_BITRATE_FIELD, ROUTING_SCOPE_FIELD, TIER_FIELD, CREATED_AT_FIELD, UPDATED_AT_FIELD,
            MESSAGES_FIELD, TAGS_FIELD, STREAM_DETAILS_FIELD, IP_ADDRESS_FIELD, ROUTED_INPUT_ARN_FIELD, MAINTENANCE_TYPE_FIELD,
            MAINTENANCE_CONFIGURATION_FIELD, MAINTENANCE_SCHEDULE_TYPE_FIELD, MAINTENANCE_SCHEDULE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String id;

    private final String state;

    private final String outputType;

    private final RouterOutputConfiguration configuration;

    private final String routedState;

    private final String regionName;

    private final String availabilityZone;

    private final Long maximumBitrate;

    private final String routingScope;

    private final String tier;

    private final Instant createdAt;

    private final Instant updatedAt;

    private final List<RouterOutputMessage> messages;

    private final Map<String, String> tags;

    private final RouterOutputStreamDetails streamDetails;

    private final String ipAddress;

    private final String routedInputArn;

    private final String maintenanceType;

    private final MaintenanceConfiguration maintenanceConfiguration;

    private final String maintenanceScheduleType;

    private final MaintenanceSchedule maintenanceSchedule;

    private RouterOutput(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.id = builder.id;
        this.state = builder.state;
        this.outputType = builder.outputType;
        this.configuration = builder.configuration;
        this.routedState = builder.routedState;
        this.regionName = builder.regionName;
        this.availabilityZone = builder.availabilityZone;
        this.maximumBitrate = builder.maximumBitrate;
        this.routingScope = builder.routingScope;
        this.tier = builder.tier;
        this.createdAt = builder.createdAt;
        this.updatedAt = builder.updatedAt;
        this.messages = builder.messages;
        this.tags = builder.tags;
        this.streamDetails = builder.streamDetails;
        this.ipAddress = builder.ipAddress;
        this.routedInputArn = builder.routedInputArn;
        this.maintenanceType = builder.maintenanceType;
        this.maintenanceConfiguration = builder.maintenanceConfiguration;
        this.maintenanceScheduleType = builder.maintenanceScheduleType;
        this.maintenanceSchedule = builder.maintenanceSchedule;
    }

    /**
     * <p>
     * The name of the router output.
     * </p>
     * 
     * @return The name of the router output.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the router output.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the router output.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The unique identifier of the router output.
     * </p>
     * 
     * @return The unique identifier of the router output.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The overall state of the router output.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RouterOutputState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The overall state of the router output.
     * @see RouterOutputState
     */
    public final RouterOutputState state() {
        return RouterOutputState.fromValue(state);
    }

    /**
     * <p>
     * The overall state of the router output.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RouterOutputState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The overall state of the router output.
     * @see RouterOutputState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The type of the router output.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #outputType} will
     * return {@link RouterOutputType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #outputTypeAsString}.
     * </p>
     * 
     * @return The type of the router output.
     * @see RouterOutputType
     */
    public final RouterOutputType outputType() {
        return RouterOutputType.fromValue(outputType);
    }

    /**
     * <p>
     * The type of the router output.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #outputType} will
     * return {@link RouterOutputType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #outputTypeAsString}.
     * </p>
     * 
     * @return The type of the router output.
     * @see RouterOutputType
     */
    public final String outputTypeAsString() {
        return outputType;
    }

    /**
     * Returns the value of the Configuration property for this object.
     * 
     * @return The value of the Configuration property for this object.
     */
    public final RouterOutputConfiguration configuration() {
        return configuration;
    }

    /**
     * <p>
     * The current state of the association between the router output and its input.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #routedState} will
     * return {@link RouterOutputRoutedState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #routedStateAsString}.
     * </p>
     * 
     * @return The current state of the association between the router output and its input.
     * @see RouterOutputRoutedState
     */
    public final RouterOutputRoutedState routedState() {
        return RouterOutputRoutedState.fromValue(routedState);
    }

    /**
     * <p>
     * The current state of the association between the router output and its input.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #routedState} will
     * return {@link RouterOutputRoutedState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #routedStateAsString}.
     * </p>
     * 
     * @return The current state of the association between the router output and its input.
     * @see RouterOutputRoutedState
     */
    public final String routedStateAsString() {
        return routedState;
    }

    /**
     * <p>
     * The AWS Region where the router output is located.
     * </p>
     * 
     * @return The AWS Region where the router output is located.
     */
    public final String regionName() {
        return regionName;
    }

    /**
     * <p>
     * The Availability Zone of the router output.
     * </p>
     * 
     * @return The Availability Zone of the router output.
     */
    public final String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * The maximum bitrate for the router output.
     * </p>
     * 
     * @return The maximum bitrate for the router output.
     */
    public final Long maximumBitrate() {
        return maximumBitrate;
    }

    /**
     * <p>
     * Indicates whether the router output is configured for Regional or global routing.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #routingScope} will
     * return {@link RoutingScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #routingScopeAsString}.
     * </p>
     * 
     * @return Indicates whether the router output is configured for Regional or global routing.
     * @see RoutingScope
     */
    public final RoutingScope routingScope() {
        return RoutingScope.fromValue(routingScope);
    }

    /**
     * <p>
     * Indicates whether the router output is configured for Regional or global routing.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #routingScope} will
     * return {@link RoutingScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #routingScopeAsString}.
     * </p>
     * 
     * @return Indicates whether the router output is configured for Regional or global routing.
     * @see RoutingScope
     */
    public final String routingScopeAsString() {
        return routingScope;
    }

    /**
     * <p>
     * The tier level of the router output.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link RouterOutputTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return The tier level of the router output.
     * @see RouterOutputTier
     */
    public final RouterOutputTier tier() {
        return RouterOutputTier.fromValue(tier);
    }

    /**
     * <p>
     * The tier level of the router output.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link RouterOutputTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return The tier level of the router output.
     * @see RouterOutputTier
     */
    public final String tierAsString() {
        return tier;
    }

    /**
     * <p>
     * The timestamp when the router output was created.
     * </p>
     * 
     * @return The timestamp when the router output was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The timestamp when the router output was last updated.
     * </p>
     * 
     * @return The timestamp when the router output was last updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the Messages property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasMessages() {
        return messages != null && !(messages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The messages associated with the router output.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMessages} method.
     * </p>
     * 
     * @return The messages associated with the router output.
     */
    public final List<RouterOutputMessage> messages() {
        return messages;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Key-value pairs that can be used to tag and organize this router output.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Key-value pairs that can be used to tag and organize this router output.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * Returns the value of the StreamDetails property for this object.
     * 
     * @return The value of the StreamDetails property for this object.
     */
    public final RouterOutputStreamDetails streamDetails() {
        return streamDetails;
    }

    /**
     * <p>
     * The IP address of the router output.
     * </p>
     * 
     * @return The IP address of the router output.
     */
    public final String ipAddress() {
        return ipAddress;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the router input associated with the output.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the router input associated with the output.
     */
    public final String routedInputArn() {
        return routedInputArn;
    }

    /**
     * <p>
     * The type of maintenance configuration applied to this router output.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #maintenanceType}
     * will return {@link MaintenanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #maintenanceTypeAsString}.
     * </p>
     * 
     * @return The type of maintenance configuration applied to this router output.
     * @see MaintenanceType
     */
    public final MaintenanceType maintenanceType() {
        return MaintenanceType.fromValue(maintenanceType);
    }

    /**
     * <p>
     * The type of maintenance configuration applied to this router output.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #maintenanceType}
     * will return {@link MaintenanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #maintenanceTypeAsString}.
     * </p>
     * 
     * @return The type of maintenance configuration applied to this router output.
     * @see MaintenanceType
     */
    public final String maintenanceTypeAsString() {
        return maintenanceType;
    }

    /**
     * <p>
     * The maintenance configuration settings applied to this router output.
     * </p>
     * 
     * @return The maintenance configuration settings applied to this router output.
     */
    public final MaintenanceConfiguration maintenanceConfiguration() {
        return maintenanceConfiguration;
    }

    /**
     * <p>
     * The type of maintenance schedule currently in effect for this router output.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #maintenanceScheduleType} will return {@link MaintenanceScheduleType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #maintenanceScheduleTypeAsString}.
     * </p>
     * 
     * @return The type of maintenance schedule currently in effect for this router output.
     * @see MaintenanceScheduleType
     */
    public final MaintenanceScheduleType maintenanceScheduleType() {
        return MaintenanceScheduleType.fromValue(maintenanceScheduleType);
    }

    /**
     * <p>
     * The type of maintenance schedule currently in effect for this router output.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #maintenanceScheduleType} will return {@link MaintenanceScheduleType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #maintenanceScheduleTypeAsString}.
     * </p>
     * 
     * @return The type of maintenance schedule currently in effect for this router output.
     * @see MaintenanceScheduleType
     */
    public final String maintenanceScheduleTypeAsString() {
        return maintenanceScheduleType;
    }

    /**
     * <p>
     * The current maintenance schedule details for this router output.
     * </p>
     * 
     * @return The current maintenance schedule details for this router output.
     */
    public final MaintenanceSchedule maintenanceSchedule() {
        return maintenanceSchedule;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(outputTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        hashCode = 31 * hashCode + Objects.hashCode(routedStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(regionName());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(maximumBitrate());
        hashCode = 31 * hashCode + Objects.hashCode(routingScopeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tierAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasMessages() ? messages() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(streamDetails());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddress());
        hashCode = 31 * hashCode + Objects.hashCode(routedInputArn());
        hashCode = 31 * hashCode + Objects.hashCode(maintenanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maintenanceConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(maintenanceScheduleTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maintenanceSchedule());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RouterOutput)) {
            return false;
        }
        RouterOutput other = (RouterOutput) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn()) && Objects.equals(id(), other.id())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(outputTypeAsString(), other.outputTypeAsString())
                && Objects.equals(configuration(), other.configuration())
                && Objects.equals(routedStateAsString(), other.routedStateAsString())
                && Objects.equals(regionName(), other.regionName())
                && Objects.equals(availabilityZone(), other.availabilityZone())
                && Objects.equals(maximumBitrate(), other.maximumBitrate())
                && Objects.equals(routingScopeAsString(), other.routingScopeAsString())
                && Objects.equals(tierAsString(), other.tierAsString()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(updatedAt(), other.updatedAt()) && hasMessages() == other.hasMessages()
                && Objects.equals(messages(), other.messages()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(streamDetails(), other.streamDetails())
                && Objects.equals(ipAddress(), other.ipAddress()) && Objects.equals(routedInputArn(), other.routedInputArn())
                && Objects.equals(maintenanceTypeAsString(), other.maintenanceTypeAsString())
                && Objects.equals(maintenanceConfiguration(), other.maintenanceConfiguration())
                && Objects.equals(maintenanceScheduleTypeAsString(), other.maintenanceScheduleTypeAsString())
                && Objects.equals(maintenanceSchedule(), other.maintenanceSchedule());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RouterOutput").add("Name", name()).add("Arn", arn()).add("Id", id())
                .add("State", stateAsString()).add("OutputType", outputTypeAsString()).add("Configuration", configuration())
                .add("RoutedState", routedStateAsString()).add("RegionName", regionName())
                .add("AvailabilityZone", availabilityZone()).add("MaximumBitrate", maximumBitrate())
                .add("RoutingScope", routingScopeAsString()).add("Tier", tierAsString()).add("CreatedAt", createdAt())
                .add("UpdatedAt", updatedAt()).add("Messages", hasMessages() ? messages() : null)
                .add("Tags", hasTags() ? tags() : null).add("StreamDetails", streamDetails()).add("IpAddress", ipAddress())
                .add("RoutedInputArn", routedInputArn()).add("MaintenanceType", maintenanceTypeAsString())
                .add("MaintenanceConfiguration", maintenanceConfiguration())
                .add("MaintenanceScheduleType", maintenanceScheduleTypeAsString())
                .add("MaintenanceSchedule", maintenanceSchedule()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "OutputType":
            return Optional.ofNullable(clazz.cast(outputTypeAsString()));
        case "Configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "RoutedState":
            return Optional.ofNullable(clazz.cast(routedStateAsString()));
        case "RegionName":
            return Optional.ofNullable(clazz.cast(regionName()));
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "MaximumBitrate":
            return Optional.ofNullable(clazz.cast(maximumBitrate()));
        case "RoutingScope":
            return Optional.ofNullable(clazz.cast(routingScopeAsString()));
        case "Tier":
            return Optional.ofNullable(clazz.cast(tierAsString()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "UpdatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "Messages":
            return Optional.ofNullable(clazz.cast(messages()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "StreamDetails":
            return Optional.ofNullable(clazz.cast(streamDetails()));
        case "IpAddress":
            return Optional.ofNullable(clazz.cast(ipAddress()));
        case "RoutedInputArn":
            return Optional.ofNullable(clazz.cast(routedInputArn()));
        case "MaintenanceType":
            return Optional.ofNullable(clazz.cast(maintenanceTypeAsString()));
        case "MaintenanceConfiguration":
            return Optional.ofNullable(clazz.cast(maintenanceConfiguration()));
        case "MaintenanceScheduleType":
            return Optional.ofNullable(clazz.cast(maintenanceScheduleTypeAsString()));
        case "MaintenanceSchedule":
            return Optional.ofNullable(clazz.cast(maintenanceSchedule()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("arn", ARN_FIELD);
        map.put("id", ID_FIELD);
        map.put("state", STATE_FIELD);
        map.put("outputType", OUTPUT_TYPE_FIELD);
        map.put("configuration", CONFIGURATION_FIELD);
        map.put("routedState", ROUTED_STATE_FIELD);
        map.put("regionName", REGION_NAME_FIELD);
        map.put("availabilityZone", AVAILABILITY_ZONE_FIELD);
        map.put("maximumBitrate", MAXIMUM_BITRATE_FIELD);
        map.put("routingScope", ROUTING_SCOPE_FIELD);
        map.put("tier", TIER_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("updatedAt", UPDATED_AT_FIELD);
        map.put("messages", MESSAGES_FIELD);
        map.put("tags", TAGS_FIELD);
        map.put("streamDetails", STREAM_DETAILS_FIELD);
        map.put("ipAddress", IP_ADDRESS_FIELD);
        map.put("routedInputArn", ROUTED_INPUT_ARN_FIELD);
        map.put("maintenanceType", MAINTENANCE_TYPE_FIELD);
        map.put("maintenanceConfiguration", MAINTENANCE_CONFIGURATION_FIELD);
        map.put("maintenanceScheduleType", MAINTENANCE_SCHEDULE_TYPE_FIELD);
        map.put("maintenanceSchedule", MAINTENANCE_SCHEDULE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RouterOutput, T> g) {
        return obj -> g.apply((RouterOutput) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RouterOutput> {
        /**
         * <p>
         * The name of the router output.
         * </p>
         * 
         * @param name
         *        The name of the router output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the router output.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the router output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The unique identifier of the router output.
         * </p>
         * 
         * @param id
         *        The unique identifier of the router output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The overall state of the router output.
         * </p>
         * 
         * @param state
         *        The overall state of the router output.
         * @see RouterOutputState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouterOutputState
         */
        Builder state(String state);

        /**
         * <p>
         * The overall state of the router output.
         * </p>
         * 
         * @param state
         *        The overall state of the router output.
         * @see RouterOutputState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouterOutputState
         */
        Builder state(RouterOutputState state);

        /**
         * <p>
         * The type of the router output.
         * </p>
         * 
         * @param outputType
         *        The type of the router output.
         * @see RouterOutputType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouterOutputType
         */
        Builder outputType(String outputType);

        /**
         * <p>
         * The type of the router output.
         * </p>
         * 
         * @param outputType
         *        The type of the router output.
         * @see RouterOutputType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouterOutputType
         */
        Builder outputType(RouterOutputType outputType);

        /**
         * Sets the value of the Configuration property for this object.
         *
         * @param configuration
         *        The new value for the Configuration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(RouterOutputConfiguration configuration);

        /**
         * Sets the value of the Configuration property for this object.
         *
         * This is a convenience method that creates an instance of the {@link RouterOutputConfiguration.Builder}
         * avoiding the need to create one manually via {@link RouterOutputConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouterOutputConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #configuration(RouterOutputConfiguration)}.
         * 
         * @param configuration
         *        a consumer that will call methods on {@link RouterOutputConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configuration(RouterOutputConfiguration)
         */
        default Builder configuration(Consumer<RouterOutputConfiguration.Builder> configuration) {
            return configuration(RouterOutputConfiguration.builder().applyMutation(configuration).build());
        }

        /**
         * <p>
         * The current state of the association between the router output and its input.
         * </p>
         * 
         * @param routedState
         *        The current state of the association between the router output and its input.
         * @see RouterOutputRoutedState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouterOutputRoutedState
         */
        Builder routedState(String routedState);

        /**
         * <p>
         * The current state of the association between the router output and its input.
         * </p>
         * 
         * @param routedState
         *        The current state of the association between the router output and its input.
         * @see RouterOutputRoutedState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouterOutputRoutedState
         */
        Builder routedState(RouterOutputRoutedState routedState);

        /**
         * <p>
         * The AWS Region where the router output is located.
         * </p>
         * 
         * @param regionName
         *        The AWS Region where the router output is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regionName(String regionName);

        /**
         * <p>
         * The Availability Zone of the router output.
         * </p>
         * 
         * @param availabilityZone
         *        The Availability Zone of the router output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * The maximum bitrate for the router output.
         * </p>
         * 
         * @param maximumBitrate
         *        The maximum bitrate for the router output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumBitrate(Long maximumBitrate);

        /**
         * <p>
         * Indicates whether the router output is configured for Regional or global routing.
         * </p>
         * 
         * @param routingScope
         *        Indicates whether the router output is configured for Regional or global routing.
         * @see RoutingScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RoutingScope
         */
        Builder routingScope(String routingScope);

        /**
         * <p>
         * Indicates whether the router output is configured for Regional or global routing.
         * </p>
         * 
         * @param routingScope
         *        Indicates whether the router output is configured for Regional or global routing.
         * @see RoutingScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RoutingScope
         */
        Builder routingScope(RoutingScope routingScope);

        /**
         * <p>
         * The tier level of the router output.
         * </p>
         * 
         * @param tier
         *        The tier level of the router output.
         * @see RouterOutputTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouterOutputTier
         */
        Builder tier(String tier);

        /**
         * <p>
         * The tier level of the router output.
         * </p>
         * 
         * @param tier
         *        The tier level of the router output.
         * @see RouterOutputTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouterOutputTier
         */
        Builder tier(RouterOutputTier tier);

        /**
         * <p>
         * The timestamp when the router output was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the router output was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The timestamp when the router output was last updated.
         * </p>
         * 
         * @param updatedAt
         *        The timestamp when the router output was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * The messages associated with the router output.
         * </p>
         * 
         * @param messages
         *        The messages associated with the router output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messages(Collection<RouterOutputMessage> messages);

        /**
         * <p>
         * The messages associated with the router output.
         * </p>
         * 
         * @param messages
         *        The messages associated with the router output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messages(RouterOutputMessage... messages);

        /**
         * <p>
         * The messages associated with the router output.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.mediaconnect.model.RouterOutputMessage.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.mediaconnect.model.RouterOutputMessage#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.mediaconnect.model.RouterOutputMessage.Builder#build()} is called
         * immediately and its result is passed to {@link #messages(List<RouterOutputMessage>)}.
         * 
         * @param messages
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.mediaconnect.model.RouterOutputMessage.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #messages(java.util.Collection<RouterOutputMessage>)
         */
        Builder messages(Consumer<RouterOutputMessage.Builder>... messages);

        /**
         * <p>
         * Key-value pairs that can be used to tag and organize this router output.
         * </p>
         * 
         * @param tags
         *        Key-value pairs that can be used to tag and organize this router output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * Sets the value of the StreamDetails property for this object.
         *
         * @param streamDetails
         *        The new value for the StreamDetails property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamDetails(RouterOutputStreamDetails streamDetails);

        /**
         * Sets the value of the StreamDetails property for this object.
         *
         * This is a convenience method that creates an instance of the {@link RouterOutputStreamDetails.Builder}
         * avoiding the need to create one manually via {@link RouterOutputStreamDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouterOutputStreamDetails.Builder#build()} is called immediately
         * and its result is passed to {@link #streamDetails(RouterOutputStreamDetails)}.
         * 
         * @param streamDetails
         *        a consumer that will call methods on {@link RouterOutputStreamDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #streamDetails(RouterOutputStreamDetails)
         */
        default Builder streamDetails(Consumer<RouterOutputStreamDetails.Builder> streamDetails) {
            return streamDetails(RouterOutputStreamDetails.builder().applyMutation(streamDetails).build());
        }

        /**
         * <p>
         * The IP address of the router output.
         * </p>
         * 
         * @param ipAddress
         *        The IP address of the router output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the router input associated with the output.
         * </p>
         * 
         * @param routedInputArn
         *        The Amazon Resource Name (ARN) of the router input associated with the output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routedInputArn(String routedInputArn);

        /**
         * <p>
         * The type of maintenance configuration applied to this router output.
         * </p>
         * 
         * @param maintenanceType
         *        The type of maintenance configuration applied to this router output.
         * @see MaintenanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceType
         */
        Builder maintenanceType(String maintenanceType);

        /**
         * <p>
         * The type of maintenance configuration applied to this router output.
         * </p>
         * 
         * @param maintenanceType
         *        The type of maintenance configuration applied to this router output.
         * @see MaintenanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceType
         */
        Builder maintenanceType(MaintenanceType maintenanceType);

        /**
         * <p>
         * The maintenance configuration settings applied to this router output.
         * </p>
         * 
         * @param maintenanceConfiguration
         *        The maintenance configuration settings applied to this router output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maintenanceConfiguration(MaintenanceConfiguration maintenanceConfiguration);

        /**
         * <p>
         * The maintenance configuration settings applied to this router output.
         * </p>
         * This is a convenience method that creates an instance of the {@link MaintenanceConfiguration.Builder}
         * avoiding the need to create one manually via {@link MaintenanceConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MaintenanceConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #maintenanceConfiguration(MaintenanceConfiguration)}.
         * 
         * @param maintenanceConfiguration
         *        a consumer that will call methods on {@link MaintenanceConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #maintenanceConfiguration(MaintenanceConfiguration)
         */
        default Builder maintenanceConfiguration(Consumer<MaintenanceConfiguration.Builder> maintenanceConfiguration) {
            return maintenanceConfiguration(MaintenanceConfiguration.builder().applyMutation(maintenanceConfiguration).build());
        }

        /**
         * <p>
         * The type of maintenance schedule currently in effect for this router output.
         * </p>
         * 
         * @param maintenanceScheduleType
         *        The type of maintenance schedule currently in effect for this router output.
         * @see MaintenanceScheduleType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceScheduleType
         */
        Builder maintenanceScheduleType(String maintenanceScheduleType);

        /**
         * <p>
         * The type of maintenance schedule currently in effect for this router output.
         * </p>
         * 
         * @param maintenanceScheduleType
         *        The type of maintenance schedule currently in effect for this router output.
         * @see MaintenanceScheduleType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceScheduleType
         */
        Builder maintenanceScheduleType(MaintenanceScheduleType maintenanceScheduleType);

        /**
         * <p>
         * The current maintenance schedule details for this router output.
         * </p>
         * 
         * @param maintenanceSchedule
         *        The current maintenance schedule details for this router output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maintenanceSchedule(MaintenanceSchedule maintenanceSchedule);

        /**
         * <p>
         * The current maintenance schedule details for this router output.
         * </p>
         * This is a convenience method that creates an instance of the {@link MaintenanceSchedule.Builder} avoiding the
         * need to create one manually via {@link MaintenanceSchedule#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MaintenanceSchedule.Builder#build()} is called immediately and
         * its result is passed to {@link #maintenanceSchedule(MaintenanceSchedule)}.
         * 
         * @param maintenanceSchedule
         *        a consumer that will call methods on {@link MaintenanceSchedule.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #maintenanceSchedule(MaintenanceSchedule)
         */
        default Builder maintenanceSchedule(Consumer<MaintenanceSchedule.Builder> maintenanceSchedule) {
            return maintenanceSchedule(MaintenanceSchedule.builder().applyMutation(maintenanceSchedule).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String id;

        private String state;

        private String outputType;

        private RouterOutputConfiguration configuration;

        private String routedState;

        private String regionName;

        private String availabilityZone;

        private Long maximumBitrate;

        private String routingScope;

        private String tier;

        private Instant createdAt;

        private Instant updatedAt;

        private List<RouterOutputMessage> messages = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private RouterOutputStreamDetails streamDetails;

        private String ipAddress;

        private String routedInputArn;

        private String maintenanceType;

        private MaintenanceConfiguration maintenanceConfiguration;

        private String maintenanceScheduleType;

        private MaintenanceSchedule maintenanceSchedule;

        private BuilderImpl() {
        }

        private BuilderImpl(RouterOutput model) {
            name(model.name);
            arn(model.arn);
            id(model.id);
            state(model.state);
            outputType(model.outputType);
            configuration(model.configuration);
            routedState(model.routedState);
            regionName(model.regionName);
            availabilityZone(model.availabilityZone);
            maximumBitrate(model.maximumBitrate);
            routingScope(model.routingScope);
            tier(model.tier);
            createdAt(model.createdAt);
            updatedAt(model.updatedAt);
            messages(model.messages);
            tags(model.tags);
            streamDetails(model.streamDetails);
            ipAddress(model.ipAddress);
            routedInputArn(model.routedInputArn);
            maintenanceType(model.maintenanceType);
            maintenanceConfiguration(model.maintenanceConfiguration);
            maintenanceScheduleType(model.maintenanceScheduleType);
            maintenanceSchedule(model.maintenanceSchedule);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(RouterOutputState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getOutputType() {
            return outputType;
        }

        public final void setOutputType(String outputType) {
            this.outputType = outputType;
        }

        @Override
        public final Builder outputType(String outputType) {
            this.outputType = outputType;
            return this;
        }

        @Override
        public final Builder outputType(RouterOutputType outputType) {
            this.outputType(outputType == null ? null : outputType.toString());
            return this;
        }

        public final RouterOutputConfiguration.Builder getConfiguration() {
            return configuration != null ? configuration.toBuilder() : null;
        }

        public final void setConfiguration(RouterOutputConfiguration.BuilderImpl configuration) {
            this.configuration = configuration != null ? configuration.build() : null;
        }

        @Override
        public final Builder configuration(RouterOutputConfiguration configuration) {
            this.configuration = configuration;
            return this;
        }

        public final String getRoutedState() {
            return routedState;
        }

        public final void setRoutedState(String routedState) {
            this.routedState = routedState;
        }

        @Override
        public final Builder routedState(String routedState) {
            this.routedState = routedState;
            return this;
        }

        @Override
        public final Builder routedState(RouterOutputRoutedState routedState) {
            this.routedState(routedState == null ? null : routedState.toString());
            return this;
        }

        public final String getRegionName() {
            return regionName;
        }

        public final void setRegionName(String regionName) {
            this.regionName = regionName;
        }

        @Override
        public final Builder regionName(String regionName) {
            this.regionName = regionName;
            return this;
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final Long getMaximumBitrate() {
            return maximumBitrate;
        }

        public final void setMaximumBitrate(Long maximumBitrate) {
            this.maximumBitrate = maximumBitrate;
        }

        @Override
        public final Builder maximumBitrate(Long maximumBitrate) {
            this.maximumBitrate = maximumBitrate;
            return this;
        }

        public final String getRoutingScope() {
            return routingScope;
        }

        public final void setRoutingScope(String routingScope) {
            this.routingScope = routingScope;
        }

        @Override
        public final Builder routingScope(String routingScope) {
            this.routingScope = routingScope;
            return this;
        }

        @Override
        public final Builder routingScope(RoutingScope routingScope) {
            this.routingScope(routingScope == null ? null : routingScope.toString());
            return this;
        }

        public final String getTier() {
            return tier;
        }

        public final void setTier(String tier) {
            this.tier = tier;
        }

        @Override
        public final Builder tier(String tier) {
            this.tier = tier;
            return this;
        }

        @Override
        public final Builder tier(RouterOutputTier tier) {
            this.tier(tier == null ? null : tier.toString());
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final List<RouterOutputMessage.Builder> getMessages() {
            List<RouterOutputMessage.Builder> result = RouterOutputMessagesCopier.copyToBuilder(this.messages);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMessages(Collection<RouterOutputMessage.BuilderImpl> messages) {
            this.messages = RouterOutputMessagesCopier.copyFromBuilder(messages);
        }

        @Override
        public final Builder messages(Collection<RouterOutputMessage> messages) {
            this.messages = RouterOutputMessagesCopier.copy(messages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder messages(RouterOutputMessage... messages) {
            messages(Arrays.asList(messages));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder messages(Consumer<RouterOutputMessage.Builder>... messages) {
            messages(Stream.of(messages).map(c -> RouterOutputMessage.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = ___mapOfStringCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = ___mapOfStringCopier.copy(tags);
            return this;
        }

        public final RouterOutputStreamDetails.Builder getStreamDetails() {
            return streamDetails != null ? streamDetails.toBuilder() : null;
        }

        public final void setStreamDetails(RouterOutputStreamDetails.BuilderImpl streamDetails) {
            this.streamDetails = streamDetails != null ? streamDetails.build() : null;
        }

        @Override
        public final Builder streamDetails(RouterOutputStreamDetails streamDetails) {
            this.streamDetails = streamDetails;
            return this;
        }

        public final String getIpAddress() {
            return ipAddress;
        }

        public final void setIpAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        @Override
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final String getRoutedInputArn() {
            return routedInputArn;
        }

        public final void setRoutedInputArn(String routedInputArn) {
            this.routedInputArn = routedInputArn;
        }

        @Override
        public final Builder routedInputArn(String routedInputArn) {
            this.routedInputArn = routedInputArn;
            return this;
        }

        public final String getMaintenanceType() {
            return maintenanceType;
        }

        public final void setMaintenanceType(String maintenanceType) {
            this.maintenanceType = maintenanceType;
        }

        @Override
        public final Builder maintenanceType(String maintenanceType) {
            this.maintenanceType = maintenanceType;
            return this;
        }

        @Override
        public final Builder maintenanceType(MaintenanceType maintenanceType) {
            this.maintenanceType(maintenanceType == null ? null : maintenanceType.toString());
            return this;
        }

        public final MaintenanceConfiguration.Builder getMaintenanceConfiguration() {
            return maintenanceConfiguration != null ? maintenanceConfiguration.toBuilder() : null;
        }

        public final void setMaintenanceConfiguration(MaintenanceConfiguration.BuilderImpl maintenanceConfiguration) {
            this.maintenanceConfiguration = maintenanceConfiguration != null ? maintenanceConfiguration.build() : null;
        }

        @Override
        public final Builder maintenanceConfiguration(MaintenanceConfiguration maintenanceConfiguration) {
            this.maintenanceConfiguration = maintenanceConfiguration;
            return this;
        }

        public final String getMaintenanceScheduleType() {
            return maintenanceScheduleType;
        }

        public final void setMaintenanceScheduleType(String maintenanceScheduleType) {
            this.maintenanceScheduleType = maintenanceScheduleType;
        }

        @Override
        public final Builder maintenanceScheduleType(String maintenanceScheduleType) {
            this.maintenanceScheduleType = maintenanceScheduleType;
            return this;
        }

        @Override
        public final Builder maintenanceScheduleType(MaintenanceScheduleType maintenanceScheduleType) {
            this.maintenanceScheduleType(maintenanceScheduleType == null ? null : maintenanceScheduleType.toString());
            return this;
        }

        public final MaintenanceSchedule.Builder getMaintenanceSchedule() {
            return maintenanceSchedule != null ? maintenanceSchedule.toBuilder() : null;
        }

        public final void setMaintenanceSchedule(MaintenanceSchedule.BuilderImpl maintenanceSchedule) {
            this.maintenanceSchedule = maintenanceSchedule != null ? maintenanceSchedule.build() : null;
        }

        @Override
        public final Builder maintenanceSchedule(MaintenanceSchedule maintenanceSchedule) {
            this.maintenanceSchedule = maintenanceSchedule;
            return this;
        }

        @Override
        public RouterOutput build() {
            return new RouterOutput(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
