/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about the server-side encryption settings for an S3 bucket or S3 object.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ServerSideEncryption implements SdkPojo, Serializable,
        ToCopyableBuilder<ServerSideEncryption.Builder, ServerSideEncryption> {
    private static final SdkField<String> ENCRYPTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("encryptionType").getter(getter(ServerSideEncryption::encryptionTypeAsString))
            .setter(setter(Builder::encryptionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionType").build()).build();

    private static final SdkField<String> KMS_MASTER_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("kmsMasterKeyId").getter(getter(ServerSideEncryption::kmsMasterKeyId))
            .setter(setter(Builder::kmsMasterKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kmsMasterKeyId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENCRYPTION_TYPE_FIELD,
            KMS_MASTER_KEY_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String encryptionType;

    private final String kmsMasterKeyId;

    private ServerSideEncryption(BuilderImpl builder) {
        this.encryptionType = builder.encryptionType;
        this.kmsMasterKeyId = builder.kmsMasterKeyId;
    }

    /**
     * <p>
     * The server-side encryption algorithm that's used when storing data in the bucket or object. If encryption is
     * disabled for the bucket or object, this value is NONE.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionType}
     * will return {@link EncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionTypeAsString}.
     * </p>
     * 
     * @return The server-side encryption algorithm that's used when storing data in the bucket or object. If encryption
     *         is disabled for the bucket or object, this value is NONE.
     * @see EncryptionType
     */
    public final EncryptionType encryptionType() {
        return EncryptionType.fromValue(encryptionType);
    }

    /**
     * <p>
     * The server-side encryption algorithm that's used when storing data in the bucket or object. If encryption is
     * disabled for the bucket or object, this value is NONE.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionType}
     * will return {@link EncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionTypeAsString}.
     * </p>
     * 
     * @return The server-side encryption algorithm that's used when storing data in the bucket or object. If encryption
     *         is disabled for the bucket or object, this value is NONE.
     * @see EncryptionType
     */
    public final String encryptionTypeAsString() {
        return encryptionType;
    }

    /**
     * <p>
     * The unique identifier for the AWS Key Management Service (AWS KMS) master key that's used to encrypt the bucket
     * or object. This value is null if AWS KMS isn't used to encrypt the bucket or object.
     * </p>
     * 
     * @return The unique identifier for the AWS Key Management Service (AWS KMS) master key that's used to encrypt the
     *         bucket or object. This value is null if AWS KMS isn't used to encrypt the bucket or object.
     */
    public final String kmsMasterKeyId() {
        return kmsMasterKeyId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(encryptionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(kmsMasterKeyId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ServerSideEncryption)) {
            return false;
        }
        ServerSideEncryption other = (ServerSideEncryption) obj;
        return Objects.equals(encryptionTypeAsString(), other.encryptionTypeAsString())
                && Objects.equals(kmsMasterKeyId(), other.kmsMasterKeyId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ServerSideEncryption").add("EncryptionType", encryptionTypeAsString())
                .add("KmsMasterKeyId", kmsMasterKeyId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "encryptionType":
            return Optional.ofNullable(clazz.cast(encryptionTypeAsString()));
        case "kmsMasterKeyId":
            return Optional.ofNullable(clazz.cast(kmsMasterKeyId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ServerSideEncryption, T> g) {
        return obj -> g.apply((ServerSideEncryption) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ServerSideEncryption> {
        /**
         * <p>
         * The server-side encryption algorithm that's used when storing data in the bucket or object. If encryption is
         * disabled for the bucket or object, this value is NONE.
         * </p>
         * 
         * @param encryptionType
         *        The server-side encryption algorithm that's used when storing data in the bucket or object. If
         *        encryption is disabled for the bucket or object, this value is NONE.
         * @see EncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionType
         */
        Builder encryptionType(String encryptionType);

        /**
         * <p>
         * The server-side encryption algorithm that's used when storing data in the bucket or object. If encryption is
         * disabled for the bucket or object, this value is NONE.
         * </p>
         * 
         * @param encryptionType
         *        The server-side encryption algorithm that's used when storing data in the bucket or object. If
         *        encryption is disabled for the bucket or object, this value is NONE.
         * @see EncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionType
         */
        Builder encryptionType(EncryptionType encryptionType);

        /**
         * <p>
         * The unique identifier for the AWS Key Management Service (AWS KMS) master key that's used to encrypt the
         * bucket or object. This value is null if AWS KMS isn't used to encrypt the bucket or object.
         * </p>
         * 
         * @param kmsMasterKeyId
         *        The unique identifier for the AWS Key Management Service (AWS KMS) master key that's used to encrypt
         *        the bucket or object. This value is null if AWS KMS isn't used to encrypt the bucket or object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsMasterKeyId(String kmsMasterKeyId);
    }

    static final class BuilderImpl implements Builder {
        private String encryptionType;

        private String kmsMasterKeyId;

        private BuilderImpl() {
        }

        private BuilderImpl(ServerSideEncryption model) {
            encryptionType(model.encryptionType);
            kmsMasterKeyId(model.kmsMasterKeyId);
        }

        public final String getEncryptionType() {
            return encryptionType;
        }

        @Override
        public final Builder encryptionType(String encryptionType) {
            this.encryptionType = encryptionType;
            return this;
        }

        @Override
        public final Builder encryptionType(EncryptionType encryptionType) {
            this.encryptionType(encryptionType == null ? null : encryptionType.toString());
            return this;
        }

        public final void setEncryptionType(String encryptionType) {
            this.encryptionType = encryptionType;
        }

        public final String getKmsMasterKeyId() {
            return kmsMasterKeyId;
        }

        @Override
        public final Builder kmsMasterKeyId(String kmsMasterKeyId) {
            this.kmsMasterKeyId = kmsMasterKeyId;
            return this;
        }

        public final void setKmsMasterKeyId(String kmsMasterKeyId) {
            this.kmsMasterKeyId = kmsMasterKeyId;
        }

        @Override
        public ServerSideEncryption build() {
            return new ServerSideEncryption(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
