/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about addresses or points of interest that match the search criteria.
 * </p>
 * <p>
 * Not all details are included with all responses. Some details may only be returned by specific data partners.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Place implements SdkPojo, Serializable, ToCopyableBuilder<Place.Builder, Place> {
    private static final SdkField<String> ADDRESS_NUMBER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AddressNumber").getter(getter(Place::addressNumber)).setter(setter(Builder::addressNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AddressNumber").build()).build();

    private static final SdkField<List<String>> CATEGORIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Categories")
            .getter(getter(Place::categories))
            .setter(setter(Builder::categories))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Categories").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> COUNTRY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Country")
            .getter(getter(Place::country)).setter(setter(Builder::country))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Country").build()).build();

    private static final SdkField<PlaceGeometry> GEOMETRY_FIELD = SdkField.<PlaceGeometry> builder(MarshallingType.SDK_POJO)
            .memberName("Geometry").getter(getter(Place::geometry)).setter(setter(Builder::geometry))
            .constructor(PlaceGeometry::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Geometry").build()).build();

    private static final SdkField<Boolean> INTERPOLATED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Interpolated").getter(getter(Place::interpolated)).setter(setter(Builder::interpolated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Interpolated").build()).build();

    private static final SdkField<String> LABEL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Label")
            .getter(getter(Place::label)).setter(setter(Builder::label))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Label").build()).build();

    private static final SdkField<String> MUNICIPALITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Municipality").getter(getter(Place::municipality)).setter(setter(Builder::municipality))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Municipality").build()).build();

    private static final SdkField<String> NEIGHBORHOOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Neighborhood").getter(getter(Place::neighborhood)).setter(setter(Builder::neighborhood))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Neighborhood").build()).build();

    private static final SdkField<String> POSTAL_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PostalCode").getter(getter(Place::postalCode)).setter(setter(Builder::postalCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PostalCode").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Region")
            .getter(getter(Place::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Region").build()).build();

    private static final SdkField<String> STREET_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Street")
            .getter(getter(Place::street)).setter(setter(Builder::street))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Street").build()).build();

    private static final SdkField<String> SUB_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubRegion").getter(getter(Place::subRegion)).setter(setter(Builder::subRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubRegion").build()).build();

    private static final SdkField<List<String>> SUPPLEMENTAL_CATEGORIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SupplementalCategories")
            .getter(getter(Place::supplementalCategories))
            .setter(setter(Builder::supplementalCategories))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupplementalCategories").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<TimeZone> TIME_ZONE_FIELD = SdkField.<TimeZone> builder(MarshallingType.SDK_POJO)
            .memberName("TimeZone").getter(getter(Place::timeZone)).setter(setter(Builder::timeZone))
            .constructor(TimeZone::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeZone").build()).build();

    private static final SdkField<String> UNIT_NUMBER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UnitNumber").getter(getter(Place::unitNumber)).setter(setter(Builder::unitNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UnitNumber").build()).build();

    private static final SdkField<String> UNIT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UnitType").getter(getter(Place::unitType)).setter(setter(Builder::unitType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UnitType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ADDRESS_NUMBER_FIELD,
            CATEGORIES_FIELD, COUNTRY_FIELD, GEOMETRY_FIELD, INTERPOLATED_FIELD, LABEL_FIELD, MUNICIPALITY_FIELD,
            NEIGHBORHOOD_FIELD, POSTAL_CODE_FIELD, REGION_FIELD, STREET_FIELD, SUB_REGION_FIELD, SUPPLEMENTAL_CATEGORIES_FIELD,
            TIME_ZONE_FIELD, UNIT_NUMBER_FIELD, UNIT_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String addressNumber;

    private final List<String> categories;

    private final String country;

    private final PlaceGeometry geometry;

    private final Boolean interpolated;

    private final String label;

    private final String municipality;

    private final String neighborhood;

    private final String postalCode;

    private final String region;

    private final String street;

    private final String subRegion;

    private final List<String> supplementalCategories;

    private final TimeZone timeZone;

    private final String unitNumber;

    private final String unitType;

    private Place(BuilderImpl builder) {
        this.addressNumber = builder.addressNumber;
        this.categories = builder.categories;
        this.country = builder.country;
        this.geometry = builder.geometry;
        this.interpolated = builder.interpolated;
        this.label = builder.label;
        this.municipality = builder.municipality;
        this.neighborhood = builder.neighborhood;
        this.postalCode = builder.postalCode;
        this.region = builder.region;
        this.street = builder.street;
        this.subRegion = builder.subRegion;
        this.supplementalCategories = builder.supplementalCategories;
        this.timeZone = builder.timeZone;
        this.unitNumber = builder.unitNumber;
        this.unitType = builder.unitType;
    }

    /**
     * <p>
     * The numerical portion of an address, such as a building number.
     * </p>
     * 
     * @return The numerical portion of an address, such as a building number.
     */
    public final String addressNumber() {
        return addressNumber;
    }

    /**
     * For responses, this returns true if the service returned a value for the Categories property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCategories() {
        return categories != null && !(categories instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Location categories that describe this Place.
     * </p>
     * <p>
     * For more information about using categories, including a list of Amazon Location categories, see <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/category-filtering.html">Categories and
     * filtering</a>, in the <i>Amazon Location Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCategories} method.
     * </p>
     * 
     * @return The Amazon Location categories that describe this Place.</p>
     *         <p>
     *         For more information about using categories, including a list of Amazon Location categories, see <a
     *         href="https://docs.aws.amazon.com/location/latest/developerguide/category-filtering.html">Categories and
     *         filtering</a>, in the <i>Amazon Location Service Developer Guide</i>.
     */
    public final List<String> categories() {
        return categories;
    }

    /**
     * <p>
     * A country/region specified using <a href="https://www.iso.org/iso-3166-country-codes.html">ISO 3166</a> 3-digit
     * country/region code. For example, <code>CAN</code>.
     * </p>
     * 
     * @return A country/region specified using <a href="https://www.iso.org/iso-3166-country-codes.html">ISO 3166</a>
     *         3-digit country/region code. For example, <code>CAN</code>.
     */
    public final String country() {
        return country;
    }

    /**
     * Returns the value of the Geometry property for this object.
     * 
     * @return The value of the Geometry property for this object.
     */
    public final PlaceGeometry geometry() {
        return geometry;
    }

    /**
     * <p>
     * <code>True</code> if the result is interpolated from other known places.
     * </p>
     * <p>
     * <code>False</code> if the Place is a known place.
     * </p>
     * <p>
     * Not returned when the partner does not provide the information.
     * </p>
     * <p>
     * For example, returns <code>False</code> for an address location that is found in the partner data, but returns
     * <code>True</code> if an address does not exist in the partner data and its location is calculated by
     * interpolating between other known addresses.
     * </p>
     * 
     * @return <code>True</code> if the result is interpolated from other known places.</p>
     *         <p>
     *         <code>False</code> if the Place is a known place.
     *         </p>
     *         <p>
     *         Not returned when the partner does not provide the information.
     *         </p>
     *         <p>
     *         For example, returns <code>False</code> for an address location that is found in the partner data, but
     *         returns <code>True</code> if an address does not exist in the partner data and its location is calculated
     *         by interpolating between other known addresses.
     */
    public final Boolean interpolated() {
        return interpolated;
    }

    /**
     * <p>
     * The full name and address of the point of interest such as a city, region, or country. For example,
     * <code>123 Any Street, Any Town, USA</code>.
     * </p>
     * 
     * @return The full name and address of the point of interest such as a city, region, or country. For example,
     *         <code>123 Any Street, Any Town, USA</code>.
     */
    public final String label() {
        return label;
    }

    /**
     * <p>
     * A name for a local area, such as a city or town name. For example, <code>Toronto</code>.
     * </p>
     * 
     * @return A name for a local area, such as a city or town name. For example, <code>Toronto</code>.
     */
    public final String municipality() {
        return municipality;
    }

    /**
     * <p>
     * The name of a community district. For example, <code>Downtown</code>.
     * </p>
     * 
     * @return The name of a community district. For example, <code>Downtown</code>.
     */
    public final String neighborhood() {
        return neighborhood;
    }

    /**
     * <p>
     * A group of numbers and letters in a country-specific format, which accompanies the address for the purpose of
     * identifying a location.
     * </p>
     * 
     * @return A group of numbers and letters in a country-specific format, which accompanies the address for the
     *         purpose of identifying a location.
     */
    public final String postalCode() {
        return postalCode;
    }

    /**
     * <p>
     * A name for an area or geographical division, such as a province or state name. For example,
     * <code>British Columbia</code>.
     * </p>
     * 
     * @return A name for an area or geographical division, such as a province or state name. For example,
     *         <code>British Columbia</code>.
     */
    public final String region() {
        return region;
    }

    /**
     * <p>
     * The name for a street or a road to identify a location. For example, <code>Main Street</code>.
     * </p>
     * 
     * @return The name for a street or a road to identify a location. For example, <code>Main Street</code>.
     */
    public final String street() {
        return street;
    }

    /**
     * <p>
     * A county, or an area that's part of a larger region. For example, <code>Metro Vancouver</code>.
     * </p>
     * 
     * @return A county, or an area that's part of a larger region. For example, <code>Metro Vancouver</code>.
     */
    public final String subRegion() {
        return subRegion;
    }

    /**
     * For responses, this returns true if the service returned a value for the SupplementalCategories property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSupplementalCategories() {
        return supplementalCategories != null && !(supplementalCategories instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Categories from the data provider that describe the Place that are not mapped to any Amazon Location categories.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupplementalCategories} method.
     * </p>
     * 
     * @return Categories from the data provider that describe the Place that are not mapped to any Amazon Location
     *         categories.
     */
    public final List<String> supplementalCategories() {
        return supplementalCategories;
    }

    /**
     * <p>
     * The time zone in which the <code>Place</code> is located. Returned only when using HERE or Grab as the selected
     * partner.
     * </p>
     * 
     * @return The time zone in which the <code>Place</code> is located. Returned only when using HERE or Grab as the
     *         selected partner.
     */
    public final TimeZone timeZone() {
        return timeZone;
    }

    /**
     * <p>
     * For addresses with multiple units, the unit identifier. Can include numbers and letters, for example
     * <code>3B</code> or <code>Unit 123</code>.
     * </p>
     * <note>
     * <p>
     * Returned only for a place index that uses Esri or Grab as a data provider. Is not returned for
     * <code>SearchPlaceIndexForPosition</code>.
     * </p>
     * </note>
     * 
     * @return For addresses with multiple units, the unit identifier. Can include numbers and letters, for example
     *         <code>3B</code> or <code>Unit 123</code>.</p> <note>
     *         <p>
     *         Returned only for a place index that uses Esri or Grab as a data provider. Is not returned for
     *         <code>SearchPlaceIndexForPosition</code>.
     *         </p>
     */
    public final String unitNumber() {
        return unitNumber;
    }

    /**
     * <p>
     * For addresses with a <code>UnitNumber</code>, the type of unit. For example, <code>Apartment</code>.
     * </p>
     * <note>
     * <p>
     * Returned only for a place index that uses Esri as a data provider.
     * </p>
     * </note>
     * 
     * @return For addresses with a <code>UnitNumber</code>, the type of unit. For example, <code>Apartment</code>.</p>
     *         <note>
     *         <p>
     *         Returned only for a place index that uses Esri as a data provider.
     *         </p>
     */
    public final String unitType() {
        return unitType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(addressNumber());
        hashCode = 31 * hashCode + Objects.hashCode(hasCategories() ? categories() : null);
        hashCode = 31 * hashCode + Objects.hashCode(country());
        hashCode = 31 * hashCode + Objects.hashCode(geometry());
        hashCode = 31 * hashCode + Objects.hashCode(interpolated());
        hashCode = 31 * hashCode + Objects.hashCode(label());
        hashCode = 31 * hashCode + Objects.hashCode(municipality());
        hashCode = 31 * hashCode + Objects.hashCode(neighborhood());
        hashCode = 31 * hashCode + Objects.hashCode(postalCode());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(street());
        hashCode = 31 * hashCode + Objects.hashCode(subRegion());
        hashCode = 31 * hashCode + Objects.hashCode(hasSupplementalCategories() ? supplementalCategories() : null);
        hashCode = 31 * hashCode + Objects.hashCode(timeZone());
        hashCode = 31 * hashCode + Objects.hashCode(unitNumber());
        hashCode = 31 * hashCode + Objects.hashCode(unitType());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Place)) {
            return false;
        }
        Place other = (Place) obj;
        return Objects.equals(addressNumber(), other.addressNumber()) && hasCategories() == other.hasCategories()
                && Objects.equals(categories(), other.categories()) && Objects.equals(country(), other.country())
                && Objects.equals(geometry(), other.geometry()) && Objects.equals(interpolated(), other.interpolated())
                && Objects.equals(label(), other.label()) && Objects.equals(municipality(), other.municipality())
                && Objects.equals(neighborhood(), other.neighborhood()) && Objects.equals(postalCode(), other.postalCode())
                && Objects.equals(region(), other.region()) && Objects.equals(street(), other.street())
                && Objects.equals(subRegion(), other.subRegion())
                && hasSupplementalCategories() == other.hasSupplementalCategories()
                && Objects.equals(supplementalCategories(), other.supplementalCategories())
                && Objects.equals(timeZone(), other.timeZone()) && Objects.equals(unitNumber(), other.unitNumber())
                && Objects.equals(unitType(), other.unitType());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Place").add("AddressNumber", addressNumber())
                .add("Categories", hasCategories() ? categories() : null).add("Country", country()).add("Geometry", geometry())
                .add("Interpolated", interpolated()).add("Label", label()).add("Municipality", municipality())
                .add("Neighborhood", neighborhood()).add("PostalCode", postalCode()).add("Region", region())
                .add("Street", street()).add("SubRegion", subRegion())
                .add("SupplementalCategories", hasSupplementalCategories() ? supplementalCategories() : null)
                .add("TimeZone", timeZone()).add("UnitNumber", unitNumber()).add("UnitType", unitType()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AddressNumber":
            return Optional.ofNullable(clazz.cast(addressNumber()));
        case "Categories":
            return Optional.ofNullable(clazz.cast(categories()));
        case "Country":
            return Optional.ofNullable(clazz.cast(country()));
        case "Geometry":
            return Optional.ofNullable(clazz.cast(geometry()));
        case "Interpolated":
            return Optional.ofNullable(clazz.cast(interpolated()));
        case "Label":
            return Optional.ofNullable(clazz.cast(label()));
        case "Municipality":
            return Optional.ofNullable(clazz.cast(municipality()));
        case "Neighborhood":
            return Optional.ofNullable(clazz.cast(neighborhood()));
        case "PostalCode":
            return Optional.ofNullable(clazz.cast(postalCode()));
        case "Region":
            return Optional.ofNullable(clazz.cast(region()));
        case "Street":
            return Optional.ofNullable(clazz.cast(street()));
        case "SubRegion":
            return Optional.ofNullable(clazz.cast(subRegion()));
        case "SupplementalCategories":
            return Optional.ofNullable(clazz.cast(supplementalCategories()));
        case "TimeZone":
            return Optional.ofNullable(clazz.cast(timeZone()));
        case "UnitNumber":
            return Optional.ofNullable(clazz.cast(unitNumber()));
        case "UnitType":
            return Optional.ofNullable(clazz.cast(unitType()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Place, T> g) {
        return obj -> g.apply((Place) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Place> {
        /**
         * <p>
         * The numerical portion of an address, such as a building number.
         * </p>
         * 
         * @param addressNumber
         *        The numerical portion of an address, such as a building number.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addressNumber(String addressNumber);

        /**
         * <p>
         * The Amazon Location categories that describe this Place.
         * </p>
         * <p>
         * For more information about using categories, including a list of Amazon Location categories, see <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/category-filtering.html">Categories and
         * filtering</a>, in the <i>Amazon Location Service Developer Guide</i>.
         * </p>
         * 
         * @param categories
         *        The Amazon Location categories that describe this Place.</p>
         *        <p>
         *        For more information about using categories, including a list of Amazon Location categories, see <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/category-filtering.html">Categories
         *        and filtering</a>, in the <i>Amazon Location Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder categories(Collection<String> categories);

        /**
         * <p>
         * The Amazon Location categories that describe this Place.
         * </p>
         * <p>
         * For more information about using categories, including a list of Amazon Location categories, see <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/category-filtering.html">Categories and
         * filtering</a>, in the <i>Amazon Location Service Developer Guide</i>.
         * </p>
         * 
         * @param categories
         *        The Amazon Location categories that describe this Place.</p>
         *        <p>
         *        For more information about using categories, including a list of Amazon Location categories, see <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/category-filtering.html">Categories
         *        and filtering</a>, in the <i>Amazon Location Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder categories(String... categories);

        /**
         * <p>
         * A country/region specified using <a href="https://www.iso.org/iso-3166-country-codes.html">ISO 3166</a>
         * 3-digit country/region code. For example, <code>CAN</code>.
         * </p>
         * 
         * @param country
         *        A country/region specified using <a href="https://www.iso.org/iso-3166-country-codes.html">ISO
         *        3166</a> 3-digit country/region code. For example, <code>CAN</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder country(String country);

        /**
         * Sets the value of the Geometry property for this object.
         *
         * @param geometry
         *        The new value for the Geometry property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder geometry(PlaceGeometry geometry);

        /**
         * Sets the value of the Geometry property for this object.
         *
         * This is a convenience method that creates an instance of the {@link PlaceGeometry.Builder} avoiding the need
         * to create one manually via {@link PlaceGeometry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PlaceGeometry.Builder#build()} is called immediately and its
         * result is passed to {@link #geometry(PlaceGeometry)}.
         * 
         * @param geometry
         *        a consumer that will call methods on {@link PlaceGeometry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #geometry(PlaceGeometry)
         */
        default Builder geometry(Consumer<PlaceGeometry.Builder> geometry) {
            return geometry(PlaceGeometry.builder().applyMutation(geometry).build());
        }

        /**
         * <p>
         * <code>True</code> if the result is interpolated from other known places.
         * </p>
         * <p>
         * <code>False</code> if the Place is a known place.
         * </p>
         * <p>
         * Not returned when the partner does not provide the information.
         * </p>
         * <p>
         * For example, returns <code>False</code> for an address location that is found in the partner data, but
         * returns <code>True</code> if an address does not exist in the partner data and its location is calculated by
         * interpolating between other known addresses.
         * </p>
         * 
         * @param interpolated
         *        <code>True</code> if the result is interpolated from other known places.</p>
         *        <p>
         *        <code>False</code> if the Place is a known place.
         *        </p>
         *        <p>
         *        Not returned when the partner does not provide the information.
         *        </p>
         *        <p>
         *        For example, returns <code>False</code> for an address location that is found in the partner data, but
         *        returns <code>True</code> if an address does not exist in the partner data and its location is
         *        calculated by interpolating between other known addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder interpolated(Boolean interpolated);

        /**
         * <p>
         * The full name and address of the point of interest such as a city, region, or country. For example,
         * <code>123 Any Street, Any Town, USA</code>.
         * </p>
         * 
         * @param label
         *        The full name and address of the point of interest such as a city, region, or country. For example,
         *        <code>123 Any Street, Any Town, USA</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder label(String label);

        /**
         * <p>
         * A name for a local area, such as a city or town name. For example, <code>Toronto</code>.
         * </p>
         * 
         * @param municipality
         *        A name for a local area, such as a city or town name. For example, <code>Toronto</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder municipality(String municipality);

        /**
         * <p>
         * The name of a community district. For example, <code>Downtown</code>.
         * </p>
         * 
         * @param neighborhood
         *        The name of a community district. For example, <code>Downtown</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder neighborhood(String neighborhood);

        /**
         * <p>
         * A group of numbers and letters in a country-specific format, which accompanies the address for the purpose of
         * identifying a location.
         * </p>
         * 
         * @param postalCode
         *        A group of numbers and letters in a country-specific format, which accompanies the address for the
         *        purpose of identifying a location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder postalCode(String postalCode);

        /**
         * <p>
         * A name for an area or geographical division, such as a province or state name. For example,
         * <code>British Columbia</code>.
         * </p>
         * 
         * @param region
         *        A name for an area or geographical division, such as a province or state name. For example,
         *        <code>British Columbia</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The name for a street or a road to identify a location. For example, <code>Main Street</code>.
         * </p>
         * 
         * @param street
         *        The name for a street or a road to identify a location. For example, <code>Main Street</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder street(String street);

        /**
         * <p>
         * A county, or an area that's part of a larger region. For example, <code>Metro Vancouver</code>.
         * </p>
         * 
         * @param subRegion
         *        A county, or an area that's part of a larger region. For example, <code>Metro Vancouver</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subRegion(String subRegion);

        /**
         * <p>
         * Categories from the data provider that describe the Place that are not mapped to any Amazon Location
         * categories.
         * </p>
         * 
         * @param supplementalCategories
         *        Categories from the data provider that describe the Place that are not mapped to any Amazon Location
         *        categories.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supplementalCategories(Collection<String> supplementalCategories);

        /**
         * <p>
         * Categories from the data provider that describe the Place that are not mapped to any Amazon Location
         * categories.
         * </p>
         * 
         * @param supplementalCategories
         *        Categories from the data provider that describe the Place that are not mapped to any Amazon Location
         *        categories.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supplementalCategories(String... supplementalCategories);

        /**
         * <p>
         * The time zone in which the <code>Place</code> is located. Returned only when using HERE or Grab as the
         * selected partner.
         * </p>
         * 
         * @param timeZone
         *        The time zone in which the <code>Place</code> is located. Returned only when using HERE or Grab as the
         *        selected partner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeZone(TimeZone timeZone);

        /**
         * <p>
         * The time zone in which the <code>Place</code> is located. Returned only when using HERE or Grab as the
         * selected partner.
         * </p>
         * This is a convenience method that creates an instance of the {@link TimeZone.Builder} avoiding the need to
         * create one manually via {@link TimeZone#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TimeZone.Builder#build()} is called immediately and its result is
         * passed to {@link #timeZone(TimeZone)}.
         * 
         * @param timeZone
         *        a consumer that will call methods on {@link TimeZone.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #timeZone(TimeZone)
         */
        default Builder timeZone(Consumer<TimeZone.Builder> timeZone) {
            return timeZone(TimeZone.builder().applyMutation(timeZone).build());
        }

        /**
         * <p>
         * For addresses with multiple units, the unit identifier. Can include numbers and letters, for example
         * <code>3B</code> or <code>Unit 123</code>.
         * </p>
         * <note>
         * <p>
         * Returned only for a place index that uses Esri or Grab as a data provider. Is not returned for
         * <code>SearchPlaceIndexForPosition</code>.
         * </p>
         * </note>
         * 
         * @param unitNumber
         *        For addresses with multiple units, the unit identifier. Can include numbers and letters, for example
         *        <code>3B</code> or <code>Unit 123</code>.</p> <note>
         *        <p>
         *        Returned only for a place index that uses Esri or Grab as a data provider. Is not returned for
         *        <code>SearchPlaceIndexForPosition</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unitNumber(String unitNumber);

        /**
         * <p>
         * For addresses with a <code>UnitNumber</code>, the type of unit. For example, <code>Apartment</code>.
         * </p>
         * <note>
         * <p>
         * Returned only for a place index that uses Esri as a data provider.
         * </p>
         * </note>
         * 
         * @param unitType
         *        For addresses with a <code>UnitNumber</code>, the type of unit. For example, <code>Apartment</code>
         *        .</p> <note>
         *        <p>
         *        Returned only for a place index that uses Esri as a data provider.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unitType(String unitType);
    }

    static final class BuilderImpl implements Builder {
        private String addressNumber;

        private List<String> categories = DefaultSdkAutoConstructList.getInstance();

        private String country;

        private PlaceGeometry geometry;

        private Boolean interpolated;

        private String label;

        private String municipality;

        private String neighborhood;

        private String postalCode;

        private String region;

        private String street;

        private String subRegion;

        private List<String> supplementalCategories = DefaultSdkAutoConstructList.getInstance();

        private TimeZone timeZone;

        private String unitNumber;

        private String unitType;

        private BuilderImpl() {
        }

        private BuilderImpl(Place model) {
            addressNumber(model.addressNumber);
            categories(model.categories);
            country(model.country);
            geometry(model.geometry);
            interpolated(model.interpolated);
            label(model.label);
            municipality(model.municipality);
            neighborhood(model.neighborhood);
            postalCode(model.postalCode);
            region(model.region);
            street(model.street);
            subRegion(model.subRegion);
            supplementalCategories(model.supplementalCategories);
            timeZone(model.timeZone);
            unitNumber(model.unitNumber);
            unitType(model.unitType);
        }

        public final String getAddressNumber() {
            return addressNumber;
        }

        public final void setAddressNumber(String addressNumber) {
            this.addressNumber = addressNumber;
        }

        @Override
        public final Builder addressNumber(String addressNumber) {
            this.addressNumber = addressNumber;
            return this;
        }

        public final Collection<String> getCategories() {
            if (categories instanceof SdkAutoConstructList) {
                return null;
            }
            return categories;
        }

        public final void setCategories(Collection<String> categories) {
            this.categories = PlaceCategoryListCopier.copy(categories);
        }

        @Override
        public final Builder categories(Collection<String> categories) {
            this.categories = PlaceCategoryListCopier.copy(categories);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder categories(String... categories) {
            categories(Arrays.asList(categories));
            return this;
        }

        public final String getCountry() {
            return country;
        }

        public final void setCountry(String country) {
            this.country = country;
        }

        @Override
        public final Builder country(String country) {
            this.country = country;
            return this;
        }

        public final PlaceGeometry.Builder getGeometry() {
            return geometry != null ? geometry.toBuilder() : null;
        }

        public final void setGeometry(PlaceGeometry.BuilderImpl geometry) {
            this.geometry = geometry != null ? geometry.build() : null;
        }

        @Override
        public final Builder geometry(PlaceGeometry geometry) {
            this.geometry = geometry;
            return this;
        }

        public final Boolean getInterpolated() {
            return interpolated;
        }

        public final void setInterpolated(Boolean interpolated) {
            this.interpolated = interpolated;
        }

        @Override
        public final Builder interpolated(Boolean interpolated) {
            this.interpolated = interpolated;
            return this;
        }

        public final String getLabel() {
            return label;
        }

        public final void setLabel(String label) {
            this.label = label;
        }

        @Override
        public final Builder label(String label) {
            this.label = label;
            return this;
        }

        public final String getMunicipality() {
            return municipality;
        }

        public final void setMunicipality(String municipality) {
            this.municipality = municipality;
        }

        @Override
        public final Builder municipality(String municipality) {
            this.municipality = municipality;
            return this;
        }

        public final String getNeighborhood() {
            return neighborhood;
        }

        public final void setNeighborhood(String neighborhood) {
            this.neighborhood = neighborhood;
        }

        @Override
        public final Builder neighborhood(String neighborhood) {
            this.neighborhood = neighborhood;
            return this;
        }

        public final String getPostalCode() {
            return postalCode;
        }

        public final void setPostalCode(String postalCode) {
            this.postalCode = postalCode;
        }

        @Override
        public final Builder postalCode(String postalCode) {
            this.postalCode = postalCode;
            return this;
        }

        public final String getRegion() {
            return region;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final String getStreet() {
            return street;
        }

        public final void setStreet(String street) {
            this.street = street;
        }

        @Override
        public final Builder street(String street) {
            this.street = street;
            return this;
        }

        public final String getSubRegion() {
            return subRegion;
        }

        public final void setSubRegion(String subRegion) {
            this.subRegion = subRegion;
        }

        @Override
        public final Builder subRegion(String subRegion) {
            this.subRegion = subRegion;
            return this;
        }

        public final Collection<String> getSupplementalCategories() {
            if (supplementalCategories instanceof SdkAutoConstructList) {
                return null;
            }
            return supplementalCategories;
        }

        public final void setSupplementalCategories(Collection<String> supplementalCategories) {
            this.supplementalCategories = PlaceSupplementalCategoryListCopier.copy(supplementalCategories);
        }

        @Override
        public final Builder supplementalCategories(Collection<String> supplementalCategories) {
            this.supplementalCategories = PlaceSupplementalCategoryListCopier.copy(supplementalCategories);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supplementalCategories(String... supplementalCategories) {
            supplementalCategories(Arrays.asList(supplementalCategories));
            return this;
        }

        public final TimeZone.Builder getTimeZone() {
            return timeZone != null ? timeZone.toBuilder() : null;
        }

        public final void setTimeZone(TimeZone.BuilderImpl timeZone) {
            this.timeZone = timeZone != null ? timeZone.build() : null;
        }

        @Override
        public final Builder timeZone(TimeZone timeZone) {
            this.timeZone = timeZone;
            return this;
        }

        public final String getUnitNumber() {
            return unitNumber;
        }

        public final void setUnitNumber(String unitNumber) {
            this.unitNumber = unitNumber;
        }

        @Override
        public final Builder unitNumber(String unitNumber) {
            this.unitNumber = unitNumber;
            return this;
        }

        public final String getUnitType() {
            return unitType;
        }

        public final void setUnitType(String unitType) {
            this.unitType = unitType;
        }

        @Override
        public final Builder unitType(String unitType) {
            this.unitType = unitType;
            return this;
        }

        @Override
        public Place build() {
            return new Place(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
