/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains error details for each device that failed to update its position.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BatchUpdateDevicePositionError implements SdkPojo, Serializable,
        ToCopyableBuilder<BatchUpdateDevicePositionError.Builder, BatchUpdateDevicePositionError> {
    private static final SdkField<String> DEVICE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeviceId").getter(getter(BatchUpdateDevicePositionError::deviceId)).setter(setter(Builder::deviceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeviceId").build()).build();

    private static final SdkField<BatchItemError> ERROR_FIELD = SdkField.<BatchItemError> builder(MarshallingType.SDK_POJO)
            .memberName("Error").getter(getter(BatchUpdateDevicePositionError::error)).setter(setter(Builder::error))
            .constructor(BatchItemError::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Error").build()).build();

    private static final SdkField<Instant> SAMPLE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("SampleTime")
            .getter(getter(BatchUpdateDevicePositionError::sampleTime))
            .setter(setter(Builder::sampleTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SampleTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DEVICE_ID_FIELD, ERROR_FIELD,
            SAMPLE_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String deviceId;

    private final BatchItemError error;

    private final Instant sampleTime;

    private BatchUpdateDevicePositionError(BuilderImpl builder) {
        this.deviceId = builder.deviceId;
        this.error = builder.error;
        this.sampleTime = builder.sampleTime;
    }

    /**
     * <p>
     * The device associated with the failed location update.
     * </p>
     * 
     * @return The device associated with the failed location update.
     */
    public final String deviceId() {
        return deviceId;
    }

    /**
     * <p>
     * Contains details related to the error code such as the error code and error message.
     * </p>
     * 
     * @return Contains details related to the error code such as the error code and error message.
     */
    public final BatchItemError error() {
        return error;
    }

    /**
     * <p>
     * The timestamp for when a position sample was attempted in <a
     * href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
     * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
     * </p>
     * 
     * @return The timestamp for when a position sample was attempted in <a
     *         href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
     *         <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
     */
    public final Instant sampleTime() {
        return sampleTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deviceId());
        hashCode = 31 * hashCode + Objects.hashCode(error());
        hashCode = 31 * hashCode + Objects.hashCode(sampleTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BatchUpdateDevicePositionError)) {
            return false;
        }
        BatchUpdateDevicePositionError other = (BatchUpdateDevicePositionError) obj;
        return Objects.equals(deviceId(), other.deviceId()) && Objects.equals(error(), other.error())
                && Objects.equals(sampleTime(), other.sampleTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BatchUpdateDevicePositionError").add("DeviceId", deviceId()).add("Error", error())
                .add("SampleTime", sampleTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DeviceId":
            return Optional.ofNullable(clazz.cast(deviceId()));
        case "Error":
            return Optional.ofNullable(clazz.cast(error()));
        case "SampleTime":
            return Optional.ofNullable(clazz.cast(sampleTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BatchUpdateDevicePositionError, T> g) {
        return obj -> g.apply((BatchUpdateDevicePositionError) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BatchUpdateDevicePositionError> {
        /**
         * <p>
         * The device associated with the failed location update.
         * </p>
         * 
         * @param deviceId
         *        The device associated with the failed location update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deviceId(String deviceId);

        /**
         * <p>
         * Contains details related to the error code such as the error code and error message.
         * </p>
         * 
         * @param error
         *        Contains details related to the error code such as the error code and error message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder error(BatchItemError error);

        /**
         * <p>
         * Contains details related to the error code such as the error code and error message.
         * </p>
         * This is a convenience that creates an instance of the {@link BatchItemError.Builder} avoiding the need to
         * create one manually via {@link BatchItemError#builder()}.
         *
         * When the {@link Consumer} completes, {@link BatchItemError.Builder#build()} is called immediately and its
         * result is passed to {@link #error(BatchItemError)}.
         * 
         * @param error
         *        a consumer that will call methods on {@link BatchItemError.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #error(BatchItemError)
         */
        default Builder error(Consumer<BatchItemError.Builder> error) {
            return error(BatchItemError.builder().applyMutation(error).build());
        }

        /**
         * <p>
         * The timestamp for when a position sample was attempted in <a
         * href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
         * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
         * </p>
         * 
         * @param sampleTime
         *        The timestamp for when a position sample was attempted in <a
         *        href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
         *        <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sampleTime(Instant sampleTime);
    }

    static final class BuilderImpl implements Builder {
        private String deviceId;

        private BatchItemError error;

        private Instant sampleTime;

        private BuilderImpl() {
        }

        private BuilderImpl(BatchUpdateDevicePositionError model) {
            deviceId(model.deviceId);
            error(model.error);
            sampleTime(model.sampleTime);
        }

        public final String getDeviceId() {
            return deviceId;
        }

        @Override
        public final Builder deviceId(String deviceId) {
            this.deviceId = deviceId;
            return this;
        }

        public final void setDeviceId(String deviceId) {
            this.deviceId = deviceId;
        }

        public final BatchItemError.Builder getError() {
            return error != null ? error.toBuilder() : null;
        }

        @Override
        public final Builder error(BatchItemError error) {
            this.error = error;
            return this;
        }

        public final void setError(BatchItemError.BuilderImpl error) {
            this.error = error != null ? error.build() : null;
        }

        public final Instant getSampleTime() {
            return sampleTime;
        }

        @Override
        public final Builder sampleTime(Instant sampleTime) {
            this.sampleTime = sampleTime;
            return this;
        }

        public final void setSampleTime(Instant sampleTime) {
            this.sampleTime = sampleTime;
        }

        @Override
        public BatchUpdateDevicePositionError build() {
            return new BatchUpdateDevicePositionError(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
