/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the anonymous access permissions for an Amazon Lightsail bucket and its objects.
 * </p>
 * <p>
 * For more information about bucket access permissions, see <a
 * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-understanding-bucket-permissions"
 * >Understanding bucket permissions in Amazon Lightsail</a> in the
 * </p>
 * <p>
 * <i>Amazon Lightsail Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AccessRules implements SdkPojo, Serializable, ToCopyableBuilder<AccessRules.Builder, AccessRules> {
    private static final SdkField<String> GET_OBJECT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("getObject").getter(getter(AccessRules::getObjectAsString)).setter(setter(Builder::getObject))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("getObject").build()).build();

    private static final SdkField<Boolean> ALLOW_PUBLIC_OVERRIDES_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("allowPublicOverrides").getter(getter(AccessRules::allowPublicOverrides))
            .setter(setter(Builder::allowPublicOverrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allowPublicOverrides").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GET_OBJECT_FIELD,
            ALLOW_PUBLIC_OVERRIDES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String getObject;

    private final Boolean allowPublicOverrides;

    private AccessRules(BuilderImpl builder) {
        this.getObject = builder.getObject;
        this.allowPublicOverrides = builder.allowPublicOverrides;
    }

    /**
     * <p>
     * Specifies the anonymous access to all objects in a bucket.
     * </p>
     * <p>
     * The following options can be specified:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>public</code> - Sets all objects in the bucket to public (read-only), making them readable by anyone in the
     * world.
     * </p>
     * <p>
     * If the <code>getObject</code> value is set to <code>public</code>, then all objects in the bucket default to
     * public regardless of the <code>allowPublicOverrides</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>private</code> - Sets all objects in the bucket to private, making them readable only by you or anyone you
     * give access to.
     * </p>
     * <p>
     * If the <code>getObject</code> value is set to <code>private</code>, and the <code>allowPublicOverrides</code>
     * value is set to <code>true</code>, then all objects in the bucket default to private unless they are configured
     * with a <code>public-read</code> ACL. Individual objects with a <code>public-read</code> ACL are readable by
     * anyone in the world.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #getObject} will
     * return {@link AccessType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #getObjectAsString}.
     * </p>
     * 
     * @return Specifies the anonymous access to all objects in a bucket.</p>
     *         <p>
     *         The following options can be specified:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>public</code> - Sets all objects in the bucket to public (read-only), making them readable by
     *         anyone in the world.
     *         </p>
     *         <p>
     *         If the <code>getObject</code> value is set to <code>public</code>, then all objects in the bucket default
     *         to public regardless of the <code>allowPublicOverrides</code> value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>private</code> - Sets all objects in the bucket to private, making them readable only by you or
     *         anyone you give access to.
     *         </p>
     *         <p>
     *         If the <code>getObject</code> value is set to <code>private</code>, and the
     *         <code>allowPublicOverrides</code> value is set to <code>true</code>, then all objects in the bucket
     *         default to private unless they are configured with a <code>public-read</code> ACL. Individual objects
     *         with a <code>public-read</code> ACL are readable by anyone in the world.
     *         </p>
     *         </li>
     * @see AccessType
     */
    public final AccessType getObject() {
        return AccessType.fromValue(getObject);
    }

    /**
     * <p>
     * Specifies the anonymous access to all objects in a bucket.
     * </p>
     * <p>
     * The following options can be specified:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>public</code> - Sets all objects in the bucket to public (read-only), making them readable by anyone in the
     * world.
     * </p>
     * <p>
     * If the <code>getObject</code> value is set to <code>public</code>, then all objects in the bucket default to
     * public regardless of the <code>allowPublicOverrides</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>private</code> - Sets all objects in the bucket to private, making them readable only by you or anyone you
     * give access to.
     * </p>
     * <p>
     * If the <code>getObject</code> value is set to <code>private</code>, and the <code>allowPublicOverrides</code>
     * value is set to <code>true</code>, then all objects in the bucket default to private unless they are configured
     * with a <code>public-read</code> ACL. Individual objects with a <code>public-read</code> ACL are readable by
     * anyone in the world.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #getObject} will
     * return {@link AccessType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #getObjectAsString}.
     * </p>
     * 
     * @return Specifies the anonymous access to all objects in a bucket.</p>
     *         <p>
     *         The following options can be specified:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>public</code> - Sets all objects in the bucket to public (read-only), making them readable by
     *         anyone in the world.
     *         </p>
     *         <p>
     *         If the <code>getObject</code> value is set to <code>public</code>, then all objects in the bucket default
     *         to public regardless of the <code>allowPublicOverrides</code> value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>private</code> - Sets all objects in the bucket to private, making them readable only by you or
     *         anyone you give access to.
     *         </p>
     *         <p>
     *         If the <code>getObject</code> value is set to <code>private</code>, and the
     *         <code>allowPublicOverrides</code> value is set to <code>true</code>, then all objects in the bucket
     *         default to private unless they are configured with a <code>public-read</code> ACL. Individual objects
     *         with a <code>public-read</code> ACL are readable by anyone in the world.
     *         </p>
     *         </li>
     * @see AccessType
     */
    public final String getObjectAsString() {
        return getObject;
    }

    /**
     * <p>
     * A Boolean value that indicates whether the access control list (ACL) permissions that are applied to individual
     * objects override the <code>getObject</code> option that is currently specified.
     * </p>
     * <p>
     * When this is true, you can use the <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html">PutObjectAcl</a> Amazon S3 API
     * action to set individual objects to public (read-only) using the <code>public-read</code> ACL, or to private
     * using the <code>private</code> ACL.
     * </p>
     * 
     * @return A Boolean value that indicates whether the access control list (ACL) permissions that are applied to
     *         individual objects override the <code>getObject</code> option that is currently specified.</p>
     *         <p>
     *         When this is true, you can use the <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html">PutObjectAcl</a> Amazon S3
     *         API action to set individual objects to public (read-only) using the <code>public-read</code> ACL, or to
     *         private using the <code>private</code> ACL.
     */
    public final Boolean allowPublicOverrides() {
        return allowPublicOverrides;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(getObjectAsString());
        hashCode = 31 * hashCode + Objects.hashCode(allowPublicOverrides());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AccessRules)) {
            return false;
        }
        AccessRules other = (AccessRules) obj;
        return Objects.equals(getObjectAsString(), other.getObjectAsString())
                && Objects.equals(allowPublicOverrides(), other.allowPublicOverrides());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AccessRules").add("GetObject", getObjectAsString())
                .add("AllowPublicOverrides", allowPublicOverrides()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "getObject":
            return Optional.ofNullable(clazz.cast(getObjectAsString()));
        case "allowPublicOverrides":
            return Optional.ofNullable(clazz.cast(allowPublicOverrides()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AccessRules, T> g) {
        return obj -> g.apply((AccessRules) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AccessRules> {
        /**
         * <p>
         * Specifies the anonymous access to all objects in a bucket.
         * </p>
         * <p>
         * The following options can be specified:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>public</code> - Sets all objects in the bucket to public (read-only), making them readable by anyone in
         * the world.
         * </p>
         * <p>
         * If the <code>getObject</code> value is set to <code>public</code>, then all objects in the bucket default to
         * public regardless of the <code>allowPublicOverrides</code> value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>private</code> - Sets all objects in the bucket to private, making them readable only by you or anyone
         * you give access to.
         * </p>
         * <p>
         * If the <code>getObject</code> value is set to <code>private</code>, and the <code>allowPublicOverrides</code>
         * value is set to <code>true</code>, then all objects in the bucket default to private unless they are
         * configured with a <code>public-read</code> ACL. Individual objects with a <code>public-read</code> ACL are
         * readable by anyone in the world.
         * </p>
         * </li>
         * </ul>
         * 
         * @param getObject
         *        Specifies the anonymous access to all objects in a bucket.</p>
         *        <p>
         *        The following options can be specified:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>public</code> - Sets all objects in the bucket to public (read-only), making them readable by
         *        anyone in the world.
         *        </p>
         *        <p>
         *        If the <code>getObject</code> value is set to <code>public</code>, then all objects in the bucket
         *        default to public regardless of the <code>allowPublicOverrides</code> value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>private</code> - Sets all objects in the bucket to private, making them readable only by you or
         *        anyone you give access to.
         *        </p>
         *        <p>
         *        If the <code>getObject</code> value is set to <code>private</code>, and the
         *        <code>allowPublicOverrides</code> value is set to <code>true</code>, then all objects in the bucket
         *        default to private unless they are configured with a <code>public-read</code> ACL. Individual objects
         *        with a <code>public-read</code> ACL are readable by anyone in the world.
         *        </p>
         *        </li>
         * @see AccessType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccessType
         */
        Builder getObject(String getObject);

        /**
         * <p>
         * Specifies the anonymous access to all objects in a bucket.
         * </p>
         * <p>
         * The following options can be specified:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>public</code> - Sets all objects in the bucket to public (read-only), making them readable by anyone in
         * the world.
         * </p>
         * <p>
         * If the <code>getObject</code> value is set to <code>public</code>, then all objects in the bucket default to
         * public regardless of the <code>allowPublicOverrides</code> value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>private</code> - Sets all objects in the bucket to private, making them readable only by you or anyone
         * you give access to.
         * </p>
         * <p>
         * If the <code>getObject</code> value is set to <code>private</code>, and the <code>allowPublicOverrides</code>
         * value is set to <code>true</code>, then all objects in the bucket default to private unless they are
         * configured with a <code>public-read</code> ACL. Individual objects with a <code>public-read</code> ACL are
         * readable by anyone in the world.
         * </p>
         * </li>
         * </ul>
         * 
         * @param getObject
         *        Specifies the anonymous access to all objects in a bucket.</p>
         *        <p>
         *        The following options can be specified:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>public</code> - Sets all objects in the bucket to public (read-only), making them readable by
         *        anyone in the world.
         *        </p>
         *        <p>
         *        If the <code>getObject</code> value is set to <code>public</code>, then all objects in the bucket
         *        default to public regardless of the <code>allowPublicOverrides</code> value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>private</code> - Sets all objects in the bucket to private, making them readable only by you or
         *        anyone you give access to.
         *        </p>
         *        <p>
         *        If the <code>getObject</code> value is set to <code>private</code>, and the
         *        <code>allowPublicOverrides</code> value is set to <code>true</code>, then all objects in the bucket
         *        default to private unless they are configured with a <code>public-read</code> ACL. Individual objects
         *        with a <code>public-read</code> ACL are readable by anyone in the world.
         *        </p>
         *        </li>
         * @see AccessType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccessType
         */
        Builder getObject(AccessType getObject);

        /**
         * <p>
         * A Boolean value that indicates whether the access control list (ACL) permissions that are applied to
         * individual objects override the <code>getObject</code> option that is currently specified.
         * </p>
         * <p>
         * When this is true, you can use the <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html">PutObjectAcl</a> Amazon S3 API
         * action to set individual objects to public (read-only) using the <code>public-read</code> ACL, or to private
         * using the <code>private</code> ACL.
         * </p>
         * 
         * @param allowPublicOverrides
         *        A Boolean value that indicates whether the access control list (ACL) permissions that are applied to
         *        individual objects override the <code>getObject</code> option that is currently specified.</p>
         *        <p>
         *        When this is true, you can use the <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html">PutObjectAcl</a> Amazon
         *        S3 API action to set individual objects to public (read-only) using the <code>public-read</code> ACL,
         *        or to private using the <code>private</code> ACL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowPublicOverrides(Boolean allowPublicOverrides);
    }

    static final class BuilderImpl implements Builder {
        private String getObject;

        private Boolean allowPublicOverrides;

        private BuilderImpl() {
        }

        private BuilderImpl(AccessRules model) {
            getObject(model.getObject);
            allowPublicOverrides(model.allowPublicOverrides);
        }

        public final String getGetObject() {
            return getObject;
        }

        public final void setGetObject(String getObject) {
            this.getObject = getObject;
        }

        @Override
        public final Builder getObject(String getObject) {
            this.getObject = getObject;
            return this;
        }

        @Override
        public final Builder getObject(AccessType getObject) {
            this.getObject(getObject == null ? null : getObject.toString());
            return this;
        }

        public final Boolean getAllowPublicOverrides() {
            return allowPublicOverrides;
        }

        public final void setAllowPublicOverrides(Boolean allowPublicOverrides) {
            this.allowPublicOverrides = allowPublicOverrides;
        }

        @Override
        public final Builder allowPublicOverrides(Boolean allowPublicOverrides) {
            this.allowPublicOverrides = allowPublicOverrides;
            return this;
        }

        @Override
        public AccessRules build() {
            return new AccessRules(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
