/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntimev2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The current intent that Amazon Lex is attempting to fulfill.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Intent implements SdkPojo, Serializable, ToCopyableBuilder<Intent.Builder, Intent> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Intent::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<Map<String, Slot>> SLOTS_FIELD = SdkField
            .<Map<String, Slot>> builder(MarshallingType.MAP)
            .memberName("slots")
            .getter(getter(Intent::slots))
            .setter(setter(Builder::slots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slots").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Slot> builder(MarshallingType.SDK_POJO)
                                            .constructor(Slot::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(Intent::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> CONFIRMATION_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("confirmationState").getter(getter(Intent::confirmationStateAsString))
            .setter(setter(Builder::confirmationState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("confirmationState").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, SLOTS_FIELD,
            STATE_FIELD, CONFIRMATION_STATE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Map<String, Slot> slots;

    private final String state;

    private final String confirmationState;

    private Intent(BuilderImpl builder) {
        this.name = builder.name;
        this.slots = builder.slots;
        this.state = builder.state;
        this.confirmationState = builder.confirmationState;
    }

    /**
     * <p>
     * The name of the intent.
     * </p>
     * 
     * @return The name of the intent.
     */
    public final String name() {
        return name;
    }

    /**
     * Returns true if the Slots property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasSlots() {
        return slots != null && !(slots instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of all of the slots for the intent. The name of the slot maps to the value of the slot. If a slot has not
     * been filled, the value is null.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSlots()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A map of all of the slots for the intent. The name of the slot maps to the value of the slot. If a slot
     *         has not been filled, the value is null.
     */
    public final Map<String, Slot> slots() {
        return slots;
    }

    /**
     * <p>
     * Contains fulfillment information for the intent.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IntentState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Contains fulfillment information for the intent.
     * @see IntentState
     */
    public final IntentState state() {
        return IntentState.fromValue(state);
    }

    /**
     * <p>
     * Contains fulfillment information for the intent.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IntentState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Contains fulfillment information for the intent.
     * @see IntentState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * Contains information about whether fulfillment of the intent has been confirmed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #confirmationState}
     * will return {@link ConfirmationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #confirmationStateAsString}.
     * </p>
     * 
     * @return Contains information about whether fulfillment of the intent has been confirmed.
     * @see ConfirmationState
     */
    public final ConfirmationState confirmationState() {
        return ConfirmationState.fromValue(confirmationState);
    }

    /**
     * <p>
     * Contains information about whether fulfillment of the intent has been confirmed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #confirmationState}
     * will return {@link ConfirmationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #confirmationStateAsString}.
     * </p>
     * 
     * @return Contains information about whether fulfillment of the intent has been confirmed.
     * @see ConfirmationState
     */
    public final String confirmationStateAsString() {
        return confirmationState;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasSlots() ? slots() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(confirmationStateAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Intent)) {
            return false;
        }
        Intent other = (Intent) obj;
        return Objects.equals(name(), other.name()) && hasSlots() == other.hasSlots() && Objects.equals(slots(), other.slots())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(confirmationStateAsString(), other.confirmationStateAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Intent").add("Name", name()).add("Slots", hasSlots() ? slots() : null)
                .add("State", stateAsString()).add("ConfirmationState", confirmationStateAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "slots":
            return Optional.ofNullable(clazz.cast(slots()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "confirmationState":
            return Optional.ofNullable(clazz.cast(confirmationStateAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Intent, T> g) {
        return obj -> g.apply((Intent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Intent> {
        /**
         * <p>
         * The name of the intent.
         * </p>
         * 
         * @param name
         *        The name of the intent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A map of all of the slots for the intent. The name of the slot maps to the value of the slot. If a slot has
         * not been filled, the value is null.
         * </p>
         * 
         * @param slots
         *        A map of all of the slots for the intent. The name of the slot maps to the value of the slot. If a
         *        slot has not been filled, the value is null.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slots(Map<String, Slot> slots);

        /**
         * <p>
         * Contains fulfillment information for the intent.
         * </p>
         * 
         * @param state
         *        Contains fulfillment information for the intent.
         * @see IntentState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IntentState
         */
        Builder state(String state);

        /**
         * <p>
         * Contains fulfillment information for the intent.
         * </p>
         * 
         * @param state
         *        Contains fulfillment information for the intent.
         * @see IntentState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IntentState
         */
        Builder state(IntentState state);

        /**
         * <p>
         * Contains information about whether fulfillment of the intent has been confirmed.
         * </p>
         * 
         * @param confirmationState
         *        Contains information about whether fulfillment of the intent has been confirmed.
         * @see ConfirmationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfirmationState
         */
        Builder confirmationState(String confirmationState);

        /**
         * <p>
         * Contains information about whether fulfillment of the intent has been confirmed.
         * </p>
         * 
         * @param confirmationState
         *        Contains information about whether fulfillment of the intent has been confirmed.
         * @see ConfirmationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfirmationState
         */
        Builder confirmationState(ConfirmationState confirmationState);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Map<String, Slot> slots = DefaultSdkAutoConstructMap.getInstance();

        private String state;

        private String confirmationState;

        private BuilderImpl() {
        }

        private BuilderImpl(Intent model) {
            name(model.name);
            slots(model.slots);
            state(model.state);
            confirmationState(model.confirmationState);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Map<String, Slot.Builder> getSlots() {
            if (slots instanceof SdkAutoConstructMap) {
                return null;
            }
            return slots != null ? CollectionUtils.mapValues(slots, Slot::toBuilder) : null;
        }

        @Override
        public final Builder slots(Map<String, Slot> slots) {
            this.slots = SlotsCopier.copy(slots);
            return this;
        }

        public final void setSlots(Map<String, Slot.BuilderImpl> slots) {
            this.slots = SlotsCopier.copyFromBuilder(slots);
        }

        public final String getState() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(IntentState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final String getConfirmationState() {
            return confirmationState;
        }

        @Override
        public final Builder confirmationState(String confirmationState) {
            this.confirmationState = confirmationState;
            return this;
        }

        @Override
        public final Builder confirmationState(ConfirmationState confirmationState) {
            this.confirmationState(confirmationState == null ? null : confirmationState.toString());
            return this;
        }

        public final void setConfirmationState(String confirmationState) {
            this.confirmationState = confirmationState;
        }

        @Override
        public Intent build() {
            return new Intent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
